/*
 *  cache-file.c
 *  mod_musicindex
 *
 *  $Id: cache-file.c 754 2007-04-29 00:57:53Z varenet $
 *
 *  Created by Thibaut VARENE on Wed Feb 23 2005.
 *  Copyright (c) 2003-2007 Thibaut VARENE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version 2.1,
 *  as published by the Free Software Foundation.
 *
 */

/**
 * @file
 * Flat file cache management subsystem.
 *
 * @author Thibaut Varene
 * @version $Revision: 754 $
 * @date 2003-2007
 *
 * @warning 'errno' seems not to be seen as volatile by the compiler, therefore we
 *	cannot do "if(mkdir(foo) && errno == EEXIST)" for instance.
 * @bug No flat file cache support for Solaris.
 *
 * @warning We do rely on success of chdir() calls.
 *
 * @todo O P T I M I Z E ! (maybe)
 */

#include "playlist.h"
#include "mod_musicindex.h"
#include "cache-file.h"

#ifdef HAVE_DIRENT_H
#include <dirent.h>	/* opendir & friends */
#endif
#include <errno.h>
#include <stdio.h>	/* fprintf / fscanf / fileno */
#include <sys/file.h>	/* flock */
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>	/* file handling */
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>	/* file handling */
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>	/* file handling */
#endif
#ifdef HAVE_STRING_H
#include <string.h>	/* strerror() */
#endif

#define CACHE_VERS	10
#define CACHE_NF	15

#define CA_OK		0
#define CA_FATAL	10
#define CA_MISSARG	20
#define CA_CREATE	30
#define CA_LOCKED	40

#define MAX_GENRE	200

/** test for "." and ".." dir entries. x is a char * */
#define ISDOT(x)	( (x)[0] == '.' && (!(x)[1] || ((x)[1] == '.' && !(x)[2])) )
/** if char *x is NULL, print a printable ascii char (white space would be trashed by fscanf alas) */
#define BLANKSTR(x)	( !(x) ? "#" : (x) )
/** Detection of our "home made" encoding of NULL strings */
#define ISBLANK(x)	( ((x)[0] == '#') && ((x)[1] == '\0') )

/**
 * Handles error for the flat file cache subsystem.
 *
 * This function handles various errors depending on errno's value.
 *
 * @param r Apache request_rec struct to handle log writings.
 * @param caller A string (eg. calling function name) used in messages sent.
 *
 * @todo Many things.
 */
static void error_handler(request_rec *r, const char *const caller)
{
	switch (errno) {
		case EPERM:
			/* The filesystem containing pathname does not support the creation of directories. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, MI_LOG_PREFIX "(%s) Can't create/delete directory.", caller);
			break;
#if 0
		case EISDIR:
			/* pathname refers to a directory.  (This is  the  non-POSIX  value returned by Linux since 2.1.132.) */
		case EINVAL:
			/* mode  requested  creation of something other than a normal file, device special file or FIFO */
		case EEXIST:
			/* pathname already exists (not necessarily as a directory). */
		case EFAULT:
			/* pathname points outside your accessible address space. */
#endif
		case EACCES:
			/* The parent directory does not allow write permission to the  process,  or  one  of  the
			directories in pathname did not allow search (execute) permission. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, MI_LOG_PREFIX "(%s) Permission denied.", caller);
			break;
		case EMFILE:
			/* Too many file descriptors in use by process. */
		case ENFILE:
			/* Too many files are currently open in the system. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, MI_LOG_PREFIX "(%s) Too many open files!", caller);
			break;
		case ENAMETOOLONG:
			/* pathname was too long. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, MI_LOG_PREFIX "(%s) Pathname was too long.", caller);
			break;
		case ENOENT:
			/* A directory component in pathname does not exist or is a dangling symbolic link. */
			break;
#if 0
		case ENOTDIR:
			/* A component used as a directory in pathname is not, in fact, a directory. */
		case ENOTEMPTY:
			/* pathname contains entries other than . and .. . */
#endif
		case ENOMEM:
			/* Insufficient kernel memory was available. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, MI_LOG_PREFIX "(%s) Out Of Memory!", caller);
			break;
		case EROFS:
			/* pathname refers to a file on a read-only filesystem. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, MI_LOG_PREFIX "(%s) Read-Only filesystem!", caller);
			break;
		case ELOOP:
			/* Too many symbolic links were encountered in resolving pathname. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, MI_LOG_PREFIX "(%s) Too many symbolic links.", caller);
			break;
		case EIO:
			/* An I/O error occured. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, MI_LOG_PREFIX "(%s) I/O error.", caller);
			break;
		case ENOSPC:
			/* The device containing pathname has no room for the new directory.
			The new directory cannot be created because the user's disk quota is exhausted. */
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, MI_LOG_PREFIX "(%s) No space left on device!", caller);
			break;
		default:
			ap_log_rerror(APLOG_MARK, APLOG_ERR, r, MI_LOG_PREFIX "(%s) - error_handler! %s", caller, strerror(errno));
			break;
	}
	return;
}

/**
 * Creates cache subdirectories.
 *
 * This subroutine takes care of creating all requested directories and
 * subdirectories if they don't already exist and if possible.
 *
 * @warning If dirpath begins with a "/" the function will return immediately.
 *
 * @param r Apache request_rec struct to handle log writings.
 * @param dirpath A string representing a path to create.
 *
 * @return 0 on succes, CA_FATAL otherwise.
 */
static short file_cache_make_dir(request_rec *r, const char *const dirpath)
{
	register unsigned short l = 0, m = 0;
	const char *tempdir = NULL;

	do {	/* We build the path subdirs by subdirs, in a "mkdir -p" fashion */
		tempdir = ap_pstrndup(r->pool, dirpath, (m + (l = strcspn(dirpath + m, "/"))));
		m += l;

		if (!l)
			break;

		/* skipping (potentially multiple) slashes */
		while (dirpath[m] == '/')
			m++;

		if (mkdir(tempdir, S_IRWXU)) {
			if (errno != EEXIST)
				goto error_out;
		}
	} while (1);

	return 0;

error_out:
	error_handler(r, __FUNCTION__);
	return CA_FATAL;
}

/**
 * Removes cache subdirectories.
 *
 * This subroutine takes care of removing any given directory and
 * its content (recursively) if any, and if possible.
 *
 * @param r Apache request_rec struct to handle log writings.
 * @param cachedir A DIR stream corresponding to the directory to remove.
 * @param curdir A string representing the absolute path of the corresponding
 *	parent directory on the "original" filesystem.
 */
static void file_cache_remove_dir(request_rec *r, DIR *cachedir, const char *const curdir)
{
	DIR		*subdir = NULL;
	struct dirent	*cachedirent = NULL;
	struct stat	origdirstat;
	const char 	*origdir = NULL;

	fchdir(dirfd(cachedir));			/* on se place dans le repertoire de cache. XXX Pas de test ici, tout est verifie avant (en principe). */
	while ((cachedirent = readdir(cachedir))) {	/* on parcourt le repertoire */
		if (ISDOT(cachedirent->d_name))		/* We'd rather avoid trying to remove the whole filesystem... */
			continue;

		if (unlink(cachedirent->d_name)) {	/* We try to remove any entry (actually we will only remove regular files) */
				if ((errno == EISDIR) || (errno == EPERM)) {
				/* On BSDs unlink() returns EPERM on non empty directories.
				 * This shouldn't lead to infloop because of subsequent tests.
				 * If it's a directory, we check that the "original" still exists.
				 * If not, we remove it recursively.
				 * Reminder: "errno == (EISDIR || EPERM)" doesn't work */
				origdir = ap_pstrcat(r->pool, curdir, "/", cachedirent->d_name, NULL);
				if (stat(origdir, &origdirstat)) {
					if (rmdir(cachedirent->d_name)) {	/* stat() sets errno. We have to split */
						if (errno == ENOTEMPTY) {			/* il est pas vide, bigre! */
							subdir = opendir(cachedirent->d_name);	/* on ouvre le vilain repertoire pour en supprimer le contenu */
							file_cache_remove_dir(r, subdir, origdir);	/* en rappelant recursivement la fonction sur son contenu. */
							closedir(subdir);			/* a noter que dans ce cas la il y a un test inutile, celui qui verifie si l'original existe tjrs. Mais bon. */
							fchdir(dirfd(cachedir));		/* on retourne au repertoire precedent */
							rmdir(cachedirent->d_name);		/* maintenant il est vide, et on peut pas avoir d'erreur vu les tests precedants */
						}
						else
							error_handler(r, __FUNCTION__);		/* Oops, on est tombe sur une merde */
					}
				}
			}
			else
				error_handler(r, __FUNCTION__);		/* Oops, on est tombe sur une merde, mais plus tot */
		}
	}

	return;
}

/**
 * Initializes flat file cache subsystem.
 *
 * Basically we do nothing more here than creating the root cache folder.
 *
 * @param r Apache request_rec struct to handle log writings.
 * @param conf configuration structure in which the path to the cache root can be found.
 *
 * @return CA_OK on succes, CA_FATAL otherwise.
 */
static short file_cache_init(request_rec *r, const mu_config *const conf)
{
	chdir("/");	/* let's pray this will never fail */
	if (file_cache_make_dir(r, (char *)(conf->cache_setup) + 1))	/* since we've chdir'd, we send the path without the leading '/' */
		goto error_out;

	return CA_OK;

error_out:
	error_handler(r, __FUNCTION__);
	return CA_FATAL;
}

/**
 * Checks if a directory already exists in the cache.
 *
 * This function takes advantage of the standard behaviour of the cache
 * backend in musicindex: this function is called when the caller wants to know
 * whether the backend handles a given directory. We always return NULL,
 * pretending we don't, as "handling" means being able to provide a full listing
 * straight out of cache, which we don't in this particular incarnation, but we
 * take the opportunity to do our own cooking with our cache data, creating the
 * directory in cache if need be, and checking for cache data sanity.
 * 
 *
 * @param r Apache request_rec struct.
 * @param pack Ignored
 * @param conf The config structure used to find out cache configuration.
 * @param names Structure containing the directory name
 * @param soptions Flags to use for created entries. Ignored.
 *
 * @return NULL.
 */
static void* cache_file_opendir(request_rec *r, mu_pack *const pack, const mu_config *const conf,
				const mu_ent_names * const names, unsigned long soptions)
{
	const char *const path = names->filename;
	DIR		*cachedir = NULL;
	struct stat	cachedirstat, dirstat;

	if (!path)
		return NULL;

	/* Making sure the cache has been initialized, initialize it otherwise.
	 * Bear in mind we're chdir'd from now on. */
	if (chdir((char *)(conf->cache_setup))) { 			/* on va dans le rep de cache */
		if (errno == ENOENT) {			/* il n'existe pas, il est temps d'initialiser le cache */
			if (file_cache_init(r, conf))
				return NULL;	/* l'init du cache a chie, c'est mauvais */
			chdir((char *)(conf->cache_setup));
		}
		else
			goto error_out;			/* un autre probleme, on degage */
	}

	/* Actually check for the directory in the cache, create it if needed.
	 * "+ 1" offset to suppress leading '/'. */
	if (!(cachedir = opendir(path + 1))) {			/* on essaye d'ouvrir le repertoire concerne dans le cache (on supprime le leading "/" */
		if (errno == ENOENT) {				/* il n'existe pas mais on peut le creer (ca correspond a ENOENT, a verifier) */
			if (file_cache_make_dir(r, path + 1))	/* on envoie le chemin prive du leading '/' */
				goto error_out;
		}
		else
			goto error_out;				/* un autre probleme, on degage */
	}
	else {	/* Checking for cache sanity. Has it expired for that folder ? If so, delete its content. */
		fstat(dirfd(cachedir), &cachedirstat);		/* recuperons les stats du repertoire cache. XXX On considere cet appel sans echec vu les tests qu'on a fait avant. */
		stat(path, &dirstat);				/* recuperons les stats du rep d'origine. XXX pas de test ici, a priori ya pas de raison qu'on puisse pas les recuperer */
		if (cachedirstat.st_mtime < dirstat.st_mtime)	/* si la date de modif du rep de cache est plus vieille que celle du rep original, alors qqc a ete ajoute ou retire ou ecrit */
			file_cache_remove_dir(r, cachedir, path);	/* alors on le vide proprement de son contenu */
		closedir(cachedir);				/* On en a fini avec le repertoire, on le referme */
	}

	return NULL;

error_out:
	error_handler(r, __FUNCTION__);
	return NULL;
}

/** struct type used to reduce memory usage */
typedef struct {
	char title[MAX_STRING];
	char album[MAX_STRING];
	char artist[MAX_STRING];
	char genre[MAX_GENRE];
} mi_data_buffer;

/**
 * Fills in the information fields about a music file from the cache.
 *
 * This function reads the tags from the cache file
 * and fills in the struct mu_ent fields accordingly.
 *
 * @warning @b MUST close in file handler on success.
 *
 * @param pool Apache pool
 * @param in Not used (except for closing).
 * @param conf MusicIndex configuration paramaters struct.
 * @param names struct names to get the current filename.
 * @param r Apache request_rec struct to handle log writings.
 *
 * @return When possible, struct mu_ent correctly set up, file stream closed,
 * 		Head pointer otherwise.
 */
static mu_ent *file_make_cache_entry(request_rec *r, apr_pool_t *pool, FILE *const in,
	const mu_config *const conf, mu_ent_names *const names)
{
	mu_ent		*p = NULL;
	short 		result = 0;
	unsigned short	track, posn, flags, cvers = 0;
	signed short	filetype;
	FILE		*cache_file = NULL;
	mi_data_buffer	*data_buffer = NULL;

	if (!conf->cache_setup)
		return p;

	/* Making sure the cache has been initialized, initialize it otherwise.
	 * Bear in mind we're chdir'd from now on. */
	if (chdir((char *)(conf->cache_setup))) {			/* on va dans le rep de cache */
		if (file_cache_init(r, conf))
			goto error_out;			/* l'init du cache a chie, c'est mauvais */
		chdir((char *)(conf->cache_setup));
	}

	/* Actually check for the file in the cache, open it if possible.
	 * "+ 1" offset to suppress leading '/'. */
	cache_file = fopen(names->filename + 1, "r");
	if (!cache_file) {			/* on essaye d'ouvrir le fichier concerne en ro */
		if (errno == ENOENT) {		/* n'existe pas encore mais on peut a priori le creer */
			return p;		/* Creation of the file is handled separately (playlist.c) */
		}
		else
			goto error_out;		/* game over */
	}

	/* We acquire a shared advisory lock on the file to be (almost) certain of its integrity.
	 * This will prevent reading from incomplete cache files. The lock in non blocking:
	 * if we can't get it, we won't wait to read the file, we'll delegate to the original handler. */
	if (flock(fileno(cache_file), LOCK_SH|LOCK_NB)) {
		fclose(cache_file);
		return p;
	}
	/* Dev note: mixing unix and std IO is ugly, but there's no flockfile() counterpart to the shared
	 * advisory lock, alas, see flockfile(3), flock(2) and lockf(3). Besides, we have to lock, since
	 * fread()/fwrite() are thread safe, but not fscanf() and fprintf() */

	p = NEW_ENT(r->pool);
	if (p) {
		data_buffer = (mi_data_buffer *)malloc(sizeof(mi_data_buffer)); /* This should save some memory */
		if (data_buffer) {
			result = fscanf(cache_file, "album: %[^\n]\nartist: %[^\n]\n"
				"title: %[^\n]\ndate: %hu\ntrack: %hu\nposn: %hu\n"
				"length: %hu\nbitrate: %lu\nfreq: %hu\nsize: %lu\n"
				"filetype: %hi\ngenre: %[^\n]\nmtime: %lu\nflags: %hx\n"
				"cvers: %hu\n",
				data_buffer->album, data_buffer->artist, data_buffer->title, &p->date, &track, &posn, &p->length,
				&p->bitrate, &p->freq, &p->size, &filetype, data_buffer->genre, &p->mtime, &flags, &cvers);
		
			/* Check whether the cache is somehow corrupted */
			if ((result != CACHE_NF) || (cvers != CACHE_VERS)) {	/* fscanf() returns the number of input items assigned */
				p = NULL;					/* hopefuly p allocs should be cleaned by apache */
			}
			else {
				p->title = ap_pstrdup(r->pool, data_buffer->title);
				if (!ISBLANK(data_buffer->album))
					p->album = ap_pstrdup(r->pool, data_buffer->album);
				if (!ISBLANK(data_buffer->artist))
					p->artist = ap_pstrdup(r->pool, data_buffer->artist);
				if (!ISBLANK(data_buffer->genre))
					p->genre = ap_pstrdup(r->pool, data_buffer->genre);

				/* We have to use that trick, fscanf won't work on char variables */
				p->filetype = filetype;
				p->flags = flags;
				p->track = track;
				p->posn = posn;
			}
		
			free(data_buffer);
		}
		else
			p = NULL;	/* something failed, return non-bogus data. p will be cleaned up by apache's GC */
	}

	/* Explicitely releasing the lock, and closing the file */
	flock(fileno(cache_file), LOCK_UN);
	fclose(cache_file);

	if (p)
		fclose(in);	/* this part of the cache subsystem is (uglily) seen as part of the playlist system,
				and has to behave as such. This is why we close the input file if we took advantage of it. */

	return p;

error_out:
	error_handler(r, __FUNCTION__);
	return p;
}

/**
 * Creates and writes cache file information.
 *
 * This function creates a new cache file (using mu_ent_names), and
 * fills it with the data contained in the mu_ent p structure.
 *
 * @param r Apache request_rec struct to handle log writings.
 * @param p A mu_ent struct containing the data to store.
 * @param conf The config structure used to find out cache configuration.
 * @param names A names structure in which the file name can be found.
 */
static void cache_file_write(request_rec *r, mu_ent *p, const mu_config *const conf, const mu_ent_names * const names)
{
	FILE	*cache_file = NULL;

	chdir((char *)conf->cache_setup);	/* Par securite on se re-chdir(). XXX Considere sans echec */
	
	cache_file = fopen(names->filename + 1, "w"); /* on ouvre le fichier en ecriture et en "truncate" (pour eviter les fichiers pourris) */

	/* let's check if something bad happened */
	if (!cache_file)
		goto error_out;

	/* We acquire an exclusive advisory lock on the file to avoid corruption by another process.
	 * This will also prevent reading from incomplete cache. see cache_read_file() comments. */
	if (flock(fileno(cache_file), LOCK_EX|LOCK_NB)) {
		fclose(cache_file);
		if (errno == EWOULDBLOCK)
			return;
		else
			goto error_out;
	}

	fprintf(cache_file, "album: %s\nartist: %s\ntitle: %s\ndate: %hu\n"
		"track: %hhu\nposn: %hhu\nlength: %hu\nbitrate: %lu\nfreq: %hu\n"
		"size: %lu\nfiletype: %hi\ngenre: %s\nmtime: %lu\nflags: %hhx\ncvers: %hu\n",
		BLANKSTR(p->album), BLANKSTR(p->artist), p->title, p->date,
		p->track, p->posn, p->length, p->bitrate, p->freq, p->size, p->filetype,
		BLANKSTR(p->genre), p->mtime, (p->flags & EF_FLAGSTOSAVE), CACHE_VERS);

	/* Explicitely releasing the lock, and closing the file */
	flock(fileno(cache_file), LOCK_UN);
	fclose(cache_file);

	return;

error_out:
	error_handler(r, __FUNCTION__);
}

static const cache_backend cache_backend_file = {
	cache_file_opendir,
	NULL,
	NULL,
	file_make_cache_entry,
	cache_file_write
};

int cache_file_setup(cmd_parms *cmd, const char *const setup_string, mu_config *const conf)
{
	static const char biniou[] = "file://";

	if (strncmp(biniou, setup_string, 7) == 0) {
		conf->cache = &cache_backend_file;
		conf->cache_setup = ap_pstrdup(cmd->pool, setup_string+6);
		return 0;
	}

	return 1;
}
