// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

#include "color.h"

namespace visual {

rgb 
rgb::unsaturate() const 
{
	const float saturation = 0.5; // cut the saturation by this factor
	float rr, gg, bb, h, s, v;
	float cmin, cmax, delta;
	int i;
	float f, p, q, t;

	// r,g,b values are from 0 to 1
	// h = [0,360], s = [0,1], v = [0,1]
	//		if s == 0, then arbitrarily set h = 0

	cmin = r;
	if (g < cmin) {
		cmin = g;
	}
	if (b < cmin) {
		cmin = b;
	}

	cmax = r;
	if (g > cmax) {
		cmax = g;
	}
	if (b > cmax) {
		cmax = b;
	}
	v = cmax;				// v

	delta = cmax - cmin;

	if( cmin == cmax ) { // completely unsaturated; color is some gray
		// if r = g = b = 0, s = 0, v in principle undefined but set to 0
		s = 0.0;
		h = 0.0;
	} 
	else {

		s = delta / cmax;		// s
		if( r == cmax )
			h = ( g - b ) / delta;		// between yellow & magenta
		else if( g == cmax )
			h = 2.0 + ( b - r ) / delta;	// between cyan & yellow
		else
			h = 4.0 + ( r - g ) / delta;	// between magenta & cyan

		if( h < 0.0 )
			h += 6.0;  // make it 0 <= h < 6
	}

	// unsaturate somewhat to make sure both eyes have something to see
	s = saturation*s;
	
	if( s == 0.0 ) {
		// achromatic (grey)
		rr = gg = bb = v;
	}
	else {
		i = static_cast<int>( h);  // h represents sector 0 to 5
		f = h - i;                 // fractional part of h
		p = v * ( 1.0 - s );
		q = v * ( 1.0 - s * f );
		t = v * ( 1.0 - s * ( 1.0 - f ) );

		switch( i ) {
			case 0:
				rr = v;
				gg = t;
				bb = p;
				break;
			case 1:
				rr = q;
				gg = v;
				bb = p;
				break;
			case 2:
				rr = p;
				gg = v;
				bb = t;
				break;
			case 3:
				rr = p;
				gg = q;
				bb = v;
				break;
			case 4:
				rr = t;
				gg = p;
				bb = v;
				break;
			default:		// case 5:
				rr = v;
				gg = p;
				bb = q;
				break;
		}
	}
	return rgb(rr, gg, bb);
}

float 
rgb::grayscale() const 
{
    const float GAMMA = 2.5;
	return std::pow( 0.299* std::pow( r, GAMMA) 
			+ 0.587* std::pow( g, GAMMA) 
			+ 0.114* std::pow( b, GAMMA), (1.0/GAMMA));
}

} // !namespace visual
