#include "libnumarray.h"
#include "tc.h"

static PyObject *pNDArrayModule;
static PyObject *pNDArrayMDict;
static PyObject *pNDArrayClass;

static PyObject *pNumArrayModule;
static PyObject *pNumArrayMDict;
static PyObject *pNumArrayClass;
static PyObject *pNumArrayNewFunc;
static PyObject *pNumArrayArrayFunc;

static PyObject *pNumericTypesModule;
static PyObject *pNumericTypesMDict;
static PyObject *pNumericTypeClass;
static PyObject *pNumericTypesTDict;

static PyObject *pUfuncModule;
static PyObject *pUfuncMDict;
static PyObject *pUfuncClass;

static PyObject *pCfuncClass;

static PyObject *pConverterModule;
static PyObject *pConverterMDict;
static PyObject *pConverterClass;

static PyObject *pOperatorModule;
static PyObject *pOperatorMDict;
static PyObject *pOperatorClass;

static PyObject *pNewMemoryFunc;
static PyObject *pHandleErrorFunc;

static PyObject *pNumType[nNumarrayType];

static PyTypeObject CfuncType;


enum {
	BOOL_SCALAR,
	INT_SCALAR,
	LONG_SCALAR,
	FLOAT_SCALAR,
	COMPLEX_SCALAR,
};

/* custom init function unuseable due to circular references */
static int libnumarray_init(void) { return 0; }   

static PyObject *
init_module(char *modulename, PyObject **pMDict)
{
	PyObject *pModule = PyImport_ImportModule(modulename);
	if (!pModule) return NULL;
	*pMDict = PyModule_GetDict(pModule);
	Py_INCREF(*pMDict);
	return pModule;
}

static PyObject *
init_object(char *objectname, PyObject *pMDict)
{
	PyObject *object = PyDict_GetItemString(pMDict, objectname);
	if (!object) return NULL;
	Py_INCREF(object);
	return object;
}

static int
init_module_class(char *modulename, PyObject **pModule, 
		  PyObject **pMDict, 
		  char *classname,  PyObject **pClass)
{
	if ((*pModule = init_module(modulename, pMDict)))
		*pClass = init_object(classname, *pMDict);
	else
		return -1;
	return 0;
}


extern void *libnumarray_API[];


static int 
deferred_libnumarray_init(void)
{
	static int initialized = 0;
	int i;

	if (initialized) return 0;

	import_libtc();

	if (init_module_class("numarray.generic", &pNDArrayModule, 
			      &pNDArrayMDict,
			      "NDArray", &pNDArrayClass) < 0)
		goto _fail;
	
	if (init_module_class("numarray", &pNumArrayModule, 
			      &pNumArrayMDict,
			      "NumArray", &pNumArrayClass) < 0)
		goto _fail;
	
	if (init_module_class("numarray.numerictypes", &pNumericTypesModule, 
			      &pNumericTypesMDict, 
			      "NumericType", &pNumericTypeClass) < 0)
		goto _fail;
	
	if (init_module_class("numarray._ufunc", &pUfuncModule, 
			      &pUfuncMDict,
			      "_ufunc", &pUfuncClass) < 0)
		goto _fail;

	pCfuncClass = (PyObject *) &CfuncType;
	Py_INCREF(pCfuncClass);
	
	if (init_module_class("numarray._operator", &pOperatorModule, 
			      &pOperatorMDict,
			      "_operator", &pOperatorClass) < 0)
		goto _fail;
	
	if (init_module_class("numarray._converter", &pConverterModule, 
			      &pConverterMDict,
			      "_converter", &pConverterClass) < 0)
		goto _fail;

	if (!(pNumArrayNewFunc = PyObject_GetAttrString(
		      pNumArrayClass, "__new__")))
		goto _fail;
	
	if (!(pNumArrayArrayFunc = init_object( "array", pNumArrayMDict)))
		goto _fail;

	if (!(pNumericTypesTDict = init_object( "typeDict", pNumericTypesMDict)))
		goto _fail;

	pNewMemoryFunc = NA_initModuleGlobal("numarray.memory","new_memory");
	if (!pNewMemoryFunc) goto _fail;

	pHandleErrorFunc = 
		NA_initModuleGlobal("numarray.ufunc", "handleError");
	if (!pHandleErrorFunc) goto _fail;
		
	
	/* Set up table of type objects */
	for(i=0; i<ELEM(pNumType); i++) {
		PyObject *typeobj = init_object(NA_typeNoToName(i),
						pNumericTypesTDict);
		if (!typeobj) return -1;
		if (typeobj) {
			Py_INCREF(typeobj);
			pNumType[i] = typeobj;
		} else {
			pNumType[i] = NULL;
		}
	}

	/* Set up _get/_set descriptor hooks for numerical types */
	for(i=0; i<nNumarrayType; i++) {
		PyArray_Descr *ptr;
		switch(i) {
		case tAny: case tObject: 
			break;
		default: 
			ptr = NA_DescrFromType( i );
			if (!ptr) {
				PyErr_Format(PyExc_RuntimeError, 
					     "error initializing array descriptors");
				goto _fail;
			}
			ptr->_get = NA_getPythonScalar;
			ptr->_set = NA_setFromPythonScalar;
			break;
		}
	}

	libnumarray_API[ 0 ]  = (void *) pNumArrayClass;

	/* _exit: */
	initialized = 1;
	return 0;

  _fail:
	initialized = 0;
	return -1;
}


/* Finalize this module. */
void 
fini_module_class(PyObject *module, PyObject *mdict, PyObject *class)
{
	Py_DECREF(module);
	Py_DECREF(mdict);
	Py_DECREF(class);
}

static void
NA_Done(void)
{
	int i;

	fini_module_class(pNDArrayModule, pNDArrayMDict, pNDArrayClass);

	fini_module_class(pNumArrayModule, pNumArrayMDict, pNumArrayClass);
	Py_DECREF(pNumArrayArrayFunc);

	fini_module_class(pOperatorModule, pOperatorMDict, pOperatorClass);

	fini_module_class(pConverterModule, pConverterMDict, pConverterClass);

	fini_module_class(pUfuncModule, pUfuncMDict, pUfuncClass);
	Py_DECREF(pCfuncClass);
	
	fini_module_class(pNumericTypesModule, pNumericTypesMDict, 
			  pNumericTypeClass);
	Py_DECREF(pNumericTypesTDict);

	for(i=0; i<ELEM(pNumType); i++) {
		Py_DECREF(pNumType[i]);
	}
}

#ifdef MS_WIN32
#pragma warning(once : 4244)
#endif

#define ELEM(x) (sizeof(x)/sizeof(x[0]))

typedef struct
{
	char *name;
	int typeno;
} NumarrayTypeNameMapping;

static PyArray_Descr descriptors[ ] = {
	{ tAny,       0,                 '*'}, 
	
	{ tBool,      sizeof(Bool),      '?'},

	{ tInt8,      sizeof(Int8),      '1'},
	{ tUInt8,     sizeof(UInt8),     'b'},

	{ tInt16,     sizeof(Int16),     's'},
	{ tUInt16,    sizeof(UInt16),    'w'},

	{ tInt32,     sizeof(Int32),     'i'},
	{ tUInt32,    sizeof(UInt32),    'u'},

	{ tInt64,     sizeof(Int64),     'N'},
	{ tUInt64,    sizeof(UInt64),    'U'},

	{ tFloat32,   sizeof(Float32),   'f'},
	{ tFloat64,   sizeof(Float64),   'd'},

	{ tComplex32, sizeof(Complex32), 'F'},
	{ tComplex64, sizeof(Complex64), 'D'}
};

static PyArray_Descr *
NA_DescrFromType(int type)
{
	if ((type >= tAny) && (type <= tComplex64)) {
		return &descriptors[ type ];
	} else {
		int i;
		for(i=0; i<ELEM(descriptors); i++)
			if (descriptors[i].type == type)
				return &descriptors[i];
	}
	PyErr_Format(
		PyExc_TypeError, 
		"NA_DescrFromType: unknown type: %d", type);
	return NULL;
}

static NumarrayTypeNameMapping NumarrayTypeNameMap[] = {
	{"Any", tAny},
	{"Bool", tBool},
	{"Int8", tInt8},
	{"UInt8", tUInt8},
	{"Int16", tInt16},
	{"UInt16", tUInt16},
	{"Int32", tInt32},
	{"UInt32", tUInt32},
	{"Int64", tInt64},
	{"UInt64", tUInt64},
	{"Float32", tFloat32},
	{"Float64", tFloat64},
	{"Complex32", tComplex32},
	{"Complex64", tComplex64},
	{"Object", tObject},
	{"Long", tLong},
};

static PyObject *
setTypeException(int type)
{
	/* Check if it is a printable character */
	if ((type >= 32) && (type <= 126)) 
		PyErr_Format(_Error, 
			     "Type object lookup returned"
			     " NULL for type \'%c\'", type);
	else
		PyErr_Format(_Error,
			     "Type object lookup returned"
			     " NULL for type %d", type);
	return NULL;
}

static PyObject *
getTypeObject(NumarrayType type) 
{
        char strcharcode[2];
        PyObject *typeobj;
	
	if (deferred_libnumarray_init() < 0) return NULL;

        if ((type >= tAny) && (type <= tObject)) {
		return pNumType[type];
	} else  {
	      /* Test if it is a Numeric charcode */
		strcharcode[0] = type; strcharcode[1] = 0;
		typeobj = PyDict_GetItemString(
			pNumericTypesTDict, strcharcode);
		return typeobj ? typeobj : setTypeException(type);
	}
}

static PyObject *
NA_getType( PyObject *type)
{
	PyObject *typeobj = NULL;
	if (deferred_libnumarray_init() < 0) goto _exit;
	if (!type) goto _exit;
	if (PyObject_IsInstance(type, pNumericTypeClass)) {
		Py_INCREF(type);
		typeobj = type;
		goto _exit;
	}
	if ((typeobj = PyDict_GetItem(pNumericTypesTDict, type))) {
		Py_INCREF(typeobj);
	} else {
	       PyErr_Format(PyExc_ValueError, "NA_getType: unknown type.");
	}
  _exit:
	return typeobj;
}

/* Look up the NumarrayType which corresponds to typename */

static int 
NA_nameToTypeNo(char *typename)
{
	int i;
	for(i=0; i<ELEM(NumarrayTypeNameMap); i++)
		if (!strcmp(typename, NumarrayTypeNameMap[i].name))
			return NumarrayTypeNameMap[i].typeno;
	return -1;
}

/* Convert NumarrayType 'typeno' into the string of the type's name. */

static char *
NA_typeNoToName(int typeno) 
{
	int i;
	PyObject *typeObj;
	int typeno2;

	for(i=0; i<ELEM(NumarrayTypeNameMap); i++)
		if (typeno == NumarrayTypeNameMap[i].typeno)
			return NumarrayTypeNameMap[i].name;

	/* Handle Numeric typecodes */
	typeObj = NA_typeNoToTypeObject(typeno);
	if (!typeObj) return 0;
	typeno2 = NA_typeObjectToTypeNo(typeObj);
	Py_DECREF(typeObj);

	return NA_typeNoToName(typeno2);
}

static PyObject *
NA_typeNoToTypeObject(int typeno)
{
	PyObject *o;
	o = getTypeObject(typeno);	
	if (o) Py_INCREF(o);
	return o;
}

static int 
NA_typeObjectToTypeNo(PyObject *typeObj)
{
	int i;
	if (deferred_libnumarray_init() < 0) return -1;
	for(i=0; i<ELEM(pNumType); i++)
		if (pNumType[i] == typeObj)
			break;
	if (i == ELEM(pNumType)) i = -1;
	return i;
}

static long 
NA_isIntegerSequence(PyObject *sequence)
{
	PyObject *o;
	long i, size, isInt = 1;
	if (!sequence) { 
		isInt = -1; 
		goto _exit; 
	}
	if (!PySequence_Check(sequence)) {
		isInt = 0;
		goto _exit;
	}
	if ((size = PySequence_Length(sequence)) < 0) {
		isInt = -1;
		goto _exit;
	}
	for(i=0; i<size; i++) {
		o = PySequence_GetItem(sequence, i);
		if (!PyInt_Check(o) && !PyLong_Check(o)) {
			isInt = 0;
			Py_XDECREF(o);
			goto _exit;
		}
		Py_XDECREF(o);
	}
  _exit:
	return isInt;
}

static PyObject *
NA_intTupleFromMaybeLongs(int len, maybelong *Longs)
{
	int i;
	PyObject *intTuple = PyTuple_New(len);
	if (!intTuple) goto _exit;
	for(i=0; i<len; i++) {
		PyObject *o = PyInt_FromLong(Longs[i]);
		if (!o) {
			Py_DECREF(intTuple);
			intTuple = NULL;
			goto _exit;
		}
		PyTuple_SET_ITEM(intTuple, i, o);
	}
  _exit:
	return intTuple;
}

static long
NA_maybeLongsFromIntTuple(int len, maybelong *arr, PyObject *sequence)
{
	long i, size = -1;
	if (!PySequence_Check(sequence)) {
		PyErr_Format(PyExc_TypeError, 
			     "NA_maybeLongsFromIntTuple: must be a sequence of integers.");
		goto _exit;
	}
	size = PySequence_Length(sequence);
	if (size < 0) {
		PyErr_Format(PyExc_RuntimeError, 
			     "NA_maybeLongsFromIntTuple: error getting sequence length.");
		size = -1;
		goto _exit;
	}
	if (size > len) {
		PyErr_Format(PyExc_ValueError, 
			     "NA_maybeLongsFromIntTuple: sequence is too long");
		size = -1;
		goto _exit;
	}
	for(i=0; i<size; i++) {
		PyObject *o = PySequence_GetItem(sequence, i);
		long value;
		if (!o || !(PyInt_Check(o) || PyLong_Check(o))) {
			PyErr_Format(PyExc_TypeError, 
				     "NA_maybeLongsFromIntTuple: non-integer in sequence.");
			Py_XDECREF(o);
			size = -1;
			goto _exit;
		}
		arr[i] = value = PyInt_AsLong(o);
		if (arr[i] != value) {
			PyErr_Format(PyExc_ValueError, 
				     "NA_maybeLongsFromIntTuple: integer value too large: %ld",
				     value);
			size = -1;
			goto _exit;
		}
		if (PyErr_Occurred()) {
			Py_DECREF(o);
			size = -1;
			goto _exit;
		}
		Py_DECREF(o);
	}
  _exit:
	return size;
}

static int
NA_intTupleProduct(PyObject  *shape, long *prod)
{
	int i, nshape, rval = -1;

	if (!PySequence_Check(shape)) {
		PyErr_Format(PyExc_TypeError, 
		     "NA_intSequenceProduct: object is not a sequence.");
		goto _exit;
	}
	nshape = PySequence_Size(shape);
	
	for(i=0, *prod=1;  i<nshape; i++) {
		PyObject *obj = PySequence_GetItem(shape, i);
		if (!obj || !(PyInt_Check(obj) || PyLong_Check(obj))) {
			PyErr_Format(PyExc_TypeError, 
			     "NA_intTupleProduct: non-integer in shape.");
			Py_XDECREF(obj);
			goto _exit;
		}
		*prod *= PyInt_AsLong(obj);
		Py_DECREF(obj);
		if (PyErr_Occurred())
			goto _exit;
	}
	rval = 0;
  _exit:
	return rval;
}

/* NA_updateDataPtr updates the "working" data buffer pointer from the array's
buffer object.  Since objects which meet the buffer API can potentially
relocate their data as a result of executing arbitrary Python code
(i.e. array.resize), NA_updateDataPtr needs to be called each time control
flow returns to C, prior to accessing and numarray data.

_data  points to an object which must meet the buffer API
data   points to the array data gotten from _data via the buffer API

*/

static PyArrayObject *
NA_updateDataPtr(PyArrayObject *me)
{
	if (!me) return me;

	if (me->_data != Py_None) {

		if (getReadBufferDataPtr (me->_data, 
					  (void **) &me->data) < 0) {
			return (PyArrayObject *) PyErr_Format(
				_Error, 
				"NA_updateDataPtr: error getting read buffer data ptr");
		}
		if (isBufferWriteable( me->_data ))
			me->flags |= WRITABLE;
		else
			me->flags &= ~WRITABLE;
	} else {
		me->data = NULL;
	}

	me->data += me->byteoffset;

	return me;
}

/* Count the number of elements in a 1D static array. */
#define ELEM(x) (sizeof(x)/sizeof(x[0]))

static int 
NA_ByteOrder(void)
{
	unsigned long byteorder_test;
	byteorder_test = 1;
	if (*((char *) &byteorder_test))
		return NUM_LITTLE_ENDIAN;
	else
		return NUM_BIG_ENDIAN;
}

/* Create a new numarray specifying all attribute values and using an object which
   meets the buffer API to store the array data.
*/
static void
_stridesFromShape(PyArrayObject *self)
{
    int i;
    if (self->nd > 0) {
	for(i=0; i<self->nd; i++)
	    self->strides[i] = self->bytestride;
	for(i=self->nd-2; i>=0; i--)
	    self->strides[i] = 
		self->strides[i+1]*self->dimensions[i+1];
	self->nstrides = self->nd;
    } else 
	self->nstrides = 0;
}

 
static PyArrayObject *
NA_NewAllFromBuffer(int ndim, maybelong *shape, NumarrayType type,
		    PyObject *bufferObject, maybelong byteoffset, maybelong bytestride,
		    int byteorder, int aligned, int writeable)
{
	PyObject *typeObject;
	PyArrayObject *self = NULL;
	long i;

	if (deferred_libnumarray_init() < 0) goto _fail;

	if (type == tAny)
		type = tDefault;
	
	if (ndim > MAXDIM) goto _fail;


	tc_start_clock("NewArray __new__");
	self = (PyArrayObject *) PyObject_CallFunction(
		pNumArrayNewFunc,"(O)", pNumArrayClass);
	if (!self) goto _fail;
	tc_stop_clock("NewArray __new__");

	tc_start_clock("NewArray type");
	typeObject = getTypeObject(type);
        if (!typeObject) {
		setTypeException(type);
		goto _fail;
	}
	if (!(self->descr = NA_DescrFromType(type))) {
		goto _fail;
	}
	tc_stop_clock("NewArray type");

	tc_start_clock("NewArray misc");
	self->nd = self->nstrides = ndim;
	for(i=0; i<ndim; i++) {
		self->dimensions[i] = shape[i];
	}
	if (bytestride == 0)
		self->bytestride = self->descr->elsize;
	else
		self->bytestride = bytestride;
	_stridesFromShape(self);

	self->byteoffset = byteoffset;
	self->byteorder = byteorder;
	self->_aligned = aligned;
	self->itemsize = self->descr->elsize;
	tc_stop_clock("NewArray misc");

	tc_start_clock("NewArray buffer");
	Py_XDECREF(self->_data);
	if ((bufferObject == Py_None) || (bufferObject == NULL)) {
		long size = self->descr->elsize;
		for(i=0; i<self->nd; i++) {
			size *= self->dimensions[i];
		}
		self->_data = PyObject_CallFunction(
			pNewMemoryFunc, "(l)", size);
		if (!self->_data) goto _fail;
	} else {
		self->_data = bufferObject;
		Py_INCREF(self->_data);
	}
	tc_stop_clock("NewArray buffer");

	tc_start_clock("NewArray update");
	if (!NA_updateDataPtr(self))
		goto _fail;
	NA_updateStatus(self);
	tc_stop_clock("NewArray update");
	return self;

  _fail:
	Py_XDECREF(self);
	return NULL;
}

/* Create a new numarray specifying all attribute values but with a C-array as buffer
   which will be copied to a Python buffer. 
*/
static PyArrayObject *
NA_NewAll(int ndim, maybelong *shape, NumarrayType type, 
	  void *buffer, maybelong byteoffset, maybelong bytestride, 
	  int byteorder, int aligned, int writeable)
{
	PyArrayObject *result = NA_NewAllFromBuffer(
		ndim, shape, type, Py_None, byteoffset, bytestride,
		byteorder, aligned, writeable);
	
	if (result) {
		if (!NA_NumArrayCheck((PyObject *) result)) {
		       PyErr_Format( PyExc_TypeError,
				     "NA_NewAll: non-NumArray result");
		       result = NULL;
		} else {
			if (buffer) {
				memcpy(result->data, buffer, NA_NBYTES(result));
			} else {
				memset(result->data, 0, NA_NBYTES(result));
			}
		}
	}
	return  result;
}

static PyArrayObject *
NA_NewAllStrides(int ndim, maybelong *shape, maybelong *strides, 
		 NumarrayType type, void *buffer, maybelong byteoffset, 
		 int byteorder, int aligned, int writeable)
{
	int i;
	PyArrayObject *result = NA_NewAll(ndim, shape, type, buffer, 
					  byteoffset, 0,
					  byteorder, aligned, writeable);
	for(i=0; i<ndim; i++)
		result->strides[i] = strides[i];
	result->nstrides = ndim;
	return result;
}

/* Create a new numarray which is initially a C_array, or which
references a C_array: aligned, !byteswapped, contiguous, ... 

Call with buffer==NULL to allocate storage.
*/
static PyArrayObject *
NA_vNewArray(void *buffer, NumarrayType type, int ndim, maybelong *shape)
{
	return (PyArrayObject *) NA_NewAll(ndim, shape, type, buffer, 0, 0, 
					   NA_ByteOrder(), 1, 1);
}

static PyArrayObject *
NA_NewArray(void *buffer, NumarrayType type, int ndim, ...)
{
	int i;
	maybelong shape[MAXDIM];
	va_list ap;
	va_start(ap, ndim);
	for(i=0; i<ndim; i++)
		shape[i] = va_arg(ap, int);  /* literals will still be ints */
	va_end(ap);
	return NA_vNewArray(buffer, type, ndim, shape);
}

/* Original deprecated versions of new array and empty array */
static PyArrayObject *
NA_New(void *buffer, NumarrayType type, int ndim, ...)
{
	int i;
	maybelong shape[MAXDIM];
	va_list ap;
	va_start(ap, ndim);
	for(i=0; i<ndim; i++)
		shape[i] = va_arg(ap, int);
	va_end(ap);
	return NA_NewAll(ndim, shape, type, buffer, 0, 0, 
			 NA_ByteOrder(), 1, 1);
}

static PyArrayObject *
NA_Empty(int ndim, maybelong *shape, NumarrayType type)
{
	return NA_NewAll(ndim, shape, type, NULL, 0, 0, 
			 NA_ByteOrder(), 1, 1);
}


/* getArray creates a new array of type 't' from the given array 'a'
using the specified 'method', probably 'new' or 'astype'. */

static PyArrayObject *
getArray(PyArrayObject *a, NumarrayType t, char *method)
{
	char *name;

	if (deferred_libnumarray_init() < 0) return NULL;

	if (t == tAny)
		t = a->descr->type_num;
	name = NA_typeNoToName(t);
	if (!name) return (PyArrayObject *) setTypeException(t);
	return (PyArrayObject *) 
		PyObject_CallMethod((PyObject *) a, method, "s", name);
}

static int 
getShape(PyObject *a, maybelong *shape, int dims)
{
	long slen;

	if (PyString_Check(a)) {
		PyErr_Format(PyExc_TypeError,
			     "getShape: numerical sequences can't contain strings.");
		return -1;
	}

	if (!PySequence_Check(a) || 
	    (NA_NDArrayCheck(a) && (PyArray(a)->nd == 0)))
		return dims;
	slen = PySequence_Length(a);
	if (slen < 0) {
		PyErr_Format(_Error,
			     "getShape: couldn't get sequence length.");
		return -1;
	}
	if (!slen) {
		*shape = 0;
		return dims+1;
	} else if (dims < MAXDIM) {
		PyObject *item0 = PySequence_GetItem(a, 0);
		if (item0) {
			*shape = PySequence_Length(a);
			dims = getShape(item0, ++shape, dims+1);
			Py_DECREF(item0);
		} else {
			PyErr_Format(_Error, 
				     "getShape: couldn't get sequence item.");
			return -1;
		}
	} else {		  
		PyErr_Format(_Error, 
		 "getShape: sequence object nested more than MAXDIM deep.");
		return -1;
	}
	return dims;
}

typedef enum {
  NOTHING,
  NUMBER,
  SEQUENCE
} SequenceConstraint;

static int 
setArrayFromSequence(PyArrayObject *a, PyObject *s, int dim, long offset)
{
	SequenceConstraint mustbe = NOTHING;
	int i, seqlen=-1, slen = PySequence_Length(s);

	if (dim > a->nd) {
		PyErr_Format(PyExc_ValueError, 
			     "setArrayFromSequence: sequence/array dimensions mismatch.");
		return -1;
	}

	if (slen != a->dimensions[dim]) {
		PyErr_Format(PyExc_ValueError,
			     "setArrayFromSequence: sequence/array shape mismatch.");
		return -1;
	}

	for(i=0; i<slen; i++) {
		PyObject *o = PySequence_GetItem(s, i);
		if (!o) {
			PyErr_SetString(_Error, 
 			   "setArrayFromSequence: Can't get a sequence item");
			return -1;
		} else if ((NA_isPythonScalar(o) || 
			    (NA_NumArrayCheck(o) && PyArray(o)->nd == 0)) && 
			   ((mustbe == NOTHING) || (mustbe == NUMBER))) {
			if (NA_setFromPythonScalar(a, offset, o) < 0)
				return -2;
			mustbe = NUMBER;
		} else if (PyString_Check(o)) {
			PyErr_SetString( PyExc_ValueError,
			"setArrayFromSequence: strings can't define numeric numarray.");
			return -3;
		} else if (PySequence_Check(o)) {
			if ((mustbe == NOTHING) || (mustbe == SEQUENCE)) {
				if (mustbe == NOTHING) {
					mustbe = SEQUENCE;
					seqlen = PySequence_Length(o);
				} else if (PySequence_Length(o) != seqlen) {
					PyErr_SetString(
						PyExc_ValueError, 
						"Nested sequences with different lengths.");
					return -5;
				}
				setArrayFromSequence(a, o, dim+1, offset);
			} else {
				PyErr_SetString(PyExc_ValueError,
						"Nested sequences with different lengths.");
				return -4;
			}
		} else {
			PyErr_SetString(PyExc_ValueError, "Invalid sequence.");
			return -6;
		}
		Py_DECREF(o);
		offset += a->strides[dim];
	}
	return 0;
}

static PyObject *
NA_setArrayFromSequence(PyArrayObject *a, PyObject *s)
{
	PyObject *rval = NULL;
	if (PySequence_Check(s))
	{
		maybelong shape[MAXDIM];
		int dims = getShape(s, shape, 0);

		if (dims < 0) 
			goto _exit;
		
		if (!NA_updateDataPtr(a)) 
			goto _exit;

		if (setArrayFromSequence(a, s, 0, 0) < 0)
			goto _exit;

		Py_INCREF(Py_None);
		rval = Py_None;
		goto _exit;
	}
	PyErr_Format(PyExc_TypeError, 
		     "NA_setArrayFromSequence: (array, seq) expected.");
  _exit:
	return rval;
}

static int
_NA_maxType(PyObject *seq, int limit)
{
	if (limit > MAXDIM) {
		PyErr_Format( PyExc_ValueError, 
			      "NA_maxType: sequence nested too deep." );
		return -1;
	}
	if (NA_NumArrayCheck(seq)) {
		switch(PyArray(seq)->descr->type_num) {
		case tBool:
			return BOOL_SCALAR;
		case tInt8:
		case tUInt8:
		case tInt16:
		case tUInt16:
		case tInt32:
		case tUInt32:
			return INT_SCALAR;
		case tInt64:
		case tUInt64:
			return LONG_SCALAR;
		case tFloat32:
		case tFloat64:
			return FLOAT_SCALAR;
		case tComplex32:
		case tComplex64:
			return COMPLEX_SCALAR;
		default:
			PyErr_Format(PyExc_TypeError, 
				     "Expecting a python numeric type, got something else.");
			return -1;
		}
	} else if (PySequence_Check(seq) && !PyString_Check(seq)) {
		long i, maxtype=BOOL_SCALAR, slen;

		slen = PySequence_Length(seq);		
		if (slen < 0) return -1;

		if (slen == 0) return INT_SCALAR;

		for(i=0; i<slen; i++) {
			long newmax;
			PyObject *o = PySequence_GetItem(seq, i);
			if (!o) return -1;
			newmax = _NA_maxType(o, limit+1);
			if (newmax  < 0) 
				return -1;
			else if (newmax > maxtype) {
				maxtype = newmax;
			}
			Py_DECREF(o);
		}
		return maxtype;
	} else {
#if PY_VERSION_HEX >= 0x02030000
		if (PyBool_Check(seq))
			return BOOL_SCALAR;
		else 
#endif
			if (PyInt_Check(seq))
			return INT_SCALAR;
		else if (PyLong_Check(seq))
			return LONG_SCALAR;
		else if (PyFloat_Check(seq))
			return FLOAT_SCALAR;
		else if (PyComplex_Check(seq))
			return COMPLEX_SCALAR;
		else {
			PyErr_Format(PyExc_TypeError, 
				     "Expecting a python numeric type, got something else.");
			return -1;
		}
	}
}

static int 
NA_maxType(PyObject *seq)
{
	int rval;
	rval = _NA_maxType(seq, 0);
	return rval;
}

NumarrayType
NA_NumarrayType(PyObject *seq)
{
	int maxtype = NA_maxType(seq);
	int rval;
	switch(maxtype) {
	case BOOL_SCALAR:
		rval = tBool;
		goto _exit;
	case INT_SCALAR:
	case LONG_SCALAR:
		rval = tLong; /* tLong corresponds to C long int,
				 not Python long int */
		goto _exit;
	case FLOAT_SCALAR:
		rval = tFloat64;
		goto _exit;
	case COMPLEX_SCALAR:
		rval = tComplex64;
		goto _exit;
	default:
		PyErr_Format(PyExc_TypeError,
			     "expecting Python numeric scalar value; got something else.");
		rval = -1;
	}
  _exit:
	return rval;
}

/* sequenceAsArray converts a python sequence (list or tuple)
into an array of the specified type and returns it.
*/
static PyArrayObject*
sequenceAsArray(PyObject *s, NumarrayType *t)
{
	if (!NA_NumArrayCheck(s))
	{
		maybelong shape[MAXDIM];
		int dims = getShape(s, shape, 0);
		PyArrayObject *array;
		
		if (dims < 0) return NULL;
		
		if (*t == tAny) {
			*t = NA_NumarrayType(s);
		}

		if (!(array = NA_vNewArray(NULL, *t, dims, shape))) 
			return NULL;
		
		if (setArrayFromSequence(array, s, 0, 0) < 0) {
			return (PyArrayObject *) PyErr_Format(_Error, 
			     "sequenceAsArray: can't convert sequence to array");
		}
		return array;
	}
	Py_INCREF(s);
	return (PyArrayObject *) s;
}

/* satisfies ensures that 'a' meets a set of requirements and matches 
the specified type.
*/
static int
satisfies(PyArrayObject *a, int requirements, NumarrayType t)
{
	int type_ok = (a->descr->type_num == t) || (t == tAny);

	if (PyArray_ISCARRAY(a))
		return type_ok;
	if (PyArray_ISBYTESWAPPED(a) && (requirements & NUM_NOTSWAPPED)) 
		return 0;
	if (!PyArray_ISALIGNED(a) && (requirements & NUM_ALIGNED)) 
		return 0;
	if (!PyArray_ISCONTIGUOUS(a) && (requirements & NUM_CONTIGUOUS))
		return 0;
	if (!PyArray_ISWRITABLE(a) && (requirements & NUM_WRITABLE))
		return 0;
	if (requirements & NUM_COPY)
		return 0;
	return type_ok;
}

/* NA_InputArray is the main input conversion routine.  NA_InputArray
 converts input array 'a' as necessary to NumarrayType 't' also guaranteeing
 that either 'a' or the converted result is contigous, aligned, and not
 byteswapped. NA_InputArray returns a pointer to a Numarray for which
 C_array is 1, and fills in 'ainfo' with the array information of the return
 value.  The return value provides a means to later deallocate any temporary
 array created by NA_InputArray, while 'ainfo' provides direct access to the
 array's "metadata" from C.  Since the reference count of the input array 'a'
 is incremented when it is directly useable by C, the return value (either 'a'
 or a temporary) should always be passed to Py_XDECREF by the calller.  Note
 that at failed sequence conversion, getNumInfo, or getArray results in the
 value NULL being returned.

1. 'a' is already c-usesable.
2. 'a' is an array, but needs conversion to be c-useable.
3. 'a' is a numeric sequence, not an array.
4. 'a' is a numeric scalar, not an array.

The contents of the resulting array are either 'a' or 'a.astype(t)'.
The return value should always be DECREF'ed by the caller.

requires is a bitmask specifying a set of requirements on the converted array.
*/
static PyArrayObject*
NA_InputArray(PyObject *a, NumarrayType t, int requires)
{
	PyArrayObject *wa = NULL;
	if (NA_isPythonScalar(a)) {
		if (t == tAny) 
			t = NA_NumarrayType(a);
		if (t < 0) goto _exit;
		wa = NA_vNewArray( NULL, t, 0, NULL);
		if (!wa) goto _exit;
		if (NA_setFromPythonScalar(wa, 0, a) < 0) {
			Py_DECREF(wa);
			wa = NULL;
		}
		goto _exit;
	} else if ((wa = sequenceAsArray(a, &t))) {
		if (!satisfies(wa, requires, t)) {
			PyArrayObject *wa2 = getArray(wa, t, "astype");
			Py_DECREF(wa);
			wa = wa2;
		}
		NA_updateDataPtr(wa);
	}
  _exit:
	return wa;
}

/* NA_OutputArray creates a C-usable temporary array similar to 'a' but of
type 't' as necessary.  If 'a' is already C-useable and of type 't', then 'a'
is returned.  In either case, 'ainfo' is filled in with the array information
for the return value.

The contents of the resulting array are undefined, assumed to be filled in by
the caller.
*/
static PyArrayObject *
NA_OutputArray(PyObject *a0, NumarrayType t, int requires)
{
	PyArrayObject *a = (PyArrayObject *) a0;

	if (!NA_NumArrayCheck(a0)  || !PyArray_ISWRITABLE(a)) {
		PyErr_Format(PyExc_TypeError, 
			     "NA_OutputArray: only writable NumArrays work for output.");
		a = NULL;
		goto _exit;
	}

	if (satisfies(a, requires, t)) {
		Py_INCREF(a0);
		NA_updateDataPtr(a);
		goto _exit;
	} else {
		PyArrayObject *shadow = getArray(a, t, "new");
		if (shadow) {
			Py_INCREF(a0);
			shadow->_shadows = a0;
		}
		a = shadow;
	}
  _exit:
	return a;
}

/* NA_OptionalOutputArray works like NA_ShadowOutput, but handles the case
where the output array 'optional' is omitted entirely at the python level,
resulting in 'optional'==Py_None.  When 'optional' is Py_None, the return
value is cloned (but with NumarrayType 't') from 'master', typically an input
array with the same shape as the output array.
*/
static PyArrayObject *
NA_OptionalOutputArray(PyObject *optional, NumarrayType t, int requires, 
		       PyArrayObject *master)
{
	if ((optional == Py_None) || (optional == NULL)) {
		PyArrayObject *rval;
		rval = getArray(master, t, "new");
		return rval;
	} else {
		return NA_OutputArray(optional, t, requires);
	}
}

/* NA_IoArray is a combination of NA_InputArray and NA_OutputArray.

Unlike NA_OutputArray, if a temporary is required it is initialized to a copy
of the input array.

Unlike NA_InputArray, deallocating any resulting temporary array results in a
copy from the temporary back to the original.
*/
static PyArrayObject *
NA_IoArray(PyObject *a, NumarrayType t, int requires)
{
	PyArrayObject *shadow = NA_InputArray(a, t, requires);

	if (!shadow) return NULL;

	/* Guard against non-writable, but otherwise satisfying requires. 
	   In this case,  shadow == a.
	*/
	if (!PyArray_ISWRITABLE(shadow)) {
		PyErr_Format(PyExc_TypeError,
			     "NA_IoArray: I/O numarray must be writable NumArrays.");
		Py_DECREF(shadow);
		shadow = NULL;
		goto _exit;
	}

	if ((shadow != (PyArrayObject *) a) && NA_NumArrayCheck(a)) {
		Py_INCREF(a);
		shadow->_shadows = a;
	}
  _exit:
	return shadow;
}

/* NA_ReturnOutput handles returning a possibly unspecified output array.  If
the array 'out' was specified on the original call to the Python wrapper
function, then the contents of any 'shadow' array are copied into 'out' as
required.  The function then returns Py_None.  If no output was specified in
the original call, then the 'shadow' array *becomes* the output and is
returned.  This results in extension functions which return Py_None when you
specify an output array, and return an array value otherwise.  These functions
also correctly handle data typing, alignment, byteswapping, and contiguity
issues. 
*/
static PyObject*
NA_ReturnOutput(PyObject *out, PyArrayObject *shadow)
{
	if ((out == Py_None) || (out == NULL)) { 
                /* default behavior: return shadow array as the result */
		return (PyObject *) shadow;  
	} else {  
		PyObject *rval;
                /* specified output behavior: return None */
		/* del(shadow) --> out.copyFrom(shadow) */
		Py_DECREF(shadow);
		Py_INCREF(Py_None);
		rval = Py_None;
		return rval;
	}
}

/* NA_ShapeEqual returns 1 if 'a' and 'b' have the same shape, 0 otherwise.
*/
static int
NA_ShapeEqual(PyArrayObject *a, PyArrayObject *b)
{
	int i;
	
	if (!NA_NDArrayCheck((PyObject *) a) || 
	    !NA_NDArrayCheck((PyObject*) b)) {
		PyErr_Format(
			PyExc_TypeError, 
			"NA_ShapeEqual: non-array as parameter.");
		return -1;
	}
	if (a->nd != b->nd)
		return 0;
	for(i=0; i<a->nd; i++)
		if (a->dimensions[i] != b->dimensions[i])
			return 0;
	return 1;
}

/* NA_ShapeLessThan returns 1 if a.shape[i] < b.shape[i] for all i, else 0.  
If they have a different number of dimensions, it compares the innermost
overlapping dimensions of each.
*/
static int
NA_ShapeLessThan(PyArrayObject *a, PyArrayObject *b)
{
        int i;
	int mindim, aoff, boff;
	if (!NA_NDArrayCheck((PyObject *) a) || 
	    !NA_NDArrayCheck((PyObject *) b)) {
		PyErr_Format(PyExc_TypeError, 
			     "NA_ShapeLessThan: non-array as parameter.");
		return -1;
	}
	mindim = MIN(a->nd, b->nd);
	aoff = a->nd - mindim;
	boff = b->nd - mindim;
	for(i=0; i<mindim; i++)
		if (a->dimensions[i+aoff] >=  b->dimensions[i+boff])
			return 0;
	return 1;
}

#define MakeChecker(name, classpointer)                    \
static int                                                 \
name##Exact(PyObject *op) {                                \
        return ((PyObject *) op->ob_type) == classpointer; \
}                                                          \
static int                                                 \
name(PyObject *op) {                                       \
      int rval = -1;                                       \
      if (deferred_libnumarray_init() < 0) goto _exit;     \
      rval = PyObject_IsInstance(op, classpointer);        \
  _exit:                                                   \
      return rval;                                         \
}

MakeChecker(NA_NDArrayCheck, pNDArrayClass)
MakeChecker(NA_NumArrayCheck, pNumArrayClass)
MakeChecker(NA_OperatorCheck, pOperatorClass)
MakeChecker(NA_ConverterCheck, pConverterClass)
MakeChecker(NA_UfuncCheck, pUfuncClass)
MakeChecker(NA_CfuncCheck, pCfuncClass)

static int
NA_ComplexArrayCheck(PyObject *a)
{
	int rval = NA_NumArrayCheck(a);
	if (rval > 0) {
		PyArrayObject *arr = (PyArrayObject *) a;
		switch(arr->descr->type_num) {
		case tComplex64: case tComplex32:
			return 1;
		default:
			return 0;
		}
	}
	return rval;
}

static PyObject * 
NA_Cast(PyArrayObject *a, int type)
{
	PyObject *rval = NULL;
	if (deferred_libnumarray_init() < 0) 
		goto _exit;
	rval = (PyObject *) getArray(a, type, "astype");
  _exit:
	return rval;
}

static int
NA_copyArray(PyArrayObject *to, const PyArrayObject *from)
{
	int rval = -1;
	PyObject *result;
	result = PyObject_CallMethod((PyObject *) to, 
				     "_copyFrom","(O)", from);
	if (!result) goto _exit;
	Py_DECREF(result);
	rval = 0;
  _exit:
	return rval;
}

static PyArrayObject *
NA_copy(PyArrayObject *from)
{
	PyArrayObject * rval;
	rval = (PyArrayObject *)
		PyObject_CallMethod((PyObject *) from, "copy", NULL);
	return rval;
}

static void
NA_stridesFromShape(int nshape, maybelong *shape, maybelong bytestride, 
		    maybelong *strides)
{
	int i;
	if (nshape > 0) {
		for(i=0; i<nshape; i++)
			strides[i] = bytestride;
		for(i=nshape-2; i>=0; i--)
			strides[i] = strides[i+1]*shape[i+1];
	} 
}

static int
NA_getByteOffset(PyArrayObject *array, int nindices, maybelong *indices, 
		 long *offset)
{
	int i;

        /* rank0 or _UBuffer */
	if ((array->nd == 0) || (array->nstrides < 0)) {
		*offset = array->byteoffset;
		return 0;
	}
	
	/* Check for indices/shape mismatch when not rank-0.
	 */
	if ((nindices > array->nd) && 
	    !((nindices == 1) && (array->nd == 0))) {
		PyErr_Format(PyExc_IndexError, "too many indices.");
		return -1;
	}

	*offset = array->byteoffset;
	for(i=0; i<nindices; i++) {
		long ix = indices[i];
		long limit = i < array->nd ? array->dimensions[i] : 0;
		if (ix < 0) 
			ix += limit;
		if ((array->strides[i] || limit) && 
		    (ix < 0 || ix >= limit)) {
			PyErr_Format(PyExc_IndexError, "Index out of range");
			return -1;
		}
		*offset += ix*array->strides[i];
	}
	return 0;
}

static int
NA_swapAxes(PyArrayObject *array, int x, int y)
{
	long temp;

	if (((PyObject *) array) == Py_None) return 0;

	if (array->nd < 2) return 0;

	if (x < 0) x += array->nd;
	if (y < 0) y += array->nd;

	if ((x < 0) || (x >= array->nd) || 
	    (y < 0) || (y >= array->nd)) {
		PyErr_Format(PyExc_ValueError, 
			     "Specified dimension does not exist");
		return -1;
	}

	temp = array->dimensions[x];
	array->dimensions[x] = array->dimensions[y];
	array->dimensions[y] = temp;
	
	temp = array->strides[x];
	array->strides[x] = array->strides[y];
	array->strides[y] = temp;

	NA_updateStatus(array);
	
	return 0;
}

static PyObject * 
NA_initModuleGlobal(char *modulename, char *globalname)
{
	PyObject *module, *dict, *global = NULL;
	module = PyImport_ImportModule(modulename);
	if (!module) {
	       PyErr_Format(PyExc_RuntimeError, 
			    "Can't import '%s' module", 
			    modulename);
	       goto _exit;
	}
	dict = PyModule_GetDict(module);
	global = PyDict_GetItemString(dict, globalname);
	if (!global) {
		PyErr_Format(PyExc_RuntimeError, 
			     "Can't find '%s' global in '%s' module.",
			     globalname, modulename);
		goto _exit;
	}
	Py_DECREF(module);
	Py_INCREF(global);
  _exit:
	return global;
}


static long
_isaligned(PyArrayObject *self)
{
	long i, ptr, alignment, aligned = 1;

	alignment = MAX(MIN(self->itemsize, MAX_ALIGN), 1);

	/* Support self-test override. */
	if (self->_aligned != MUST_BE_COMPUTED) {
		return self->_aligned;
	}
	ptr = (long) self->data;
	aligned = (ptr % alignment) == 0;
	for (i=0; i <self->nd; i++)
		aligned &= ((self->strides[i] % alignment) == 0);
	return aligned != 0;
}

static long
_iscontiguous(PyArrayObject *self)
{
	long i, ndim, nstrides;

	/* Support self-test override. */
	if (self->_contiguous != MUST_BE_COMPUTED) {
		return self->_contiguous;
	}

	ndim = self->nd;
	nstrides = self->nstrides;

	/* rank-0 numarray are always contiguous */
	if (ndim == 0) return 1;

	/* Strides must be in decreasing order. ndim >= 1 */
	for(i=0; i<ndim-1; i++)
		if (self->strides[i] != 
		    self->strides[i+1]*self->dimensions[i+1])
			return 0;

	/* Broadcast numarray have 0 in some stride and are discontiguous */
	for(i=0; i<nstrides-1; i++)
		if (!self->strides[i])
			return 0;

	if ((self->strides[nstrides-1] == self->itemsize) &&
	    (self->bytestride == self->itemsize))
		return 1;

	if ((self->strides[nstrides-1] == 0) && (nstrides > 1))
		return 1;
	
	return 0;
}

static void
NA_updateAlignment(PyArrayObject *self)
{
	if (_isaligned(self))
		self->flags |= ALIGNED;
	else
		self->flags &= ~ALIGNED;

}

static void
NA_updateContiguous(PyArrayObject *self)
{
	if (_iscontiguous(self))
		self->flags |= CONTIGUOUS;
	else
		self->flags &= ~CONTIGUOUS;
}

static int 
_isbyteswapped(PyArrayObject *self)
{
	int syslittle = (NA_ByteOrder() == NUM_LITTLE_ENDIAN);
	int selflittle = (self->byteorder == NUM_LITTLE_ENDIAN);
	int byteswapped = (syslittle != selflittle);
	return byteswapped;
}

static void
NA_updateByteswap(PyArrayObject *self)
{
	if (!_isbyteswapped(self))
		self->flags |= NOTSWAPPED;
	else
		self->flags &= ~NOTSWAPPED;
}

static void 
NA_updateStatus(PyArrayObject *self)
{
	NA_updateAlignment(self);
	NA_updateContiguous(self);
	NA_updateByteswap(self);
}

static char *
NA_getArrayData(PyArrayObject *obj)
{
	if (!NA_NDArrayCheck((PyObject *) obj)) {
		PyErr_Format(PyExc_TypeError, 
			     "expected an NDArray");
	}
	if (!NA_updateDataPtr(obj))
		return NULL;
	return obj->data;
}

/* The following function has much platform dependent code since
** there is no platform-independent way of checking Floating Point
** status bits
*/

static int 
NA_checkFPErrors(void)
{
#ifdef __osf__
#ifdef __alpha
	unsigned long fpstatus;
#endif
#else
	int fpstatus;
#endif
	int retstatus;
/* MS Windows -----------------------------------------------------*/
#if defined(_MSC_VER)
#include <float.h>
	fpstatus = (int) _clear87();
	retstatus = pyFPE_DIVIDE_BY_ZERO * ((SW_ZERODIVIDE & fpstatus) != 0)
		+ pyFPE_OVERFLOW       * ((SW_OVERFLOW & fpstatus)   != 0)
		+ pyFPE_UNDERFLOW      * ((SW_UNDERFLOW & fpstatus)  != 0)
		+ pyFPE_INVALID        * ((SW_INVALID & fpstatus)    != 0);

/* Solaris --------------------------------------------------------*/
/* --------ignoring SunOS ieee_flags approach, someone else can
**         deal with that! */
#elif defined(sun)
#include <ieeefp.h>
/* #include <math.h> */
	fpstatus = (int) fpgetsticky();
	retstatus = pyFPE_DIVIDE_BY_ZERO * ((FP_X_DZ  & fpstatus) != 0)
		+ pyFPE_OVERFLOW       * ((FP_X_OFL & fpstatus) != 0)
		+ pyFPE_UNDERFLOW      * ((FP_X_UFL & fpstatus) != 0)
		+ pyFPE_INVALID        * ((FP_X_INV & fpstatus) != 0);
	(void) fpsetsticky(0);

/* Linux ----------------------------------------------------------*/
#elif defined(linux) || defined(darwin) || defined(__CYGWIN__)
#if defined(__GLIBC__)
#include <fenv.h>
#elif defined(__CYGWIN__)
#include <mingw/fenv.h>
#elif defined(darwin)
#include <CoreServices/CoreServices.h>
#else
#include <i386/fenv.h>
#endif

	fpstatus = (int) fetestexcept(
		FE_DIVBYZERO | FE_OVERFLOW | FE_UNDERFLOW | FE_INVALID);
	retstatus = pyFPE_DIVIDE_BY_ZERO * ((FE_DIVBYZERO  & fpstatus) != 0)
		+ pyFPE_OVERFLOW       * ((FE_OVERFLOW   & fpstatus) != 0)
		+ pyFPE_UNDERFLOW      * ((FE_UNDERFLOW  & fpstatus) != 0)
		+ pyFPE_INVALID        * ((FE_INVALID    & fpstatus) != 0);
	(void) feclearexcept(FE_DIVBYZERO | FE_OVERFLOW | FE_UNDERFLOW | FE_INVALID);

/* Tru64 ----------------------------------------------------------*/
#elif defined(__alpha)
#ifdef __osf__
#include <machine/fpu.h>
	fpstatus = ieee_get_fp_control(); 
	/* clear status bits as well as disable exception mode if on */
	ieee_set_fp_control( 0 );
	retstatus = pyFPE_DIVIDE_BY_ZERO* (int)((IEEE_STATUS_DZE  & fpstatus) != 0)
		+ pyFPE_OVERFLOW      * (int)((IEEE_STATUS_OVF  & fpstatus) != 0)
		+ pyFPE_UNDERFLOW     * (int)((IEEE_STATUS_UNF  & fpstatus) != 0)
		+ pyFPE_INVALID       * (int)((IEEE_STATUS_INV  & fpstatus) != 0); 
#endif
#else
	retstatus = 0;
#endif
	return retstatus;
}

static void
NA_clearFPErrors()
{
	NA_checkFPErrors();
}

static int
NA_checkAndReportFPErrors(char *name)
{
	int error = NA_checkFPErrors();
	if (error) {
		PyObject *ans;
		char msg[128];
		if (deferred_libnumarray_init() < 0) 
			return -1;
		strcpy(msg, " in ");
		strncat(msg, name, 100);
		ans = PyObject_CallFunction(pHandleErrorFunc, "(is)", error, msg);
		if (!ans) return -1;
		Py_DECREF(ans); /* Py_None */
	}
	return 0;
}

/*
 * Local Variables:
 * mode: C
 * c-file-style: "python"
 * End:
 */
