# Session.py
#
# Copyright 2002 Wichert Akkerman <wichert@deephackmode.org>
#
# This file is free software; you can redistribute it and/or modify it
# under the terms of version 2 of the GNU General Public License as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# Calculate shared library dependencies

"""Session storage class

When dealing with multiple simultaneous sessions as is common with web-based
programming it is very useful to store session information on the server. To
do this a special kind of database is needed which allows for multiple readers
and writers in parallel.

This module implements such a database: it uses samba's tdb library to provide
the on-disk storage and adds a shelf-like interface on top of it.
"""

__docformat__	= "epytext en"

import os, shelve, UserDict
import tdb

class Session(shelve.Shelf):
	"""Session storage

	This class acts very much like a selve.Shelf class, except for two
	differences:
	
	  1. it only supports tdb databases
	  2. You can only store dictionaries in the top dictionary
	
	It works very simple::

	  session=Session("session.db")
	  session[sessionkey]["hostname"]=os.environ["REMOTE_ADDR"]
	  session[sessionkey]["username"]="username"
	"""
	def __init__(self, file):
		self.db=tdb.open(file, open_flags=os.O_CREAT|os.O_RDWR)
		shelve.Shelf.__init__(self, self.db)
	

	def __getitem__(self, key):
		return _SubSession(self, key)



class _SubSession(UserDict.UserDict):
	def __init__(self, session, key):
		self.session=session
		self.key=key
		try:
			self.data=shelve.Shelf.__getitem__(session, key)
		except KeyError:
			self.data={}
	

	def __sync(self):
		shelve.Shelf.__setitem__(self.session, self.key, self.data)


	def __setitem__(self, key, item):
		UserDict.UserDict.__setitem__(self, key, item)
		self.__sync()


	def __delitem__(self, key):
		UserDict.UserDict.__delitem__(self, key)
		self.__sync()


	def clear(self):
		UserDict.UserDict.clear()
		self.__sync()


	def update(self, dict):
		UserDict.UserDict.self(self, dict)
		self.__sync()


	def copy(self):
		import copy
		return copy.copy(self)


