// silencedetecter.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
 
#include "silencedetecter.h"
#ifdef __GNUG__
//#pragma implementation
#endif
#include "sound.h"
#include "request.h"
#include "requester.h"
 
Modifier* SilenceDetecter::createUndo() {
	return nil;
}

class SilenceDetectRequester : public TitledRequester {
	friend class SilenceDetecter;
protected:
	SilenceDetectRequester(const char* title, SilenceDetecter* e)
		: TitledRequester(title), _client(e) {}
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	SilenceDetecter* _client;
	double	_frameDuration;
	TimeVal	_silenceDuration;
};

void
SilenceDetectRequester::configureRequest(Request* request) {
	request->appendLabel("Search for section with levels below specified threshold.");
	request->appendValue("Silence threshold (dB):",
			     &_client->_attenuationThreshold,
			    Range(-96.0, -3.0),
			    true);
	request->appendChoice("Threshold Type:", "|Relative to Sound Level|Absolute|",
						  &_client->_thresholdMode, true);
	Sound *sound = (Sound *) _client->target();
	_frameDuration = (double) _client->_frameSize / sound->sRate();
	request->appendValue("Examine sound in frames of length (seconds):",
						 &_frameDuration,
			     		 Range(0.001, sound->length()/(2.0 * sound->sRate())));
	_silenceDuration = (double) _client->_silenceLength / sound->sRate();
	request->appendValue("Minimum duration of silence to detect (MM:SS):",
						 &_silenceDuration, Range(0.1, sound->duration()));
}

boolean
SilenceDetectRequester::confirmValues() {
	Sound *sound = (Sound *) _client->target();
	_client->_frameSize = int(0.5 + (double) _frameDuration * sound->sRate());
	_client->_silenceLength = int(0.5 + (double) _silenceDuration * sound->sRate());
	return true;
}

//********

double SilenceDetecter::_savedAttenuationThreshold = -60.0;
ChoiceValue SilenceDetecter::_savedThresholdMode = SilenceDetecter::Relative;
int SilenceDetecter::_savedFrameSize = 0;
int SilenceDetecter::_savedSilenceLength = 0;

SilenceDetecter::SilenceDetecter(Data* d) : DataModifier(d, false),
	  _attenuationThreshold(_savedAttenuationThreshold),
	  _thresholdMode(_savedThresholdMode),
	  _frameSize(_savedFrameSize == 0 ? int(0.01 * d->sRate()) : _savedFrameSize),
	  _silenceLength(_savedSilenceLength == 0 ? int(0.5 * d->sRate()) : _savedSilenceLength),
	  _silenceOffset(-1) {}

SilenceDetecter::SilenceDetecter(Data* d,
					 double attenThreshold,
					 int framesize, int silenceLen)
		: DataModifier(d, false), _attenuationThreshold(attenThreshold),
		_thresholdMode(Absolute),
		  _frameSize(_frameSize), _silenceLength(silenceLen), _silenceOffset(-1) {
	initialize();
}

void
SilenceDetecter::initialize() {
	Super::initialize();
}

Requester *
SilenceDetecter::createRequester() {
	return new SilenceDetectRequester("Search for Silence:", this);
}

void
SilenceDetecter::saveConfig() {
    _savedAttenuationThreshold = _attenuationThreshold;
	_savedThresholdMode = _thresholdMode;
    _savedFrameSize = _frameSize;
	_savedSilenceLength = _silenceLength;
}

int
SilenceDetecter::doApply(Data *data) {
	const int length = data->length();
	const int chans = data->channels();
	int silenceFrameCount = 0, noiseSectionCount = 0;
	const int slices = 1 + length / _frameSize;

	// Get db gains for entire selection, one per every _frameSize frames.
	Data *dbGains = new Data(DoubleData, slices, chans);
	dbGains->ref();
	for (int chan = 0; chan < chans; ++chan) {
		Data *evpChannel = dbGains->clone(dbGains->frameRange(), Range(chan,chan));
		data->getEnvelope(evpChannel, chan, Decibels);
	}
	int section = 0;
	double runningGain = 0;
	_silenceOffset = -1;	// reset
	for (int frame = 0; frame < length; frame += _frameSize) {
		double sectionGain = 0.0;
		for (int chan = 0; chan < chans; ++chan) {
			// Values from getEnvelope are from 0 to +96.0 db
			double channelGain = dbGains->get(section, chan) - 96.0;
			if (channelGain < sectionGain) {
				sectionGain = channelGain;
			}
		}
		if (frame == 0)
			runningGain = sectionGain;	// first call
//		printf("section gain [%d] = %g, running gain = %g\n",
//			   section, sectionGain, runningGain);
		++section;
		
		double threshold = (_thresholdMode == Absolute) ?
			_attenuationThreshold : runningGain - _attenuationThreshold;

		// If section gain is less than the threshold, store the current offset.
		// Increment silence counter, and return if we exceed the requested
		// length.  Else, reset the offset and frame counts and continue.
		if (sectionGain <= threshold) {
			if (_silenceOffset == -1) {
//				printf("\tPossible beginning of silence.\n");
				_silenceOffset = frame;
			}
			silenceFrameCount += _frameSize;
//			printf("\tsilenceFrameCount = %d\n", silenceFrameCount);
			if (silenceFrameCount >= _silenceLength) {
//				printf("\tFOUND ONE at offset %d\n", _silenceOffset);
				break;
			}
		}
		// If section gain rises suddenly above running gain, assume we are now
		// at the end of a (non-qualifying) silence, and reset the running gain.
		else if (sectionGain > runningGain) {
			if (noiseSectionCount > 0) {
//				printf("got two sections with level above running level -- reset\n");
				runningGain = sectionGain;
			}
			else {
				++noiseSectionCount;
			}
			_silenceOffset = -1;
			silenceFrameCount = 0;
		}
		// If section gain is simply above the threshold, reset if we were doing
		// a silence test, else just update the running gain.
		else {
			if (_silenceOffset != -1) {
//				printf("\tSilence ended at %d (needed %d) -- resetting\n",
//					   silenceFrameCount, _silenceLength);
				_silenceOffset = -1;
				silenceFrameCount = 0;
			}
			else
				runningGain = (runningGain * 0.99) + (sectionGain * 0.01);
		}
	}
	Resource::unref(dbGains);
	return (_silenceOffset >= 0);
}
