//
// cppopt.cpp
// C++ Argument Parsing
//
// Copyright 2001 Zed C. Pobre
// Licensed to the public under the terms of the GNU GPL
//
// See cppopt.hpp for notes
//

#include <iostream>
#include <map>
#include <string>
#include <sstream>
#include <vector>
#include "cppopt.h"

namespace cppopt
{
    using namespace std;

    // ===================================================================== //
    //
    // Functions, constructor, destructor of the class Options
    //

    Options::Options()
    {
	clear();
	return;
    }
    Options::~Options()
    {}

    void Options::clear()
    {
	preopt.erase(preopt.begin(),preopt.end());
	shortopt.erase(shortopt.begin(),shortopt.end());
	longopt.erase(longopt.begin(),longopt.end());
	postopt.erase();
	return;
    }

    bool Options::empty()
    {
	if(!preopt.empty()) return false;
	if(!shortopt.empty()) return false;
	if(!longopt.empty()) return false;
	if(!target.empty()) return false;
	if(!postopt.empty()) return false;
	return true;
    }

    bool Options::haspreopt() const
    {
	return !preopt.empty();
    }
    bool Options::haspreopt(string opt)
    {
	vector<string>::iterator iter;
	bool retval = false;
	for(iter = preopt.begin(); iter != preopt.end(); iter++)
	{
	    if(*iter == opt) retval = true;
	}
	return retval;
    }
    
    bool Options::hasshortopt()
    {
	return !shortopt.empty();
    }
    bool Options::hasshortopt(char opt)
    {
	map<char,string>::iterator iter;
	bool retval = false;
	for(iter = shortopt.begin(); iter != shortopt.end(); iter++)
	{
	    if(iter->first == opt) retval = true;
	}
	return retval;
    }

    bool Options::haslongopt()
    {
	return !longopt.empty();
    }
    bool Options::haslongopt(string opt)
    {
	map<string,string>::iterator iter;
	bool retval = false;
	for(iter = longopt.begin(); iter != longopt.end(); iter++)
	{
	    if(iter->first == opt) retval = true;
	}
	return retval;
    }

    bool Options::hastarget() const
    {
	return !target.empty();
    }
    bool Options::hastarget(string opt)
    {
	vector<string>::iterator iter;
	bool retval = false;
	for(iter = target.begin(); iter != target.end(); iter++)
	{
	    if(*iter == opt) retval = true;
	}
	return retval;
    }

    bool Options::haspostopt() const
    {
	return !postopt.empty();
    }

    void Options::listall()
    {

	listpreopts();
	listshort();
	listlong();
	listtargets();

	if(!postopt.empty()) cout << "--\n\"" << postopt << "\"" << endl << endl;

	return;
    }

    void Options::listpreopts()
    {
	vector<string>::iterator iter;
	cout << "Pre-Options:" << endl;
	for(iter = preopt.begin(); iter != preopt.end(); iter++)
	{
	    cout << "  \"" << *iter << "\"" << endl;
	}
	cout << endl;
	return;
    }

    void Options::listshort()
    {
	map<char,string>::iterator iter;

	cout << "Short Options:" << endl;
	for(iter = shortopt.begin(); iter != shortopt.end(); iter++)
	{
	    cout << "  \"" << iter->first << "\"";
	    cout << " = \"" << iter->second << "\"" << endl;
	}

	cout << endl;
	return;
    }

    void Options::listlong()
    {
	map<string,string>::iterator iter;

	cout << "Long Options:" << endl;
	for(iter = longopt.begin(); iter != longopt.end(); iter++)
	{
	    cout << "  \"" << iter->first << "\"";
	    cout << " = \"" << iter->second << "\"" << endl;
	}

	cout << endl;
	return;
    }

    void Options::listtargets()
    {
	vector<string>::iterator iter;
	cout << "Targets:" << endl;
	for(iter = target.begin(); iter != target.end(); iter++)
	{
	    cout << "  \"" << *iter << "\"" << endl;
	}
	cout << endl;
	return;
    }


    // ===================================================================== //
    //
    // Global functions
    //

    string stripwhitespace(string unstripped)
    {
	string::size_type index1,index2;

	index1 = unstripped.find_first_not_of(" \t");
	if(index1 == string::npos)
	{
	    unstripped.erase();
	    return unstripped;
	}
	index2 = unstripped.find_last_not_of(" \t");

	return unstripped.substr(index1,index2-index1+1);
    }

    bool isopt(string optstring)
    {
	if(optstring.size() < 1) return false;
	if(optstring[0] == '-') return true;
	return false;
    }

    bool islongopt(string optstring)
    {
	if(optstring.size() < 2) return false;
	if((optstring[0] == '-') && (optstring[1] == '-')) return true;
	return false;
    }

    vector<string> tokenize(string tokenstring)
    {
	vector<string> retval;

	string token;
	istringstream tokenstream;

	tokenstream.str(stripwhitespace(tokenstring));

	while(tokenstream >> token)
	{
	    retval.push_back(token);
	}

// An alternate implementation that doesn't use streams, for people
// who don't have compilers that deal with stringstream
//
//	string::size_type index1, index2;
//
//	index1 = optstring.find_first_not_of(" \t",0);
//	index2 = optstring.find_first_of(" \t",index1);
//
//	while(index1 != string::npos)
//	{
//	    if(index2 == string::npos)
//	    {
//		cout << "DEBUG: last token = " << optstring.substr(index1) << endl;
//		optvector.push_back(optstring.substr(index1));
//	    }
//	    else
//	    {
//		cout << "DEBUG: token = " << optstring.substr(index1, index2-index1) << endl;
//		optvector.push_back(optstring.substr(index1, index2-index1));
//	    }
//	    index1 = optstring.find_first_not_of(" \t",index2);
//	    index2 = optstring.find_first_of(" \t",index1);
//	}

	return retval;
    }

    Options parse(string optstring, ParseMode mode)
    {
	vector<string> optvector;
	optvector = tokenize(optstring); 
	return parse(optvector,mode);
    }

    // This is the old autoparser, which is being left in now just for
    // reference since it is better handled by tokenizing the string
    // and passing the token vector to the vector parser.
    //
    Options parsestringold(string optstring, ParseMode mode)
    {
	string::size_type optindex1,optindex2;
	string tempstring;
	char shortopt;
	string longopt;
	string arg;
	Options retval;

	optstring = stripwhitespace((const string) optstring);

	if(optstring.empty()) return retval;

	// Check for (and handle) preopts, removing them from optstring
	retval.preopt = parse_preopt(optstring);

	while(!optstring.empty())
	{
	    if(optstring[0] == '-')
	    {
		if(optstring[1] == '-')
		{
		    // We have a "--" as an option.  Check to see if
		    // any characters follow it.  If none do and
		    // postopts are enabled, ignore it and return
		    // everything that has been parsed so far.  If
		    // postopts are disabled, set a null longopt and
		    // then return.
		    //
		    if(optstring.size() < 3)
		    {
			if(mode == NOPOSTOPT) retval.longopt[""] = "";
			return retval;
		    }
		    if( (optstring[2] == ' ') || (optstring[2] == '\t') )
		    {
			// "-- " has been found, indicating a postopt,
			// if postopts have not been disabled.  If
			// they have not been disabled, set the
			// postopt to the remaining string and return
			// what has been parsed so far.  If they have
			// been disabled, attach any following value
			// to a null longopt.
			//
			
			if(mode != NOPOSTOPT)
			{
			    retval.postopt = optstring.substr(3);
			    return retval;
			}

			// Remove the "-- " token, strip whitespace,
			// and check to see what's left.
			//
			optstring = stripwhitespace(optstring.substr(3));
			if(optstring.empty()) return retval;
			if(optstring[0] == '-') continue;
			
			optindex1 = optstring.find_first_of(" \t");
			if(optindex1 == string::npos)
			{
			    retval.longopt[""] = optstring;
			    return retval;
			}
			retval.longopt[""] = optstring.substr(0,optindex1);
			optstring = stripwhitespace(optstring.substr(optindex1));
			continue;
		    }
		    
		    optindex1 = optstring.find_first_of(" \t=",3);
		    optindex2 = optstring.find_first_not_of(" \t=",optindex1);
		    longopt = optstring.substr(2,optindex1-2);
		    
		    if(optstring[optindex1] != '=')
		    {
			if(optindex2 == string::npos)
			{
			    retval.longopt[longopt] = "";
			    return retval;
			}
			else if(optstring[optindex2] == '-')
			{
			    retval.longopt[longopt]="";
			    optstring = optstring.substr(optindex2);
			    continue;
			}
			else
			{
//			    cout << "DEBUG: parse: --" << longopt << " with arg" << endl;
			    optindex1 = optstring.find_first_of(" \t",optindex2);
			    if(optindex1 == string::npos)
			    {
				arg = optstring.substr(optindex2);
				retval.longopt[longopt] = arg;
				return retval;
			    }
			    else 
			    {
				arg = optstring.substr(optindex2,optindex1-optindex2);
				retval.longopt[longopt] = arg;
				optstring = stripwhitespace(optstring.substr(optindex1));
//				cout << "DEBUG: remaining optstring = \"" << optstring << "\"" << endl;
				continue;
			    }
			}
		    }
		    else
		    {
//			cout << "DEBUG: parse: --longopt=" << endl;
			if(optindex2 == string::npos)
			{
//			    cout << "DEBUG: parse: --longopt=[NULL][EOL]" << endl;
			    retval.longopt[longopt]="";
			    return retval;
			}
			optindex1 = optstring.find_first_of(" \t",optindex2);
			if(optindex1 == string::npos)
			{
			    arg = optstring.substr(optindex2);
			    retval.longopt[longopt]=arg;
			    return retval;
			}
			else
			{
			    arg = optstring.substr(optindex2,optindex1-optindex2);
			    retval.longopt[longopt]=arg;
			    optstring = stripwhitespace(optstring.substr(optindex1));
//			    cout << "DEBUG: remaining optstring = \"" << optstring << "\"" << endl;
			    continue;
			}
		    } // optstring[optindex1] == '='
		} // optstring[0] == '-' && optstring[1] == '-'
		
		if(optstring.size() < 2)
		{
		    retval.shortopt['-'] = "";
		    return retval;
		}		
		if( (optstring[1] == ' ') || (optstring[1] == '\t') )
		{
		    retval.shortopt['-'] = "";
		    optstring = stripwhitespace(optstring.substr(2));
//		    cout << "DEBUG: remaining optstring = \"" << optstring << "\"" << endl;
		    continue;
		}
		shortopt = optstring[1];
		if(optstring[2] == '=')
		{
//		    cout << "DEBUG: -" << shortopt << "=" << endl;
		    optindex1 = optstring.find_first_of(" \t");
		    if(optindex1 == string::npos)
		    {
			retval.shortopt[shortopt] = optstring.substr(3);
			return retval;
		    }
		    else
		    {
//			cout << "DEBUG: -" << shortopt << " = \"" << optstring.substr(3) << "\"" << endl;
			retval.shortopt[shortopt] = optstring.substr(3,optindex1-3);
			optstring = stripwhitespace(optstring.substr(optindex1));
			continue;
		    }
		}
		
		if( (optstring[2] == ' ') || (optstring[2] == '\t') )
		{
		    optstring = stripwhitespace(optstring.substr(3));
		    if(optstring[0] == '-')
		    {
			retval.shortopt[shortopt] = "";
			continue;
		    }
		    optindex1 = optstring.find_first_of(" \t");
		    if(optindex1 == string::npos)
		    {
			retval.shortopt[shortopt] = optstring;
			return retval;
		    }
		    else
		    {
			retval.shortopt[shortopt] = optstring.substr(0,optindex1);
			optstring = stripwhitespace(optstring.substr(optindex1));
			continue;
		    }
		}
		
		optindex1 = optstring.find_first_of(" \t");
		if(optindex1 == string::npos)
		{
		    retval.shortopt[shortopt] = optstring.substr(2);
		    return retval;
		}
		else
		{
		    retval.shortopt[shortopt] = optstring.substr(2,optindex1-2);
		    optstring = stripwhitespace(optstring.substr(optindex1));
		    continue;
		}
	    } // optstring[0] == '-'
	    
	    optindex1 = optstring.find_first_of(" \t");
	    if(optindex1 == string::npos)
	    {
		retval.target.push_back(optstring);
		return retval;
	    }
	    else
	    {
		retval.target.push_back(optstring.substr(0,optindex1));
		optstring = stripwhitespace(optstring.substr(optindex1));
//		cout << "DEBUG: target: optstring = \"" << optstring << "\"" << endl;
		continue;
	    }
	    
	    break;
	} // while(!optstring.empty())
	
	return retval;
    }


    Options parse(int argc, char **argv, ParseMode mode)
    {
	int ctr;
	string optstring;
	vector<string> optvector;
	Options retval;

	if(argc < 2)
	{
	    return retval;
	}
    
	for(ctr = 2; ctr < argc; ctr++)
	{
	    optstring = optstring + " " + argv[ctr];
	}

	for(ctr = 1; ctr < argc; ctr++)
	{
	    optvector.push_back(argv[ctr]);
	}

	return parse(optvector, mode);
    }

    
    Options parse(vector<string> optvector, ParseMode mode)
    {
	Options retval;
	vector<string>::iterator optiter;
	string optstring;
	char shortopt;
	string longopt, arg;
	string::size_type optindex1, optindex2;
	unsigned short int ctr;

	retval.preopt = parse_preopt(optvector);

	optiter = optvector.begin();
	
	while(optiter != optvector.end())
	{
	    optstring = *optiter;
	    
	    if(optstring.empty())
	    {
		optvector.erase(optiter);
	    }
	    else if(*optiter == "--")
	    {
		optvector.erase(optiter);
		if(mode != NOPOSTOPT)
		    retval.postopt = parse_postopt(optvector);
	    }
	    else if(islongopt(optstring)) // Long option
	    {
		optindex1 = optstring.find_first_of(" \t=",3);
		optindex2 = optstring.find_first_not_of(" \t=",optindex1);
		longopt = optstring.substr(2,optindex1-2);
		
		if(optindex2 == string::npos)
		{
		    if( ((optiter+1) != optvector.end()) && 
			!(isopt(*(optiter+1))) )
		    {
			retval.longopt[longopt] = *(optiter+1);
			optvector.erase(optiter+1);
		    }
		    else
		    {
			retval.longopt[longopt] = "";
		    }
		}
		else
		{
		    arg = optstring.substr(optindex2);
		    retval.longopt[longopt] = arg;
		}
		optvector.erase(optiter);
	    }	
	    else if(isopt(optstring)) // Short option
	    {
		if(optstring.size() < 2) // Null shortopt
		{
		    retval.shortopt['-'] = "";
		}
		else if(optstring.size() == 2) // Standalone shortopt
		{
		    shortopt = optstring[1];
		    if( ((optiter+1) != optvector.end()) && 
			!(isopt(*(optiter+1))) )
		    {
			retval.shortopt[shortopt] = *(optiter+1);
			optvector.erase(optiter+1);
		    }
		}
		else
		{
		    // Strip out the dash
		    optstring = optstring.substr(1);

		    // What's left at this point is a string of short
		    // options, the last of which might take an
		    // argument.  Cycle though all but the last and
		    // put them into optionless short arguments.
		    
		    optindex1 = optstring.find_first_of(" \t=",0);
		    optindex2 = optstring.find_first_not_of(" \t=",optindex1);
		    if(optindex1 == string::npos) optindex1 = optstring.size();

		    for(ctr = 0; ctr < (optindex1-1); ctr++)
		    {
			shortopt = optstring[ctr];
			retval.shortopt[shortopt] = "";
		    }
		    shortopt = optstring[(optindex1-1)];
		    if(optindex2 == string::npos)
		    {
			if( ((optiter+1) != optvector.end()) && 
			    !(isopt(*(optiter+1))) )
			{
			    retval.shortopt[shortopt] = *(optiter+1);
			    optvector.erase(optiter+1);
			}
			else
			{
			    retval.shortopt[shortopt] = "";
			}
		    }
		    else
		    {
			arg = optstring.substr(optindex2);
			retval.shortopt[shortopt] = arg;
		    }
		}
		optvector.erase(optiter);
	    }
	    else  // Next element is a target
	    {
		retval.target.push_back(*optiter);
		optvector.erase(optiter);
	    }
	} // while(optiter != optvector.end())
	return retval;
    }


    // parse_preopt(string &optstring)
    //
    // Takes an option string that will be modified to remove any
    // preopts found, and returns a vector<string> containing those
    // preopts.  If nothing is found, returns an empty string.
    //
    vector<string> parse_preopt(string &optstring)
    {
	vector<string> retval;
	string::size_type optindex1,optindex2;
	string tempstring;

	if(optstring.empty()) return retval;

	if(optstring[0] != '-')
	{
	    // There is a preopt.  Check to see if there are any
	    // switches at all, and move only the preopts into a
	    // temporary string for special processing.
	    //
	    optindex1 = optstring.find(" -");
	    if(optindex1 == string::npos)
	    {
		// No switches found.  The entire string can be moved,
		// and the optstring can be cleared to break the loop.
		//
		tempstring = optstring;
		optstring.erase();
	    }
	    else
	    {
		// Switches were found.  Move all of the preopts in
		// one lump to tempstring for special processing, and
		// remove them from optstring, leaving optstring[0] at
		// the first '-'
		// 
		tempstring = optstring.substr(0,optindex1);
		optstring = optstring.substr(optindex1+1);
	    }
	    
	    optindex1 = 0;
	    while(optindex1 != string::npos)
	    {
		// Parse the preopt tempstring
		//
		optindex2 = tempstring.find_first_of(" \t",optindex1);
		if(optindex2 == string::npos)
		{
		    retval.push_back(tempstring.substr(optindex1));
		    optindex1 = optindex2;
		}
		else
		{
		    retval.push_back(tempstring.substr(optindex1,optindex2-optindex1));
		    optindex1 = tempstring.find_first_not_of(" \t",optindex2);
		}
	    }
	    
	    if(optstring.empty()) return retval;
	    
	} // if(optstring[0] != '-')
	// The element was an option, so there were no preopts.
	return retval;
    } // parse_preopt(string &optstring)


    // parse_preopt(vector<string> &optvector)
    //
    // Takes an option vector that will be modified to remove any
    // preopts found, and returns a vector<string> containing those
    // preopts.  If nothing is found, returns an empty string vector.
    //
    vector<string> parse_preopt(vector<string> &optvector)
    {
	vector<string> retval;
	vector<string>::iterator iter;
	string tempstring;

	if(optvector.empty()) return retval;
	
	iter = optvector.begin();
	tempstring = *iter;
	while(tempstring[0] != '-')
	{
	    retval.push_back(*iter);
	    optvector.erase(iter);
	    if(iter == optvector.end()) break;
	    tempstring = *iter;
	}
	return retval;
    } // parse_preopt(vector<string> &optvector)


    // parse_postopt(vector<string> &optvector)
    //
    // Takes an option vector that has been checked to contain only
    // postopts, concatenates those postopts separated by spaces, and
    // returns the resulting string as the postopt string.  The vector
    // is erased in this process.
    //
    string parse_postopt(vector<string> &optvector)
    {
	string retval;
	vector<string>::iterator iter;

	if(optvector.empty()) return retval;

	iter = optvector.begin();
	retval = *iter;
	optvector.erase(iter);
	while(iter != optvector.end())
	{
	    retval += " ";
	    retval += *iter;
	    optvector.erase(iter);
	}
	return retval;
    } // parse_postopt(vector<string> &optvector)

    
} // namespace cppopt
