/*-------------------------------------------------------------------------
DFA.java -- Generation of the Scanner Automaton
Compiler Generator Coco/R,
Copyright (c) 1990, 2004 Hanspeter Moessenboeck, University of Linz
extended by M. Loeberbauer & A. Woess, Univ. of Linz
ported from C# to Java by Wolfgang Ahorner
with improvements by Pat Terry, Rhodes University

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

As an exception, it is allowed to write an extension of Coco/R that is
used as a plugin in non-free software.

If not otherwise stated, any source code generated by Coco/R (other than
Coco/R itself) does not fall under the GNU General Public License.
------------------------------------------------------------------------*/

package Coco;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.Reader;                /* pdt */
import java.io.BufferedReader;        /* pdt */
import java.io.FileReader;            /* pdt */
import java.io.IOException;
import java.io.PrintWriter;           /* pdt */
import java.io.BufferedWriter;        /* pdt */
import java.io.FileWriter;            /* pdt */
import java.util.BitSet;
import java.util.Map;

//-----------------------------------------------------------------------------
//  State
//-----------------------------------------------------------------------------

class State {               // state of finite automaton
  public static int lastNr; // highest state number
  public int nr;            // state number
  public Action firstAction;// to first action of this state
  public Symbol endOf;      // recognized token if state is final
  public boolean ctx;       // true if state is reached via contextTrans
  public State next;

  public State() {
    nr = ++lastNr;
  }

  public void AddAction(Action act) {
    Action lasta = null, a = firstAction;
    while (a != null && act.typ >= a.typ) {lasta = a; a = a.next;}
    // collecting classes at the beginning gives better performance
    act.next = a;
    if (a==firstAction) firstAction = act; else lasta.next = act;
  }

  public void DetachAction(Action act) {
    Action lasta = null, a = firstAction;
    while (a != null && a != act) {lasta = a; a = a.next;}
    if (a != null)
      if (a == firstAction) firstAction = a.next; else lasta.next = a.next;
  }

  public Action TheAction(char ch) {
    BitSet s;
    for (Action a = firstAction; a != null; a = a.next)
      if (a.typ == Node.chr && ch == a.sym) return a;
      else if (a.typ == Node.clas) {
        s = CharClass.Set(a.sym);
        if (s.get(ch)) return a;
      }
    return null;
  }

  public void MeltWith(State s) { // copy actions of s to state
    Action a;
    for (Action action = s.firstAction; action != null; action = action.next) {
      a = new Action(action.typ, action.sym, action.tc);
      a.AddTargets(action);
      AddAction(a);
    }
  }

}

//-----------------------------------------------------------------------------
//  Action
//-----------------------------------------------------------------------------

class Action {      // action of finite automaton
  public int typ;         // type of action symbol: clas, chr
  public int sym;         // action symbol
  public int tc;          // transition code: normalTrans, contextTrans
  public Target target;   // states reached from this action
  public Action next;

  public Action(int typ, int sym, int tc) {
    this.typ = typ; this.sym = sym; this.tc = tc;
  }

  public void AddTarget(Target t) { // add t to the action.targets
    Target last = null;
    Target p = target;
    while (p != null && t.state.nr >= p.state.nr) {
      if (t.state == p.state) return;
      last = p; p = p.next;
    }
    t.next = p;
    if (p == target) target = t; else last.next = t;
  }

  public void AddTargets(Action a) { // add copy of a.targets to action.targets
    for (Target p = a.target; p != null; p = p.next) {
      Target t = new Target(p.state);
      AddTarget(t);
    }
    if (a.tc == Node.contextTrans) tc = Node.contextTrans;
  }

  public BitSet Symbols() {
    BitSet s;
    if (typ == Node.clas)
      s = (BitSet) CharClass.Set(sym).clone();
    else {
      s = new BitSet(CharClass.charSetSize); s.set(sym);
    }
    return s;
  }

  public void ShiftWith(BitSet s) {
    if (Sets.Elements(s) == 1) {
      typ = Node.chr; sym = Sets.First(s);
    } else {
      CharClass c = CharClass.Find(s);
      if (c == null) c = new CharClass("#", s); // class with dummy name
      typ = Node.clas; sym = c.n;
    }
  }

  //public void GetTargetStates(out BitArray targets, out Symbol endOf, out bool ctx) {
  public boolean GetTargetStates(Object[] param) {
    // compute the set of target states
    BitSet targets = new BitSet(DFA.maxStates);
    Symbol endOf = null;
    boolean ctx = false;
    for (Target t = target; t != null; t = t.next) {
      int stateNr = t.state.nr;
      if (stateNr <= DFA.lastSimState) targets.set(stateNr);
      else {
        try {
          targets.or(Melted.Set(stateNr));
        } catch (Exception e) {
            System.out.println(e.getMessage());
            Errors.count++;
        }
      }
      if (t.state.endOf != null)
        if (endOf == null || endOf == t.state.endOf)
          endOf = t.state.endOf;
        else {
          System.out.println("Tokens " + endOf.name + " and " + t.state.endOf.name +
            " cannot be distinguished");
          Errors.count++;
        }
      if (t.state.ctx) {
        ctx = true;
        // The following check seems to be unnecessary. It reported an error
        // if a symbol + context was the prefix of another symbol, e.g.
        //   s1 = "a" "b" "c".
        //   s2 = "a" CONTEXT("b").
        // But this is ok.
        // if (t.state.endOf != null) {
        //   Console.WriteLine("Ambiguous context clause");
        //   Errors.count++;
        // }
      }
    }
    param[0] = targets;
    param[1] = endOf;
    return ctx;
  }

}

//-----------------------------------------------------------------------------
//  Target
//-----------------------------------------------------------------------------

class Target {          // set of states that are reached by an action
  public State state;   // target state
  public Target next;

  public Target (State s) {
    state = s;
  }
}

//-----------------------------------------------------------------------------
//  Melted
//-----------------------------------------------------------------------------

class Melted {          // info about melted states
  public static Melted first; // head of melted state list
  public BitSet set;          // set of old states
  public State state;         // new state
  public Melted next;

  public Melted(BitSet set, State state) {
    this.set = set; this.state = state;
    this.next = first; first = this;
  }

  public static BitSet Set(int nr) throws Exception {
    Melted m = first;
    while (m != null) {
      if (m.state.nr == nr) return m.set; else m = m.next;
    }
    throw new Exception("-- compiler error in Melted.Set");
  }

  public static Melted StateWithSet(BitSet s) {
    for (Melted m = first; m != null; m = m.next)
      if (Sets.Equals(s, m.set)) return m;
    return null;
  }

}

//-----------------------------------------------------------------------------
//  Comment
//-----------------------------------------------------------------------------

class Comment {         // info about comment syntax
  public static Comment first;  // list of comments
  public String start;
  public String stop;
  public boolean nested;
  public Comment next;

  static String Str(Node p) {
    StringBuffer s = new StringBuffer();
    while (p != null) {
      if (p.typ == Node.chr) {
        s.append((char)p.val);
      } else if (p.typ == Node.clas) {
        BitSet set = CharClass.Set(p.val);
        if (Sets.Elements(set) != 1) Parser.SemErr("character set contains more than 1 character");
        s.append((char)Sets.First(set));
      } else Parser.SemErr("comment delimiters must not be structured");
      p = p.next;
    }
    if (s.length() == 0 || s.length() > 2) {
      Parser.SemErr("comment delimiters must be 1 or 2 characters long");
      s = new StringBuffer("?");
    }
    return s.toString();
  }

  public Comment(Node from, Node to, boolean nested) {
    start = Str(from);
    stop = Str(to);
    this.nested = nested;
    this.next = first; first = this;
  }

}

//-----------------------------------------------------------------------------
//  DFA
//-----------------------------------------------------------------------------

public class DFA {
  public static int maxStates;
  public static final int EOF = -1;

  public static State firstState;
  public static State lastState;      // last allocated state
  public static int lastSimState;     // last non melted state
  public static Reader fram;          // scanner frame input     /* pdt */
  public static PrintWriter gen;      // generated scanner file  /* pdt */
  public static Symbol curSy;         // current token to be recognized (in FindTrans)
  public static Node curGraph;        // start of graph for current token (in FindTrans)
  public static boolean ignoreCase;   // true if input should be treated case-insensitively
  public static boolean dirtyDFA;     // DFA may become nondeterministic in MatchLiteral
  public static boolean hasCtxMoves;  // DFA has context transitions
  static String srcName;              // name of the attributed grammar file
  static String srcDir;               // directory of attributed grammar file

  // AH 20031017
  private static int framRead() {
    try {
      return fram.read();
    } catch (java.io.IOException e) {
      Errors.Exception("-- error reading Scanner.frame");
      return EOF;
    }
  }

  //---------- Output primitives
  private static String Ch(char ch) {
    if (ch < ' ' || ch >= 127 || ch == '\'' || ch == '\\') {
      return Integer.toString((int)ch);
    }
    else return "'" + ch + "'";
  }

  private static String ChCond(char ch) {
    return ("ch == " + Ch(ch));
  }

  private static void PutRange(BitSet s) {
    int[] lo = new int[32];
    int[] hi = new int[32];
    // fill lo and hi
    int max = CharClass.charSetSize;
    int top = -1;
    int i = 0;
    while (i < max) {
      if (s.get(i)) {
        top++; lo[top] = i; i++;
        while (i < max && s.get(i)) i++;
        hi[top] = i-1;
      } else i++;
    }
    // print ranges
    if (top == 1 && lo[0] == 0 && hi[1] == max-1 && hi[0]+2 == lo[1]) {
      BitSet s1 = new BitSet(max); s1.set(hi[0]+1);
      gen.print("!"); PutRange(s1);  gen.print(" && ch != Buffer.EOF");
    } else {
      gen.print("(");
      for (i = 0; i <= top; i++) {
        if (hi[i] == lo[i]) gen.print("ch == " + Ch((char)lo[i]));
        else if (lo[i] == 0) gen.print("ch <= " + Ch((char)hi[i]));
        else if (hi[i] == max-1) gen.print("ch >= " + Ch((char)lo[i]));
        else gen.print("ch >= " + Ch((char)lo[i]) + " && ch <= " + Ch((char)hi[i]));
        if (i < top) gen.print(" || ");
      }
      gen.print(")");
    }
  }

  //---------- String handling
  static char Hex2Char(String s) {
    int val = 0;
    for (int i = 0; i < s.length(); i++) {
      char ch = s.charAt(i);
      if ('0' <= ch && ch <= '9') val = 16 * val + (ch - '0');
      else if ('a' <= ch && ch <= 'f') val = 16 * val + (10 + ch - 'a');
      else if ('A' <= ch && ch <= 'F') val = 16 * val + (10 + ch - 'A');
      else Parser.SemErr("bad escape sequence in string or character");
    }
    if (val > CharClass.charSetSize) /* pdt */
      Parser.SemErr("bad escape sequence in string or character");
    return (char) (val % CharClass.charSetSize);
  }

  static String Char2Hex(char ch) {
    //StringWriter w = new StringWriter();
    //w.write("\\u{0:x4}", (int)ch);
    //return w.toString();
    String hex = Integer.toHexString((int)ch);
    for (int i = hex.length(); i < 4; i++) hex = '0' + hex;
    return "\\u" + hex;
  }

  public static String Unescape (String s) {
    /* replaces escape sequences in s by their Unicode values. */
    StringBuffer buf = new StringBuffer();
    int i = 0;
    while (i < s.length()) {
      if (s.charAt(i) == '\\') {
        switch (s.charAt(i+1)) {
          case '\\': buf.append('\\'); i += 2; break;
          case '\'': buf.append('\''); i += 2; break;
          case '\"': buf.append('\"'); i += 2; break;
          case 'r': buf.append('\r'); i += 2; break;
          case 'n': buf.append('\n'); i += 2; break;
          case 't': buf.append('\t'); i += 2; break;
          case '0': buf.append('\0'); i += 2; break;
          case 'b': buf.append('\b'); i += 2; break;
          case 'f': buf.append('\f'); i += 2; break;
          case 'u': case 'x':
            if (i + 6 <= s.length()) {
              buf.append(Hex2Char(s.substring(i+2, i+6))); i += 6; break;
            } else {
              Parser.SemErr("bad escape sequence in string or character");
              i = s.length(); break;
            }
          default: Parser.SemErr("bad escape sequence in string or character"); i += 2; break;
        }
      } else {
        buf.append(s.charAt(i));
        i++;
      }
    }
    return buf.toString();
  }

  public static String Escape (String s) {
    StringBuffer buf = new StringBuffer();
    char ch;
    for (int i = 0; i < s.length(); i++) {
      ch = s.charAt(i);
      switch(ch) {
        case '\\': buf.append("\\\\"); break;
        case '\'': buf.append("\\'"); break;
        case '\"': buf.append("\\\""); break;
        case '\t': buf.append("\\t"); break;
        case '\r': buf.append("\\r"); break;
        case '\n': buf.append("\\n"); break;
        default:
          if (ch < ' ' || ch > '\u007f') buf.append(Char2Hex(ch));
          else buf.append(ch);
          break;
      }
    }
    return buf.toString();
  }

  //---------- State handling
  static State NewState() {
    State s = new State();
    if (firstState == null) firstState = s; else lastState.next = s;
    lastState = s;
    return s;
  }

  static void NewTransition(State from, State to, int typ, int sym, int tc) {
    if (to == firstState) Parser.SemErr("token must not start with an iteration");
    Target t = new Target(to);
    Action a = new Action(typ, sym, tc); a.target = t;
    from.AddAction(a);
    if (typ == Node.clas) curSy.tokenKind = Symbol.classToken;
  }

  static void CombineShifts() {
    State state;
    Action a, b, c;
    BitSet seta, setb;
    for (state = firstState; state != null; state = state.next) {
      for (a = state.firstAction; a != null; a = a.next) {
        b = a.next;
        while (b != null)
          if (a.target.state == b.target.state && a.tc == b.tc) {
            seta = a.Symbols(); setb = b.Symbols();
            seta.or(setb);
            a.ShiftWith(seta);
            c = b; b = b.next; state.DetachAction(c);
          } else b = b.next;
      }
    }
  }

  static void FindUsedStates(State state, BitSet used) {
    if (used.get(state.nr)) return;
    used.set(state.nr);
    for (Action a = state.firstAction; a != null; a = a.next)
      FindUsedStates(a.target.state, used);
  }

  static void DeleteRedundantStates() {
    State[] newState = new State[State.lastNr + 1];
    BitSet used = new BitSet(State.lastNr + 1);
    FindUsedStates(firstState, used);
    // combine equal final states
    for (State s1 = firstState.next; s1 != null; s1 = s1.next) // firstState cannot be final
      if (used.get(s1.nr) && s1.endOf != null && s1.firstAction == null && !s1.ctx)
        for (State s2 = s1.next; s2 != null; s2 = s2.next)
          if (used.get(s2.nr) && s1.endOf == s2.endOf && s2.firstAction == null & !s2.ctx) {
            used.set(s2.nr, false); newState[s2.nr] = s1;
          }
    for (State state = firstState; state != null; state = state.next)
      if (used.get(state.nr))
        for (Action a = state.firstAction; a != null; a = a.next)
          if (!used.get(a.target.state.nr))
            a.target.state = newState[a.target.state.nr];
    // delete unused states
    lastState = firstState; State.lastNr = 0; // firstState has number 0
    for (State state = firstState.next; state != null; state = state.next)
      if (used.get(state.nr)) {state.nr = ++State.lastNr; lastState = state;}
      else lastState.next = state.next;
  }

  static State TheState(Node p) {
    State state;
    if (p == null) {state = NewState(); state.endOf = curSy; return state;}
    else return p.state;
  }

  static void Step(State from, Node p, BitSet stepped) {
    if (p == null) return;
    stepped.set(p.n);
    switch (p.typ) {
      case Node.clas: case Node.chr: {
        NewTransition(from, TheState(p.next), p.typ, p.val, p.code);
        break;
      }
      case Node.alt: {
        Step(from, p.sub, stepped); Step(from, p.down, stepped);
        break;
      }
      case Node.iter: case Node.opt: {
        if (p.next != null && !stepped.get(p.next.n)) Step(from, p.next, stepped);
        Step(from, p.sub, stepped);
        break;
      }
    }
  }

  static void NumberNodes(Node p, State state) {
    /* Assigns a state n.state to every node n. There will be a transition from
       n.state to n.next.state triggered by n.val. All nodes in an alternative
       chain are represented by the same state.
    */
    if (p == null) return;
    if (p.state != null) return; // already visited;
    if (state == null) state = NewState();
    p.state = state;
    if (Node.DelGraph(p)) state.endOf = curSy;
    switch (p.typ) {
      case Node.clas: case Node.chr: {
        NumberNodes(p.next, null);
        break;
      }
      case Node.opt: {
        NumberNodes(p.next, null); NumberNodes(p.sub, state);
        break;
      }
      case Node.iter: {
        NumberNodes(p.next, state); NumberNodes(p.sub, state);
        break;
      }
      case Node.alt: {
        NumberNodes(p.sub, state); NumberNodes(p.down, state);
        break;
      }
    }
  }

  static void FindTrans (Node p, boolean start, BitSet marked) {
    if (p == null || marked.get(p.n)) return;
    marked.set(p.n);
    if (start) Step(p.state, p, new BitSet(Node.nodes.size())); // start of group of equally numbered nodes
    switch (p.typ) {
      case Node.clas: case Node.chr: {
        FindTrans(p.next, true, marked);
        break;
      }
      case Node.opt: {
        FindTrans(p.next, true, marked); FindTrans(p.sub, false, marked);
        break;
      }
      case Node.iter: {
        FindTrans(p.next, false, marked); FindTrans(p.sub, false, marked);
        break;
      }
      case Node.alt: {
        FindTrans(p.sub, false, marked); FindTrans(p.down, false, marked);
        break;
      }
    }
  }

  public static void ConvertToStates(Node p, Symbol sym) {
    curGraph = p; curSy = sym;
    if (Node.DelGraph(curGraph)) Parser.SemErr("token might be empty");
    NumberNodes(curGraph, firstState);
    FindTrans(curGraph, true, new BitSet(Node.nodes.size()));
  }

  // match string against current automaton; store it either as a fixedToken or as a litToken
  public static void MatchLiteral(String s, Symbol sym) {
    s = Unescape(s.substring(1, s.length()-1));
    int i, len = s.length();
    State state = firstState;
    Action a = null;
    for (i = 0; i < len; i++) { // try to match s against existing DFA
      a = state.TheAction(s.charAt(i));
      if (a == null) break;
      state = a.target.state;
    }
    // if s was not totally consumed or leads to a non-final state => make new DFA from it
    if (i != len || state.endOf == null) {
      state = firstState; i = 0; a = null;
      dirtyDFA = true;
    }
    for (; i < len; i++) { // make new DFA for s[i..len-1]
      State to = NewState();
      NewTransition(state, to, Node.chr, s.charAt(i), Node.normalTrans);
      state = to;
    }
    Symbol matchedSym = state.endOf;
    if (state.endOf == null) {
      state.endOf = sym;
    } else if (matchedSym.tokenKind == Symbol.fixedToken || (a != null && a.tc == Node.contextTrans)) {
      // s matched a token with a fixed definition or a token with an appendix that will be cut off
      Parser.SemErr("tokens " + sym.name + " and " + matchedSym.name + " cannot be distinguished");
    } else { // matchedSym == classToken || classLitToken
      matchedSym.tokenKind = Symbol.classLitToken;
      sym.tokenKind = Symbol.litToken;
    }
  }

  static void SplitActions(State state, Action a, Action b) {
    Action c; BitSet seta, setb, setc;
    seta = a.Symbols(); setb = b.Symbols();
    if (Sets.Equals(seta, setb)) {
      a.AddTargets(b);
      state.DetachAction(b);
    } else if (Sets.Includes(seta, setb)) {
      setc = (BitSet)seta.clone(); Sets.Subtract(setc, setb);
      b.AddTargets(a);
      a.ShiftWith(setc);
    } else if (Sets.Includes(setb, seta)) {
      setc = (BitSet)setb.clone(); Sets.Subtract(setc, seta);
      a.AddTargets(b);
      b.ShiftWith(setc);
    } else {
      setc = (BitSet)seta.clone(); setc.and(setb);
      Sets.Subtract(seta, setc);
      Sets.Subtract(setb, setc);
      a.ShiftWith(seta);
      b.ShiftWith(setb);
      c = new Action(0, 0, Node.normalTrans);  // typ and sym are set in ShiftWith
      c.AddTargets(a);
      c.AddTargets(b);
      c.ShiftWith(setc);
      state.AddAction(c);
    }
  }

  private static boolean Overlap(Action a, Action b) {
    BitSet seta, setb;
    if (a.typ == Node.chr)
      if (b.typ == Node.chr) return a.sym == b.sym;
      else {setb = CharClass.Set(b.sym); return setb.get(a.sym);}
    else {
      seta = CharClass.Set(a.sym);
      if (b.typ ==Node.chr) return seta.get(b.sym);
      else {setb = CharClass.Set(b.sym); return Sets.Intersect(seta, setb);}
    }
  }

  static boolean MakeUnique(State state) { // return true if actions were split
    boolean changed = false;
    for (Action a = state.firstAction; a != null; a = a.next)
      for (Action b = a.next; b != null; b = b.next)
        if (Overlap(a, b)) {SplitActions(state, a, b); changed = true;}
    return changed;
  }

  static void MeltStates(State state) {
    boolean changed, ctx;
    BitSet targets;
    Symbol endOf;
    for (Action action = state.firstAction; action != null; action = action.next) {
      if (action.target.next != null) {
        //action.GetTargetStates(out targets, out endOf, out ctx);
        Object[] param = new Object[2];
        ctx = action.GetTargetStates(param);
        targets = (BitSet)param[0];
        endOf = (Symbol)param[1];
        //
        Melted melt = Melted.StateWithSet(targets);
        if (melt == null) {
          State s = NewState(); s.endOf = endOf; s.ctx = ctx;
          for (Target targ = action.target; targ != null; targ = targ.next)
            s.MeltWith(targ.state);
          do {changed = MakeUnique(s);} while (changed);
          melt = new Melted(targets, s);
        }
        action.target.next = null;
        action.target.state = melt.state;
      }
    }
  }

  static void FindCtxStates() {
    for (State state = firstState; state != null; state = state.next)
      for (Action a = state.firstAction; a != null; a = a.next)
        if (a.tc == Node.contextTrans) a.target.state.ctx = true;
  }

  public static void MakeDeterministic() {
    State state;
    boolean changed;
    lastSimState = lastState.nr;
    maxStates = 2 * lastSimState; // heuristic for set size in Melted.set
    FindCtxStates();
    for (state = firstState; state != null; state = state.next)
      do {changed = MakeUnique(state);} while (changed);
    for (state = firstState; state != null; state = state.next)
      MeltStates(state);
    DeleteRedundantStates();
    CombineShifts();
  }

  public static void PrintStates() {
    Trace.WriteLine();
    Trace.WriteLine("---------- states ----------");
    for (State state = firstState; state != null; state = state.next) {
      boolean first = true;
      if (state.endOf == null) Trace.Write("               ");
      else Trace.Write("E(" + Node.Name(state.endOf.name) + ")", 12);
      Trace.Write(state.nr + ":", 3);
      if (state.firstAction == null) Trace.WriteLine();
      for (Action action = state.firstAction; action != null; action = action.next) {
        if (first) {Trace.Write(" "); first = false;} else Trace.Write("                   ");
        if (action.typ == Node.clas)
          Trace.Write(((CharClass)CharClass.classes.get(action.sym)).name);
        else Trace.Write(Ch((char)action.sym), 3);
        for (Target targ = action.target; targ != null; targ = targ.next)
          Trace.Write(Integer.toString(targ.state.nr), 3);
        if (action.tc == Node.contextTrans) Trace.WriteLine(" context"); else Trace.WriteLine();
      }
    }
    Trace.WriteLine();
    Trace.WriteLine("---------- character classes ----------");
    CharClass.WriteClasses();
  }

  static void GenComBody(Comment com) {
    gen.println("\t\t\tfor(;;) {");
    gen.print  ("\t\t\t\tif (" + ChCond(com.stop.charAt(0)) + ") "); gen.println("{");
    if (com.stop.length() == 1) {
      gen.println("\t\t\t\t\tlevel--;");
      gen.println("\t\t\t\t\tif (level == 0) { oldEols = line - line0; NextCh(); return true; }");
      gen.println("\t\t\t\t\tNextCh();");
    } else {
      gen.println("\t\t\t\t\tNextCh();");
      gen.println("\t\t\t\t\tif (" + ChCond(com.stop.charAt(1)) + ") {");
      gen.println("\t\t\t\t\t\tlevel--;");
      gen.println("\t\t\t\t\t\tif (level == 0) { oldEols = line - line0; NextCh(); return true; }");
      gen.println("\t\t\t\t\t\tNextCh();");
      gen.println("\t\t\t\t\t}");
    }
    if (com.nested) {
      gen.print  ("\t\t\t\t}"); gen.println(" else if (" + ChCond(com.start.charAt(0)) + ") {");
      if (com.start.length() == 1)
        gen.println("\t\t\t\t\tlevel++; NextCh();");
      else {
        gen.println("\t\t\t\t\tNextCh();");
        gen.print  ("\t\t\t\t\tif (" + ChCond(com.start.charAt(1)) + ") "); gen.println("{");
        gen.println("\t\t\t\t\t\tlevel++; NextCh();");
        gen.println("\t\t\t\t\t}");
      }
    }
    gen.println(    "\t\t\t\t} else if (ch == Buffer.EOF) return false;");
    gen.println(    "\t\t\t\telse NextCh();");
    gen.println(    "\t\t\t}");
  }

  static void GenComment(Comment com, int i) {
    gen.println();
    gen.print  ("\tstatic boolean Comment" + i + "() "); gen.println("{");
    gen.println("\t\tint level = 1, line0 = line, lineStart0 = lineStart;");
    if (com.start.length() == 1) {
      gen.println("\t\tNextCh();");
      GenComBody(com);
    } else {
      gen.println("\t\tNextCh();");
      gen.print  ("\t\tif (" + ChCond(com.start.charAt(1)) + ") "); gen.println("{");
      gen.println("\t\t\tNextCh();");
      GenComBody(com);
      gen.println("\t\t} else {");
      gen.println("\t\t\tif (ch==EOL) {line--; lineStart = lineStart0;}");
      gen.println("\t\t\tpos = pos - 2; Buffer.setPos(pos+1); NextCh();");
      gen.println("\t\t}");
      gen.println("\t\treturn false;");
    }
    gen.println("\t}");
  }

  static void CopyFramePart(String stop) {
    char startCh = stop.charAt(0);
    int endOfStopString = stop.length() - 1;
    int ch = framRead();
    while (ch != EOF)
      if (ch == startCh) {
        int i = 0;
        do {
          if (i == endOfStopString) return; // stop[0..i] found
          ch = framRead(); i++;
        } while (ch == stop.charAt(i));
        // stop[0..i-1] found; continue with last read character
        gen.print(stop.substring(0, i));
      } else {
        gen.print((char)ch); ch = framRead();
      }
    Errors.Exception(" -- incomplete or corrupt scanner frame file");
  }

  static String SymName(Symbol sym) {
    if (Character.isLetter(sym.name.charAt(0))) { // real name value is stored in Tab.literals
      //foreach (DictionaryEntry e in Tab.literals)
      java.util.Set es = Tab.literals.entrySet();
      java.util.Iterator iter = es.iterator();
      Map.Entry me = null;
      while (iter.hasNext()) {
        me = (Map.Entry)iter.next();
        if ((Symbol)me.getValue() == sym) return (String)me.getKey();
      }
    }
    return sym.name;
  }

  static void GenLiterals () {
    gen.print("String lit = t.val");
    if (ignoreCase) {
      gen.print(".toLowerCase()");
    }
    gen.println(";"); gen.print("\t\t");
    boolean first = true;
    Symbol sym;
    for (int i = 0; i < Symbol.terminals.size(); i++) {
      sym = (Symbol)Symbol.terminals.get(i);
      if (sym.tokenKind == Symbol.litToken) {
        String name = SymName(sym);
        if (ignoreCase) name = name.toLowerCase();
        // sym.name stores literals with quotes, e.g. "\"Literal\"",
        if (!first) {
          gen.println(); gen.print("\t\telse ");
        } else first = false;
        gen.print("if (lit.compareTo(" + name + ") == 0) t.kind = " + sym.n + ";");
      }
    }
  }

  static void WriteState(State state) {
    Symbol endOf = state.endOf;
    gen.println("\t\t\t\tcase " + state.nr + ":");
    boolean ctxEnd = state.ctx;
    for (Action action = state.firstAction; action != null; action = action.next) {
      if (action == state.firstAction) gen.print("\t\t\t\t\tif (");
      else gen.print("\t\t\t\t\telse if (");
      if (action.typ == Node.chr) gen.print(ChCond((char)action.sym));
      else PutRange(CharClass.Set(action.sym));
      gen.print(") {");
      if (action.tc == Node.contextTrans) {
        gen.print("apx++; "); ctxEnd = false;
      } else if (state.ctx)
        gen.print("apx = 0; ");
      if (ignoreCase) gen.print("tval[tlen++] = valCh; "); else gen.print("tval[tlen++] = ch; ");
      gen.println("NextCh(); state = " + action.target.state.nr + "; break;}");
    }
    if (state.firstAction == null)
      gen.print("\t\t\t\t\t{");
    else
      gen.print("\t\t\t\t\telse {");
    if (ctxEnd) { // final context state: cut appendix
      gen.println();
      gen.println("\t\t\t\t\ttlen -= apx;");
      gen.println("\t\t\t\t\tpos = pos - apx - 1; line = t.line;");
      gen.println("\t\t\t\t\tBuffer.setPos(pos+1); NextCh();");
      gen.print  ("\t\t\t\t\t");
    }
    if (endOf == null) {
      gen.println("t.kind = noSym; done = true; break;}");
    } else {
      gen.print("t.kind = " + endOf.n + "; ");
      if (endOf.tokenKind == Symbol.classLitToken) {
        gen.println("t.val = new String(tval, 0, tlen); CheckLiteral(); return t;}");
      } else {
        gen.println("done = true; break;}");
      }
    }
  }

  static void FillStartTab(int[] startTab) {
    for (Action action = firstState.firstAction; action != null; action = action.next) {
      int targetState = action.target.state.nr;
      if (action.typ == Node.chr) startTab[action.sym] = targetState;
      else {
        BitSet s = CharClass.Set(action.sym);
        for (int i = 0; i < s.size(); i++)
          if (s.get(i)) startTab[i] = targetState;
      }
    }
  }

  static void OpenGen(boolean backUp) {
    try {
      String fn = srcDir + "Scanner.java";
      File f = new File(fn);
      if (backUp && f.exists()) {
        File old = new File(fn + ".old");
        old.delete(); f.renameTo(old);
      }
      gen = new PrintWriter(new BufferedWriter(new FileWriter(fn, false))); /* pdt */
    } catch (Exception e) {
      Errors.Exception("-- Cannot generate scanner file.");
    }
  }

  public static void WriteScanner() {
    int i, j;
    int[] startTab = new int[CharClass.charSetSize];
    String fr = srcDir + "Scanner.frame";
    if (!new File(fr).exists()) {
      if (Tab.frameDir != null) fr = Tab.frameDir.trim() + System.getProperty("file.separator") + "Scanner.frame";
      if (!new File(fr).exists()) Errors.Exception("-- Cannot find Scanner.frame");
    }
    try {
      fram = new BufferedReader(new FileReader(fr)); /* pdt */
    } catch (FileNotFoundException e) {
      Errors.Exception("-- Cannot open Scanner.frame.");
    }
    OpenGen(true);
    if (dirtyDFA) MakeDeterministic();
    FillStartTab(startTab);
    CopyFramePart("-->begin");
    if (!srcName.toLowerCase().endsWith("coco.atg")) {
      gen.close();
      OpenGen(false);
    }
    /* add package name, if it exists */
    if (Tab.nsName != null && Tab.nsName.length() > 0) {
      gen.print("package ");
      gen.print(Tab.nsName);
      gen.println(";");
    }
    CopyFramePart("-->declarations");
    gen.println("\tstatic final int charSetSize = " + CharClass.charSetSize + ";");
    gen.println("\tstatic final int maxT = " + (Symbol.terminals.size() - 1) + ";");
    gen.println("\tstatic final int noSym = " + Tab.noSym.n + ";");
    gen.println("\tstatic short[] start = {");
    for (i = 0; i < CharClass.charSetSize / 16; i++) {
      gen.print("\t");
      for (j = 0; j < 16; j++) {
        gen.print(Trace.formatString(Integer.toString(startTab[16*i+j]), 3));
        gen.print(",");
      }
      gen.println();
    }
    gen.println("\t  -1};");
    if (ignoreCase)
      gen.print("\tstatic char valCh;       // current input character (for token.val)");
    CopyFramePart("-->initialization");
    gen.print  ("\t\t");
    j = 0;
    for (i = 0; i < Tab.ignored.size(); i++) {
      if (Tab.ignored.get(i)) {
        gen.print("ignore.set(" + i + "); ");
        if (++j % 4 == 0) { gen.println(); gen.print("\t\t"); }
      }
    }
    CopyFramePart("-->casing");
    if (ignoreCase) {
      gen.println("\t\tvalCh = ch;");
      gen.print  ("\t\tif (ch != Buffer.EOF) ch = Character.toLowerCase(ch);");
    }
    CopyFramePart("-->comments");
    Comment com = Comment.first; i = 0;
    while (com != null) {
      GenComment(com, i);
      com = com.next; i++;
    }
    CopyFramePart("-->literals"); GenLiterals();
    CopyFramePart("-->scan1");
    if (Comment.first!=null) {
      gen.print("\t\tif (");
      com = Comment.first; i = 0;
      while (com != null) {
        gen.print(ChCond(com.start.charAt(0)));
        gen.print(" && Comment" + i + "()");
        if (com.next != null) gen.print(" ||");
        com = com.next; i++;
      }
      gen.print(") return NextToken();");
    }
    if (hasCtxMoves) { gen.println(); gen.print("\t\tint apx = 0;"); } /* pdt */
    CopyFramePart("-->scan2");
    if (ignoreCase) gen.print("\t\ttval[tlen++] = valCh; NextCh();");
    else gen.print("\t\ttval[tlen++] = ch; NextCh();");
    CopyFramePart("-->scan3");
    for (State state = firstState.next; state != null; state = state.next)
      WriteState(state);
    CopyFramePart("$$$");
    gen.close();
  }

  public static void Init (String file, String dir) {
    srcName = file;
    srcDir = dir;
    firstState = null; lastState = null; State.lastNr = -1;
    firstState = NewState();
    Melted.first = null; Comment.first = null;
    ignoreCase = false;
    dirtyDFA = false;
    hasCtxMoves = false;
  }

}
