#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>

#include "common.h"
#include "shmem.h"
#include "config.h"
#include "parseprg.h"
#include "prgcommon.h"

//-----------------------------------------------------------------------------

int main(int argc, char* argv[]) {
  char* ipcid=DEFAULT_DAEMONFILE;
  char* program=0;
  char* file=0;
  char readonly=0;

  for(;;) {
    static option long_options[]=
      {
       {"daemon",  1,0,'d'},
       {"eval",    1,0,'e'},
       {"file",    1,0,'f'},
       {"readonly",0,0,'r'}
      };
     
     int c=getopt_long(argc,argv,"d:e:f:r",long_options,0);
     if(c==EOF) break;
     
     switch(c) {
       case 'd':
         ipcid=optarg;
	 break;
       case 'e':
         program=optarg;
	 break;
       case 'f':
         file=optarg;
	 break;
       case 'r':
         readonly=1;
         break;
	 
       // Error situations:
       case ':':
         fprintf(stderr,"Missing parameter for option\n");
	 return 1;
       case '?':
         return 1; // The getopt library prints an error message
	 break;	 
       default:
         // Should not happen, I guess
	 fprintf(stderr,"Error parsing commandline arguments\n");
	 return 1;
       
     }
  }
  
  if(optind!=argc) {
    fprintf(stderr,"Unknown argument(s) given\n");
    return 1;
  }
  
  if(program && file) {
    fprintf(stderr,"Program can not be read from the command line and a file at the same time\n");
    return 1;
  }
  
  // Read program from stdin if not given on command line:
  DynString ds;
  if(!program) {
    FILE* f=stdin; 
    int c;
    if(file) {
      // A filename is given:
      f=fopen(file,"r");
      if(!f) {
        fprintf(stderr,"Could not open file: %s\n",file);
        return 1;
      }
    }
    while((c=getc(f))!=EOF) ds.add(c);
    program=ds.getString();
  }  
    
  // The parameters are now read.
  
  // Try to compile the program:
  ScriptExec se(readonly);
  char* c=se.setProgram(1,program);
  if(c) {
    fprintf(stderr,"%s\n",c);
    return 1;
  }
  
  // The program is compiled - try to fetch and lock the shared memory
  // with write access:
  ShMemClient clnt;
  c=clnt.init(ipcid,readonly);
  if(c) {
    fprintf(stderr,"%s\n",c);
    return 1;
  }
  
  ServerMem* ca=(ServerMem*)clnt.Lock();
  if(ca==0) {
    fprintf(stderr,"%s\n",LockErrString);
    return 1;
  }
  
  // Check that no wrong adressing mode is used:
  c=se.assert_cit(ca->cit);

  // And execute the program:
  if(!c) {
    Time t;
    gettimeofday(&t.tv,0);
    c=se.Consume(ca,t);
  }

  // Unlock the memory:
  clnt.UnLock(ca);  

  if(c) {    
    fprintf(stderr,"%s\n",c);
    return 1;
  }
}
