/*
 * bridge.cc - Monitored file (socket, pipe...) bridge
 * $Id: bridge.cc,v 1.8 2003/01/26 09:09:07 rdenisc Exp $
 */

/***********************************************************************
 *  Copyright (C) 2002-2003 Rmi Denis-Courmont.                       *
 *  This program is free software; you can redistribute and/or modify  *
 *  it under the terms of the GNU General Public License as published  *
 *  by the Free Software Foundation; version 2 of the license.         *
 *                                                                     *
 *  This program is distributed in the hope that it will be useful,    *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of     *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.               *
 *  See the GNU General Public License for more details.               *
 *                                                                     *
 *  You should have received a copy of the GNU General Pulic License   *
 *  along with this program; if not, you can get it from:              *
 *  http://www.gnu.org/copyleft/gpl.html                               *
 ***********************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <string.h> /* memcpy() */
#ifdef HAVE_LIMITS_H
# include <limits.h> /* SHRT_MAX */
#endif
#ifdef HAVE_UNISTD_H
# include <sys/time.h> /* struct timeval (unused, but needed for select) */
# include <unistd.h> /* close(), select(), read(), write() */
#endif
#ifdef HAVE_SYS_SOCKET_H
# include <sys/socket.h> /* shutdown() */
#endif

#include "log.h"
#include "bridge.h"

/*
 * Finds a pair of fds (one for reading, one for writing) in fd vectors
 * readfds and writefds, both of length <len>.
 *
 * Loops for ever if len is nul (that's not a bug, that's logic).
 * Returns a non-negative value (`val') on success, -1 on select() error.
 *
 * On return, readfds[index] is readable, writefds[index] is writeable,
 * exceptflag is true if an exceptionnal condition occured on
 * readfds[index] rather than a readability event.
 *
 * Function undefined if len is negative or exceeds MAXFILE.
 */
inline int
select_fd_pair (const int *readfds, const int *writefds, int len,
		int *exceptflag)
{
	while (1)
	{
		fd_set readset, writeset, exceptset;
		int maxreadfd = -1, maxwritefd = -1, i, s2;

		FD_ZERO (&readset);
		FD_ZERO (&writeset);

		for (i = 0; i < len; i++)
		{
			int fd;

			fd = readfds[i];
			if (fd != -1)
			{
				FD_SET (fd, &readset);
				if (maxreadfd < fd)
					maxreadfd = fd;
			}

			fd = writefds[i];
			if (fd != -1) {
				FD_SET (fd, &writeset);
				if (maxwritefd < fd)
					maxwritefd = fd;
			}
		}
		memcpy (&exceptset, &readset, sizeof(exceptset));


		//  Finds readable stream(s)...
		if ((select (++maxreadfd, &readset, NULL, &exceptset, NULL)
			== -1)
		// ...and then writeable stream(s). 
		 || ((s2 = select (++maxwritefd, NULL, &writeset, NULL, NULL))
			 == -1))
			return -1;

		for (i = 0; s2 > 0; i++)
		{
			int fd;

			fd = writefds[i];
			if (fd != -1)
				if (FD_ISSET (fd, &writeset))
				{
					s2--;
					fd = readfds[i];
					if (FD_ISSET (fd, &exceptset))
					{
						*exceptflag = 1;
						return i;
					}
					else if (FD_ISSET (fd, &readset))
					{
						*exceptflag = 0;
						return i;
					}
				}
		}
	}
}

static int
times_in_array(int *array, int len, int elem)
{
	int c = 0, i;

	for (i = 0; i < len; i++)
		if (array[i] == elem)
			c++;
	return c;
}

static int
spare_close(int *readfds, int *writefds, int len, int fd, int how)
{
	if (times_in_array (how ? writefds : readfds, len, fd) == 1) {
		if (times_in_array (how ? readfds : writefds, len, fd) == 0)
			return close(fd);
		else
			return shutdown (fd, how);
	}
	return 0;
}


#define MAX_PACKET_SIZE (SHRT_MAX > 65535) ? 65535 : SHRT_MAX /* (bytes) */

/*
 * Operates a bridge between fd[0](input)/fd[2](output) and another
 * between fd[1](input)/fd[3](output) and display any transmitted data
 * to each streams in the NULL-terminated stream list <logs>.
 *
 * Note: no assumption is made about the transport protocol used,
 * but the use of shutdown() assumes we work with sockets
 * (shutdown() will otherwise silently fail -- not a big problem).
 *
 * fd is modified: closed descriptors are replaced by (-1).
 * In case of error, some of them might not have been closed.
 * Do it yourself.
 */

#define blen 2 /* number of half-duplex bridges */
#define readfds fds
#define writefds (fds+blen)
int
monitor_bridge(int fds[4], DataLogList *logs, long limit)
{
	long totalcount = 0;
	int bcount = blen;

	do
	{
		int oob;

		// What should we do?
		int index = select_fd_pair (readfds, writefds, blen, &oob);
		if (index < 0)
			return 0; // most likely: errno = EINTR
	
		// Processes data
		char buf[MAX_PACKET_SIZE];
			
		// Reads data
		int rfd = readfds[index];
		int len = (oob)
			? recv (rfd, buf, sizeof(buf), MSG_OOB)
			: read (rfd, buf, sizeof(buf));
			
		switch (len)
		{
			case -1:// usually: EINTR or ETIMEDOUT
			case 0: // end-of-file
				spare_close (readfds, writefds, blen, rfd, 0);
				readfds[index] = -1;
				spare_close (readfds, writefds, blen,
						writefds[index], 1);
				writefds[index] = -1;
				bcount--;

				if (index & 1)
					logs->ShutdownServer ();
				else
					logs->ShutdownClient ();
				break;

			default:
				if (index & 1)
					logs->WriteServerData (buf, len, oob);
				else
					logs->WriteClientData (buf, len, oob);

				totalcount += len;
				if (totalcount < 0)
					totalcount = LONG_MAX;
					
				/* Sends data to the real destination */
				int check = (oob)
					? send (writefds[index], buf, len,
						MSG_OOB)
					: write (writefds[index], buf, len);
				if (check != len)
				{
					// usually either EINTR or ETIMEDOUT
					spare_close (readfds, writefds, blen,
							rfd, 0);
					readfds[index] = -1;
					spare_close (readfds, writefds, blen,
							writefds[index], 1);
					writefds[index] = -1;
					bcount--;
				}
		}
	}
	while (bcount && ((limit == -1) || (limit > totalcount)));

	return 0;
}

