/*
 * SpanDSP - a series of DSP components for telephony
 *
 * v22bis_rx.c - ITU V.22bis modem receive part
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2004 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: v22bis_rx.c,v 1.9 2005/01/17 13:12:15 steveu Exp $
 */

/*! \file */

/* THIS IS A WORK IN PROGRESS - NOT YET FUNCTIONAL! */

#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "spandsp/telephony.h"
#include "spandsp/power_meter.h"
#include "spandsp/arctan2.h"
#include "spandsp/complex.h"
#include "spandsp/dds.h"
#include "spandsp/complex_filters.h"

#include "spandsp/v29rx.h"
#include "spandsp/v22bis.h"

#define ms_to_symbols(t)    (((t)*600)/1000)

enum
{
    V22BIS_TRAINING_STAGE_NORMAL_OPERATION,
    V22BIS_TRAINING_STAGE_SYMBOL_ACQUISITION,
    V22BIS_TRAINING_STAGE_LOG_PHASE,
    V22BIS_TRAINING_STAGE_UNSCRAMBLED_ONES,
    V22BIS_TRAINING_STAGE_UNSCRAMBLED_0011,
    V22BIS_TRAINING_STAGE_SCRAMBLED_ONES_AT_1200,
    V22BIS_TRAINING_STAGE_SCRAMBLED_ONES_AT_2400,
    V22BIS_TRAINING_STAGE_WAIT_FOR_START_1,
    V22BIS_TRAINING_STAGE_WAIT_FOR_START_2,
    V22BIS_TRAINING_STAGE_PARKED
};

static const complex_t v22bis_constellation[16] =
{
    { 1.0,  1.0},
    { 1.0,  3.0},
    { 3.0,  1.0},
    { 3.0,  3.0},
    {-1.0,  1.0},
    {-1.0,  3.0},
    {-3.0,  1.0},
    {-3.0,  3.0},
    {-1.0, -1.0},
    {-1.0, -3.0},
    {-3.0, -1.0},
    {-3.0, -3.0},
    { 1.0, -1.0},
    { 1.0, -3.0},
    { 3.0, -1.0},
    { 3.0, -3.0}
};

/* The following is not an issue for multi-threading, as this is a global table
   of constants. The first thread generates it. A second thread, starting around
   the same time as the first, might duplicate the work. The outcome will still be
   the same. */
uint8_t space_map_v22bis[30][30];

/* Raised root cosine pulse shaping; Beta = 0.75; 4 symbols either
   side of the centre. We cannot simplify this by using only half
   the filter, as each variant are each skewed by a n/48 of a
   sample. Only one is symmetric. */
#define PULSESHAPER_GAIN    39.98830768
static const float pulseshaper[12][107] =
{
    {
         0.0068651997,    /* Filter 0 */
         0.0062915835,
         0.0046119232,
         0.0020123423,
        -0.0011552408,
        -0.0044167427,
        -0.0072431051,
        -0.0091323275,
        -0.0096944718,
        -0.0087262175,
        -0.0062619777,
        -0.0025911137,
         0.0017647892,
         0.0061148322,
         0.0096957609,
         0.0117908030,
         0.0118535101,
         0.0096172077,
         0.0051709167,
        -0.0010141249,
        -0.0081171458,
        -0.0150613428,
        -0.0206605082,
        -0.0238019964,
        -0.0236417305,
        -0.0197822920,
        -0.0124043050,
        -0.0023248041,
         0.0090360327,
         0.0197872093,
         0.0277922967,
         0.0309679234,
         0.0276228481,
         0.0167957625,
        -0.0014545369,
        -0.0258516454,
        -0.0538343018,
        -0.0816438998,
        -0.1045738681,
        -0.1173682629,
        -0.1147371949,
        -0.0919395612,
        -0.0453713347,
         0.0269077959,
         0.1247742540,
         0.2458213451,
         0.3853580168,
         0.5366399562,
         0.6913219290,
         0.8400950581,
         0.9734507804,
         1.0824972516,
         1.1597460164,
         1.1997879651,
         1.1997879651,
         1.1597460164,
         1.0824972516,
         0.9734507804,
         0.8400950581,
         0.6913219290,
         0.5366399562,
         0.3853580168,
         0.2458213451,
         0.1247742540,
         0.0269077959,
        -0.0453713347,
        -0.0919395612,
        -0.1147371949,
        -0.1173682629,
        -0.1045738681,
        -0.0816438998,
        -0.0538343018,
        -0.0258516454,
        -0.0014545369,
         0.0167957625,
         0.0276228481,
         0.0309679234,
         0.0277922967,
         0.0197872093,
         0.0090360327,
        -0.0023248041,
        -0.0124043050,
        -0.0197822920,
        -0.0236417305,
        -0.0238019964,
        -0.0206605082,
        -0.0150613428,
        -0.0081171458,
        -0.0010141249,
         0.0051709167,
         0.0096172077,
         0.0118535101,
         0.0117908030,
         0.0096957609,
         0.0061148322,
         0.0017647892,
        -0.0025911137,
        -0.0062619777,
        -0.0087262175,
        -0.0096944718,
        -0.0091323275,
        -0.0072431051,
        -0.0044167427,
        -0.0011552408,
         0.0020123423,
         0.0046119232,
         0.0062915835
    },
    {
         0.0063828958,    /* Filter 1 */
         0.0061924839,
         0.0044265854,
         0.0017641827,
        -0.0014305915,
        -0.0046762332,
        -0.0074426800,
        -0.0092340768,
        -0.0096734310,
        -0.0085756893,
        -0.0059963784,
        -0.0022456493,
         0.0021378724,
         0.0064531574,
         0.0099360952,
         0.0118794442,
         0.0117562309,
         0.0093271121,
         0.0047127189,
        -0.0015844813,
        -0.0087180003,
        -0.0155956820,
        -0.0210301162,
        -0.0239236242,
        -0.0234629996,
        -0.0192948248,
        -0.0116516965,
        -0.0014035339,
         0.0099838342,
         0.0205891726,
         0.0282682673,
         0.0309571951,
         0.0270129860,
         0.0155492505,
        -0.0032797140,
        -0.0280909843,
        -0.0562179397,
        -0.0838112545,
        -0.1061014746,
        -0.1178088441,
        -0.1136664386,
        -0.0890058445,
        -0.0403428967,
         0.0341038349,
         0.1340260686,
         0.2568236164,
         0.3976222979,
         0.5495242559,
         0.7040774266,
         0.8519258656,
         0.9835810789,
         1.0902391332,
         1.1645611093,
         1.2013364399,
         1.1979597716,
         1.1546710455,
         1.0745333436,
         0.9631508525,
         0.8281568285,
         0.6785255167,
         0.5237798314,
         0.3731762346,
         0.2349486709,
         0.1156847469,
         0.0198907757,
        -0.0502200247,
        -0.0947072384,
        -0.1156672473,
        -0.1168202457,
        -0.1029759004,
        -0.0794429837,
        -0.0514500185,
        -0.0236379069,
         0.0003272036,
         0.0179900146,
         0.0281801584,
         0.0309329459,
         0.0272825627,
         0.0189661487,
         0.0080841442,
        -0.0032368983,
        -0.0131377986,
        -0.0202449519,
        -0.0237944348,
        -0.0236571793,
        -0.0202736773,
        -0.0145176219,
        -0.0075152540,
        -0.0004502946,
         0.0056168341,
         0.0098917320,
         0.0119346156,
         0.0116878849,
         0.0094450088,
         0.0057711396,
         0.0013917087,
        -0.0029317209,
        -0.0065190612,
        -0.0088662073,
        -0.0097047455,
        -0.0090212533,
        -0.0070369571,
        -0.0041542150,
        -0.0008805752,
         0.0022564855,
         0.0047907753,
         0.0063828958
    },
    {
         0.0064663699,    /* Filter 2 */
         0.0060856570,
         0.0042349258,
         0.0015122568,
        -0.0017063243,
        -0.0049323766,
        -0.0076354145,
        -0.0093263247,
        -0.0096415750,
        -0.0084147230,
        -0.0057225097,
        -0.0018956946,
         0.0025105195,
         0.0067856700,
         0.0101656332,
         0.0119535662,
         0.0116427277,
         0.0090216168,
         0.0042426325,
        -0.0021607907,
        -0.0093171358,
        -0.0161199478,
        -0.0213819140,
        -0.0240216914,
        -0.0232581792,
        -0.0187828526,
        -0.0108806473,
        -0.0004740836,
         0.0109263386,
         0.0213707716,
         0.0287093418,
         0.0308999646,
         0.0263502995,
         0.0142508746,
        -0.0051471849,
        -0.0303540411,
        -0.0585984121,
        -0.0859420842,
        -0.1075555867,
        -0.1181390134,
        -0.1124525092,
        -0.0859044561,
        -0.0351341894,
         0.0414781213,
         0.1434380645,
         0.2679520756,
         0.4099645924,
         0.5624274958,
         0.7167864433,
         0.8636438240,
         0.9935369350,
         1.0977552210,
         1.1691139460,
         1.2026044259,
         1.1958527683,
         1.1493387006,
         1.0663512781,
         0.9526861769,
         0.8161166340,
         0.6656937456,
         0.5109490692,
         0.3610813578,
         0.2242089021,
         0.1067595615,
         0.0130534338,
        -0.0548895888,
        -0.0973105896,
        -0.1164591206,
        -0.1161677943,
        -0.1013107028,
        -0.0772114421,
        -0.0490675632,
        -0.0214515922,
         0.0020644278,
         0.0191316703,
         0.0286852394,
         0.0308530946,
         0.0267402172,
         0.0181272599,
         0.0071293664,
        -0.0041388439,
        -0.0138515330,
        -0.0206825338,
        -0.0239212039,
        -0.0234895666,
        -0.0198702242,
        -0.0139652141,
        -0.0069130014,
         0.0001064491,
         0.0060500970,
         0.0101505319,
         0.0119996156,
         0.0115709461,
         0.0091842263,
         0.0054225271,
         0.0010190658,
        -0.0032671125,
        -0.0067673936,
        -0.0089955710,
        -0.0097043122,
        -0.0089010399,
        -0.0068245087,
        -0.0038889617,
        -0.0006068945,
         0.0024963679,
         0.0049629856,
         0.0064663699
    },
    {
         0.0065419639,    /* Filter 3 */
         0.0059711715,
         0.0040371168,
         0.0012568206,
        -0.0019821343,
        -0.0051848641,
        -0.0078210472,
        -0.0094089041,
        -0.0095988674,
        -0.0082434313,
        -0.0054406297,
        -0.0015416245,
         0.0028822889,
         0.0071119276,
         0.0103840047,
         0.0120129410,
         0.0115129677,
         0.0087009126,
         0.0037610662,
        -0.0027424678,
        -0.0099138651,
        -0.0166334531,
        -0.0217153278,
        -0.0240958492,
        -0.0230272354,
        -0.0182467088,
        -0.0100918611,
         0.0004625281,
         0.0118623246,
         0.0221307429,
         0.0291144076,
         0.0307954723,
         0.0256345668,
         0.0129010914,
        -0.0070557432,
        -0.0326388750,
        -0.0609731532,
        -0.0880334001,
        -0.1089330711,
        -0.1183558242,
        -0.1110929950,
        -0.0826338469,
        -0.0297447937,
         0.0490297716,
         0.1530080038,
         0.2792032143,
         0.4223803384,
         0.5753443962,
         0.7294434063,
         0.8752435393,
         1.0033136073,
         1.1050418520,
         1.1734022760,
         1.2035912921,
         1.1934680027,
         1.1437516105,
         1.0579550234,
         0.9420617007,
         0.8039799584,
         0.6528321585,
         0.4981528063,
         0.3490777124,
         0.2136052448,
         0.0980005994,
         0.0063963183,
        -0.0593807489,
        -0.0997514077,
        -0.1171153925,
        -0.1154139348,
        -0.0995814019,
        -0.0749521818,
        -0.0466893607,
        -0.0192944649,
         0.0037561192,
         0.0202204526,
         0.0291384627,
         0.0307292358,
         0.0261664295,
         0.0172718131,
         0.0061728835,
        -0.0050296928,
        -0.0145448943,
        -0.0210947985,
        -0.0240221572,
        -0.0232995727,
        -0.0194507616,
        -0.0134048171,
        -0.0063110582,
         0.0006555592,
         0.0064703485,
         0.0103934735,
         0.0120485952,
         0.0114402560,
         0.0089138082,
         0.0050694436,
         0.0006472911,
        -0.0035969389,
        -0.0070067514,
        -0.0091142336,
        -0.0096932432,
        -0.0087718820,
        -0.0066060378,
        -0.0036212950,
        -0.0003344956,
         0.0027317518,
         0.0051284067,
         0.0065419639
    },
    {
         0.0066096450,    /* Filter 4 */
         0.0058491052,
         0.0038333384,
         0.0009981360,
        -0.0022577140,
        -0.0054333894,
        -0.0079993228,
        -0.0094816578,
        -0.0095452840,
        -0.0080619399,
        -0.0051510077,
        -0.0011838217,
         0.0032527359,
         0.0074314908,
         0.0105908493,
         0.0120573548,
         0.0113669364,
         0.0083652090,
         0.0032684457,
        -0.0033289154,
        -0.0105074975,
        -0.0171355155,
        -0.0220297989,
        -0.0241457716,
        -0.0227701632,
        -0.0176867587,
        -0.0092860707,
         0.0014052606,
         0.0127905605,
         0.0228678280,
         0.0294823738,
         0.0306429970,
         0.0248656183,
         0.0115004181,
        -0.0090041199,
        -0.0349434874,
        -0.0633395530,
        -0.0900821895,
        -0.1102307969,
        -0.1184563610,
        -0.1095855440,
        -0.0791925527,
        -0.0241743943,
         0.0567577892,
         0.1627335380,
         0.2905734259,
         0.4348648999,
         0.5882696362,
         0.7420427392,
         0.8867196544,
         1.0129064289,
         1.1120954696,
         1.1774239776,
         1.2042965476,
         1.1908066594,
         1.1379125263,
         1.0493486446,
         0.9312824336,
         0.7917523088,
         0.6399462815,
         0.4853961268,
         0.3371695409,
         0.2031408018,
         0.0894096495,
        -0.0000801335,
        -0.0636943253,
        -0.1020315628,
        -0.1176386914,
        -0.1145617146,
        -0.0977911177,
        -0.0726680782,
        -0.0443177867,
        -0.0171682285,
         0.0054013243,
         0.0212561427,
         0.0295402474,
         0.0305622686,
         0.0255623848,
         0.0164010779,
         0.0052158650,
        -0.0059085213,
        -0.0152172988,
        -0.0214815377,
        -0.0240974418,
        -0.0230876329,
        -0.0190159138,
        -0.0128371304,
        -0.0057100875,
         0.0011965024,
         0.0068772498,
         0.0106204417,
         0.0120816570,
         0.0112960971,
         0.0086341567,
         0.0047123396,
         0.0002768111,
        -0.0039208595,
        -0.0072369232,
        -0.0092221326,
        -0.0096716217,
        -0.0086339827,
        -0.0063818272,
        -0.0033515279,
        -0.0000636719,
         0.0029624060,
         0.0052868993,
         0.0066096450
    },
    {
         0.0066693889,    /* Filter 5 */
         0.0057195448,
         0.0036237788,
         0.0007364699,
        -0.0025327544,
        -0.0056776483,
        -0.0081699926,
        -0.0095444384,
        -0.0094808126,
        -0.0078703867,
        -0.0048539236,
        -0.0008226763,
         0.0036214137,
         0.0077439242,
         0.0107858160,
         0.0120866090,
         0.0112046371,
         0.0080147345,
         0.0027652130,
        -0.0039195251,
        -0.0110973383,
        -0.0176254587,
        -0.0223247838,
        -0.0241711570,
        -0.0224869875,
        -0.0171033989,
        -0.0084640378,
         0.0023530521,
         0.0137098056,
         0.0235807746,
         0.0298121729,
         0.0304418569,
         0.0240433372,
         0.0100494334,
        -0.0109909829,
        -0.0372658233,
        -0.0656949589,
        -0.0920854182,
        -0.1114456382,
        -0.1184377422,
        -0.1079278668,
        -0.0755791966,
        -0.0184227802,
         0.0646610651,
         0.1726122077,
         0.3020590079,
         0.4474135694,
         0.6011978558,
         0.7545788655,
         0.8980668519,
         1.0223108104,
         1.1189126259,
         1.1811770597,
         1.2047198413,
         1.1878700602,
         1.1318243204,
         1.0405363008,
         0.9203534442,
         0.7794392121,
         0.6270416209,
         0.4726840584,
         0.3253610010,
         0.1928185710,
         0.0809883877,
        -0.0063755943,
        -0.0678312349,
        -0.1041530002,
        -0.1180316937,
        -0.1136142006,
        -0.0959429613,
        -0.0703619740,
        -0.0419551660,
        -0.0150745255,
         0.0069991529,
         0.0222385798,
         0.0298910596,
         0.0303531236,
         0.0249292833,
         0.0155163223,
         0.0042594649,
        -0.0067744317,
        -0.0158681939,
        -0.0218425746,
        -0.0241472318,
        -0.0228542022,
        -0.0185663160,
        -0.0122628545,
        -0.0051107450,
         0.0017287596,
         0.0072704799,
         0.0108313403,
         0.0120989200,
         0.0111387641,
         0.0083456809,
         0.0043516660,
        -0.0000919530,
        -0.0042385431,
        -0.0074577092,
        -0.0093192183,
        -0.0096395417,
        -0.0084875536,
        -0.0061521639,
        -0.0030799735,
         0.0002052867,
         0.0031881057,
         0.0054383331,
         0.0066693889
    },
    {
         0.0067211807,    /* Filter 6 */
         0.0055825860,
         0.0034086335,
         0.0004720945,
        -0.0028069448,
        -0.0059173393,
        -0.0083328144,
        -0.0095971086,
        -0.0094054532,
        -0.0076689221,
        -0.0045496683,
        -0.0004585855,
         0.0039878737,
         0.0080487964,
         0.0109685642,
         0.0121005200,
         0.0110260917,
         0.0076497362,
         0.0022518263,
        -0.0045136780,
        -0.0116826903,
        -0.0181026129,
        -0.0225997550,
        -0.0241717279,
        -0.0221777630,
        -0.0164970575,
        -0.0076265522,
         0.0033048207,
         0.0146188112,
         0.0242683382,
         0.0301027619,
         0.0301914108,
         0.0231676605,
         0.0085487775,
        -0.0130149372,
        -0.0396037715,
        -0.0680366770,
        -0.0940400326,
        -0.1125744759,
        -0.1182971220,
        -0.1061177384,
        -0.0717924901,
        -0.0124898463,
         0.0727383765,
         0.1826414446,
         0.3136561630,
         0.4600215701,
         0.6141236600,
         0.7670462121,
         0.9092798577,
         1.0315222429,
         1.1254899840,
         1.1846596628,
         1.2048609626,
         1.1846596628,
         1.1254899840,
         1.0315222429,
         0.9092798577,
         0.7670462121,
         0.6141236600,
         0.4600215701,
         0.3136561630,
         0.1826414446,
         0.0727383765,
        -0.0124898463,
        -0.0717924901,
        -0.1061177384,
        -0.1182971220,
        -0.1125744759,
        -0.0940400326,
        -0.0680366770,
        -0.0396037715,
        -0.0130149372,
         0.0085487775,
         0.0231676605,
         0.0301914108,
         0.0301027619,
         0.0242683382,
         0.0146188112,
         0.0033048207,
        -0.0076265522,
        -0.0164970575,
        -0.0221777630,
        -0.0241717279,
        -0.0225997550,
        -0.0181026129,
        -0.0116826903,
        -0.0045136780,
         0.0022518263,
         0.0076497362,
         0.0110260917,
         0.0121005200,
         0.0109685642,
         0.0080487964,
         0.0039878737,
        -0.0004585855,
        -0.0045496683,
        -0.0076689221,
        -0.0094054532,
        -0.0095971086,
        -0.0083328144,
        -0.0059173393,
        -0.0028069448,
         0.0004720945,
         0.0034086335,
         0.0055825860,
         0.0067211807
    },
    {
         0.0067650141,    /* Filter 7 */
         0.0054383331,
         0.0031881057,
         0.0002052867,
        -0.0030799735,
        -0.0061521639,
        -0.0084875536,
        -0.0096395417,
        -0.0093192183,
        -0.0074577092,
        -0.0042385431,
        -0.0000919530,
         0.0043516660,
         0.0083456809,
         0.0111387641,
         0.0120989200,
         0.0108313403,
         0.0072704799,
         0.0017287596,
        -0.0051107450,
        -0.0122628545,
        -0.0185663160,
        -0.0228542022,
        -0.0241472318,
        -0.0218425746,
        -0.0158681939,
        -0.0067744317,
         0.0042594649,
         0.0155163223,
         0.0249292833,
         0.0303531236,
         0.0298910596,
         0.0222385798,
         0.0069991529,
        -0.0150745255,
        -0.0419551660,
        -0.0703619740,
        -0.0959429613,
        -0.1136142006,
        -0.1180316937,
        -0.1041530002,
        -0.0678312349,
        -0.0063755943,
         0.0809883877,
         0.1928185710,
         0.3253610010,
         0.4726840584,
         0.6270416209,
         0.7794392121,
         0.9203534442,
         1.0405363008,
         1.1318243204,
         1.1878700602,
         1.2047198413,
         1.1811770597,
         1.1189126259,
         1.0223108104,
         0.8980668519,
         0.7545788655,
         0.6011978558,
         0.4474135694,
         0.3020590079,
         0.1726122077,
         0.0646610651,
        -0.0184227802,
        -0.0755791966,
        -0.1079278668,
        -0.1184377422,
        -0.1114456382,
        -0.0920854182,
        -0.0656949589,
        -0.0372658233,
        -0.0109909829,
         0.0100494334,
         0.0240433372,
         0.0304418569,
         0.0298121729,
         0.0235807746,
         0.0137098056,
         0.0023530521,
        -0.0084640378,
        -0.0171033989,
        -0.0224869875,
        -0.0241711570,
        -0.0223247838,
        -0.0176254587,
        -0.0110973383,
        -0.0039195251,
         0.0027652130,
         0.0080147345,
         0.0112046371,
         0.0120866090,
         0.0107858160,
         0.0077439242,
         0.0036214137,
        -0.0008226763,
        -0.0048539236,
        -0.0078703867,
        -0.0094808126,
        -0.0095444384,
        -0.0081699926,
        -0.0056776483,
        -0.0025327544,
         0.0007364699,
         0.0036237788,
         0.0057195448,
         0.0067650141
    },
    {
         0.0068008916,    /* Filter 8 */
         0.0052868993,
         0.0029624060,
        -0.0000636719,
        -0.0033515279,
        -0.0063818272,
        -0.0086339827,
        -0.0096716217,
        -0.0092221326,
        -0.0072369232,
        -0.0039208595,
         0.0002768111,
         0.0047123396,
         0.0086341567,
         0.0112960971,
         0.0120816570,
         0.0106204417,
         0.0068772498,
         0.0011965024,
        -0.0057100875,
        -0.0128371304,
        -0.0190159138,
        -0.0230876329,
        -0.0240974418,
        -0.0214815377,
        -0.0152172988,
        -0.0059085213,
         0.0052158650,
         0.0164010779,
         0.0255623848,
         0.0305622686,
         0.0295402474,
         0.0212561427,
         0.0054013243,
        -0.0171682285,
        -0.0443177867,
        -0.0726680782,
        -0.0977911177,
        -0.1145617146,
        -0.1176386914,
        -0.1020315628,
        -0.0636943253,
        -0.0000801335,
         0.0894096495,
         0.2031408018,
         0.3371695409,
         0.4853961268,
         0.6399462815,
         0.7917523088,
         0.9312824336,
         1.0493486446,
         1.1379125263,
         1.1908066594,
         1.2042965476,
         1.1774239776,
         1.1120954696,
         1.0129064289,
         0.8867196544,
         0.7420427392,
         0.5882696362,
         0.4348648999,
         0.2905734259,
         0.1627335380,
         0.0567577892,
        -0.0241743943,
        -0.0791925527,
        -0.1095855440,
        -0.1184563610,
        -0.1102307969,
        -0.0900821895,
        -0.0633395530,
        -0.0349434874,
        -0.0090041199,
         0.0115004181,
         0.0248656183,
         0.0306429970,
         0.0294823738,
         0.0228678280,
         0.0127905605,
         0.0014052606,
        -0.0092860707,
        -0.0176867587,
        -0.0227701632,
        -0.0241457716,
        -0.0220297989,
        -0.0171355155,
        -0.0105074975,
        -0.0033289154,
         0.0032684457,
         0.0083652090,
         0.0113669364,
         0.0120573548,
         0.0105908493,
         0.0074314908,
         0.0032527359,
        -0.0011838217,
        -0.0051510077,
        -0.0080619399,
        -0.0095452840,
        -0.0094816578,
        -0.0079993228,
        -0.0054333894,
        -0.0022577140,
         0.0009981360,
         0.0038333384,
         0.0058491052,
         0.0068008916
    },
    {
         0.0068288246,    /* Filter 9 */
         0.0051284067,
         0.0027317518,
        -0.0003344956,
        -0.0036212950,
        -0.0066060378,
        -0.0087718820,
        -0.0096932432,
        -0.0091142336,
        -0.0070067514,
        -0.0035969389,
         0.0006472911,
         0.0050694436,
         0.0089138082,
         0.0114402560,
         0.0120485952,
         0.0103934735,
         0.0064703485,
         0.0006555592,
        -0.0063110582,
        -0.0134048171,
        -0.0194507616,
        -0.0232995727,
        -0.0240221572,
        -0.0210947985,
        -0.0145448943,
        -0.0050296928,
         0.0061728835,
         0.0172718131,
         0.0261664295,
         0.0307292358,
         0.0291384627,
         0.0202204526,
         0.0037561192,
        -0.0192944649,
        -0.0466893607,
        -0.0749521818,
        -0.0995814019,
        -0.1154139348,
        -0.1171153925,
        -0.0997514077,
        -0.0593807489,
         0.0063963183,
         0.0980005994,
         0.2136052448,
         0.3490777124,
         0.4981528063,
         0.6528321585,
         0.8039799584,
         0.9420617007,
         1.0579550234,
         1.1437516105,
         1.1934680027,
         1.2035912921,
         1.1734022760,
         1.1050418520,
         1.0033136073,
         0.8752435393,
         0.7294434063,
         0.5753443962,
         0.4223803384,
         0.2792032143,
         0.1530080038,
         0.0490297716,
        -0.0297447937,
        -0.0826338469,
        -0.1110929950,
        -0.1183558242,
        -0.1089330711,
        -0.0880334001,
        -0.0609731532,
        -0.0326388750,
        -0.0070557432,
         0.0129010914,
         0.0256345668,
         0.0307954723,
         0.0291144076,
         0.0221307429,
         0.0118623246,
         0.0004625281,
        -0.0100918611,
        -0.0182467088,
        -0.0230272354,
        -0.0240958492,
        -0.0217153278,
        -0.0166334531,
        -0.0099138651,
        -0.0027424678,
         0.0037610662,
         0.0087009126,
         0.0115129677,
         0.0120129410,
         0.0103840047,
         0.0071119276,
         0.0028822889,
        -0.0015416245,
        -0.0054406297,
        -0.0082434313,
        -0.0095988674,
        -0.0094089041,
        -0.0078210472,
        -0.0051848641,
        -0.0019821343,
         0.0012568206,
         0.0040371168,
         0.0059711715,
         0.0068288246
    },
    {
         0.0068488331,    /* Filter 10 */
         0.0049629856,
         0.0024963679,
        -0.0006068945,
        -0.0038889617,
        -0.0068245087,
        -0.0089010399,
        -0.0097043122,
        -0.0089955710,
        -0.0067673936,
        -0.0032671125,
         0.0010190658,
         0.0054225271,
         0.0091842263,
         0.0115709461,
         0.0119996156,
         0.0101505319,
         0.0060500970,
         0.0001064491,
        -0.0069130014,
        -0.0139652141,
        -0.0198702242,
        -0.0234895666,
        -0.0239212039,
        -0.0206825338,
        -0.0138515330,
        -0.0041388439,
         0.0071293664,
         0.0181272599,
         0.0267402172,
         0.0308530946,
         0.0286852394,
         0.0191316703,
         0.0020644278,
        -0.0214515922,
        -0.0490675632,
        -0.0772114421,
        -0.1013107028,
        -0.1161677943,
        -0.1164591206,
        -0.0973105896,
        -0.0548895888,
         0.0130534338,
         0.1067595615,
         0.2242089021,
         0.3610813578,
         0.5109490692,
         0.6656937456,
         0.8161166340,
         0.9526861769,
         1.0663512781,
         1.1493387006,
         1.1958527683,
         1.2026044259,
         1.1691139460,
         1.0977552210,
         0.9935369350,
         0.8636438240,
         0.7167864433,
         0.5624274958,
         0.4099645924,
         0.2679520756,
         0.1434380645,
         0.0414781213,
        -0.0351341894,
        -0.0859044561,
        -0.1124525092,
        -0.1181390134,
        -0.1075555867,
        -0.0859420842,
        -0.0585984121,
        -0.0303540411,
        -0.0051471849,
         0.0142508746,
         0.0263502995,
         0.0308999646,
         0.0287093418,
         0.0213707716,
         0.0109263386,
        -0.0004740836,
        -0.0108806473,
        -0.0187828526,
        -0.0232581792,
        -0.0240216914,
        -0.0213819140,
        -0.0161199478,
        -0.0093171358,
        -0.0021607907,
         0.0042426325,
         0.0090216168,
         0.0116427277,
         0.0119535662,
         0.0101656332,
         0.0067856700,
         0.0025105195,
        -0.0018956946,
        -0.0057225097,
        -0.0084147230,
        -0.0096415750,
        -0.0093263247,
        -0.0076354145,
        -0.0049323766,
        -0.0017063243,
         0.0015122568,
         0.0042349258,
         0.0060856570,
         0.0068488331
    },
    {
         0.0068609458,    /* Filter 11 */
         0.0047907753,
         0.0022564855,
        -0.0008805752,
        -0.0041542150,
        -0.0070369571,
        -0.0090212533,
        -0.0097047455,
        -0.0088662073,
        -0.0065190612,
        -0.0029317209,
         0.0013917087,
         0.0057711396,
         0.0094450088,
         0.0116878849,
         0.0119346156,
         0.0098917320,
         0.0056168341,
        -0.0004502946,
        -0.0075152540,
        -0.0145176219,
        -0.0202736773,
        -0.0236571793,
        -0.0237944348,
        -0.0202449519,
        -0.0131377986,
        -0.0032368983,
         0.0080841442,
         0.0189661487,
         0.0272825627,
         0.0309329459,
         0.0281801584,
         0.0179900146,
         0.0003272036,
        -0.0236379069,
        -0.0514500185,
        -0.0794429837,
        -0.1029759004,
        -0.1168202457,
        -0.1156672473,
        -0.0947072384,
        -0.0502200247,
         0.0198907757,
         0.1156847469,
         0.2349486709,
         0.3731762346,
         0.5237798314,
         0.6785255167,
         0.8281568285,
         0.9631508525,
         1.0745333436,
         1.1546710455,
         1.1979597716,
         1.2013364399,
         1.1645611093,
         1.0902391332,
         0.9835810789,
         0.8519258656,
         0.7040774266,
         0.5495242559,
         0.3976222979,
         0.2568236164,
         0.1340260686,
         0.0341038349,
        -0.0403428967,
        -0.0890058445,
        -0.1136664386,
        -0.1178088441,
        -0.1061014746,
        -0.0838112545,
        -0.0562179397,
        -0.0280909843,
        -0.0032797140,
         0.0155492505,
         0.0270129860,
         0.0309571951,
         0.0282682673,
         0.0205891726,
         0.0099838342,
        -0.0014035339,
        -0.0116516965,
        -0.0192948248,
        -0.0234629996,
        -0.0239236242,
        -0.0210301162,
        -0.0155956820,
        -0.0087180003,
        -0.0015844813,
         0.0047127189,
         0.0093271121,
         0.0117562309,
         0.0118794442,
         0.0099360952,
         0.0064531574,
         0.0021378724,
        -0.0022456493,
        -0.0059963784,
        -0.0085756893,
        -0.0096734310,
        -0.0092340768,
        -0.0074426800,
        -0.0046762332,
        -0.0014305915,
         0.0017641827,
         0.0044265854,
         0.0061924839,
         0.0068609458
    }
};

float v22bis_rx_carrier_frequency(v22bis_state_t *s)
{
    return s->rx_carrier_phase_rate*(float) SAMPLE_RATE/(65536.0*65536.0);
}
/*- End of function --------------------------------------------------------*/

float v22bis_rx_symbol_timing_correction(v22bis_state_t *s)
{
    return s->gardner_total_correction;
}
/*- End of function --------------------------------------------------------*/

float v22bis_rx_signal_power(v22bis_state_t *s)
{
    return power_meter_dbm0(&s->rx_power);
}
/*- End of function --------------------------------------------------------*/

int v22bis_rx_equalizer_state(v22bis_state_t *s, complex_t **coeffs)
{
    *coeffs = s->eq_coeff;
    return 2*V22BIS_EQUALIZER_LEN + 1;
}
/*- End of function --------------------------------------------------------*/

static void equalizer_reset(v22bis_state_t *s, float delta)
{
    int i;

    /* Start with an equalizer based on everything being perfect */
    for (i = 0;  i < 2*V22BIS_EQUALIZER_LEN + 1;  i++)
        s->eq_coeff[i] = complex_set(0.0, 0.0);
    s->eq_coeff[V22BIS_EQUALIZER_LEN] = complex_set(3.0, 0.0);
    for (i = 0;  i <= V22BIS_EQUALIZER_MASK;  i++)
        s->eq_buf[i] = complex_set(0.0, 0.0);
#if 1
s->eq_coeff[0] = complex_set(0.02665,        -0.00066);
s->eq_coeff[1] = complex_set(-0.08924,        -0.00912);
s->eq_coeff[2] = complex_set(0.04198,         0.00116);
s->eq_coeff[3] = complex_set(0.12208,         0.01135);
s->eq_coeff[4] = complex_set(-0.08063,        -0.00694);
s->eq_coeff[5] = complex_set(-0.22126,        -0.01338);
s->eq_coeff[6] = complex_set(0.00853,         0.01998);
s->eq_coeff[7] = complex_set(2.44700,         0.04613);
s->eq_coeff[8] = complex_set(0.02272,         0.02039);
s->eq_coeff[9] = complex_set(-0.22893,        -0.01379);
s->eq_coeff[10] = complex_set(-0.09634,        -0.01114);
s->eq_coeff[11] = complex_set(0.12721,         0.00782);
s->eq_coeff[12] = complex_set(0.05212,         0.00679);
s->eq_coeff[13] = complex_set(-0.09209,        -0.00284);
s->eq_coeff[14] = complex_set(0.02134,        -0.00133);
#endif
    s->eq_put_step = 20 - 1;
    s->eq_step = 0;
    s->eq_delta = delta/(2*V22BIS_EQUALIZER_LEN + 1);
}
/*- End of function --------------------------------------------------------*/

static complex_t equalizer_get(v22bis_state_t *s)
{
    int i;
    int p;
    complex_t z;
    complex_t z1;

    /* Get the next equalized value. */
    z = complex_set(0.0, 0.0);
    for (i = 0;  i < 2*V22BIS_EQUALIZER_LEN + 1;  i++)
    {
        p = (s->eq_step + i) & V22BIS_EQUALIZER_MASK;
        z1 = complex_mul(&s->eq_coeff[i], &s->eq_buf[p]);
        z = complex_add(&z, &z1);
    }
    return z;
}
/*- End of function --------------------------------------------------------*/

static inline int find_quadrant(complex_t *z)
{
    int b1;
    int b2;

    /* Split the space along the two diagonals. */
    b1 = (z->im > z->re);
    b2 = (z->im < -z->re);
    return (b2 << 1) | (b1 ^ b2);
}
/*- End of function --------------------------------------------------------*/

static void tune_equalizer(v22bis_state_t *s, const complex_t *z, const complex_t *target)
{
    int i;
    int p;
    complex_t ez;
    complex_t z1;

    /* Find the x and y mismatch from the exact constellation position. */
    ez = complex_sub(target, z);
    ez.re *= s->eq_delta;
    ez.im *= s->eq_delta;

    for (i = 0;  i <= 2*V22BIS_EQUALIZER_LEN;  i++)
    {
        p = (s->eq_step + i) & V22BIS_EQUALIZER_MASK;
        z1 = complex_conj(&s->eq_buf[p]);
        z1 = complex_mul(&ez, &z1);
        s->eq_coeff[i] = complex_add(&s->eq_coeff[i], &z1);
        /* If we don't leak a little bit we seem to get some wandering adaption */
        s->eq_coeff[i].re *= 0.9999;
        s->eq_coeff[i].im *= 0.9999;
    }
}
/*- End of function --------------------------------------------------------*/

static __inline__ void track_carrier(v22bis_state_t *s, const complex_t *z, const complex_t *target)
{
    complex_t zz;

    zz = complex_conj(target);
    zz = complex_mul(z, &zz);
    
    /* For small errors the imaginary part of zz is now proportional to the phase error,
       for any particular target. However, the different amplitudes of the various target
       positions scale things. */

    s->rx_carrier_phase_rate += 500.0*zz.im;
    s->rx_carrier_phase += 1000000.0*zz.im;
    //fprintf(stderr, "Im = %15.5f   f = %15.5f\n", zz.im, s->rx_carrier_phase_rate*8000.0/(65536.0*65536.0));
}
/*- End of function --------------------------------------------------------*/

static inline void put_bit(v22bis_state_t *s, int bit)
{
    int out_bit;

    bit &= 1;

    /* Descramble the bit */
    s->rx_scramble_reg = (s->rx_scramble_reg << 1) | bit;
    out_bit = (bit ^ (s->rx_scramble_reg >> 15) ^ (s->rx_scramble_reg >> 18)) & 1;
    if (s->rx_scrambler_pattern_count >= 64)
    {
        out_bit ^= 1;
        s->rx_scrambler_pattern_count = 0;
    }
    if (bit)
        s->rx_scrambler_pattern_count++;
    else
        s->rx_scrambler_pattern_count = 0;

    s->put_bit(s->user_data, out_bit);
}
/*- End of function --------------------------------------------------------*/

static void decode_baud(v22bis_state_t *s, complex_t *z)
{
    static const uint8_t phase_steps[4] =
    {
        0, 2, 3, 1
    };
    int nearest;
    int raw_bits;
    int i;
    int re;
    int im;

    if (s->bit_rate == 2400)
    {
        re = (z->re + 3.0)*3.0;
        if (re > 29)
            re = 29;
        else if (re < 0)
            re = 0;
        im = (z->im + 3.0)*3.0;
        if (im > 29)
            im = 29;
        else if (im < 0)
            im = 0;
        nearest = space_map_v22bis[re][im];
        put_bit(s, nearest >> 3);
        put_bit(s, nearest >> 2);
        raw_bits = phase_steps[(nearest - s->rx_constellation_state) & 3];
        put_bit(s, raw_bits);
        put_bit(s, raw_bits >> 1);
    }
    else
    {
        nearest = find_quadrant(z);
        raw_bits = phase_steps[(nearest - s->rx_constellation_state) & 3];
        put_bit(s, raw_bits);
        put_bit(s, raw_bits >> 1);
    }
    //track_carrier(s, z, &v22bis_constellation[nearest]);
    //tune_equalizer(s, z, &v22bis_constellation[nearest]);
    s->rx_constellation_state = nearest;
}
/*- End of function --------------------------------------------------------*/

static void process_baud(v22bis_state_t *s, const complex_t *sample)
{
    complex_t z;
    complex_t zz;
    const complex_t *target;
    float p;
    float q;
    int bit;
    int i;
    int j;
    float phase;
    int32_t angle;
    int32_t ang;

    s->rx_rrc_filter[s->rx_rrc_filter_step].re =
    s->rx_rrc_filter[s->rx_rrc_filter_step + V22BIS_RX_FILTER_STEPS].re = sample->re;
    s->rx_rrc_filter[s->rx_rrc_filter_step].im =
    s->rx_rrc_filter[s->rx_rrc_filter_step + V22BIS_RX_FILTER_STEPS].im = sample->im;
    if (++s->rx_rrc_filter_step >= V22BIS_RX_FILTER_STEPS)
        s->rx_rrc_filter_step = 0;

#if 0
    for (j = 0;  j < 12;  j++)
    {
        z = complex_set(0.0, 0.0);
        for (i = 0;  i < V22BIS_RX_FILTER_STEPS;  i++)
        {
            z.re += pulseshaper[j][i]*s->rx_rrc_filter[i + s->rx_rrc_filter_step].re;
            z.im += pulseshaper[j][i]*s->rx_rrc_filter[i + s->rx_rrc_filter_step].im;
        }
        z.re *= 1.0/PULSESHAPER_GAIN;
        z.im *= 1.0/PULSESHAPER_GAIN;
        printf("ZZZ %15.5f %15.5f\n", z.re, z.im);
    }
#endif

    /* Put things into the equalization buffer at T/2 rate. The Gardner algorithm
       will fiddle the step to align this with the bits. */
    if ((s->eq_put_step -= 12) > 0)
    {
        //fprintf(stderr, "Samp, %f, %f, %f, 0, 0x%X\n", z.re, z.im, sqrt(z.re*z.re + z.im*z.im), s->eq_put_step);
        return;
    }

    /* This is our interpolation filter, as well as our demod filter. */
    j = -s->eq_put_step;
    if (j > 12 - 1)
        j = 12 - 1;
    z = complex_set(0.0, 0.0);
    for (i = 0;  i < V22BIS_RX_FILTER_STEPS;  i++)
    {
        z.re += pulseshaper[j][i]*s->rx_rrc_filter[i + s->rx_rrc_filter_step].re;
        z.im += pulseshaper[j][i]*s->rx_rrc_filter[i + s->rx_rrc_filter_step].im;
    }
    z.re *= 1.0/PULSESHAPER_GAIN;
    z.im *= 1.0/PULSESHAPER_GAIN;

    s->eq_put_step += 80;

    /* Add a sample to the equalizer's circular buffer, but don't calculate anything
       at this time. */
    s->eq_buf[s->eq_step].re = z.re;
    s->eq_buf[s->eq_step].im = z.im;
    s->eq_step = (s->eq_step + 1) & V22BIS_EQUALIZER_MASK;

    /* On alternate insertions we have a whole baud and must process it. */
    if ((++s->rx_baud_phase & 1))
        return;

    /* Perform a Gardner test for baud alignment on the three most recent samples. */
    p = s->eq_buf[(s->eq_step - 3) & V22BIS_EQUALIZER_MASK].re
      - s->eq_buf[(s->eq_step - 1) & V22BIS_EQUALIZER_MASK].re;
    p *= s->eq_buf[(s->eq_step - 2) & V22BIS_EQUALIZER_MASK].re;

    q = s->eq_buf[(s->eq_step - 3) & V22BIS_EQUALIZER_MASK].im
      - s->eq_buf[(s->eq_step - 1) & V22BIS_EQUALIZER_MASK].im;
    q *= s->eq_buf[(s->eq_step - 2) & V22BIS_EQUALIZER_MASK].im;

    p += q;
    s->gardner_integrate += ((p + q) > 0.0)  ?  s->gardner_step  :  -s->gardner_step;

    if (abs(s->gardner_integrate) >= 16)
    {
fprintf(stderr, "Gardner kick %d\n", s->gardner_integrate);
        /* This integrate and dump approach avoids rapid changes of the equalizer put step.
           Rapid changes, without hysteresis, are bad. They degrade the equalizer performance
           when the true symbol boundary is close to a sample boundary. */
        s->eq_put_step += (s->gardner_integrate/16);
        s->gardner_total_correction += (s->gardner_integrate/16);
        if (s->qam_report)
            s->qam_report(s->qam_user_data, NULL, NULL, s->gardner_integrate);
        s->gardner_integrate = 0;
    }

    z = equalizer_get(s);

fprintf(stderr, "VVV %p %d\n", s->user_data, s->rx_training);
    switch (s->rx_training)
    {
    case V22BIS_TRAINING_STAGE_NORMAL_OPERATION:
        /* Normal operation. */
        decode_baud(s, &z);
        target = &v22bis_constellation[s->rx_constellation_state];
        break;
    case V22BIS_TRAINING_STAGE_SYMBOL_ACQUISITION:
        /* Allow time for the Gardner algorithm to settle the symbol timing. */
        target = &z;
        if (++s->rx_training_count >= 59)
        {
            if (s->caller)
                s->rx_training = V22BIS_TRAINING_STAGE_UNSCRAMBLED_ONES;
            else
                s->rx_training = V22BIS_TRAINING_STAGE_UNSCRAMBLED_0011;
            break;
        }

        /* QAM and Gardner only play nicely with heavy damping, so we need to change to
           a slow rate of symbol timing adaption. However, it must not be so slow that it
           cannot track the worst case timing error specified in V.22bis. This should be 0.01%,
           but since we might be off in the opposite direction from the source, the total
           error could be higher. */
        if (s->rx_training_count == 30)
            s->gardner_step = 16;
        else if (s->rx_training_count == 45)
            s->gardner_step = 4;
        else if (s->rx_training_count == 58)
            s->gardner_step = 4;
        break;
    case V22BIS_TRAINING_STAGE_UNSCRAMBLED_ONES:
        target = &z;
        angle = arctan2(z.im, z.re);
        if (++s->rx_training_count >= 60  &&  s->rx_training_count <= 63)
        {
            /* Record the current phase angle */
            /* We record 4 of these, as we may be seeing phase alternations, or phases steping
               round in 90 degree steps. */
            s->angles[s->rx_training_count - 60] =
            s->start_angles[s->rx_training_count - 60] = angle;
fprintf(stderr, "WWW %p 0x%08x %d\n", s->user_data, angle, s->rx_training_count);
            break;
        }
        ang = angle - s->angles[(s->rx_training_count - 64) & 0xF];
fprintf(stderr, "XXX %p 0x%08x 0x%08x 0x%08x %d\n", s->user_data, ang, angle, s->angles[(s->rx_training_count - 64) & 0xF], s->rx_training_count);
        s->angles[(s->rx_training_count - 60) & 0xF] = angle;
printf("TWIDDLING THUMBS - %d\n", s->rx_training_count);
        if (s->rx_training_count == ms_to_symbols(155 + 456))
            s->detected_unscrambled_ones = TRUE;
        if (s->rx_training_count == ms_to_symbols(155 + 457))
            s->rx_training = V22BIS_TRAINING_STAGE_UNSCRAMBLED_0011;
        //if (((ang <= 0x20000000  &&  ang >= -0x20000000)  ||  s->rx_training_count < 80)
        //    &&
        if (s->rx_training_count != ms_to_symbols(200))
            break;
fprintf(stderr, "YYY %p 0x%x %d\n", s->user_data, ang, s->rx_training_count);
        /* We seem to have a phase reversal */
        /* Slam the carrier frequency into line, based on the total phase drift over the last
           section. Use the shift from the odd bits and the shift from the even bits to get
           better jitter suppression. We need to scale here, or at the maximum specified
           frequency deviation we could overflow, and get a silly answer. */
        /* Step back a few symbols so we don't get ISI distorting things. */
        i = (s->rx_training_count - 60 - 8) & ~3;
        j = i & 0xF;
        ang = (s->angles[j] - s->start_angles[0])/i;
            + (s->angles[j + 1] - s->start_angles[1])/i;
            + (s->angles[j + 2] - s->start_angles[2])/i;
            + (s->angles[j + 3] - s->start_angles[3])/i;
        fprintf(stderr, "Coarse carrier frequency %7.2f (%d)\n", s->rx_carrier_phase_rate*8000.0/(65536.0*65536.0), s->rx_training_count);
printf("0x%X 0x%X 0x%X 0x%X\n", s->start_angles[0], s->start_angles[1], s->start_angles[2], s->start_angles[3]);
printf("0x%X 0x%X 0x%X 0x%X\n", s->angles[j], s->angles[(j + 1) & 0xF], s->angles[(j + 2) & 0xF], s->angles[(j + 3) & 0xF]);
#if 0
        s->rx_carrier_phase_rate += 3*(ang/40);
        /* Make a step shift in the phase, to pull it into line. We need to rotate the RRC filter
           buffer and the equalizer buffer, as well as the carrier phase, for this to play out
           nicely. */
        zz = complex_set(cos(angle*2.0*3.14159/(65536.0*65536.0)), -sin(angle*2.0*3.14159/(65536.0*65536.0)));
        for (i = 0;  i < 2*V22BIS_RX_FILTER_STEPS;  i++)
            s->rx_rrc_filter[i] = complex_mul(&s->rx_rrc_filter[i], &zz);
        for (i = 0;  i <= V22BIS_EQUALIZER_MASK;  i++)
            s->eq_buf[i] = complex_mul(&s->eq_buf[i], &zz);
        s->rx_carrier_phase += angle;
#endif
        break;
    case V22BIS_TRAINING_STAGE_UNSCRAMBLED_0011:
        target = &z;
s->rx_training = V22BIS_TRAINING_STAGE_NORMAL_OPERATION;
        if (++s->rx_training_count > ms_to_symbols(800))
            s->detected_unscrambled_0011_ending = TRUE;
        break;
    case V22BIS_TRAINING_STAGE_SCRAMBLED_ONES_AT_1200:
        target = &z;
        if (++s->rx_training_count > ms_to_symbols(900))
            s->detected_scrambled_ones_or_zeros_at_1200bps = TRUE;
        break;
    case V22BIS_TRAINING_STAGE_PARKED:
    default:
        /* We failed to train! */
        /* Park here until the carrier drops. */
        target = &z;
        break;
    }
    if (s->qam_report)
        s->qam_report(s->qam_user_data, &z, target, s->rx_constellation_state);
}
/*- End of function --------------------------------------------------------*/

int v22bis_rx(v22bis_state_t *s, const int16_t *amp, int len)
{
    int i;
    int16_t sample;
    complex_t z;
    int32_t power;
    float x;

    for (i = 0;  i < len;  i++)
    {
        sample = amp[i];
        power = power_meter_update(&(s->rx_power), sample);
        if (s->carrier_present)
        {
            /* Look for power below -48dBm0 to turn the carrier off */
            if (power < s->carrier_off_power)
            {
                v22bis_rx_restart(s, s->bit_rate);
                s->put_bit(s->user_data, PUTBIT_CARRIER_DOWN);
                continue;
            }
        }
        else
        {
            /* Look for power exceeding -43dBm0 to turn the carrier on */
            if (power < s->carrier_on_power)
                continue;
            s->carrier_present = TRUE;
            s->put_bit(s->user_data, PUTBIT_CARRIER_UP);
        }
        if (s->rx_training != V22BIS_TRAINING_STAGE_PARKED)
        {
            /* Only spend effort processing this data if the modem is not
               parked, after training failure. */
            if (s->rx_training == V22BIS_TRAINING_STAGE_SYMBOL_ACQUISITION)
            {
                /* Only AGC during the initial training */
                s->agc_scaling = 3.60/sqrt(power);
            }
            x = sample*s->agc_scaling;
            /* Shift to baseband */
            z = dds_complexf(&(s->rx_carrier_phase), s->rx_carrier_phase_rate);
            z.re *= x;
            z.im *= x;
            process_baud(s, &z);
        }
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int v22bis_rx_restart(v22bis_state_t *s, int bit_rate)
{
    int i;

    /* If bit_rate is 2400, the real bit rate is negotiated. If bit_rate
       is 1200, the real bit rate is forced to 1200. */
    s->bit_rate = bit_rate;

    memset(s->rx_rrc_filter, 0, sizeof(s->rx_rrc_filter));
    s->rx_rrc_filter_step = 0;

    s->rx_scramble_reg = 0;
    s->rx_scrambler_pattern_count = 0;
    s->rx_training = V22BIS_TRAINING_STAGE_SYMBOL_ACQUISITION;
    s->rx_training_count = 0;
    s->carrier_present = FALSE;

    if (s->caller)
        s->rx_carrier_phase_rate = dds_phase_stepf(2400.0);
    else
        s->rx_carrier_phase_rate = dds_phase_stepf(1200.0);
    s->rx_carrier_phase = 0;
    power_meter_init(&(s->rx_power), 5);
    s->carrier_on_power = power_meter_level(-43);
    s->carrier_off_power = power_meter_level(-48);
    s->agc_scaling = 0.0005;

    s->rx_constellation_state = 0;

    equalizer_reset(s, 0.25);

    s->gardner_integrate = 0;
    s->gardner_step = 64;
    s->rx_baud_phase = 0;
    return 0;
}
/*- End of function --------------------------------------------------------*/

void v22bis_rx_set_qam_report_handler(v22bis_state_t *s, qam_report_handler_t *handler, void *user_data)
{
    s->qam_report = handler;
    s->qam_user_data = user_data;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
