#ifndef VISUAL_CURVE_H
#define VISUAL_CURVE_H

// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

#include "displaylist.h"
#include "num_util.h"

using boost::python::list;
using boost::python::numeric::array;
using boost::python::tuple;
using boost::python::handle;

namespace visual {

class curve : public DisplayObject
{
 private:
	// The pos and color arrays are always overallocated to make appends
	// faster.  Whenever they are read from Python, we return a slice into the
	// array that starts at its beginning and runs up to the last used position
	// in the array.  This is simmilar to many implementations of std::vector<>.
	array pos;
	array color;
	bool antialias;
	bool degenerate;
	double radius;
	// the space allocated for storage so far
	size_t preallocated_size;
	// the number of vectors currently occupying the allocated storage.
	size_t count;

	int curve_slice[512];
	float curve_sc[8];     // 8 == curve_around * 2
 public:
	
	curve();
	curve( const curve& other);

	void append_rgb( vector, double, double, double);
	void append( vector _pos, rgb _color); // Append a single position with new color.
	void append( vector _pos); // Append a single position element, extend color.
	virtual void glRender( rView& view);
	virtual void refreshCache();
	
	boost::python::object get_pos(void);
	boost::python::object get_color(void);
	
	inline bool get_antialias( void) { return antialias; }
	inline double get_radius( void) { return radius; }

	void set_pos( array pos); // An Nx3 array
	void set_pos_l( const list& pos); // A list of vector
	void set_pos_v( const vector& pos); // Interpreted as an initial append().
	void set_color( array color); // An Nx3 array of color
	void set_color_l( const list& color); // A list of vectors
	void set_color_t( const tuple& color); // A single tuple
	void set_color_v( const vector& color); // A single vector
	
	void set_antialias( bool);
	void set_radius( const double& r);
	void set_red( const array& red);
	void set_red_l( const list& red);
	void set_red_d( const double red);
	void set_blue( const array& blue);
	void set_blue_l( const list& blue);
	void set_blue_d( const double blue);
	void set_green( const array& green);
	void set_green_l( const list& green);
	void set_green_d( const double green);
	void set_x( const array& x);
	void set_x_l( const list& x);
	void set_x_d( const double x);
	void set_y( const array& y);
	void set_y_l( const list& y);
	void set_y_d( const double y);
	void set_z( const array& z);
	void set_z_l( const list& z);
	void set_z_d( const double z);
	
 private:
	void thinline( rView& view);
	void thickline( rView& view);
	// Verify that the pos and color arrays have room for the requested length
	// if not, they are grown as required and the old data is copied over.
	void set_length( size_t new_length);
	void set_one_dim_color( int dim, double value);
	void set_one_dim_color( int dim, const array& value);
	void set_one_dim_pos( int dim, double value);
	void set_one_dim_pos( int dim, const array& value);
};

void curve_init_type();

} // !namespace visual

#endif // !VISUAL_CURVE_H
