/*****
*
* Copyright (C) 2003 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

#ifdef __linux__
# define _XOPEN_SOURCE 600
# include <stdio.h>
# undef _XOPEN_SOURCE
#else
# include <stdio.h>
#endif /* __linux__ */

#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>

/*
 * glibc2 won't define strptime()
 * unless _XOPEN_SOURCE is defined.
 */
 #include <time.h>


#include <libprelude/prelude-log.h>
#include "log-common.h"



static int format_syslog_header(const char *buf, struct timeval *tv, char host[256], char tag[33]) 
{
        char *log;
        int ret, i;
        time_t now;
        struct tm localtime;

        if ( ! buf )
                return -1;
        
        /*
         * We first get the localtime from this system,
         * so that all the struct tm member are filled.
         *
         * As syslog header doesn't contain a complete timestamp, this
         * avoid us some computation error.
         */
        now = time(NULL);
        if ( ! localtime_r(&now, &localtime) ) {
                log(LOG_ERR, "error getting local time.\n");
                return -1;
        }

        /*
         * Now, let's format the timestamp provided in the syslog message.
         * strptime() return a pointer to the first non matched character.
         */
        log = strptime(buf, "%b %d %H:%M:%S", &localtime);
        if ( ! log ) 
                return -1;
        
        ret = sscanf(log, "%255s %32s", host, tag);
        if ( ret != 2 ) 
                return -1;

        /*
         * the tag end as soon as we meet a non alpha numeric character.
         */
        for ( i = 0; i < strlen(tag); i++ ) {
                
                if ( ! isalnum((int) tag[i]) ) {
                        tag[i] = '\0';
                        break;
                }
        }

        /*
         * convert back to a timeval.
         */
        (*tv).tv_usec = 0;
        (*tv).tv_sec = mktime(&localtime);

        return 0;
}




static char *get_hostname(void) 
{
        int ret;
        char out[256];
                
        ret = gethostname(out, sizeof(out));
        if ( ret < 0 )
                return NULL;
        
        return strdup(out);
}




log_container_t *log_container_new(void)
{
        log_container_t *lc;

        lc = calloc(1, sizeof(*lc));
        if ( ! lc ) {
                log(LOG_ERR, "memory exhausted.\n");
                return NULL;
        }

        gettimeofday(&lc->tv, NULL);
        lc->target_hostname = get_hostname();
        
        return lc;
}




int log_container_set_log(log_container_t *lc, const char *entry) 
{
        int ret;
        char host[256], tag[33];
                
        lc->log = strdup(entry);
        if ( ! lc->log ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }

        ret = format_syslog_header(entry, &lc->tv, host, tag);
        if ( ret < 0 ) {
                gettimeofday(&lc->tv, NULL);
                return 0;
        }
        
        lc->target_program = strdup(tag);
        if ( ! lc->target_program ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }

        if ( lc->target_hostname )
                free(lc->target_hostname);
        
        lc->target_hostname = strdup(host);
        if ( ! lc->target_hostname ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }
        
        return 0;
}




int log_container_set_source(log_container_t *log, const char *source) 
{
        log->source = strdup(source);
        if ( ! log->source ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }

        return 0;
}




void log_container_delete(log_container_t *lc)
{
        if ( lc->target_hostname )
                free(lc->target_hostname);

        if ( lc->target_program )
                free(lc->target_program);

        if ( lc->source )
                free(lc->source);

        if ( lc->log )
                free(lc->log);
        
	free(lc);
}






