/***************************************************************************
                          xmlutils.cpp  -  description
                             -------------------
    begin                : Mit Mai 7 2003
    copyright            : (C) 2003 by Dominik Seichter
    email                : domseichter@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "xmlutils.h"
#include "definition.h"
#include "mycanvasview.h"
#include "mycanvasitem.h"
#include "labelutils.h"
#include "measurements.h"

// Qt includes
#include <qdom.h>

// KDE includes
#include <klocale.h>
#include <kmessagebox.h>

XMLUtils::XMLUtils()
{
    // read barcodes saved by kbarcode < 1.3.0
    if( !legacy.count() ) {
        legacy.insert( "0", "any" );
        legacy.insert( "1", "ean" );
        legacy.insert( "2", "upc" );
        legacy.insert( "3", "isbn" );
        legacy.insert( "4", "code39" );
        legacy.insert( "5", "code128" );
        legacy.insert( "6", "code128c" );
        legacy.insert( "7", "code128b" );
        legacy.insert( "8", "i25" );
        legacy.insert( "9", "i28raw" );
        legacy.insert( "10", "cbr" );
        legacy.insert( "11", "msi" );
        legacy.insert( "12", "pls" );
        legacy.insert( "13", "code93" );
        legacy.insert( "14", "msi" );
        legacy.insert( "15", "code39 -c" );
        legacy.insert( "16", "i25 -c" );
    }
}

XMLUtils::~XMLUtils()
{
}

QMap<QString,QString> XMLUtils::legacy;

void XMLUtils::writeXMLColor( QDomElement* tag, const QString & prefix, QColor c ) const
{
    tag->setAttribute( prefix + "r", c.red() );
    tag->setAttribute( prefix + "g", c.green() );
    tag->setAttribute( prefix + "b", c.blue() );
}

QColor XMLUtils::readXMLColor( QDomElement* tag, const QString & prefix, QColor c )
{
    int r = c.red();
    int g = c.green();
    int b = c.blue();

    r = tag->attribute( prefix + "r", QString("%1").arg( r ) ).toInt();
    g = tag->attribute( prefix + "g", QString("%1").arg( g ) ).toInt();
    b = tag->attribute( prefix + "b", QString("%1").arg( b ) ).toInt();

    return QColor( r, g, b);
}

void XMLUtils::writeXMLRect( QDomElement* tag, QRect r, MyCanvasView* cv ) const
{
    if( cv ) {
        QPoint translation = cv->getTranslation();
        r.moveBy( -translation.x(), -translation.y() );
    }
    
    tag->setAttribute( "x", r.x() );
    tag->setAttribute( "y", r.y() );
    tag->setAttribute( "width", r.width() );
    tag->setAttribute( "height", r.height() );

    LabelUtils l;
    tag->setAttribute( "x_mm", l.pixelToMm( r.x(), cv, LabelUtils::DpiX ) );
    tag->setAttribute( "y_mm", l.pixelToMm( r.y(), cv, LabelUtils::DpiY ) );
    tag->setAttribute( "width_mm", l.pixelToMm( r.width(), cv, LabelUtils::DpiX ) );
    tag->setAttribute( "height_mm", l.pixelToMm( r.height(), cv, LabelUtils::DpiY ) );
}

QRect XMLUtils::readXMLRect( QDomElement* tag, MyCanvasView* cv )
{
    QRect r;
    int x = tag->attribute( "x", "0" ).toInt();
    int y = tag->attribute( "y", "0" ).toInt();

    if( cv ) {
        QPoint translation  = cv->getTranslation();
        x += translation.x();
        y += translation.y();
    }

    r.setX( x );
    r.setY( y );    
    r.setWidth( tag->attribute( "width", "-1" ).toInt() );
    r.setHeight( tag->attribute( "height", "-1" ).toInt() );
    return r;
}

void XMLUtils::writeDefinition( QDomElement* tag, Definition* d ) const
{
    // do not delete measurements!
    Measurements* m = d->getMeasurements();

    tag->setAttribute("producer", d->getProducer());
    tag->setAttribute("type", d->getType());

    tag->setAttribute("width", m->widthMM());
    tag->setAttribute("height", m->heightMM());
    tag->setAttribute("gap_left", m->gapLeftMM());
    tag->setAttribute("gap_top", m->gapTopMM());
    tag->setAttribute("gap_v", m->gapVMM());
    tag->setAttribute("gap_h", m->gapHMM());
    tag->setAttribute("num_v", m->numV());
    tag->setAttribute("num_h", m->numH());
}

Definition* XMLUtils::readDefinition( QDomElement* tag, QWidget* parent )
{
    int label_def_id = tag->text().toInt();
    QString producer = tag->attribute("producer", "" );
    QString type = tag->attribute("type", "" );

    // The label was written by KBarcode <= 1.2.0
    Definition* d = new Definition( label_def_id );
    if( type.isEmpty() && producer.isEmpty() ) {
        KMessageBox::information( parent,
            i18n("<qt>This appears file appears to be created by an older version of KBarcode.<br>"
                 "Please check if the used label definition is correct:<br>"
                 "<b>") + d->getProducer() + " " + d->getType() + "</b><br>"
                 "You can change the label definition at <i>Edit->Change Label...</i></qt>");

        return d;
    }

    // The id does not match the correct type and producer,
    // try to find the correct one
    if( (d->getType() != type || d->getProducer() != producer) &&
        !(type.isEmpty() && producer.isEmpty() ) ) {

        int id = label_def_id;
        QStringList p = Definition::getProducers();
        if( !p.contains( producer ) ) {
            for( unsigned int i = 0; i < p.count(); i++ ) {
                id = Definition::getClosest( p[i], type );
                if( id >= 0 )
                    break;
            }
        } else
            id = Definition::getClosest( producer, type );

        if( id != label_def_id && id >= 0 ) {
            d->setId( id );
            return d;
        }
    }

    // Everything failed
    // Read the measurements from the file
    Measurements m;
    m.setWidthMM( tag->attribute("width", I2S( m.widthMM() ) ).toDouble() );
    m.setHeightMM( tag->attribute("height", I2S( m.heightMM() )).toDouble() );
    m.setGapLeftMM( tag->attribute("gap_left", I2S( m.gapLeftMM() )).toDouble() );
    m.setGapTopMM( tag->attribute("gap_top", I2S( m.gapTopMM() )).toDouble() );
    m.setGapVMM( tag->attribute("gap_v", I2S( m.gapVMM() )).toDouble() );
    m.setGapHMM( tag->attribute("gap_h", I2S( m.gapHMM() )).toDouble() );
    m.setNumV( tag->attribute("num_v", I2S( m.numH() )).toInt() );
    m.setNumH( tag->attribute("num_h", I2S( m.numV() )).toInt() );

    d->setId( Definition::write( &m, type, producer ) );
        
    return d;
}

void XMLUtils::writeBarcode( QDomElement* tag, const barcodeData* data, const CanvasBarcode* code, const MyCanvasView* cv, bool cache ) const
{
    if( code && cv && !cache ) {
        QPoint p( int(code->x() - cv->getTranslation().x()), int(code->y() - cv->getTranslation().y()) );
        LabelUtils l;
        tag->setAttribute( "x_mm", l.pixelToMm( p.x(), cv, LabelUtils::DpiX ) );
        tag->setAttribute( "y_mm", l.pixelToMm( p.y(), cv, LabelUtils::DpiY ) );
    
        tag->setAttribute( "x", p.x() );
        tag->setAttribute( "y", p.y() );
        tag->setAttribute( "z", code->z() );
    }

    if( !cache ) {
        /*
         * This values are not needed for the barcode cache.
         */
        tag->setAttribute( "margin", data->margin );
        tag->setAttribute( "rotation", data->rotation );
        tag->setAttribute( "cut", data->cut );
        tag->setAttribute( "caption", data->xml.caption );

        /*
         * This values are only needed for !cache and for sequences
         */
        tag->setAttribute( "sequenceenabled", data->sequence.enabled );
        if( data->sequence.enabled ) {
            tag->setAttribute( "sequencemode", data->sequence.mode );
            tag->setAttribute( "sequencestep", data->sequence.step );
            tag->setAttribute( "sequencestart", data->sequence.start );
        }
    }

    tag->setAttribute( "type", data->type );
    tag->setAttribute( "text", data->text );
    tag->setAttribute( "scale", data->scale*1000 );

    if( BarCode::hasFeature( data->type, PDF417BARCODE ) ) {
        tag->setAttribute( "pdf417.row", data->pdf417.row );
        tag->setAttribute( "pdf417.col", data->pdf417.col );
        tag->setAttribute( "pdf417.err", data->pdf417.err );
    }

    if( BarCode::hasFeature( data->type, DATAMATRIX ) )
        tag->setAttribute( "datamatrix.size", data->datamatrix.size );

    if( BarCode::hasFeature( data->type, TBARCODE ) ) {
        tag->setAttribute( "tbarcode.modulewidth", data->tbarcode.modulewidth );
        tag->setAttribute( "tbarcode.escape", data->tbarcode.escape );
        tag->setAttribute( "tbarcode.above", data->tbarcode.above );
        tag->setAttribute( "tbarcode.autocorrect", data->tbarcode.autocorrect );
        tag->setAttribute( "tbarcode.checksum", data->tbarcode.checksum );
    }
        
    QDomElement texttag = tag->ownerDocument().createElement( "value" );
    texttag.appendChild( tag->ownerDocument().createTextNode( data->value ) );

    tag->appendChild( texttag );
}

barcodeData XMLUtils::readBarcode( QDomElement* tag, MyCanvasView* cv )
{
    barcodeData bcode;
    BarCode::fillDefault( &bcode );

    if( tag->tagName() == "barcode" ) {
        bcode.margin = tag->attribute("margin", "10" ).toInt();
        bcode.rotation = tag->attribute("rotation", "0" ).toInt();
        bcode.scale = tag->attribute("scale", "1000" ).toDouble() / 1000;
        bcode.cut = tag->attribute("cut", "1.0" ).toDouble();
        bcode.type = tag->attribute("type", "code39" );

        /*
         * check for encoding types saved by kbarcode <= 1.2.0
         */
         if( legacy.contains( bcode.type ) )
            bcode.type = legacy[bcode.type];

        bcode.text = tag->attribute("text", "0" ).toInt();
        bcode.xml.caption = tag->attribute("caption", "Static" );

        bcode.xml.x = tag->attribute( "x", "0" ).toInt();
        bcode.xml.y = tag->attribute( "y", "0" ).toInt();

        if( cv ) {
            /*
            // check if positions are already saved as mm
            if( tag->attribute( "x_mm", "MM" ) != "MM" ) {
                LabelUtils l;
                bcode.xml.x = l.mmToPixel( (int)tag->attribute( "x_mm", "0" ).toInt(), cv, LabelUtils::DpiX );
                bcode.xml.y = l.mmToPixel( (int)tag->attribute( "y_mm", "0" ).toInt(), cv, LabelUtils::DpiY );
            }
            */
            bcode.xml.x += cv->getTranslation().x();
            bcode.xml.y += cv->getTranslation().y();
        }

        if( BarCode::hasFeature( bcode.type, PDF417BARCODE ) ) {
            bcode.pdf417.row = tag->attribute( "pdf417.row", "24" ).toInt();
            bcode.pdf417.col = tag->attribute( "pdf417.col", "8" ).toInt();
            bcode.pdf417.err = tag->attribute( "pdf417.err", "5" ).toInt();
        }

        if( BarCode::hasFeature( bcode.type, DATAMATRIX ) )
            bcode.datamatrix.size = tag->attribute( "datamatrix.size", "0" ).toInt();

        if( BarCode::hasFeature( bcode.type, TBARCODE ) ) {
            bcode.tbarcode.modulewidth  = tag->attribute( "tbarcode.modulewidth", "0.353" ).toDouble();
            bcode.tbarcode.escape = tag->attribute( "tbarcode.escape", "0" ).toInt();
            bcode.tbarcode.above = tag->attribute( "tbarcode.above", "0" ).toInt();
            bcode.tbarcode.autocorrect = tag->attribute ( "tbarcode.autocorrect", "0" ).toInt();
            bcode.tbarcode.checksum = tag->attribute( "tbarcode.checksum", "0" ).toInt();
        }

        bcode.sequence.enabled = tag->attribute( "sequenceenabled", "0" ).toInt();
        if( bcode.sequence.enabled ) {
            bcode.sequence.mode = tag->attribute( "sequencemode", "0" ).toInt();
            bcode.sequence.step = tag->attribute( "sequencestep", "1" ).toInt();
            bcode.sequence.start = tag->attribute( "sequencestart", "0" ).toInt();
        }

        QDomNode n = tag->firstChild();
        while( !n.isNull() ) {
            QDomElement e = n.toElement(); // try to convert the node to an element.
            if( !e.isNull() )
                if( e.tagName() == "value" )
                    bcode.value = e.text();

            n = n.nextSibling();
        }
    }

    return bcode;
}
