// --------------------------------------------------------------------
// The Ipe document, with PDF support
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipemodel.h"

#include "ipeprefs.h"
#include "ipeq.h"

#include "ipelatex.h"

#include <qbuffer.h>
#include <qfile.h>
#include <qdir.h>
#include <qtextstream.h>

// --------------------------------------------------------------------

/*! \class IpeModel
  \brief Support for loading/saving/running Latex on IpeDocument.

  This class supports running pdflatex to generate the PDF output for
  text objects, and adds I/O using QFile's to IpeDocument.

  All members of the class are static.

 */

IpeDocument *IpeModel::New(const QString &fname, int &reason)
{
  QFile f(fname);
  reason = -1;
  if (!f.open(IO_ReadOnly))
    return 0;
  XmlQSource source(&f);
  IpeDocument::TFormat format = IpeDocument::FileFormat(source);
  f.at(0); // rewind
  IpeDocument *self = IpeDocument::New(source, format, reason);
  f.close();
  return self;
}

bool IpeModel::Save(const IpeDocument *doc, const QString &fname,
		    IpeString creator, IpeDocument::TFormat format,
		    uint flags)
{
  QFile f(fname);
  if (!f.open(IO_WriteOnly))
    return 0;
  IpeQStream stream(&f);
  bool result = doc->Save(stream, creator, format, flags,
			  IpePreferences::Static()->iCompressLevel);
  f.close();
  return result;
}

bool IpeModel::AddStyleSheet(IpeDocument *doc, const QString &fname)
{
  QFile file(fname);
  if (!file.open(IO_ReadOnly))
    return false;
  XmlQSource source(&file);
  bool result = doc->AddStyleSheet(source);
  file.close();
  return result;
}

// --------------------------------------------------------------------

//! Run PdfLatex
/*! Prepend \a styleDirs to Latex include path. */
int IpeModel::RunLatex(IpeDocument *doc, bool needLatex,
		       const QStringList &styleDirs, QString &texLog)
{
  IpePreferences *prefs = IpePreferences::Static();
  texLog = QString::null;
  IpeLatex converter;
  int count = 0;
  for (IpeDocument::iterator it = doc->begin(); it != doc->end(); ++it)
    count += converter.ScanPage(*it);
  if (count == 0)
    return ErrNoText;
  // First we need a temporary directory
  QDir temp = QDir(prefs->iLatexDir);
  if (!temp.exists())
    // create Pdflatex directory if it doesn't exist
    QDir().mkdir(temp.absPath());
  if (!temp.exists())
    return ErrNoDir;

  QString texFile = temp.filePath("text.tex");
  QString pdfFile = temp.filePath("text.pdf");
  QString logFile = temp.filePath("text.log");
  QFile::remove(logFile);

  QFile file(texFile);
  if(!file.open(IO_WriteOnly))
    return -1;
  IpeQStream stream(&file);
  int err = converter.CreateLatexSource(stream, doc->Properties().iPreamble,
					doc->StyleSheet());
  file.close();

  if (err < 0)
    return ErrWritingSource;
  if (err == 0 && !needLatex)
    return ErrAlreadyHaveForm;
  // Latex source has been prepared correctly

  // set up TEXINPUTS (but special handling for MikTeX below)
  QCString ti = "TEXINPUTS=";
  bool putTi = false;
  if (styleDirs.count() > 0) {
    putTi = true;
    for (uint i = 0; i < styleDirs.count(); ++i)
      ti.append((styleDirs[i] + ":").local8Bit());
  }
  if (!prefs->iTexInputs.empty()) {
    putTi = true;
    ti.append(prefs->iTexInputs.CString());
  }
  if (putTi)
    putenv(qstrdup(ti.data()));

#ifdef WIN32
  QString s = temp.filePath("runlatex.bat");
  QFile f(s);
  f.open(IO_WriteOnly);
  QTextStream batFile(&f);
  QString dirPath = QDir::convertSeparators(temp.path());
  if (dirPath.length() > 2 && dirPath[1] == ':')
    batFile << dirPath.left(2) << "\r\n";
  batFile << "cd \"" << dirPath << "\"\r\n";
  batFile << "\"" << prefs->iPdfLatex << "\"";
  if (prefs->iMikTeX) {
    for (uint i = 0; i < styleDirs.count(); ++i) {
      batFile << " -include-directory=\""
	     << QDir::convertSeparators(styleDirs[i]) << "\"";
    }
  }
  batFile << " text.tex\r\n";
  f.close();
  s = "call \"" + QDir::convertSeparators(s) + "\"";
  system(s.local8Bit());
#else
  QCString s = "cd " + QFile::encodeName(temp.path()) + "; " +
    QFile::encodeName(prefs->iPdfLatex) + " text.tex";
  system(s.data());
#endif
  // Check log file for Pdflatex version and errors
  QFile log(logFile);
  if (!log.open(IO_ReadOnly))
    return ErrLatex;
  texLog = QString(log.readAll());
  log.close();
  if (texLog.left(14) != "This is pdfTeX" &&
      texLog.left(15) != "This is pdfeTeX")
    return ErrLatex;
  prefs->iMikTeX = (texLog.left(80).find("MikTeX") >= 0);
  int i = texLog.find('-');
  if (i < 0)
    return ErrLatex;
  QString version = texLog.mid(i+1, 30);
  ipeDebug("pdfTeX version %s", version.latin1());
  if (version[0] == '1' && '0' <= version[1] && version[1] <= '3')
    return ErrOldPdfLatex;
  if (version.left(2) == "14" && version.left(3) < "14f")
    return ErrOldPdfLatex;
  // Check for error
  if (texLog.find("\n!") >= 0)
    return ErrLatex;

  QFile pdfF(pdfFile);
  if (!pdfF.open(IO_ReadOnly))
    return false;
  XmlQSource source(&pdfF);
  if (converter.ReadPdf(source) && converter.UpdateTextObjects()) {
    doc->SetFontPool(converter.TakeFontPool());
    ipeDebug("Pdflatex run completed successfully");
    return ErrNone;
  }
  return ErrLatexOutput;
}

// --------------------------------------------------------------------
