// --------------------------------------------------------------------
// Overlay for scissors
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipescissors.h"
#include "ipecanvas.h"

#include <qevent.h>
#include <qpainter.h>

// --------------------------------------------------------------------

/*! \class IpeScissors
  \brief Overlay for cutting IpePath objects into pieces.
*/

//! IpeScissors starts selection.
IpeScissors::IpeScissors(IpeCanvas *canvas, IpePath *path,
			 IpePage *page, IpePage::iterator it)
  : IpeOverlay(canvas), iPage(page), iIt(it)
{
  assert(path->NumSubPaths() == 1);
  iSP = path->SubPath(0);
  iCanvas->SetDoubleBuffer(true);
  iCanvas->SetDimmed(true);
  iM = path->Matrix();
  iNoCuts = iSP->Closed() ? 2 : 1;
  iSSP = iSP->AsSegs();
  iEll = iSP->AsEllipse();
  iCSP = iSP->AsClosedSpline();
  if (iEll) {
    iPointOn = iEll->Matrix() * IpeVector(1,0);
    iPos[0] = iPointOn;
    iPos[1] = iEll->Matrix() * IpeVector(-1,0);
  } else if (iCSP) {
    std::vector<IpeBezier> bez;
    iCSP->Beziers(bez);
    iPointOn = bez[0].iV[0];
    iPos[0] = iPointOn;
    iPos[1] = bez[0].iV[3];
  } else {
    assert(iSSP);
    iPointOn = iSSP->Segment(0).CP(0);
    iPos[0] = iPointOn;
    iPos[1] = iSSP->Segment(0).Last();
  }
  iMoving = -1;
  Explain();
}

void IpeScissors::Explain() const
{
  QString s;
  s += QObject::tr("Space: cut object here");
  iCanvas->Message(s);
}

void IpeScissors::NewObj(IpeSubPath *sp, bool second)
{
  IpePath *path = iIt->Object()->AsPath();
  IpeAllAttributes attr;
  attr.iStroke = path->Stroke();
  attr.iFill = path->Fill();
  attr.iLineWidth = path->LineWidth();
  attr.iDashStyle = path->DashStyle();
  IpePath *obj = new IpePath(attr);
  if (second)
    obj->SetForwardArrow(path->ForwardArrow());
  if (!second)
    obj->SetBackwardArrow(path->BackwardArrow());
  obj->SetStrokeStyle(path->StrokeStyle());
  obj->AddSubPath(sp);
  obj->SetMatrix(iM);
  iPage->push_back(IpePgObject(IpePgObject::ESecondary, 0, obj));
}

bool IpeScissors::CutEllipse()
{
  IpeSegmentSubPath *sp1 = new IpeSegmentSubPath;
  sp1->AppendArc(iEll->Matrix(), iPos[0], iPos[1]);
  IpeSegmentSubPath *sp2 = new IpeSegmentSubPath;
  sp2->AppendArc(iEll->Matrix(), iPos[1], iPos[0]);
  NewObj(sp1, false);
  NewObj(sp2, true);
  return true;
}

int IpeScissors::FindPoint(const IpeVector &v)
{
  int seg = 0;
  double d = 1.0;
  double d1;
  int beg = iSSP->Closed() ? -1 : 0;
  IpeVector u[2];
  for (int i = beg; i < iSSP->NumSegments(); ++i) {
    IpePathSegment segm = (i < 0) ? iSSP->ClosingSegment(u) :
      iSSP->Segment(i);
    d1 = segm.Distance(v, IpeMatrix(), d);
    if (d1 < d) {
      d = d1;
      seg = i;
    }
  }
  return seg;
}

bool IpeScissors::Cut()
{
  /* iPos are already in path coordinate system!
  IpeMatrix inv = iM.Inverse();
  iPos[0] = inv * iPos[0];
  iPos[1] = inv * iPos[1];
  */

  if (iEll)
    return CutEllipse();
  else if (iCSP)
    return CutClosedSpline();
  else if (iSP->Closed())
    return CutClosedSegments();
  else
    return CutSegments();
}

bool IpeScissors::CutClosedSpline()
{
  iCanvas->Message(QObject::tr("Cutting splines not yet implemented"));
  return false;
}

void IpeScissors::Swap()
{
  IpeVector t = iPos[0];
  iPos[0] = iPos[1];
  iPos[1] = t;
}

void IpeScissors::AppendSegs(IpeSegmentSubPath *sp, int i0, int i1)
{
  for (int i = i0; i <= i1; ++i)
    sp->Append(iSSP->Segment(i));
}

bool IpeScissors::SplitSegment(IpeSegmentSubPath *sp1, IpeSegmentSubPath *sp2,
			       const IpeVector &pos, int seg)
{
  IpeVector u[2];
  IpePathSegment segm = (seg < 0) ? iSSP->ClosingSegment(u) :
    iSSP->Segment(seg);
  if (segm.Type() == IpePathSegment::ESegment) {
    sp2->AppendSegment(pos, segm.CP(1));
    sp1->AppendSegment(segm.CP(0), pos);
  } else if (segm.Type() == IpePathSegment::EArc) {
    sp2->AppendArc(segm.Matrix(), pos, segm.CP(1));
    sp1->AppendArc(segm.Matrix(), segm.CP(0), pos);
  } else {
    iCanvas->Message(QObject::tr("Cannot cut this object at this position"));
    return false;
  }
  return true;
}

bool IpeScissors::CutClosedSegments()
{
  int seg1 = FindPoint(iPos[0]);
  int seg2 = FindPoint(iPos[1]);

  if (seg2 < seg1) {
    Swap();
    int t = seg1;
    seg1 = seg2;
    seg2 = t;
  }

  ipeDebug("Splitting segments %d and %d", seg1, seg2);

  IpeAutoPtr<IpeSegmentSubPath> sp1(new IpeSegmentSubPath);
  IpeAutoPtr<IpeSegmentSubPath> sp2(new IpeSegmentSubPath);

  IpeVector u[2];
  IpePathSegment closing = iSSP->ClosingSegment(u);

  if (seg1 == seg2) {
    // set up sp1 to be the short segment, sp2 the long segment
    IpePathSegment segm = (seg1 < 0) ? closing : iSSP->Segment(seg1);

    if (segm.Type() == IpePathSegment::ESegment) {
      if ((iPos[1] - segm.CP(0)).SqLen() < (iPos[0] - segm.CP(0)).SqLen())
	Swap();
      sp1->AppendSegment(iPos[0], iPos[1]);
      sp2->AppendSegment(iPos[1], segm.CP(1));
      AppendSegs(sp2.Ptr(), seg1 + 1, iSSP->NumSegments() - 1);
      if (seg1 >= 0) {
	sp2->Append(closing);
	AppendSegs(sp2.Ptr(), 0, seg1 - 1);
      }
      sp2->AppendSegment(segm.CP(0), iPos[0]);
    } else if (segm.Type() == IpePathSegment::EArc) {
      IpeMatrix inv = segm.Matrix().Inverse();
      IpeAngle alpha = (inv * segm.CP(0)).Angle();
      IpeAngle alpha0 = (inv * iPos[0]).Angle();
      IpeAngle alpha1 = (inv * iPos[1]).Angle();
      if (alpha1.LiesBetween(alpha, alpha0))
	Swap();
      sp1->AppendArc(segm.Matrix(), iPos[0], iPos[1]);
      sp2->AppendArc(segm.Matrix(), iPos[1], segm.CP(1));
      AppendSegs(sp2.Ptr(), seg1 + 1, iSSP->NumSegments() - 1);
      sp2->Append(closing);
      AppendSegs(sp2.Ptr(), 0, seg1 - 1);
      sp2->AppendArc(segm.Matrix(), segm.CP(0), iPos[0]);
    } else {
      iCanvas->Message
	(QObject::tr("Cannot cut this object at this position"));
      return false;
    }
  } else {
    // two different segments: sp1 from seg1 to seg2, sp2 from seg2 to seg1
    IpeSegmentSubPath sp1a;
    IpeSegmentSubPath sp2a;

    if (!SplitSegment(&sp1a, sp1.Ptr(), iPos[0], seg1) ||
	!SplitSegment(&sp2a, sp2.Ptr(), iPos[1], seg2))
      return false;

    AppendSegs(sp1.Ptr(), seg1 + 1, seg2 - 1);
    AppendSegs(sp2.Ptr(), seg2 + 1, iSSP->NumSegments() - 1);
    if (seg1 >= 0) {
      sp2->Append(closing);
      AppendSegs(sp2.Ptr(), 0, seg1 - 1);
    }
    sp1->Append(sp2a.Segment(0));
    sp2->Append(sp1a.Segment(0));
  }

  NewObj(sp1.Take(), false);
  NewObj(sp2.Take(), true);
  return true;
}

bool IpeScissors::CutSegments()
{
  int seg = FindPoint(iPos[0]);
  ipeDebug("Splitting segment %d", seg);

  IpeAutoPtr<IpeSegmentSubPath> sp1(new IpeSegmentSubPath);
  IpeAutoPtr<IpeSegmentSubPath> sp2(new IpeSegmentSubPath);
  IpeSegmentSubPath spa;

  if (!SplitSegment(&spa, sp2.Ptr(), iPos[0], seg))
    return false;

  AppendSegs(sp2.Ptr(), seg + 1, iSSP->NumSegments() - 1);
  AppendSegs(sp1.Ptr(), 0, seg - 1);
  sp1->Append(spa.Segment(0));

  NewObj(sp1.Take(), false);
  NewObj(sp2.Take(), true);
  return true;
}

void IpeScissors::KeyPress(QKeyEvent *ev)
{
  if (ev->key() == Qt::Key_Space) {
    if (Cut()) {
      iPage->erase(iIt);
      iPage->EnsurePrimarySelection();
      iPage->SetEdited(true);
      iCanvas->FinishOverlay();
    }
    ev->accept();
  } else
    ev->ignore();
}

void IpeScissors::Draw(QPaintEvent *, QPainter *qPainter) const
{
  IpeOverlayPainter painter(iCanvas->StyleSheet(), qPainter);
  painter.Transform(iCanvas->CanvasTfm());
  painter.Transform(iM);

  qPainter->setPen(Qt::magenta);
  qPainter->setBrush(QBrush::NoBrush);

  painter.NewPath();
  iSP->Draw(painter);
  painter.DrawPath();
  for (int i = 0; i < iNoCuts; ++i) {
    qPainter->setPen(QPen::NoPen);
    qPainter->setBrush(Qt::red);
    IpeVector v = painter.Matrix() * iPos[i];
    int x = int(v.iX);
    int y = int(v.iY);
    qPainter->drawEllipse(x - 4, y - 4, 9, 9);
    if (i == iMoving) {
      qPainter->setPen(Qt::red);
      qPainter->setBrush(Qt::NoBrush);
      qPainter->drawEllipse(x - 8, y - 8, 17, 17);
    }
  }
}

void IpeScissors::MouseMove(QMouseEvent *)
{
  if (iMoving >= 0) {
    IpeVector mouse = iCanvas->Pos();
    IpeVector pos = iPointOn;
    double bound = (mouse - iPointOn).Len();
    iSP->SnapBnd(mouse, iM, pos, bound);
    iPos[iMoving] = iM.Inverse() * pos;
    iCanvas->UpdateOverlay();
  }
}

void IpeScissors::MousePress(QMouseEvent *ev)
{
  IpeVector v = iCanvas->Pos();
  if (iNoCuts > 1) {
    if ((v - iPos[0]).SqLen() < (v - iPos[1]).SqLen())
      iMoving = 0;
    else
      iMoving = 1;
  } else
    iMoving = 0;
  MouseMove(ev);
  Explain();
}

void IpeScissors::MouseRelease(QMouseEvent *ev)
{
  if (iMoving >= 0)
    MouseMove(ev);
  iMoving = -1;
  Explain();
}

// --------------------------------------------------------------------
