#include <math.h>
#include <gtk/gtk.h>

#define DETAIL(xx) ((detail) && (!strcmp(xx, detail)))

extern GtkStyleClass gtkbe_default_class;

/* internal functions */
static void         draw_hline(GtkStyle * style,
			       GdkWindow * window,
			       GtkStateType state_type,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       gint x1,
			       gint x2,
			       gint y);
static void         draw_vline(GtkStyle * style,
			       GdkWindow * window,
			       GtkStateType state_type,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       gint y1,
			       gint y2,
			       gint x);
static void         draw_shadow(GtkStyle * style,
				GdkWindow * window,
				GtkStateType state_type,
				GtkShadowType shadow_type,
				GdkRectangle * area,
				GtkWidget * widget,
				gchar * detail,
				gint x,
				gint y,
				gint width,
				gint height);

static void         draw_polygon(GtkStyle * style,
				 GdkWindow * window,
				 GtkStateType state_type,
				 GtkShadowType shadow_type,
				 GdkRectangle * area,
				 GtkWidget * widget,
				 gchar * detail,
				 GdkPoint * point,
				 gint npoints,
				 gint fill);
static void         draw_arrow(GtkStyle * style,
			       GdkWindow * window,
			       GtkStateType state_type,
			       GtkShadowType shadow_type,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       GtkArrowType arrow_type,
			       gint fill,
			       gint x,
			       gint y,
			       gint width,
			       gint height);
static void         draw_diamond(GtkStyle * style,
				 GdkWindow * window,
				 GtkStateType state_type,
				 GtkShadowType shadow_type,
				 GdkRectangle * area,
				 GtkWidget * widget,
				 gchar * detail,
				 gint x,
				 gint y,
				 gint width,
				 gint height);
static void         draw_oval(GtkStyle * style,
			      GdkWindow * window,
			      GtkStateType state_type,
			      GtkShadowType shadow_type,
			      GdkRectangle * area,
			      GtkWidget * widget,
			      gchar * detail,
			      gint x,
			      gint y,
			      gint width,
			      gint height);
static void         draw_string(GtkStyle * style,
				GdkWindow * window,
				GtkStateType state_type,
				GdkRectangle * area,
				GtkWidget * widget,
				gchar * detail,
				gint x,
				gint y,
				const gchar * string);
static void         draw_box(GtkStyle * style,
			     GdkWindow * window,
			     GtkStateType state_type,
			     GtkShadowType shadow_type,
			     GdkRectangle * area,
			     GtkWidget * widget,
			     gchar * detail,
			     gint x,
			     gint y,
			     gint width,
			     gint height);
static void         draw_flat_box(GtkStyle * style,
				  GdkWindow * window,
				  GtkStateType state_type,
				  GtkShadowType shadow_type,
				  GdkRectangle * area,
				  GtkWidget * widget,
				  gchar * detail,
				  gint x,
				  gint y,
				  gint width,
				  gint height);
static void         draw_check(GtkStyle * style,
			       GdkWindow * window,
			       GtkStateType state_type,
			       GtkShadowType shadow_type,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       gint x,
			       gint y,
			       gint width,
			       gint height);
static void         draw_option(GtkStyle * style,
				GdkWindow * window,
				GtkStateType state_type,
				GtkShadowType shadow_type,
				GdkRectangle * area,
				GtkWidget * widget,
				gchar * detail,
				gint x,
				gint y,
				gint width,
				gint height);
static void         draw_cross(GtkStyle * style,
			       GdkWindow * window,
			       GtkStateType state_type,
			       GtkShadowType shadow_type,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       gint x,
			       gint y,
			       gint width,
			       gint height);
static void         draw_ramp(GtkStyle * style,
			      GdkWindow * window,
			      GtkStateType state_type,
			      GtkShadowType shadow_type,
			      GdkRectangle * area,
			      GtkWidget * widget,
			      gchar * detail,
			      GtkArrowType arrow_type,
			      gint x,
			      gint y,
			      gint width,
			      gint height);
static void         draw_tab(GtkStyle * style,
			     GdkWindow * window,
			     GtkStateType state_type,
			     GtkShadowType shadow_type,
			     GdkRectangle * area,
			     GtkWidget * widget,
			     gchar * detail,
			     gint x,
			     gint y,
			     gint width,
			     gint height);
static void         draw_shadow_gap(GtkStyle * style,
				    GdkWindow * window,
				    GtkStateType state_type,
				    GtkShadowType shadow_type,
				    GdkRectangle * area,
				    GtkWidget * widget,
				    gchar * detail,
				    gint x,
				    gint y,
				    gint width,
				    gint height,
				    GtkPositionType gap_side,
				    gint gap_x,
				    gint gap_width);
static void         draw_box_gap(GtkStyle * style,
				 GdkWindow * window,
				 GtkStateType state_type,
				 GtkShadowType shadow_type,
				 GdkRectangle * area,
				 GtkWidget * widget,
				 gchar * detail,
				 gint x,
				 gint y,
				 gint width,
				 gint height,
				 GtkPositionType gap_side,
				 gint gap_x,
				 gint gap_width);
static void         draw_extension(GtkStyle * style,
				   GdkWindow * window,
				   GtkStateType state_type,
				   GtkShadowType shadow_type,
				   GdkRectangle * area,
				   GtkWidget * widget,
				   gchar * detail,
				   gint x,
				   gint y,
				   gint width,
				   gint height,
				   GtkPositionType gap_side);
static void         draw_focus(GtkStyle * style,
			       GdkWindow * window,
			       GdkRectangle * area,
			       GtkWidget * widget,
			       gchar * detail,
			       gint x,
			       gint y,
			       gint width,
			       gint height);
static void         draw_slider(GtkStyle * style,
				GdkWindow * window,
				GtkStateType state_type,
				GtkShadowType shadow_type,
				GdkRectangle * area,
				GtkWidget * widget,
				gchar * detail,
				gint x,
				gint y,
				gint width,
				gint height,
				GtkOrientation orientation);
static void         draw_handle(GtkStyle * style,
				GdkWindow * window,
				GtkStateType state_type,
				GtkShadowType shadow_type,
				GdkRectangle * area,
				GtkWidget * widget,
				gchar * detail,
				gint x,
				gint y,
				gint width,
				gint height,
				GtkOrientation orientation);

/* internal data structs */

GtkStyleClass       gtkbe_default_class =
{
  2,
  2,
  draw_hline,
  draw_vline,
  draw_shadow,
  draw_polygon,
  draw_arrow,
  draw_diamond,
  draw_oval,
  draw_string,
  draw_box,
  draw_flat_box,
  draw_check,
  draw_option,
  draw_cross,
  draw_ramp,
  draw_tab,
  draw_shadow_gap,
  draw_box_gap,
  draw_extension,
  draw_focus,
  draw_slider,
  draw_handle
};

static void
draw_hline(GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   gint x1,
	   gint x2,
	   gint y)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if (area)
    {
      gdk_gc_set_clip_rectangle(style->light_gc[state_type], area);
      gdk_gc_set_clip_rectangle(style->dark_gc[state_type], area);
    }

  gdk_draw_line(window, style->dark_gc[state_type], x1, y, x2, y);
  gdk_draw_line(window, style->light_gc[state_type], x1, y + 1, x2, y + 1);

  if (area)
    {
      gdk_gc_set_clip_rectangle(style->light_gc[state_type], NULL);
      gdk_gc_set_clip_rectangle(style->dark_gc[state_type], NULL);
    }
}

static void
draw_vline(GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   gint y1,
	   gint y2,
	   gint x)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if (area)
    {
      gdk_gc_set_clip_rectangle(style->light_gc[state_type], area);
      gdk_gc_set_clip_rectangle(style->dark_gc[state_type], area);
    }

  gdk_draw_line(window, style->dark_gc[state_type], x, y1, x, y2);
  gdk_draw_line(window, style->light_gc[state_type], x + 1, y1, x + 1, y2);

  if (area)
    {
      gdk_gc_set_clip_rectangle(style->light_gc[state_type], NULL);
      gdk_gc_set_clip_rectangle(style->dark_gc[state_type], NULL);
    }
}

static void
draw_shadow(GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    gint x,
	    gint y,
	    gint width,
	    gint height)
{
  GdkGC              *gc1 = NULL;
  GdkGC              *gc2 = NULL;
  GdkGC		     *gc3 = NULL;
  GdkGC              *gc4 = NULL;
  gint                thickness_light;
  gint                thickness_dark;
  gint                i;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    gdk_window_get_size(window, &width, &height);
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  if(DETAIL("entry"))
  {
    gint entry_width, entry_height;

    gdk_window_get_size(window, &entry_width, &entry_height);
    if(width != entry_width)
    {
      width += 2;
      height += 2;
      x -= 1;
      y -= 1;
    }
  }

  switch (shadow_type)
    {
    case GTK_SHADOW_NONE:
      return;
    case GTK_SHADOW_IN:
      gc1 = style->dark_gc[state_type];
      gc2 = style->black_gc;
      gc3 = style->bg_gc[state_type];
      gc4 = style->light_gc[state_type];
      break;
    case GTK_SHADOW_ETCHED_OUT:
      gc1 = style->light_gc[state_type];
      gc2 = style->dark_gc[state_type];
      gc3 = style->light_gc[state_type];
      gc4 = style->dark_gc[state_type];
      break;
    case GTK_SHADOW_OUT:
      gc1 = style->dark_gc[state_type];
      gc2 = style->light_gc[state_type];
      gc3 = style->dark_gc[state_type];
      gc4 = style->black_gc;
      break;
    case GTK_SHADOW_ETCHED_IN:
      gc1 = style->dark_gc[state_type];
      gc2 = style->light_gc[state_type];
      gc3 = style->dark_gc[state_type];
      gc4 = style->light_gc[state_type];
      break;
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle(gc1, area);
      gdk_gc_set_clip_rectangle(gc2, area);
      gdk_gc_set_clip_rectangle(gc3, area);
      gdk_gc_set_clip_rectangle(gc4, area);
    }
      
  if (DETAIL("button") || DETAIL("buttondefault") || DETAIL("togglebutton"))
  {
      gdk_draw_line(window, gc1,
		    x + 1, y, x + width - 2, y);
      gdk_draw_line(window, gc1,
		    x, y + 1, x, y + height - 2);
      gdk_draw_line(window, gc2,
		    x+1, y+1, x + width - 2, y+1);
      gdk_draw_line(window, gc2,
		    x+1, y+1, x+1, y + height - 2);

      gdk_draw_line(window, gc3,
		    x+1, y + height-2, x + width-2, y + height-2);
      gdk_draw_line(window, gc3,
		    x + width-2, y+1, x + width-2, y + height-2);
      gdk_draw_line(window, gc4,
		    x + 1, y+height-1, x + width-2, y+height-1);
      gdk_draw_line(window, gc4,
		    x+width-1, y + 1, x+width-1, y + height-2);

      if(shadow_type != GTK_SHADOW_ETCHED_IN)
      {
        gdk_draw_line(window, gc2,
		      x+2, y+2, x + width - 3, y+2);
        gdk_draw_line(window, gc2,
		      x+2, y+2, x+2, y + height - 3);

        gdk_draw_line(window, gc3,
		      x+2, y + height-3, x + width-3, y + height-3);
        gdk_draw_line(window, gc3,
		      x + width-3, y+2, x + width-3, y + height-3);
      }
  }
  else
  {
      gdk_draw_line(window, gc1,
		    x , y, x + width - 1, y);
      gdk_draw_line(window, gc1,
		    x, y, x, y + height - 1);
      gdk_draw_line(window, gc2,
		    x+1, y+1, x + width - 2, y+1);
      gdk_draw_line(window, gc2,
		    x+1, y+1, x+1, y + height - 2);

      gdk_draw_line(window, gc3,
		    x+1, y + height-2, x + width-2, y + height-2);
      gdk_draw_line(window, gc3,
		    x + width-2, y+1, x + width-2, y + height-2);
      gdk_draw_line(window, gc4,
		    x, y+height-1, x + width-1, y+height-1);
      gdk_draw_line(window, gc4,
		    x+width-1, y, x+width-1, y + height-1);
  }
  
  if (area)
    {
      gdk_gc_set_clip_rectangle(gc1, NULL);
      gdk_gc_set_clip_rectangle(gc2, NULL);
      gdk_gc_set_clip_rectangle(gc3, NULL);
      gdk_gc_set_clip_rectangle(gc4, NULL);
    }
}

static void
draw_polygon(GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail,
	     GdkPoint * points,
	     gint npoints,
	     gint fill)
{
#ifndef M_PI
#define M_PI    3.14159265358979323846
#endif /* M_PI */
#ifndef M_PI_4
#define M_PI_4  0.78539816339744830962
#endif /* M_PI_4 */

  static const gdouble pi_over_4 = M_PI_4;
  static const gdouble pi_3_over_4 = M_PI_4 * 3;

  GdkGC              *gc1;
  GdkGC              *gc2;
  GdkGC              *gc3;
  GdkGC              *gc4;
  gdouble             angle;
  gint                xadjust;
  gint                yadjust;
  gint                i;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);
  g_return_if_fail(points != NULL);

  switch (shadow_type)
    {
    case GTK_SHADOW_IN:
      gc1 = style->light_gc[state_type];
      gc2 = style->dark_gc[state_type];
      gc3 = style->light_gc[state_type];
      gc4 = style->dark_gc[state_type];
      break;
    case GTK_SHADOW_ETCHED_IN:
      gc1 = style->light_gc[state_type];
      gc2 = style->dark_gc[state_type];
      gc3 = style->dark_gc[state_type];
      gc4 = style->light_gc[state_type];
      break;
    case GTK_SHADOW_OUT:
      gc1 = style->dark_gc[state_type];
      gc2 = style->light_gc[state_type];
      gc3 = style->dark_gc[state_type];
      gc4 = style->light_gc[state_type];
      break;
    case GTK_SHADOW_ETCHED_OUT:
      gc1 = style->dark_gc[state_type];
      gc2 = style->light_gc[state_type];
      gc3 = style->light_gc[state_type];
      gc4 = style->dark_gc[state_type];
      break;
    default:
      return;
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle(gc1, area);
      gdk_gc_set_clip_rectangle(gc2, area);
      gdk_gc_set_clip_rectangle(gc3, area);
      gdk_gc_set_clip_rectangle(gc4, area);
    }

  if (fill)
    gdk_draw_polygon(window, style->bg_gc[state_type], TRUE, points, npoints);

  npoints--;

  for (i = 0; i < npoints; i++)
    {
      if ((points[i].x == points[i + 1].x) &&
	  (points[i].y == points[i + 1].y))
	{
	  angle = 0;
	}
      else
	{
	  angle = atan2(points[i + 1].y - points[i].y,
			points[i + 1].x - points[i].x);
	}

      if ((angle > -pi_3_over_4) && (angle < pi_over_4))
	{
	  if (angle > -pi_over_4)
	    {
	      xadjust = 0;
	      yadjust = 1;
	    }
	  else
	    {
	      xadjust = 1;
	      yadjust = 0;
	    }

	  gdk_draw_line(window, gc1,
			points[i].x - xadjust, points[i].y - yadjust,
			points[i + 1].x - xadjust, points[i + 1].y - yadjust);
	  gdk_draw_line(window, gc3,
			points[i].x, points[i].y,
			points[i + 1].x, points[i + 1].y);
	}
      else
	{
	  if ((angle < -pi_3_over_4) || (angle > pi_3_over_4))
	    {
	      xadjust = 0;
	      yadjust = 1;
	    }
	  else
	    {
	      xadjust = 1;
	      yadjust = 0;
	    }

	  gdk_draw_line(window, gc4,
			points[i].x + xadjust, points[i].y + yadjust,
			points[i + 1].x + xadjust, points[i + 1].y + yadjust);
	  gdk_draw_line(window, gc2,
			points[i].x, points[i].y,
			points[i + 1].x, points[i + 1].y);
	}
    }
  if (area)
    {
      gdk_gc_set_clip_rectangle(gc1, NULL);
      gdk_gc_set_clip_rectangle(gc2, NULL);
      gdk_gc_set_clip_rectangle(gc3, NULL);
      gdk_gc_set_clip_rectangle(gc4, NULL);
    }
}

static void
draw_arrow(GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GtkShadowType shadow_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   GtkArrowType arrow_type,
	   gint fill,
	   gint x,
	   gint y,
	   gint width,
	   gint height)
{
  GdkGC              *gc1;
  GdkGC              *gc2;
  gint                half_width;
  gint                half_height;
  GdkPoint            points[3];

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    gdk_window_get_size(window, &width, &height);
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  if(DETAIL("spinbutton") || GTK_IS_SCROLLBAR(widget) || GTK_IS_NOTEBOOK(widget))
  {
    gtk_paint_box (style, window, state_type, shadow_type, area, widget,
                   detail, x, y, width, height);

    width -= 4;
    height -= 4;
    x += 2;
    y += 2;
  }

  gc1 = style->light_gc[state_type];
  gc2 = style->dark_gc[state_type];

  if (area)
    {
      gdk_gc_set_clip_rectangle(gc1, area);
      gdk_gc_set_clip_rectangle(gc2, area);
    }

  half_width = width / 2;
  half_height = height / 2;

  switch(arrow_type)
    {
      case GTK_ARROW_UP:
	  points[0].x = x + half_width + 1;
	  points[0].y = y + 1;
	  points[1].x = x + 1;
	  points[1].y = y + height - 1;
	  points[2].x = x + width - 1;
	  points[2].y = y + height - 1;

 	  gdk_draw_polygon(window, gc1, FALSE, points, 3);

	  points[0].x = x + half_width;
	  points[0].y = y;
	  points[1].x = x;
	  points[1].y = y + height - 2;
	  points[2].x = x + width - 2;
	  points[2].y = y + height - 2;

 	  gdk_draw_polygon(window, gc2, FALSE, points, 3);
 	  break;

    case GTK_ARROW_DOWN:
	  points[0].x = x + width - 1;
	  points[0].y = y + 1;
	  points[1].x = x + 1;
	  points[1].y = y + 1;
	  points[2].x = x + half_width + 1;
	  points[2].y = y + height - 1;

	  gdk_draw_polygon(window, gc1, FALSE, points, 3);

	  points[0].x = x + width - 2;
	  points[0].y = y;
	  points[1].x = x;
	  points[1].y = y;
	  points[2].x = x + half_width;
	  points[2].y = y + height - 2;

	  gdk_draw_polygon(window, gc2, FALSE, points, 3);
	  break;
	  
    case GTK_ARROW_LEFT:
	  points[0].x = x + width - 1;
	  points[0].y = y + 1;
	  points[1].x = x + 1;
	  points[1].y = y + half_height + 1;
	  points[2].x = x + width - 1;
	  points[2].y = y + height - 1;

	  gdk_draw_polygon(window, gc1, FALSE, points, 3);

	  points[0].x = x + width - 2;
	  points[0].y = y;
	  points[1].x = x;
	  points[1].y = y + half_height;
	  points[2].x = x + width - 2;
	  points[2].y = y + height - 2;

	  gdk_draw_polygon(window, gc2, FALSE, points, 3);
	  break;

    case GTK_ARROW_RIGHT:
	  points[0].x = x + width - 1;
	  points[0].y = y + half_height + 1;
	  points[1].x = x + 1;
	  points[1].y = y + 1;
	  points[2].x = x + 1;
	  points[2].y = y + height - 1;

	  gdk_draw_polygon(window, gc1, FALSE, points, 3);

	  points[0].x = x + width - 2;
	  points[0].y = y + half_height;
	  points[1].x = x;
	  points[1].y = y;
	  points[2].x = x;
	  points[2].y = y + height - 2;

	  gdk_draw_polygon(window, gc2, FALSE, points, 3);
          break;
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle(gc1, NULL);
      gdk_gc_set_clip_rectangle(gc2, NULL);
    }
}

static void
draw_diamond(GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail,
	     gint x,
	     gint y,
	     gint width,
	     gint height)
{
  gint                half_width;
  gint                half_height;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    gdk_window_get_size(window, &width, &height);
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  half_width = width / 2;
  half_height = height / 2;

  if (area)
    {
      gdk_gc_set_clip_rectangle(style->light_gc[state_type], area);
      gdk_gc_set_clip_rectangle(style->bg_gc[state_type], area);
      gdk_gc_set_clip_rectangle(style->dark_gc[state_type], area);
      gdk_gc_set_clip_rectangle(style->black_gc, area);
    }
  switch (shadow_type)
    {
    case GTK_SHADOW_IN:
      gdk_draw_line(window, style->light_gc[state_type],
		    x + 2, y + half_height,
		    x + half_width, y + height - 2);
      gdk_draw_line(window, style->light_gc[state_type],
		    x + half_width, y + height - 2,
		    x + width - 2, y + half_height);
      gdk_draw_line(window, style->light_gc[state_type],
		    x + 1, y + half_height,
		    x + half_width, y + height - 1);
      gdk_draw_line(window, style->light_gc[state_type],
		    x + half_width, y + height - 1,
		    x + width - 1, y + half_height);
      gdk_draw_line(window, style->light_gc[state_type],
		    x, y + half_height,
		    x + half_width, y + height);
      gdk_draw_line(window, style->light_gc[state_type],
		    x + half_width, y + height,
		    x + width, y + half_height);

      gdk_draw_line(window, style->dark_gc[state_type],
		    x + 2, y + half_height,
		    x + half_width, y + 2);
      gdk_draw_line(window, style->dark_gc[state_type],
		    x + half_width, y + 2,
		    x + width - 2, y + half_height);
      gdk_draw_line(window, style->dark_gc[state_type],
		    x + 1, y + half_height,
		    x + half_width, y + 1);
      gdk_draw_line(window, style->dark_gc[state_type],
		    x + half_width, y + 1,
		    x + width - 1, y + half_height);
      gdk_draw_line(window, style->dark_gc[state_type],
		    x, y + half_height,
		    x + half_width, y);
      gdk_draw_line(window, style->dark_gc[state_type],
		    x + half_width, y,
		    x + width, y + half_height);
      break;
    case GTK_SHADOW_OUT:
      gdk_draw_line(window, style->dark_gc[state_type],
		    x + 2, y + half_height,
		    x + half_width, y + height - 2);
      gdk_draw_line(window, style->dark_gc[state_type],
		    x + half_width, y + height - 2,
		    x + width - 2, y + half_height);
      gdk_draw_line(window, style->dark_gc[state_type],
		    x + 1, y + half_height,
		    x + half_width, y + height - 1);
      gdk_draw_line(window, style->dark_gc[state_type],
		    x + half_width, y + height - 1,
		    x + width - 1, y + half_height);
      gdk_draw_line(window, style->dark_gc[state_type],
		    x, y + half_height,
		    x + half_width, y + height);
      gdk_draw_line(window, style->dark_gc[state_type],
		    x + half_width, y + height,
		    x + width, y + half_height);

      gdk_draw_line(window, style->light_gc[state_type],
		    x + 2, y + half_height,
		    x + half_width, y + 2);
      gdk_draw_line(window, style->light_gc[state_type],
		    x + half_width, y + 2,
		    x + width - 2, y + half_height);
      gdk_draw_line(window, style->light_gc[state_type],
		    x + 1, y + half_height,
		    x + half_width, y + 1);
      gdk_draw_line(window, style->light_gc[state_type],
		    x + half_width, y + 1,
		    x + width - 1, y + half_height);
      gdk_draw_line(window, style->light_gc[state_type],
		    x, y + half_height,
		    x + half_width, y);
      gdk_draw_line(window, style->light_gc[state_type],
		    x + half_width, y,
		    x + width, y + half_height);
      break;
    default:
      break;
    }
  if (area)
    {
      gdk_gc_set_clip_rectangle(style->light_gc[state_type], NULL);
      gdk_gc_set_clip_rectangle(style->bg_gc[state_type], NULL);
      gdk_gc_set_clip_rectangle(style->dark_gc[state_type], NULL);
      gdk_gc_set_clip_rectangle(style->black_gc, NULL);
    }
}

static void
draw_oval(GtkStyle * style,
	  GdkWindow * window,
	  GtkStateType state_type,
	  GtkShadowType shadow_type,
	  GdkRectangle * area,
	  GtkWidget * widget,
	  gchar * detail,
	  gint x,
	  gint y,
	  gint width,
	  gint height)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);
}

static void
draw_string(GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    gint x,
	    gint y,
	    const gchar * string)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if (area)
    {
      gdk_gc_set_clip_rectangle(style->white_gc, area);
      gdk_gc_set_clip_rectangle(style->fg_gc[state_type], area);
    }

  if (state_type == GTK_STATE_INSENSITIVE)
  {
    gdk_draw_string(window, style->font, style->light_gc[state_type], x + 1, y + 1, string);
    gdk_draw_string(window, style->font, style->dark_gc[state_type], x, y, string);
  }
  else
    gdk_draw_string(window, style->font, style->fg_gc[state_type], x, y, string);

  if (area)
    {
      gdk_gc_set_clip_rectangle(style->white_gc, NULL);
      gdk_gc_set_clip_rectangle(style->fg_gc[state_type], NULL);
    }
}

static void
draw_box(GtkStyle * style,
	 GdkWindow * window,
	 GtkStateType state_type,
	 GtkShadowType shadow_type,
	 GdkRectangle * area,
	 GtkWidget * widget,
	 gchar * detail,
	 gint x,
	 gint y,
	 gint width,
	 gint height)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    gdk_window_get_size(window, &width, &height);
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  if(DETAIL("button"))
  {
    gint button_height, button_width;
    
    gdk_window_get_size(window, &button_width, &button_height);
    if(width != button_width)
    {
       width += 2;
       height += 2;
       x -= 1;
       y -= 1;
    }
  }

  if(DETAIL("buttondefault"))
  {
    state_type = GTK_STATE_PRELIGHT;
    shadow_type = GTK_SHADOW_OUT;
    x += 1;
    y += 1;
    width -= 1;
    height -= 1;
  }

  if(DETAIL("button") && state_type==GTK_STATE_INSENSITIVE)
    shadow_type = GTK_SHADOW_ETCHED_IN;

  if(DETAIL("trough"))
    state_type = GTK_STATE_ACTIVE;

  if ((!style->bg_pixmap[state_type]) ||
      (gdk_window_get_type(window) == GDK_WINDOW_PIXMAP))
    {
      if (area)
	{
	  gdk_gc_set_clip_rectangle(style->bg_gc[state_type], area);
	}
      gdk_draw_rectangle(window, style->bg_gc[state_type], TRUE,
			 x+1, y+1, width-2, height-2);
      if (area)
	{
	  gdk_gc_set_clip_rectangle(style->bg_gc[state_type], NULL);
	}
    }
  else
    gtk_style_apply_default_pixmap(style, window, state_type, area, x+1, y+1, width-2, height-2);

  if(!DETAIL("menuitem"))
  {
    gtk_paint_shadow(style, window, state_type, shadow_type, area, widget,
                     detail, x, y, width, height);
  }

  if(DETAIL("slider") && GTK_IS_SCROLLBAR(widget))
  {
    if(width > height && width > 20)
    {
      draw_vline(style, window, state_type, area, widget, detail,
	         2,
	         height - 3, width / 2 - 4);
      draw_vline(style, window, state_type, area, widget, detail,
	         2,
	         height - 3, width / 2);
      draw_vline(style, window, state_type, area, widget, detail,
	         2,
	         height - 3, width / 2 + 4);
    }
    else if(height > 20)
    {
      draw_hline(style, window, state_type, area, widget, detail,
	         2,
	         width - 3, height / 2 - 4);
      draw_hline(style, window, state_type, area, widget, detail,
	         2,
	         width - 3, height / 2);
      draw_hline(style, window, state_type, area, widget, detail,
	         2,
	         width - 3, height / 2 + 4);
    }
  }
}

static void
draw_flat_box(GtkStyle * style,
	      GdkWindow * window,
	      GtkStateType state_type,
	      GtkShadowType shadow_type,
	      GdkRectangle * area,
	      GtkWidget * widget,
	      gchar * detail,
	      gint x,
	      gint y,
	      gint width,
	      gint height)
{
  GdkGC              *gc1;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    gdk_window_get_size(window, &width, &height);
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  gc1 = style->bg_gc[state_type];

  if (DETAIL("text") && (state_type == GTK_STATE_SELECTED))
    gc1 = style->bg_gc[GTK_STATE_SELECTED];
  if (DETAIL("entry_bg"))
    gc1 = style->base_gc[state_type];
  if (DETAIL("viewportbin"))
    gc1 = style->bg_gc[GTK_STATE_NORMAL];
  if (DETAIL("trough"))
    gc1 = style->bg_gc[GTK_STATE_INSENSITIVE];

  if ((!style->bg_pixmap[state_type]) || (gc1 != style->bg_gc[state_type]) ||
      (gdk_window_get_type(window) == GDK_WINDOW_PIXMAP))
    {
      if (area)
	{
	  gdk_gc_set_clip_rectangle(gc1, area);
	}
      gdk_draw_rectangle(window, gc1, TRUE,
			 x, y, width, height);

      if ((detail) && (!strcmp("tooltip", detail)))
	gdk_draw_rectangle(window, style->black_gc, FALSE,
			   x, y, width - 1, height - 1);

      if (area)
	{
	  gdk_gc_set_clip_rectangle(gc1, NULL);
	}
    }
  else
    gtk_style_apply_default_pixmap(style, window, state_type, area, x, y, width, height);
}

static void
draw_check(GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GtkShadowType shadow_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   gint x,
	   gint y,
	   gint width,
	   gint height)
{ 
  x -= 2;
  y -= 2;
  width += 3;
  height += 3;

  if (area)
    {
      gdk_gc_set_clip_rectangle(style->base_gc[state_type], area);
      gdk_gc_set_clip_rectangle(style->bg_gc[GTK_STATE_SELECTED], area);
    }

  gdk_draw_rectangle(window, style->base_gc[state_type], TRUE,
                     x+1, y+1, width-2, height-2);

  if(shadow_type == GTK_SHADOW_IN)
  {
    gdk_draw_line(window, style->bg_gc[GTK_STATE_SELECTED], 
                  x+3, y+3, x+width-4, y+height-4);
    gdk_draw_line(window, style->bg_gc[GTK_STATE_SELECTED],
		  x+4, y+3, x+width-4, y+height-5);
    gdk_draw_line(window, style->bg_gc[GTK_STATE_SELECTED],
                  x+3, y+4, x+width-5, y+height-4);
    gdk_draw_line(window, style->bg_gc[GTK_STATE_SELECTED],
                  x+width-4, y+3, x+3, y+height-4);
    gdk_draw_line(window, style->bg_gc[GTK_STATE_SELECTED],
                  x+width-5, y+3, x+3, y+height-5);
    gdk_draw_line(window, style->bg_gc[GTK_STATE_SELECTED],
                  x+width-4, y+4, x+4, y+height-4);
  }
  gtk_paint_shadow(style, window, GTK_STATE_NORMAL, GTK_SHADOW_IN, area, widget, detail,
		   x, y, width, height);

  if (area)
    {
      gdk_gc_set_clip_rectangle(style->base_gc[GTK_STATE_NORMAL], NULL);
      gdk_gc_set_clip_rectangle(style->bg_gc[GTK_STATE_SELECTED], NULL);
    }
}

static void
draw_option(GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    gint x,
	    gint y,
	    gint width,
	    gint height)
{
  GdkGC              *gc1;
  GdkGC              *gc2;
  GdkGC              *gc3;
  GdkGC              *gc4;
  GdkGC			 *gc5;
  GdkGC			 *gc6;

  x -= 1;
  y -= 1;
  width += 2;
  height += 2;

  gc1 = style->light_gc[GTK_STATE_NORMAL];
  gc2 = style->bg_gc[GTK_STATE_NORMAL];
  gc3 = style->dark_gc[GTK_STATE_NORMAL];
  gc4 = style->black_gc;
  gc5 = style->base_gc[state_type];
  gc6 = style->bg_gc[GTK_STATE_SELECTED];

  if (area)
    {
      gdk_gc_set_clip_rectangle(gc1, area);
      gdk_gc_set_clip_rectangle(gc2, area);
      gdk_gc_set_clip_rectangle(gc3, area);
      gdk_gc_set_clip_rectangle(gc4, area);
      gdk_gc_set_clip_rectangle(gc5, area);
      gdk_gc_set_clip_rectangle(gc6, area);
    }

  gdk_draw_arc(window, gc3, FALSE, x, y, width, height, 45 * 64, 225 * 64);
  gdk_draw_arc(window, gc3, TRUE, x, y, width, height, 45 * 64, 225 * 64);
  gdk_draw_arc(window, gc1, FALSE, x, y, width, height, 225 * 64, 180 * 64);
  gdk_draw_arc(window, gc1, TRUE, x, y, width, height, 225 * 64, 180 * 64);
  gdk_draw_arc(window, gc4, FALSE, x + 1, y + 1, width - 2, height - 2, 45 * 64, 225 * 64);
  gdk_draw_arc(window, gc4, TRUE, x + 1, y + 1, width - 2, height - 2, 45 * 64, 225 * 64);
  gdk_draw_arc(window, gc2, FALSE, x + 1, y + 1, width - 2, height - 2, 225 * 64, 180 * 64);
  gdk_draw_arc(window, gc2, TRUE, x + 1, y + 1, width - 2, height - 2, 225 * 64, 180 * 64);
  gdk_draw_arc(window, gc5, FALSE, x + 2, y + 2, width - 4, height - 4, 0, 360 * 64);
  gdk_draw_arc(window, gc5, TRUE, x + 2, y + 2, width - 4, height - 4, 0, 360 * 64);

  if (shadow_type == GTK_SHADOW_IN)
    {
      gdk_draw_arc(window, gc6, FALSE, x + 4, y + 4, width - 8, height - 8, 0, 360 * 64);
      gdk_draw_arc(window, gc6, TRUE, x + 4, y + 4, width - 8, height - 8, 0, 360 * 64);
    }

  if (area)
    {
      gdk_gc_set_clip_rectangle(gc1, NULL);
      gdk_gc_set_clip_rectangle(gc2, NULL);
      gdk_gc_set_clip_rectangle(gc3, NULL);
      gdk_gc_set_clip_rectangle(gc4, NULL);
      gdk_gc_set_clip_rectangle(gc5, NULL);
      gdk_gc_set_clip_rectangle(gc6, NULL);
    }
}

static void
draw_cross(GtkStyle * style,
	   GdkWindow * window,
	   GtkStateType state_type,
	   GtkShadowType shadow_type,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   gint x,
	   gint y,
	   gint width,
	   gint height)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);
}

static void
draw_ramp(GtkStyle * style,
	  GdkWindow * window,
	  GtkStateType state_type,
	  GtkShadowType shadow_type,
	  GdkRectangle * area,
	  GtkWidget * widget,
	  gchar * detail,
	  GtkArrowType arrow_type,
	  gint x,
	  gint y,
	  gint width,
	  gint height)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);
}

static void
draw_tab(GtkStyle * style,
	 GdkWindow * window,
	 GtkStateType state_type,
	 GtkShadowType shadow_type,
	 GdkRectangle * area,
	 GtkWidget * widget,
	 gchar * detail,
	 gint x,
	 gint y,
	 gint width,
	 gint height)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  gtk_paint_box(style, window, state_type, shadow_type, area, widget, detail,
		x, y, width, height);
}

static void
draw_shadow_gap(GtkStyle * style,
		GdkWindow * window,
		GtkStateType state_type,
		GtkShadowType shadow_type,
		GdkRectangle * area,
		GtkWidget * widget,
		gchar * detail,
		gint x,
		gint y,
		gint width,
		gint height,
		GtkPositionType gap_side,
		gint gap_x,
		gint gap_width)
{
  GdkRectangle        rect;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  gtk_paint_shadow(style, window, state_type, shadow_type, area, widget, detail,
		   x, y, width, height);

  switch (gap_side)
    {
    case GTK_POS_TOP:
      rect.x = x + gap_x;
      rect.y = y;
      rect.width = gap_width;
      rect.height = 2;
      break;
    case GTK_POS_BOTTOM:
      rect.x = x + gap_x;
      rect.y = y + height - 2;
      rect.width = gap_width;
      rect.height = 2;
      break;
    case GTK_POS_LEFT:
      rect.x = x;
      rect.y = y + gap_x;
      rect.width = 2;
      rect.height = gap_width;
      break;
    case GTK_POS_RIGHT:
      rect.x = x + width - 2;
      rect.y = y + gap_x;
      rect.width = 2;
      rect.height = gap_width;
      break;
    }

  gtk_style_apply_default_pixmap(style, window, state_type, area,
				 rect.x, rect.y, rect.width, rect.height);
}

static void
draw_box_gap(GtkStyle * style,
	     GdkWindow * window,
	     GtkStateType state_type,
	     GtkShadowType shadow_type,
	     GdkRectangle * area,
	     GtkWidget * widget,
	     gchar * detail,
	     gint x,
	     gint y,
	     gint width,
	     gint height,
	     GtkPositionType gap_side,
	     gint gap_x,
	     gint gap_width)
{
  GdkRectangle        rect;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  gtk_paint_box(style, window, state_type, shadow_type, area, widget, detail,
		x, y, width, height);

  switch (gap_side)
    {
    case GTK_POS_TOP:
      rect.x = x + gap_x + 2;
      rect.y = y;
      rect.width = gap_width - 4;
      rect.height = 2;
      gdk_draw_point(window, style->dark_gc[state_type],
		     rect.x-2, rect.y);
      gdk_draw_line(window, style->light_gc[state_type],
		    rect.x-1, rect.y, rect.x-1, rect.y + 1);
      break;
    case GTK_POS_BOTTOM:
      rect.x = x + gap_x + 2;
      rect.y = y + height - 2;
      rect.width = gap_width - 4;
      rect.height = 2;
      gdk_draw_line(window, style->dark_gc[state_type],
                    rect.x-2, rect.y, rect.x-2, rect.y+1);
      gdk_draw_line(window, style->light_gc[state_type],
                    rect.x-1, rect.y, rect.x-1, rect.y+1);
      gdk_draw_line(window, style->dark_gc[state_type],
                    rect.x + rect.width, rect.y, rect.x + rect.width, rect.y+1);
      break;
    case GTK_POS_LEFT:
      rect.x = x;
      rect.y = y + gap_x + 2;
      rect.width = 2;
      rect.height = gap_width - 4;
      gdk_draw_point(window, style->dark_gc[state_type],
                     rect.x, rect.y-2);
      gdk_draw_line(window, style->light_gc[state_type],
                    rect.x, rect.y-1, rect.x+1, rect.y-1);
      break;
    case GTK_POS_RIGHT:
      rect.x = x + width - 2;
      rect.y = y + gap_x + 2;
      rect.width = 2;
      rect.height = gap_width - 4;
      gdk_draw_line(window, style->dark_gc[state_type],
                    rect.x, rect.y-2, rect.x+1, rect.y-2);
      gdk_draw_line(window, style->light_gc[state_type],
	            rect.x, rect.y-1, rect.x+1, rect.y-1);
      gdk_draw_line(window, style->dark_gc[state_type],
                    rect.x, rect.y+rect.height, rect.x+1, rect.y+rect.height);
      break;
    }

  gtk_style_apply_default_pixmap(style, window, state_type, area,
				 rect.x, rect.y, rect.width, rect.height);
}

static void
draw_extension(GtkStyle * style,
	       GdkWindow * window,
	       GtkStateType state_type,
	       GtkShadowType shadow_type,
	       GdkRectangle * area,
	       GtkWidget * widget,
	       gchar * detail,
	       gint x,
	       gint y,
	       gint width,
	       gint height,
	       GtkPositionType gap_side)
{
  GdkRectangle        rect;
  GdkGC			  *gc1, *gc2, *gc3, *gc4;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  gc1 = style->dark_gc[state_type];
  gc2 = style->light_gc[state_type];
  gc3 = style->dark_gc[state_type];
  gc4 = style->black_gc;

  switch (gap_side)
    {
    case GTK_POS_TOP:
      gtk_paint_flat_box(style, window, state_type, shadow_type, area, widget, detail,
		x, y, width, height-1);
      gdk_draw_line(window, gc1,
		    x, y, x, y + height - 2);
      gdk_draw_line(window, gc2,
		    x+1, y, x+1, y + height - 3);
      gdk_draw_line(window, gc3,
		    x+1, y + height-2, x + width-2, y + height-2);
      gdk_draw_line(window, gc3,
		    x + width-2, y, x + width-2, y + height-2);
      gdk_draw_line(window, gc4,
		    x+1, y+height-1, x + width-2, y+height-1);
      gdk_draw_line(window, gc4,
		    x+width-1, y, x+width-1, y + height-2);
      break;
    case GTK_POS_BOTTOM:
      gtk_paint_flat_box(style, window, state_type, shadow_type, area, widget, detail,
		x, y+1, width, height-1);
      gdk_draw_line(window, gc1,
		    x+1, y, x + width - 2, y);
      gdk_draw_line(window, gc1,
		    x, y+1, x, y + height - 1);
      gdk_draw_line(window, gc2,
		    x+1, y+1, x + width - 2, y+1);
      gdk_draw_line(window, gc2,
		    x+1, y+1, x+1, y + height - 1);
      gdk_draw_line(window, gc3,
		    x + width-2, y+1, x + width-2, y + height-1);
      gdk_draw_line(window, gc4,
		    x+width-1, y+1, x+width-1, y + height-1);
      break;
    case GTK_POS_LEFT:
      gtk_paint_flat_box(style, window, state_type, shadow_type, area, widget, detail,
		x, y, width-1, height);
      gdk_draw_line(window, gc1,
		    x, y, x + width - 2, y);
      gdk_draw_line(window, gc2,
		    x, y+1, x + width - 2, y+1);
      gdk_draw_line(window, gc3,
		    x, y + height-2, x + width-2, y + height-2);
      gdk_draw_line(window, gc3,
		    x + width-2, y+1, x + width-2, y + height-2);
      gdk_draw_line(window, gc4,
		    x+1, y+height-1, x + width-2, y+height-1);
      gdk_draw_line(window, gc4,
		    x+width-1, y+1, x+width-1, y + height-2);
      break;
    case GTK_POS_RIGHT:
      gtk_paint_flat_box(style, window, state_type, shadow_type, area, widget, detail,
		x+1, y, width-1, height);
      gdk_draw_line(window, gc1,
		    x+1, y, x + width - 1, y);
      gdk_draw_line(window, gc1,
		    x, y+1, x, y + height - 2);
      gdk_draw_line(window, gc2,
		    x+1, y+1, x + width - 1, y+1);
      gdk_draw_line(window, gc2,
		    x+1, y+1, x+1, y + height - 2);
      gdk_draw_line(window, gc3,
		    x+1, y + height-2, x + width-1, y + height-2);
      gdk_draw_line(window, gc4,
		    x+1, y+height-1, x + width-1, y+height-1);
      break;
    }
}

static void
draw_focus(GtkStyle * style,
	   GdkWindow * window,
	   GdkRectangle * area,
	   GtkWidget * widget,
	   gchar * detail,
	   gint x,
	   gint y,
	   gint width,
	   gint height)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);
}

static void
draw_slider(GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    gint x,
	    gint y,
	    gint width,
	    gint height,
	    GtkOrientation orientation)
{
  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    gdk_window_get_size(window, &width, &height);
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  gtk_draw_box(style, window, state_type, shadow_type, x, y,
	       width, height);

  if (orientation == GTK_ORIENTATION_HORIZONTAL)
  {
    draw_vline(style, window, state_type, area, widget, detail,
	       2,
	       height - 3, width / 2 - 4);
    draw_vline(style, window, state_type, area, widget, detail,
	       2,
	       height - 3, width / 2);
    draw_vline(style, window, state_type, area, widget, detail,
	       2,
	       height - 3, width / 2 + 4);
  }
  else
  {
    draw_hline(style, window, state_type, area, widget, detail,
	       2,
	       width - 3, height / 2 - 4);
    draw_hline(style, window, state_type, area, widget, detail,
	       2,
	       width - 3, height / 2);
    draw_hline(style, window, state_type, area, widget, detail,
	       2,
	       width - 3, height / 2 + 4);
  }
}

static void
draw_handle(GtkStyle * style,
	    GdkWindow * window,
	    GtkStateType state_type,
	    GtkShadowType shadow_type,
	    GdkRectangle * area,
	    GtkWidget * widget,
	    gchar * detail,
	    gint x,
	    gint y,
	    gint width,
	    gint height,
	    GtkOrientation orientation)
{
  gint                xx, yy;
  GdkGC              *light_gc, *dark_gc;
  GdkRectangle        dest;

  g_return_if_fail(style != NULL);
  g_return_if_fail(window != NULL);

  if ((width == -1) && (height == -1))
    gdk_window_get_size(window, &width, &height);
  else if (width == -1)
    gdk_window_get_size(window, &width, NULL);
  else if (height == -1)
    gdk_window_get_size(window, NULL, &height);

  gtk_paint_box(style, window, state_type, shadow_type, area, widget,
		detail, x, y, width, height);

  light_gc = style->light_gc[state_type];
  dark_gc = style->dark_gc[state_type];

  dest.x = x + 2;
  dest.y = y + 2;
  dest.width = width - 4;
  dest.height = height - 4;

  gdk_gc_set_clip_rectangle(light_gc, &dest);
  gdk_gc_set_clip_rectangle(dark_gc, &dest);

  for (yy = y + 2; yy < (y + height - 2); yy += 3)
    for (xx = x + 2; xx < (x + width - 2); xx += 6)
      {
	gdk_draw_point(window, light_gc, xx, yy);
	gdk_draw_point(window, dark_gc, xx + 1, yy + 1);

	gdk_draw_point(window, light_gc, xx + 3, yy + 1);
	gdk_draw_point(window, dark_gc, xx + 4, yy + 2);
      }
  gdk_gc_set_clip_rectangle(light_gc, NULL);
  gdk_gc_set_clip_rectangle(dark_gc, NULL);
}
