/**
 * Copyright (C) 2003 Billy Biggs <vektor@dumbterm.net>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <stdio.h>
#include <OpenEXR/ImfCRgbaFile.h>
#include "exroutput.h"

struct exroutput_s
{
    ImfOutputFile *file;
    ImfRgba *base;
    int width;
    int height;
    int ypos;
};

exroutput_t *exroutput_new( const char *filename, int width, int height )
{
    exroutput_t *exroutput = malloc( sizeof( exroutput_t ) );
    ImfHeader *header;

    if( !exroutput ) {
        fprintf( stderr, "exroutput: Cannot allocate memory.\n" );
        return 0;
    }

    exroutput->ypos = 0;
    exroutput->width = width;
    exroutput->height = height;

    exroutput->base = malloc( width * sizeof( ImfRgba ) );
    header = ImfNewHeader();
    if( !header || !exroutput->base ) {
        fprintf( stderr, "exroutput: Cannot allocate memory.\n" );
        if( header ) ImfDeleteHeader( header );
        if( exroutput->base ) free( exroutput->base );
        free( exroutput );
        return 0;
    }

    ImfHeaderSetDataWindow( header, 0, 0, width - 1, height - 1 );
    ImfHeaderSetDisplayWindow( header, 0, 0, width - 1, height - 1 );
    ImfHeaderSetCompression( header, IMF_PIZ_COMPRESSION );
    ImfHeaderSetLineOrder( header, IMF_INCREASING_Y );

    exroutput->file = ImfOpenOutputFile( filename, header, IMF_WRITE_RGBA );
    ImfDeleteHeader( header );

    if( !exroutput->file ) {
        fprintf( stderr, "exroutput: %s\n", ImfErrorMessage() );
        free( exroutput );
        return 0;
    }

    return exroutput;
}

void exroutput_delete( exroutput_t *exroutput )
{
    ImfCloseOutputFile( exroutput->file );
    free( exroutput->base );
    free( exroutput );
}

void exroutput_scanline( exroutput_t *exroutput, double *scanline )
{
    ImfRgba *curscanline = exroutput->base;
    int i;

    for( i = 0; i < exroutput->width; i++ ) {
        ImfHalf half;
        float cur;

        cur = (float) scanline[ (i * 4) + 0 ];
        ImfFloatToHalf( cur, &half );
        curscanline->r = half;

        cur = (float) scanline[ (i * 4) + 1 ];
        ImfFloatToHalf( cur, &half );
        curscanline->g = half;

        cur = (float) scanline[ (i * 4) + 2 ];
        ImfFloatToHalf( cur, &half );
        curscanline->b = half;

        cur = (float) scanline[ (i * 4) + 3 ];
        ImfFloatToHalf( cur, &half );
        curscanline->a = half;
        curscanline++;
    }
    ImfOutputSetFrameBuffer( exroutput->file,
                             exroutput->base - (exroutput->ypos * exroutput->width),
                             1, exroutput->width );
    ImfOutputWritePixels( exroutput->file, 1 );
    exroutput->ypos++;
}

