/* $Id: fsu_mount.c,v 1.17 2008/11/20 18:50:20 stacktic Exp $ */

/*
 * Copyright (c) 2008 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/mount.h>

#include <err.h>
#include <errno.h>
#include <stdbool.h>
#include <string.h>

#include <isofs/cd9660/cd9660_mount.h>
#include <fs/efs/efs_mount.h>
#include <ufs/ufs/ufsmount.h> /* ext2fs, ffs, lfs */
#include <fs/hfs/hfs.h>
#include <msdosfs/msdosfsmount.h>
#include <ntfs/ntfsmount.h>
#include <fs/tmpfs/tmpfs_args.h>
#include <fs/udf/udf_mount.h>

#include <rump/ukfs.h>

#include <fsu_utils.h>

#include "mount_cd9660.h"
#include "mount_efs.h"
#include "mount_ext2fs.h"
#include "mount_ffs.h"
#include "mount_hfs.h"
#include "mount_lfs.h"
#include "mount_msdos.h"
#include "mount_ntfs.h"
#include "mount_tmpfs.h"
#include "mount_udf.h"
#include "rump_syspuffs.h"

#include "fsu_mount.h"

#define MAX_MNT_ARGS (15)

struct fsu_fsalias_s {
	char *fsa_name;
	char *fsa_path;
	char *fsa_type;
	char *fsa_mntopt;
	struct fsu_fsalias_s *fsa_next;
};

union fsu_args {
	struct iso_args args_cd9660;
	struct efs_args args_efs;
	struct ufs_args args_ufs; /* ext2fs, ffs, lfs, ufs */
	struct hfs_args args_hfs;
	struct msdosfs_args args_msdos;
	struct ntfs_args args_ntfs;
	struct tmpfs_args args_tmpfs;
	struct udf_args args_udf;
	struct syspuffs_args args_syspuffs;
} args;

typedef int (*parseargs_fp)(int, char **, void *, int *, char *, char *);

typedef struct fsu_fs_s {
	const char *fs_name;
	void *fs_args;
	unsigned int fs_args_size;
	parseargs_fp fs_parseargs;
	unsigned int fs_flags;
#define FS_NO_AUTODETECT (1)
} fsu_fs_t;

fsu_fs_t fslist[] = {
	{ MOUNT_CD9660,	&args.args_cd9660,	sizeof(struct iso_args),
	  (parseargs_fp)mount_cd9660_parseargs,		0 },
	{ MOUNT_EXT2FS,	&args.args_ufs,		sizeof(struct ufs_args),
	  (parseargs_fp)mount_ext2fs_parseargs,		0 },
	{ MOUNT_FFS,	&args.args_ufs,		sizeof(struct ufs_args),
	  (parseargs_fp)mount_ffs_parseargs,		0 },
	{ MOUNT_HFS,	&args.args_hfs,		sizeof(struct hfs_args),
	  (parseargs_fp)mount_hfs_parseargs,		0 },
	{ MOUNT_LFS,	&args.args_ufs,		sizeof(struct ufs_args),
	  (parseargs_fp)mount_lfs_parseargs,		0 },
	{ MOUNT_MSDOS,	&args.args_msdos,	sizeof(struct msdosfs_args),
	  (parseargs_fp)mount_msdos_parseargs,		0 },
	{ MOUNT_NTFS,	&args.args_ntfs,	sizeof(struct ntfs_args),
	  (parseargs_fp)mount_ntfs_parseargs,		0 },
	{ MOUNT_UDF,	&args.args_udf,		sizeof(struct udf_args),
	  (parseargs_fp)mount_udf_parseargs,		0 },
	{ MOUNT_EFS,	&args.args_efs,		sizeof(struct efs_args),
	  (parseargs_fp)mount_efs_parseargs,		0 },
	{ MOUNT_TMPFS,	&args.args_tmpfs,	sizeof(struct tmpfs_args),
	  (parseargs_fp)mount_tmpfs_parseargs,		0 },
	{ MOUNT_PUFFS,	&args.args_syspuffs,	sizeof(struct syspuffs_args),
	  (parseargs_fp)mount_syspuffs_parseargs,	FS_NO_AUTODETECT },
	{ NULL,		NULL,			0,  NULL, 0 }
};

static struct fsu_fsalias_s *alias_head, *alias_tail;
static char *fsdevice, *fstype;

static int build_alias_list(void);
static void free_alias_list(void);
static struct ukfs *mount_alias(struct fsu_fsalias_s *, char *);
static struct ukfs *mount_fstype(fsu_fs_t *, char *, char *);

/*
 * Tries to mount an image.
 * if the fstype is not given try every supported types.
 */
struct ukfs *
fsu_mount(int *argc, char **argv[], char **fstp, char **fsdp)
{
	struct ukfs *ukfs;
	fsu_fs_t *fst;
	struct fsu_fsalias_s *cural;
	int ch, idx;
	char *mntopts;

	if (*argc < 2) {
		errno = EINVAL;
		return NULL;
	}

	mntopts = fstype = fsdevice = NULL;
	fst = NULL;
	ukfs = NULL;

	ukfs_init();
	ukfs_modload_dir("/usr/lib");

	/*
	 * command [-m "mnt_args"] [[-t] fstype] [-f] fsdevice command_args
	 */
	while ((ch = getopt(*argc, *argv, "f:m:t:")) != -1) {
		switch (ch) {
		case 'f':
			fsdevice = optarg;
			break;
		case 'm':
			mntopts = optarg;
			break;
		case 't':
			fstype = optarg;
			break;
		case '?':
		default:
			return NULL;
		}
	}
	idx = optind;

	if (fstype != NULL) {
		for (fst = fslist; fst->fs_name != NULL; ++fst)
			if (strcmp(fstype, fst->fs_name) == 0)
				break;

		if (fst->fs_name == NULL) {
			fprintf(stderr, "%s: filesystem not supported\n",
				fstype);
			return NULL;
		}
	}
	if (fst == NULL && idx < *argc) {
		for (fst = fslist; fst->fs_name != NULL; ++fst) {
			if (fst->fs_flags & FS_NO_AUTODETECT)
				continue;
			if (strcmp((*argv)[idx], fst->fs_name) == 0)	
				break;
		}
		if (fst->fs_name != NULL)
			idx++;
		else
			fst = NULL;
	}

       	if (fsdevice == NULL) {
		if (idx < *argc)
			fsdevice = (*argv)[idx++];
		else
			return NULL;

		build_alias_list();
		for (cural = alias_head; cural != NULL;
		     cural = cural->fsa_next) {
			if (cural->fsa_name != NULL &&
			    strcmp(cural->fsa_name, fsdevice) == 0) {
				ukfs = mount_alias(cural, mntopts);
				fstype = cural->fsa_type;
				fsdevice = cural->fsa_path;
				goto out;
			}
		}
		free_alias_list();
	}

	ukfs = mount_fstype(fst, fsdevice, mntopts);
out:
	if (--idx > 0) {
		(*argv)[idx] = (*argv)[0];
		*argv += idx;
		*argc -= idx;
		optind = optreset = 1;
	}
	if (fstp != NULL)
		*fstp = fstype;
	if (fsdp !=  NULL)
		*fsdp = fsdevice;

	return ukfs;
}

static struct ukfs
*mount_fstype(fsu_fs_t *fs, char *fsdev, char *mntopts)
{
	struct ukfs *ukfs;
	int rv, mntflags, mntargc;
	char canon_dev[MAXPATHLEN], canon_dir[MAXPATHLEN];
	char *mntargv[MAX_MNT_ARGS];

	mntflags = 0;
	mntargv[0] = strdup(getprogname());
	mntargc = 1;
	ukfs = NULL;

	if (mntopts != NULL) {
		fsu_str2arg(mntopts, &mntargc, mntargv + 1,
			    (unsigned)sizeof(mntargv) - 4);
		mntargc++;
	}

	/* only NULL for puffs */
	if (fsdev != NULL)
		mntargv[mntargc++] = fsdev;
	mntargv[mntargc++] = strdup("/");
	mntargv[mntargc] = NULL;

	/* filesystem given */
	if (fs != NULL) {
		optind = optreset = 1;
		rv = fs->fs_parseargs(mntargc, mntargv, fs->fs_args, &mntflags,
				     canon_dev, canon_dir);
		optind = optreset = 1;
		if (rv != 0)
			return NULL;
		
		ukfs = ukfs_mount(fs->fs_name, canon_dev, canon_dir, mntflags,
				  fs->fs_args, fs->fs_args_size);
		if (ukfs == NULL)
			fprintf(stderr, "%s is not a valid %s image\n",
				fsdev, fs->fs_name);
		goto out;
	}

	/* filesystem not given (auto detection) */
	for (fs = fslist; ukfs == NULL && fs->fs_name != NULL; ++fs) {
		if (fs->fs_flags & FS_NO_AUTODETECT)
			continue;
		mntflags = 0;
		optind = optreset = 1;
		rv = fs->fs_parseargs(mntargc, mntargv, fs->fs_args, &mntflags,
				      canon_dev, canon_dir);
		optind = optreset = 1;
		if (rv != 0)
			continue;
		
		ukfs = ukfs_mount(fs->fs_name, canon_dev, canon_dir, mntflags,
				  fs->fs_args, fs->fs_args_size);
	}

out:
	free(mntargv[0]);
	free(mntargv[mntargc - 1]);
	if (fs->fs_name != NULL)
		fstype = strdup(fs->fs_name);

	return ukfs;
}

static struct ukfs
*mount_alias(struct fsu_fsalias_s *al, char *mntopts)
{
	struct ukfs *ukfs;
	fsu_fs_t *cur;
	int rv, mntflags, mntargc;
	char canon_dev[MAXPATHLEN], canon_dir[MAXPATHLEN];
	char *mntargv[MAX_MNT_ARGS];

	ukfs = NULL;
	mntflags = 0;
	mntargv[0] = strdup(getprogname());
	mntargc = 1;

	if (al->fsa_mntopt != NULL) {
		fsu_str2arg(mntopts, &rv, mntargv + mntargc,
			    (int)sizeof(mntargv) - mntargc);
		mntargc += rv;

		if (mntargc >= sizeof(mntargv)) {
			free(mntargv[0]);
			return NULL;
		}
	}

	if (mntopts != NULL) {
		fsu_str2arg(mntopts, &rv, mntargv + mntargc,
			    (int)sizeof(mntargv) - mntargc);
		mntargc += rv;
	}

	if (mntargc + 2 > sizeof(mntargv)) {
		free(mntargv[0]);
		return NULL;
	}

	if (al->fsa_path != NULL)
		mntargv[mntargc++] = al->fsa_path;
	mntargv[mntargc++] = strdup("/");
	mntargv[mntargc] = NULL;
	
	for (cur = fslist; cur->fs_name != NULL; ++cur) {
		if (cur->fs_flags & FS_NO_AUTODETECT)
			continue;
		if (strcmp(cur->fs_name, al->fsa_type) == 0)
			break;
	}

	if (cur->fs_name != NULL) {
		rv = cur->fs_parseargs(mntargc, mntargv, cur->fs_args,
				       &mntflags, canon_dev, canon_dir);
		if (rv == 0)
			ukfs = ukfs_mount(cur->fs_name, canon_dev, canon_dir,
					  mntflags, cur->fs_args,
					  cur->fs_args_size);
	}

	free(mntargv[mntargc - 1]);
	free(mntargv[0]);
	if (cur->fs_name != NULL)
		fstype = strdup(cur->fs_name);
	return ukfs;

}

static int
build_alias_list(void)
{
	struct stat sb;
	struct fsu_fsalias_s *new;
	FILE *fd;
	int i, rv;
	size_t len, off;
	char buf[8192], file[PATH_MAX + 1], *fp, *home;

	fp = getenv("HOME");
	if (fp == NULL) {
		warn("getenv");
		return -1;
	}

	len = strlcpy(file, fp, PATH_MAX + 1);
	file[len] = '/';
	file[len + 1] = '\0';
	strlcat(file, ".fsurc", PATH_MAX + 1);

	/* no file no alias loaded */
	if (stat(file, &sb) != 0 || (fd = fopen(file, "r")) == NULL)
		return -1;

	rv = 0;
	new = NULL;
	for (;;) {
		/* get a line */
		for (i = 0; i < sizeof(buf); ++i) {
			buf[i] = fgetc(fd);
			if (buf[i] == '\n' || feof(fd)) {
				buf[i] = '\0';
				break;
			}
		}

		if (buf[0] == '#')
			continue;
		if (i == 0 && feof(fd))
			break;

		if (new == NULL) {
			new = malloc(sizeof(struct fsu_fsalias_s));
			if (new == NULL) {
				warn("malloc");
				rv = 1;
				goto out;
			}
		}

		fp = strtok(buf, ":");
		if (fp == NULL)
			continue;
		if (fp[0] == '-') {
			fprintf(stderr,
			  "alias starting with a dash ('-') are not suported");
			continue;
		}
		new->fsa_name = strdup(fp);

		fp = strtok(NULL, ":");
		if (fp == NULL)
			continue;

		if (fp[0] == '~' && fp[1] == '/') {
			home = getenv("HOME");

			if (home == NULL) {
				warn("getenv");
				continue;
			}
			len = strlen(home) + strlen(fp);
			new->fsa_path = malloc(len);
			if (new->fsa_path == NULL) {
				warn("malloc");
				continue;
			}
			off = strlcpy(new->fsa_path, home, len);
			new->fsa_path[off] = '/';
			new->fsa_path[off + 1] = '\0';
			strlcat(new->fsa_path, fp + 2, len);
		} else
			new->fsa_path = strdup(fp);

		fp = strtok(NULL, ":");
		if (fp == NULL)
			continue;
		new->fsa_type = strdup(fp);

		fp = strtok(NULL, ":");
		if (fp == NULL)
			new->fsa_mntopt = NULL;
		else
			new->fsa_mntopt = strdup(fp);

		new->fsa_next = NULL;
		
		if (alias_head == NULL)
			alias_head = alias_tail = new;
		else
			alias_tail = alias_tail->fsa_next = new;
		new = NULL;
	}

out:
	fclose(fd);
	if (new != NULL)
		free(new);
	return rv;
}

static void
free_alias_list(void)
{
	struct fsu_fsalias_s *cur;

	while ((cur = alias_head) != NULL) {
		alias_head = alias_head->fsa_next;
		free(cur->fsa_name);
		free(cur->fsa_path);
		free(cur->fsa_type);
		if (cur->fsa_mntopt != NULL)
			free(cur->fsa_mntopt);
		free(cur);
	}
}

const char
*fsu_mount_usage(void)
{

	return "[-m \"mnt_args\"] [[-t] fstype] [-f] fsdevice";
}
