/*	$Id: fsu_write.c,v 1.8 2008/10/09 19:23:53 stacktic Exp $	*/

/*
 * Copyright (c) 2008 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <err.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <rump/ukfs.h>

#include <fsu_file.h>

#ifndef FSU_CONSOLE
#include <fsu_mount.h>

static struct ukfs *ukfs;

static void	unmount_ukfs(void);
#else /* !FSU_CONSOLE */

#define main(a, b) fsu_write_main(struct ukfs *ukfs, a, b)

int fsu_write_main(struct ukfs *, int, char *[]);

static char *progname;
#endif /* !FSU_CONSOLE */

static void	usage(void);
int		fsu_write(struct ukfs *, int, const char *, int);

int
main(int argc, char *argv[])
{
	int append, rv;

#ifndef FSU_CONSOLE
	setprogname(argv[0]);

	if (argc < 2)
		usage();

	ukfs = fsu_mount(&argc, &argv, NULL, NULL);
	if (ukfs == NULL) {
		warn("fsu_mount");
		usage();
		/* NOTREACHED */
	}
	atexit(unmount_ukfs);
#else
	progname = argv[0];
#endif
	
	append = 0;
	while ((rv = getopt(argc, argv, "a")) != -1) {
		switch(rv) {
		case 'a':
			append = 1;
			break;

		case '?':
		default:
			usage();
			return EXIT_FAILURE;
		}
	}

	if (optind >= argc) {
		usage();
		return EXIT_FAILURE;
	}

	rv = fsu_write(ukfs, STDIN_FILENO, argv[optind], append);

	return rv != 0;
}

int
fsu_write(struct ukfs *fs, int fd, const char *fname, int append)
{
	int rd, wr;
	size_t total;
	uint8_t buf[8192];
	FSU_FILE *fdout;

	if (fname == NULL)
		return -1;

	fdout = fsu_fopen(fs, fname, append ? "a" : "w");
	if (fdout == NULL) {
		warn("%s", fname);
		return -1;
	}

	total = 0;
	do {
		rd = read(fd, buf, sizeof(buf));
		if (rd == -1) {
			warn("read");
			goto out;
		}
		wr = fsu_fwrite(buf, 1, rd, fdout);
		if (wr == -1 || wr != rd) {
			warn("%s", fname);
			goto out;
		}
		total += wr;
	} while (rd > 0 && errno != ENOSPC);

out:
	fsu_fclose(fdout);
	return total;
}

static void
usage(void)
{

#ifdef FSU_CONSOLE
	fprintf(stderr, "usage: %s [-a] file\n", progname);
#else
	fprintf(stderr, "usage: %s %s [-a] file\n",
		getprogname(), fsu_mount_usage());

	exit(EXIT_FAILURE);
#endif
}


#ifndef FSU_CONSOLE
static void
unmount_ukfs(void)
{
	
	ukfs_release(ukfs, 0);
}
#endif
