/* $Id: fsu_mv.c,v 1.5 2008/10/08 18:44:40 stacktic Exp $ */

/*
 * Copyright (c) 2008 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/stat.h>
#include <sys/syslimits.h>

#include <err.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <rump/ukfs.h>

#include <fsu_mount.h>

#include "fsu_mv.h"

static int is_user_ok(const char *);
static int move_to_dir(struct ukfs *, const char *, const char *, int);
static int move_to_file(struct ukfs *, const char *, const char *, int);
static void usage(void);
int fsu_mv_main(struct ukfs *, int, char *[]);
static char *progname;

int
fsu_mv_main(struct ukfs *fs, int argc, char *argv[])
{
	int cur_arg, flags = 0, rv;

	progname = argv[0];

	flags = fsu_mv_parse_arg(&argc, &argv);
	if (flags == -1 || argc < 2) {
		usage();
		return -1;
	}

	for (rv = 0, cur_arg = 0; cur_arg < argc-1; cur_arg++)
		rv |= fsu_mv(fs, argv[cur_arg], argv[argc-1], flags);

	return rv;
}

int
fsu_mv_parse_arg(int *argc, char ***argv)
{
	int flags, rv;

	flags = 0;
	while ((rv = getopt(*argc, *argv, "iv")) != -1) {
		switch (rv) {
		case 'i':
			flags |= FSU_MV_INTERACTIVE;
			break;
		case 'v':
			flags |= FSU_MV_VERBOSE;
			break;
		case '?':
		default:
			return -1;
		}
	}
	*argc -= optind;
	*argv += optind;

	return flags;
}


int
fsu_mv(struct ukfs *fs, const char *from, const char *to, int flags)
{
	int rv;
	struct stat file_stat;
	
	rv = ukfs_stat(fs, to, &file_stat);
	if (rv == -1 || !S_ISDIR(file_stat.st_mode))
		return move_to_file(fs, from, to, flags);

	return move_to_dir(fs, from, to, flags);
}

static int
is_user_ok(const char *to)
{
	int ch, ch2;

	printf("Overwrite %s? ", to);
	if ((ch = getchar()) != EOF && ch != '\n')
		while ((ch2 = getchar()) != EOF && ch2 != '\n')
			continue;
	
	if (ch != 'y' && ch != 'Y')
		return 0;
	return 1;
}

static int
move_to_dir(struct ukfs *fs, const char *from, const char *to, int flags)
{
	const char *p;
	int rv;
	char path[PATH_MAX + 1];
	struct stat file_stat;
	size_t len;

	len = strlen(to);
	rv = strlcpy(path, to, PATH_MAX + 1);
	if (rv != len) {
		warn("%s", to);
		return -1;
	}

	rv = ukfs_stat(fs, from, &file_stat);
	if (rv == -1) {
		warn("%s", from);
		return -1;
	}
	
	p = strrchr(from, '/');
	if (p == NULL)
		p = from;
	else
		++p;
	
	rv = strlcat(path, p, PATH_MAX + 1);
	if (rv == -1) {
		warn("%s/%s", path, p);
		return -1;
	}

	if (flags & FSU_MV_VERBOSE)
		printf("%s -> %s\n", from, path);

	if (flags & FSU_MV_INTERACTIVE) {
		rv = ukfs_stat(fs, path, &file_stat);
		if (rv != -1 && !is_user_ok(to))
			return 0;
	}

	rv = ukfs_rename(fs, from, path);
	if (rv == -1) {
		warn("%s", from);
		return -1;
	}
	return 0;
}

static int
move_to_file(struct ukfs *fs, const char *from, const char *to, int flags)
{
	int rv;
	struct stat file_stat;

	if (flags & FSU_MV_VERBOSE)
		printf("%s -> %s\n", from, to);
	
	if (flags & FSU_MV_INTERACTIVE) {
		rv = ukfs_stat(fs, to, &file_stat);
		if (rv != -1 && !is_user_ok(to))
			return 0;
	}
	rv = ukfs_rename(fs, from, to);
	if (rv == -1) {
		warn("%s or %s", from, to);
		return -1;
	}
	return 0;
}

static void
usage(void)
{

#ifdef FSU_CONSOLE
	fprintf(stderr, "usage: \t%s [-iv] source target\n"
		"\t%s [-iv] source ... directory\n",
		progname, progname);
#else
	fprintf(stderr, "usage: \t%s %s [-iv] source target\n"
		"       \t%s %s [-iv] source ... directory\n",
		getprogname(), fsu_mount_usage(),
		getprogname(), fsu_mount_usage());
	exit(EXIT_FAILURE);
#endif
}
