/* $Id: fsu_ln.c,v 1.6 2008/10/08 18:44:40 stacktic Exp $ */

/*
 * Copyright (c) 2008 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/stat.h>
#include <sys/syslimits.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <err.h>
#include <errno.h>

#include <rump/ukfs.h>

#include "fsu_ln.h"

#ifndef FSU_CONSOLE
#include <fsu_mount.h>

static struct ukfs *ukfs;

static void	unmount_ukfs(void);
#else /* !FSU_CONSOLE */

#define main(a, b) fsu_ln_main(struct ukfs *ukfs, a, b)

int fsu_ln_main(struct ukfs *, int, char *[]);

static char *progname;
#endif /* !FSU_CONSOLE */

static int linkit(struct ukfs *, const char *, const char *, int, int);
static void usage(void);
static int fsu_ln_parse_arg(int *, char ***);
static int fsu_ln(struct ukfs *, const char *, const char *, int);

int
main(int argc, char *argv[])
{
	int cur_arg, flags, rv;
	struct stat file_stat;

#ifdef FSU_CONSOLE
	progname = argv[0];
#endif
	setprogname(argv[0]);

#ifndef FSU_CONSOLE
	if (argc < 2)
		usage();

	ukfs = fsu_mount(&argc, &argv, NULL, NULL);
	if (ukfs == NULL) {
		warn("fsu_mount");
		usage();
		/* NOTREACHED */
	}
	atexit(unmount_ukfs);
#endif

	flags = fsu_ln_parse_arg(&argc, &argv);
	if (flags == -1 || argc < 1) {
		usage();
		return -1;
	}

	switch (argc) {
	case 1:
		return fsu_ln(ukfs, argv[0], ".", flags);
	case 2:
		return fsu_ln(ukfs, argv[0], argv[1], flags);
	}

	rv = ukfs_stat(ukfs, argv[argc-1], &file_stat);
	if (rv == -1 || !S_ISDIR(file_stat.st_mode)) {
		usage();
		return -1;
	}
	for (rv = 0, cur_arg = 0; cur_arg < argc - 1; ++cur_arg)
		rv |= fsu_ln(ukfs, argv[cur_arg], argv[argc - 1], flags);

	return rv;
}

static int
fsu_ln_parse_arg(int *argc, char ***argv)
{
	int flags, rv;

	flags = 0;
	while ((rv = getopt(*argc, *argv, "fhinsv")) != -1) {
		switch (rv) {
		case 'f':
			flags |= FSU_LN_FORCE;
			flags &= ~FSU_LN_INTERACTIVE;
			break;
		case 'h':
		case 'n':
			flags |= FSU_LN_NOT_FOLLOW_SYMLNK;
			break;
		case 'i':
			flags |= FSU_LN_INTERACTIVE;
			flags &= ~FSU_LN_FORCE;
			break;
		case 's':
			flags |= FSU_LN_SYMBOLIC;
			break;
		case 'v':
			flags |= FSU_LN_VERBOSE;
			break;
		case '?':
		default:
			return -1;
		}
	}
	*argv += optind;
	*argc -= optind;

	return flags;
}

static int
fsu_ln(struct ukfs *fs, const char *from, const char *to, int flags)
{
	struct stat sb;
	int isdir, rv;

	if (flags & FSU_LN_NOT_FOLLOW_SYMLNK) {
		rv = ukfs_lstat(fs, to, &sb);
		if (rv == 0 && S_ISLNK(sb.st_mode)) {
			errno = ENOTDIR;
			warn("%s", to);
			return -1;
		}
	}

	rv = ukfs_stat(fs, to, &sb);
	if (rv != 0)
		return linkit(fs, from, to, 0, flags);

	isdir = S_ISDIR(sb.st_mode);
	if (!isdir) {
		errno = ENOTDIR;
		warn("%s", to);
		return -1;
	}
	return linkit(fs, to, from, isdir, flags);
}
/* Adapted from bin/ln.c */
/*
 * Copyright (c) 1987, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
static int
linkit(struct ukfs *fs, const char *target, const char *source, int isdir,
       int flags)
{
	struct stat sb;
	const char *p;
	char linkch, path[PATH_MAX];
	int ch, exists, first, rv;
	
	if (!(flags & FSU_LN_SYMBOLIC)) {
		/* If target doesn't exist, quit now. */
		if (ukfs_stat(fs, target, &sb)) {
			warn("%s", target);
			return (1);
		}
	}

	/* If the source is a directory (and not a symlink if hflag),
	   append the target's name. */
	if (isdir ||
	    (!ukfs_lstat(fs, source, &sb) && S_ISDIR(sb.st_mode)) ||
	    (!(flags & FSU_LN_NOT_FOLLOW_SYMLNK) &&
	     !ukfs_stat(fs, source, &sb) && S_ISDIR(sb.st_mode))) {
		if ((p = strrchr(target, '/')) == NULL)
			p = target;
		else
			++p;
		snprintf(path, sizeof(path), "%s/%s", source, p);
		source = path;
	}

	exists = ukfs_lstat(fs, source, &sb) == 0;

	/*
	 * If the file exists, then unlink it forcibly if -f was specified
	 * and interactively if -i was specified.
	 */
	if ((flags & FSU_LN_FORCE) && exists) {
		if (ukfs_remove(fs, source)) {
			warn("%s", source);
			return (1);
		}
	} else if ((flags & FSU_LN_INTERACTIVE) && exists) {
		fflush(stdout);
		fprintf(stderr, "replace %s? ", source);

		first = ch = getchar();
		while (ch != '\n' && ch != EOF)
			ch = getchar();
		if (first != 'y' && first != 'Y') {
			fprintf(stderr, "not replaced\n");
			return (1);
		}

		if (ukfs_remove(fs, source)) {
			warn("%s", source);
			return (1);
		}
	}

	/* Attempt the link. */
	if (flags & FSU_LN_SYMBOLIC) {
		linkch = '-';
		rv = ukfs_symlink(fs, target, source);
	} else {
		linkch = '=';
		rv = ukfs_link(fs, target, source);
	}

	if (rv != 0) {
		warn("%s", source);
		return (1);
	}
	if (flags & FSU_LN_VERBOSE)
		printf("%s %c> %s\n", source, linkch, target);

	return (0);
}

static void
usage(void)
{

#ifdef FSU_CONSOLE
	fprintf(stderr, "usage:\t%s [-fhinsv] source_file [target_file]\n"
		"usage:\t%s [-fhinsv] source_file ... target_dir\n",
		progname, progname);
#else
	fprintf(stderr,
		"usage:\t%s %s [-fhinsv] source_file [target_file]\n"
		"usage:\t%s %s [-fhinsv] source_file ... target_dir\n",
		getprogname(), fsu_mount_usage(),
		getprogname(), fsu_mount_usage());
	exit(EXIT_FAILURE);
#endif
}

#ifndef FSU_CONSOLE
void
unmount_ukfs(void)
{
	
	ukfs_release(ukfs, 0);
}
#endif
