/* $Id: fsu_exec.c,v 1.6 2008/10/08 18:44:40 stacktic Exp $ */

/*
 * Copyright (c) 2008 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/syslimits.h>
#include <sys/wait.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <rump/ukfs.h>

#include <fsu_mount.h>

#ifndef FSU_CONSOLE
static struct ukfs *ukfs;

static void	unmount_ukfs(void);
#else /* !FSU_CONSOLE */

#define main(a, b) fsu_exec_main(struct ukfs *ukfs, a, b)

int fsu_exec_main(struct ukfs *, int, char *[]);

static char *progname;
#endif /* !FSU_CONSOLE */

static void usage(void);

#define BUFSIZE (8192)

static int copy_file(struct ukfs *, const char *, const char *, bool);
static void usage(void);

int
main(int argc, char **argv)
{
	char tmpfname[PATH_MAX];
	int len, rv;
	struct stat sb;
	pid_t child;
	char *from;

#ifndef FSU_CONSOLE
	if (argc < 2)
		usage();

	ukfs = fsu_mount(&argc, &argv, NULL, NULL);
	if (ukfs == NULL) {
		warn("fsu_mount");
		usage();
		/* NOTREACHED */
	}
	atexit(unmount_ukfs);
#else
	progname = argv[0];
#endif

	if (argc < 3) {
		usage();
		return -1;
	}
	++argv;
	--argc;

	from = argv[argc - 1];
	argv[argc - 1] = tmpfname;

	rv = snprintf(tmpfname, PATH_MAX-1, "/tmp/fsutmp.%i", getpid());
	if (rv <= 0) {
		fprintf(stderr, "error in snprintf\n");
		return -1;
	}

	if (stat(tmpfname, &sb) == 0)
		unlink(tmpfname);

	copy_file(ukfs, from, tmpfname, true);

	child = fork();
	switch (child) {
	case -1:
		warn("fork");
		rv = -1;
		goto out;
	case 0:
		execvp(argv[0], argv);
		exit(EXIT_FAILURE);
	default:
		waitpid(child, &len, 0);
	}

	copy_file(ukfs, tmpfname, from, false);

out:
	unlink(tmpfname);
	if (WIFEXITED(len))
		return WEXITSTATUS(len);
	return -1;
}

/*
 * get == true -> Copying from the image to the real fs
 * get == false -> Copying from the real fs to the image
 */
static int
copy_file(struct ukfs *fs, const char *from, const char *to, bool get)
{
	uint8_t buf[BUFSIZE];
	ssize_t rd, wr;
	off_t off;
	int fd, rv;
	struct stat from_stat;

	fd = -1;

	if (!get)
		rv = stat(from, &from_stat);
	else
		rv = ukfs_stat(fs, from, &from_stat);
	if (rv == -1) {
		warn("%s", from);
		return -1;
	}

	if (get)
		fd = rv = open(to, O_WRONLY|O_CREAT|O_EXCL, 0777);
	else {
		fd = rv = open(from, O_RDONLY, from_stat.st_mode);
		ukfs_remove(fs, to);
		rv = ukfs_create(fs, to, from_stat.st_mode);
	}
	if (rv == -1) {
		warn("%s", to);
		return -1;
	}
	
	off = 0;
	do {
		if (!get)
			rd = read(fd, buf, BUFSIZE);
		else
			rd = ukfs_read(fs, from, off, buf, BUFSIZE);
		if (rd == -1) {
			warn("%s", from);
			rv = -1;
			goto out;
		}
		if (get)
			wr = write(fd, buf, rd);
		else
			wr = ukfs_write(fs, to, off, buf, rd);
		if (wr == -1 || wr != rd) {
			warn("%s", to);
			rv = -1;
			goto out;
		}
		off += rd;
	} while (rd == BUFSIZE);

	rv = 0;
out:

	close(fd);
	return rv;
}

static void
usage(void)
{

#ifdef FSU_CONSOLE
	fprintf(stderr, "usage: %s command file...\n", progname);
#else
	fprintf(stderr, "usage: %s %s command file...\n",
		getprogname(), fsu_mount_usage());
	exit(EXIT_FAILURE);
#endif
}


#ifndef FSU_CONSOLE
void
unmount_ukfs(void)
{
	
	ukfs_release(ukfs, 0);
}
#endif
