/* $Id: commands.c,v 1.6 2008/10/10 05:46:54 stacktic Exp $ */

/*
 * Copyright (c) 2008 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/stat.h>
#include <sys/syslimits.h>
#include <sys/types.h>
#include <sys/wait.h>

#include <assert.h>
#include <dirent.h>
#include <err.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <rump/ukfs.h>

#include "fsu_utils.h"
#include "fsu_dir.h"

#include "extern.h"

#define BUFSIZE (8192)

static int fsu_get_cwd(struct ukfs *, char *);

/* current working directory and last working directory */
char cwd[PATH_MAX + 1], old_wd[PATH_MAX + 1];

/* head and tail for the alias linked list */
alias_t *al_head, *al_tail;

command_t commands[] = {
	{ "alias",	fsu_alias_main	 },
	{ "cat",	fsu_cat_main	 },
	{ "cd",		fsu_cd_main	 },
	{ "chflags",	fsu_chflags_main },
	{ "chmod",	fsu_chmod_main	 },
	{ "chown",	fsu_chown_main	 },
	{ "cp",		fsu_cp_main	 },
	{ "diff",	fsu_diff_main	 },
	{ "dir",	fsu_ls_main	 },
	{ "du",		fsu_du_main	 },
	{ "ecp",	fsu_ecp_main	 },
	{ "exit",	fsu_exit_main	 },
	{ "exec",	fsu_exec_main	 },
	{ "find",	fsu_find_main	 },
	{ "help",	fsu_help_main	 },
	{ "ln",		fsu_ln_main	 },
	{ "ls",		fsu_ls_main	 },
	{ "mkdir",	fsu_mkdir_main	 },
	{ "mkfifo",	fsu_mkfifo_main	 },
	{ "mknod",	fsu_mknod_main	 },
	{ "mv",		fsu_mv_main	 },
	{ "pwd",	fsu_pwd_main	 },
	{ "quit",	fsu_exit_main	 },
	{ "rm",		fsu_rm_main	 },
	{ "rmdir",	fsu_rmdir_main	 },
	{ "touch",	fsu_touch_main	 },
	{ "umask",	fsu_umask_main	 },
	{ "unalias",	fsu_unalias_main },
	{ "write",	fsu_write_main	 },
	{ NULL,		NULL		 }
};

int
fsu_alias_main(struct ukfs *fs, int argc, char *argv[])
{
	alias_t *cur;
	int i;

	/*
	 * No arguments: printing alias list.
	 * One argument: if this argument match an alias print it.
	 */
	if (argc == 1) {
		for (cur = al_head; cur != NULL; cur = cur->al_next) {
			printf("%s=\"", cur->al_name);
			for (i = 0; i < cur->al_argc; ++i)
				printf("%s ", cur->al_argv[i]);
			printf("\b\"\n");
		}
		return 0;
	} else if (argc == 2) {
		for (cur = al_head; cur != NULL; cur = cur->al_next) {
			if (strcmp(cur->al_name, argv[1]) == 0) {
				printf("%s=\"", cur->al_name);
				for (i = 0; i < cur->al_argc; ++i)
					printf("%s ", cur->al_argv[i]);
				printf("\b\"\n");
				break;
			}
		}
		return 0;
	}
	
	/* check whether this alias exists */
	for (cur = al_head; cur != NULL; cur = cur->al_next) {
		if (strcmp(cur->al_name, argv[1]) == 0) {
			fprintf(stderr, "alias %s: already exists\n",
				cur->al_name);
			return -1;
		}
	}

	/* this alias does not exists, add it to the linked list */
	cur = malloc(sizeof(alias_t));
	if (cur == NULL) {
		warn("malloc");
		return -1;
	}

	cur->al_name = strdup(argv[1]);
	if (cur->al_name == NULL) {
		warn("strdup");
		goto mem3;
	}

	cur->al_argc = argc - 2;
	cur->al_argv = malloc(cur->al_argc * sizeof(char *));
	if (cur->al_argv == NULL) {
		warn("malloc");
		goto mem2;
	}

	for (i = 2; i < argc; ++i) {
		cur->al_argv[i - 2] = strdup(argv[i]);
		if (cur->al_argv[i - 2] == NULL) {
			warn("strdup");
			goto mem1;
		}
	}

	cur->al_next = NULL;

	if (al_tail != NULL)
		al_tail = al_tail->al_next = cur;
	else	
		al_tail = al_head = cur;

	return 0;	

mem1:
	for (i -= 3; i >= 0; --i)
		free(cur->al_argv[i]);
	free(cur->al_argv);
mem2:
	free(cur->al_name);
mem3:
	free(cur);

	return -1;
}

/* Removes an alias from the linked list */
int
fsu_unalias_main(struct ukfs *fs, int argc, char *argv[])
{
	alias_t *cur, *tmp;
	int i;

	if (argc < 2) {
		fprintf(stderr, "Usage: unalias alias...\n");
		return -1;
	}
	if (al_head == NULL) {
		fprintf(stderr, "unalias: alias list is empty\n");
		return -1;
	}
	
	for (++argv; *argv != NULL; ++argv) {
		tmp = NULL;
		if (strcmp(*argv, al_head->al_name) == 0) {
			tmp = al_head;
			al_head = al_head->al_next;
			if (al_head == NULL)
				al_tail = NULL;
		} else
			for (cur = al_head;
			     cur->al_next != NULL;
			     cur = cur->al_next)
				if (strcmp(cur->al_next->al_name, *argv) == 0) {
					tmp = cur->al_next;
					cur->al_next = cur->al_next->al_next;
					break;
				}
		if (tmp == NULL) {
			fprintf(stderr, "%s: no such alias\n", *argv);
			continue;
		}
		
		/* freeing */
		free(tmp->al_name);
		for (i = 0; i < tmp->al_argc; ++i)
			free(tmp->al_argv[i]);
		free(tmp->al_argv);
		free(tmp);
		tmp = NULL;
	}
	return 0;
}

/*
 * function for the cd command, supports "cd", "cd -" and "cd pathname"
 */
int
fsu_cd_main(struct ukfs *fs, int argc, char *argv[])
{
	int rv;
	char tmp[PATH_MAX + 1];
	
	if (argc == 1) {
		/* "cd" go to root directory */
		rv = strlcpy(old_wd, cwd, PATH_MAX + 1);
		if (rv == -1) {
			warn("%s: %s", argv[0], cwd);
			return -1;
		}
		cwd[0] = '/';
		cwd[1] = '\0';
	} else if (argc == 2 && strcmp(argv[1], "-") == 0) {
		/* "cd -" go to previous directory (swap cwd and old_cwd) */
		rv = strlcpy(tmp, old_wd, PATH_MAX + 1);
		if (rv == -1) {
			warn("%s: %s", argv[0], old_wd);
			return -1;
		}

		strlcpy(old_wd, cwd, PATH_MAX + 1);
		if (rv == -1) {
			warn("%s: %s", argv[0], cwd);
			strlcpy(old_wd, tmp, PATH_MAX + 1);
			return -1;
		}

		rv = strlcpy(cwd, tmp, PATH_MAX + 1);
		if (rv == -1) {
			warn("%s: %s", argv[0], tmp);
			strlcpy(cwd, old_wd, PATH_MAX + 1);
			strlcpy(old_wd, tmp, PATH_MAX + 1);
			return -1;
		}
		printf("%s\n", cwd);
	} else if (argc == 2) {
		/* check whether the given directory exists. */
		rv = ukfs_chdir(fs, argv[1]);
		if (rv == -1) {
			ukfs_chdir(fs, cwd);
			warn("%s: %s", argv[0], argv[1]);
			return -1;
		}
		strlcpy(old_wd, cwd, PATH_MAX + 1);

		fsu_get_cwd(fs, cwd);
		
		return 0;
	} else {
		fprintf(stderr, "usage: cd path\n");
		return -1;
	}

	rv = ukfs_chdir(fs, cwd);
	if (rv == -1) {
		warn("%s: %s", argv[0], cwd);
		return -1;
	}
	return 0;
}

int
fsu_help_main(struct ukfs *fs, int argc, char *argv[])
{
	command_t *com;
	
	for (com = commands; com->c_name != NULL; com++)
		printf("%s\n", com->c_name);
	return 0;
}

int
fsu_exit_main(struct ukfs *fs, int argc, char *argv[])
{

	ukfs_release(fs, 0);
	exit(EXIT_SUCCESS);
}

int
fsu_umask_main(struct ukfs *fs, int argc, char **argv)
{
	mode_t mode, *set;

	umask((mode = umask(0)));
	switch (argc) {
	case 1:
		printf("%04o\n", mode);
		break;
	case 2:
		set = setmode(argv[1]);
		mode = getmode(set, ~mode);
		free(set);
		umask(~mode);
		break;
	default:
		fprintf(stderr, "usage: %s [mode]\n", argv[0]);
		return -1;
	}
	return 0;
}

int
fsu_pwd_main(struct ukfs *fs, int argc, char *argv[])
{

	printf("%s\n", cwd);
	return 0;
}

static int
fsu_get_cwd(struct ukfs *fs, char *path)
{
	char *npath;

	npath = fsu_getcwd(fs);
	strlcpy(path, npath, PATH_MAX + 1);
	free(npath);
	return 0;
}
