#
# Copyright (c) 2010 The NetBSD Foundation, Inc.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND
# CONTRIBUTORS ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS BE LIABLE FOR ANY
# DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
# GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
# IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
# OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
# IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#

# TODO: The name of the binary should be set during the build to let tests work
# when automake's program renaming features are used.
SHELLS='shells'

common_head() {
    # Once atf 0.8 is released, descriptions will be optional.  Do not bother
    # to set redundant texts for them, as we will kill this anyway soon. 
    atf_set "descr" "XXX"
    atf_set "require.progs" "${SHELLS}"
}

check_showed_usage() {
    if ! grep "Usage: ${SHELLS}" stderr >/dev/null; then
        cat stderr
        atf_fail "Expected usage message, but it was not shown or is invalid"
    fi
}

check_error() {
    if ! grep "^${SHELLS}: $1" stderr >/dev/null; then
        cat stderr
        atf_fail "Expected error message \"$1\" not found in stderr"
    fi
}

check_usage_error() {
    local message="${1}"; shift
    atf_check -s eq:1 -o empty -e save:stderr "${SHELLS}" "${@}"
    check_showed_usage
    check_error ".*${message}"
}

check_no_file() {
    atf_check -s eq:1 -o empty -e save:stderr "${SHELLS}" \
        -K non-existent "${@}"
    check_error "Cannot find the shells file 'non-existent'"
}

compare_files() {
    if [ "${1}" = - ]; then
        cat >expected
        atf_check -s eq:0 -o file:expected -e empty cat "${2}"
    else
        atf_check -s eq:0 -o file:"${1}" -e empty cat "${2}"
    fi
}

create_shell() {
    local name="${1:-binary}"
    touch "${name}"
    chmod +x "${name}"
    echo $(pwd)/"${name}"
}

# -------------------------------------------------------------------------
# Generic command line tests.
# -------------------------------------------------------------------------

atf_test_case usage
usage_head() { common_head; }
usage_body() {
    atf_check -s eq:1 -o empty -e save:stderr "${SHELLS}"
    atf_check -s eq:0 -o ignore -e empty grep "${SHELLS}.*add" stderr
    atf_check -s eq:0 -o ignore -e empty grep "${SHELLS}.*check" stderr
    atf_check -s eq:0 -o ignore -e empty grep "${SHELLS}.*list" stderr
    atf_check -s eq:0 -o ignore -e empty grep "${SHELLS}.*remove" stderr
}

atf_test_case no_args
no_args_head() { common_head; }
no_args_body() {
    atf_check -s eq:1 -o empty -e save:stderr "${SHELLS}"
    check_showed_usage
}

atf_test_case unknown_command
unknown_command_head() { common_head; }
unknown_command_body() {
    check_usage_error "Unknown command 'foo'" foo
}

atf_test_case unknown_command_with_args
unknown_command_with_args_head() { common_head; }
unknown_command_with_args_body() {
    check_usage_error "Unknown command 'foo'" foo arg1 arg2
}

# -------------------------------------------------------------------------
# Add command.
# -------------------------------------------------------------------------

atf_test_case add_bad_args
add_bad_args_head() { common_head; }
add_bad_args_body() {
    check_usage_error "'add'.*one argument" add
    check_usage_error "'add'.*one argument" add arg1 arg2
}

atf_test_case add_no_file
add_no_file_head() { common_head; }
add_no_file_body() {
    local shell="$(create_shell)"
    atf_check -s eq:0 -o empty -e empty "${SHELLS}" -K non-existent add \
        "${shell}"
    compare_files - non-existent <<EOF
${shell}
EOF
}

atf_test_case add_invalid
add_invalid_head() { common_head; }
add_invalid_body() {
    atf_check -s eq:1 -o empty -e save:stderr "${SHELLS}" -K non-existent \
        add bin/sh
    check_error "Invalid shell 'bin/sh'.*not.*absolute"
}

atf_test_case add_nonexistent
add_nonexistent_head() { common_head; }
add_nonexistent_body() {
    atf_check -s eq:1 -o empty -e save:stderr "${SHELLS}" -K non-existent \
        add $(pwd)/shell
    check_error "Cannot add non-existent shell"
}

atf_test_case add_duplicate
add_duplicate_head() { common_head; }
add_duplicate_body() {
    local shell="$(create_shell)"
    cat >db <<EOF
# foo bar
${shell}
# baz
EOF
    atf_check -s eq:1 -o empty -e save:stderr "${SHELLS}" -K db add "${shell}"
    check_error ".*'${shell}'.*already.*database"
}

atf_test_case add_some
add_some_head() { common_head; }
add_some_body() {
    local shell1="$(create_shell binary1)"
    local shell2="$(create_shell binary2)"
    cp "$(atf_get_srcdir)/shells.gold" db
    atf_check -s eq:0 -o empty -e empty "${SHELLS}" -K db add "${shell1}"
    atf_check -s eq:0 -o empty -e empty "${SHELLS}" -K db add "${shell2}"

    cp "$(atf_get_srcdir)/shells.gold" expdb
    echo "${shell1}" >>expdb
    echo "${shell2}" >>expdb
    compare_files expdb db
}

atf_test_case add_create_fail
add_create_fail_head() { common_head; }
add_create_fail_body() {
    local shell="$(create_shell)"
    mkdir dir; chmod 555 dir
    atf_check -s eq:1 -o empty -e save:stderr "${SHELLS}" -K dir/db \
        add "${shell}"
    check_error "Failed to create"
}

atf_test_case add_change_fail
add_change_fail_head() { common_head; }
add_change_fail_body() {
    local shell="$(create_shell)"
    touch db; chmod 444 db
    atf_check -s eq:1 -o empty -e save:stderr "${SHELLS}" -K db \
        add "${shell}"
    check_error "Failed to add"
}

# -------------------------------------------------------------------------
# Check command.
# -------------------------------------------------------------------------

atf_test_case check_bad_args
check_bad_args_head() { common_head; }
check_bad_args_body() {
    check_usage_error "'check'.*one argument" check
    check_usage_error "'check'.*one argument" check arg1 arg2
}

atf_test_case check_no_file
check_no_file_head() { common_head; }
check_no_file_body() {
    check_no_file check /bin/sh1
}

atf_test_case check_true
check_true_head() { common_head; }
check_true_body() {
    for shell in /bin/sh1 /usr/bin/sh6; do
        atf_check -s eq:0 -o empty -e empty "${SHELLS}" \
            -K "$(atf_get_srcdir)/shells.gold" check "${shell}"
    done
}

atf_test_case check_false
check_false_head() { common_head; }
check_true_body() {
    for shell in /bin/sh /bin/sh2; do
        atf_check -s eq:1 -o empty -e empty "${SHELLS}" \
            -K "$(atf_get_srcdir)/shells.gold" check "${shell}"
    done
}

# -------------------------------------------------------------------------
# List command.
# -------------------------------------------------------------------------

atf_test_case list_bad_args
list_bad_args_head() { common_head; }
list_bad_args_body() {
    check_usage_error "'list'.*zero arguments" list arg1
}

atf_test_case list_no_file
list_no_file_head() { common_head; }
list_no_file_body() {
    check_no_file list
}

atf_test_case list_some
list_some_head() { common_head; }
list_some_body() {
    cat >expout <<EOF
/bin/sh1
/bin/sh3
/bin/sh4
/bin/sh5
/usr/bin/sh2
/usr/bin/sh6
EOF
    atf_check -s eq:0 -o file:expout -e empty "${SHELLS}" \
        -K "$(atf_get_srcdir)/shells.gold" list
}

# -------------------------------------------------------------------------
# Remove command.
# -------------------------------------------------------------------------

atf_test_case remove_bad_args
remove_bad_args_head() { common_head; }
remove_bad_args_body() {
    check_usage_error "'remove'.*one argument" remove
    check_usage_error "'remove'.*one argument" remove arg1 arg2
}

atf_test_case remove_no_file
remove_no_file_head() { common_head; }
remove_no_file_body() {
    check_no_file remove /bin/sh
}

atf_test_case remove_invalid
remove_invalid_head() { common_head; }
remove_invalid_body() {
    atf_check -s eq:1 -o empty -e save:stderr "${SHELLS}" -K non-existent \
        remove bin/sh
    check_error "Invalid shell 'bin/sh'.*not.*absolute"
}

atf_test_case remove_missing
remove_missing_head() { common_head; }
remove_missing_body() {
    atf_check -s eq:1 -o empty -e save:stderr "${SHELLS}" \
        -K "$(atf_get_srcdir)/shells.gold" remove /bin/sh6
    check_error ".*'/bin/sh6'.*not.*database"
}

atf_test_case remove_some
remove_some_head() { common_head; }
remove_some_body() {
    cp "$(atf_get_srcdir)/shells.gold" db
    chmod +x db # This is to check that permissions are kept.
    atf_check -s eq:0 -o empty -e empty "${SHELLS}" -K db remove /bin/sh3
    atf_check -s eq:0 -o empty -e empty "${SHELLS}" -K db remove /bin/sh1
    atf_check -s eq:0 -o empty -e empty "${SHELLS}" -K db remove /usr/bin/sh6

    cat "$(atf_get_srcdir)/shells.gold" | grep -v /bin/sh1 | grep -v /bin/sh3 \
        | grep -v /usr/bin/sh6 >db2
    compare_files db2 db

    test -x db || atf_fail "shells file permissions were not respected"
}

atf_test_case remove_last
remove_last_head() { common_head; }
remove_last_body() {
    cat >db <<EOF
/bin/sh
EOF
    atf_check -s eq:0 -o empty -e empty "${SHELLS}" -K db remove /bin/sh
    test -f db && atf_fail "Empty shells file left behind"
}

atf_test_case remove_commented
remove_commented_head() { common_head; }
remove_commented_body() {
    cat >db <<EOF
/bin/sh
#/bin/sh
EOF
    cat >db2 <<EOF
#/bin/sh
EOF
    atf_check -s eq:0 -o empty -e empty "${SHELLS}" -K db remove /bin/sh
    compare_files db db2
}

# -------------------------------------------------------------------------
# Main.
# -------------------------------------------------------------------------

atf_init_test_cases() {
    atf_add_test_case usage
    atf_add_test_case no_args
    atf_add_test_case unknown_command
    atf_add_test_case unknown_command_with_args

    atf_add_test_case add_bad_args
    atf_add_test_case add_no_file
    atf_add_test_case add_invalid
    atf_add_test_case add_nonexistent
    atf_add_test_case add_duplicate
    atf_add_test_case add_some
    atf_add_test_case add_create_fail
    atf_add_test_case add_change_fail

    atf_add_test_case check_bad_args
    atf_add_test_case check_no_file
    atf_add_test_case check_true
    atf_add_test_case check_false

    atf_add_test_case list_bad_args
    atf_add_test_case list_no_file
    atf_add_test_case list_some

    atf_add_test_case remove_bad_args
    atf_add_test_case remove_no_file
    atf_add_test_case remove_invalid
    atf_add_test_case remove_missing
    atf_add_test_case remove_some
    atf_add_test_case remove_last
    atf_add_test_case remove_commented
}

# vim: syntax=sh:expandtab:shiftwidth=4:softtabstop=4
