#include <stdio.h>
#include <unistd.h>

#include <sys/cdefs.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ras.h>

#include <machine/lock.h>

#define NLOOPS	(10000000)

void noinline_simple_lock(__cpu_simple_lock_t *lkp);
void __ras_simple_lock(__cpu_simple_lock_t *lkp);
void __ras_simple_lock_start(void);
void __ras_simple_lock_end(void);

#define lock_init(lkp)			\
	do { *(lkp) = 0; } while (0)
#define lock_unlock(lkp)		\
	do { *(lkp) = 0; } while (0)


double wtime(void)
{
	static long offset1;
	static long offset2;
	static char first = 'y';
	static double c0 = 1.0e-06;
	struct timeval tp;
	struct timezone tzp;
	long time_in_seconds;
	long time_in_microseconds;
	double wall_time;

	if (first == 'y') {
		first = 'n';
		if (gettimeofday(&tp, &tzp) == -1)
			perror("gettimeofday");
		offset1 = tp.tv_sec;
		offset2 = tp.tv_usec;
	}
	if (gettimeofday(&tp, &tzp) == -1)
		perror("gettimeofday");
	time_in_seconds = tp.tv_sec - offset1;
	time_in_microseconds = tp.tv_usec - offset2;
	wall_time = time_in_seconds + c0 * time_in_microseconds;
	return (wall_time);
}


int
main(void)
{
	double t0, t1, t2, t3, t4;
	__cpu_simple_lock_t lk;
	int i;

	printf("Registering restartable atomic sequences\n");

	rasctl((caddr_t)__ras_simple_lock_start,
	    (size_t)(__ras_simple_lock_end - __ras_simple_lock_start),
	    RAS_INSTALL);

	/*
	 * Time the loop/unlock overhead.
	 */
	printf("Timing unlock overhead\n");
	t0 = wtime();
	for (i=0; i<NLOOPS; i++) {
		lock_unlock(&lk);
	}
	t1 = wtime() - t0;

	/*
	 * RAS lock.
	 */
	printf("Timing RAS locks\n");
	lock_init(&lk);
	t0 = wtime();
	for (i=0; i<NLOOPS; i++) {
		__ras_simple_lock(&lk);
		lock_unlock(&lk);
	}
	t2 = wtime() - t0;

	/*
	 * CPU lock (inlined).
	 */
	printf("Timing CPU locks (inlined)\n");
	lock_init(&lk);
	t0 = wtime();
	for (i=0; i<NLOOPS; i++) {
		__cpu_simple_lock(&lk);
		lock_unlock(&lk);
	}
	t3 = wtime() - t0;

	/*
	 * CPU lock (not inlined).
	 */
	printf("Timing CPU locks (not inlined)\n");
	lock_init(&lk);
	t0 = wtime();
	for (i=0; i<NLOOPS; i++) {
		noinline_simple_lock(&lk);
		lock_unlock(&lk);
	}
	t4 = wtime() - t0;

	printf("unlock overhead: %f s (%f us/loop)\n",
	    t1, 1.0e6*t1/NLOOPS);

	printf("RAS: %f s (%f us/loop)\n",
	    t2-t1, 1.0e6*(t2-t1)/NLOOPS);

	printf("cpu locks (inlined): %f s (%f us/loop)\n",
	    t3-t1, 1.0e6*(t3-t1)/NLOOPS);

	printf("cpu locks (not inlined): %f s (%f us/loop)\n",
	    t4-t1, 1.0e6*(t4-t1)/NLOOPS);

	return (0);
}
