/******************************************************************************
 * mach64 Chapter 6 sample code                                               *
 *                                                                            *
 * compare.c - This program uses the mach64 engine to illustrate how the      *
 * compare feature is used.                                                   *
 *                                                                            *
 * Copyright (c) 1994-1998 ATI Technologies Inc.  All rights reserved.        *
 ******************************************************************************/

#include <stdio.h>
#include <i86.h>
#include "..\util\atim64.h"
#include "..\util\defines.h"
#include "..\util\main.h"

/******************************************************************************
 * Main Program to demonstrate rectangle drawing                              *
 *  Function: The rectangle pattern from rect.c is drawn on the screen.       *
 *            The compare feature of the mach64 is demonstrated by            *
 *            drawing a LIGHT GREEN rectangle over most of the existing       *
 *            screen, but comparing it with the destination being equal       *
 *            to LIGHT BLUE.                                                  *
 *    Inputs: Arguments for mode spatial and colour resolution                *
 *   Outputs: NONE                                                            *
 ******************************************************************************/

void main (int argc, char *argv[])
{
    int width;                          // Width of drawing area.
    int height;                         // Height of drawing area.
    int i;                              // Counter for rectangle rows.
    int j;                              // Counter for rectangle columns.
    int n;                              // # rectangles in x and y direction.
    int c_width;                        // Width of compare rectangle.
    int c_height;                       // Height of compare rectangle.

    printf ("mach64 Chapter 6 sample code\n"
            "\n"
            "compare.c\n"
            "The rectangle pattern from rect.c is drawn on the screen.  The compare\n"
            "feature of the mach64 is demonstrated by drawing a LIGHT GREEN\n"
            "rectangle over most of the existing screen, but comparing it with the\n"
            "destination being equal to LIGHT BLUE.  This results in the light blue\n"
            "rectangle on screen to remain while all other destination colours\n"
            "change to light green.\n"
            "\n"
            "Spatial resolution (640, 800, 1024, 1280, 1600) and Colour Depth\n"
            "(8, 15, 16, 24, 32) should be passed as arguments.\n"
            "Default setting is 640x480 spatial resolution and 8bpp pixel depth.\n");

    // Batch command to detect the mach64, perform a hardware query, Save old
    // mode information, process mode info arguments, load and set mode, enable
    // aperture, set up palettes, initialize engine to known state, and reset
    // all engine queues.

    start (argc, argv);

    // Check for 24 bpp mode - Compare destination is not directly
    // supported in 24 bpp modes.

    if (MODE_INFO.bpp == 24)
    {
        // Disable accelerator mode and switch back to VGA text mode.
        finish ();
        printf ("Compare destination is not supported in 24 bpp modes.\n");
        exit (1);
    } // if

    // Set up values.
    width = 640;                        // Less than or equal to xres.
    height = 480;                       // Less than or equal to yres.
    n = 4;                              // > 0 ,<= min (width, height).
    c_width = width*5/6;                // Less than or equal to xres.
    c_height = height*5/6;              // Less than or equal to yres.

    // Sample drawing routines.
    // 16 different coloured rectangles are drawn on the screen corresponding
    // to the first 16 EGA standard palette colours.

    clear_screen (0, 0, MODE_INFO.xres, MODE_INFO.yres);
    for (i=0; i<n; i++)
    {
        for (j=0; j<n; j++)
        {
            wait_for_fifo (1);
            regw (DP_FRGD_CLR, get_colour_code ((n*i + j)%NUM_COLOURS));
            draw_rectangle (((MODE_INFO.xres - width)/2) + (j*width/n),
                            ((MODE_INFO.yres - height)/2) + (i*height/n),
                            width/n, height/n);
        } // for
    } // for

    // Wait for a carriage return (show screen before operation).
    getch ();

    // Compare colour = LIGHTBLUE.
    wait_for_fifo (3);
    regw (CLR_CMP_CLR, get_colour_code (LIGHTBLUE));

    // Allow all bits to pass in comparison.
    regw (CLR_CMP_MASK, 0xFFFFFFFF);

    // Compare function = CLR_CMP_CLR (i.e., light blue).
    // Compare source = destination.
    regw (CLR_CMP_CNTL, COMPARE_EQUAL | COMPARE_DESTINATION);

    // Draw LIGHTGREEN rectangle over the existing rectangles.
    wait_for_fifo (1);
    regw (DP_FRGD_CLR, get_colour_code (LIGHTGREEN));
    draw_rectangle ((MODE_INFO.xres - c_width)/2, (MODE_INFO.yres - c_height)/2,
                    c_width, c_height);

    // Wait for a carriage return.
    getch ();

    // Batch command to restore old mode.
    finish ();

    exit (0);                           // No errors.

} // main
