/* 
** Copyright 2004, Marcus Overhagen. All rights reserved.
** Distributed under the terms of the MIT license.
**
** This file implements a image format conversion from
** planar YCbCr420 format to linear RGB32 format.
** Lookup tables are used to avoid doing multiplications
** or conditial if() checks inside the inner loop.
** These tables have been generated by gen_lookup.cpp
*/

#include <strings.h>
#include <stdio.h>
#include "gfx_conv_c.h"

extern const int32 Cr_Rtab[];
extern const int32 Cr_Gtab[];
extern const int32 Cb_Gtab[];
extern const int32 Cb_Btab[];
extern const int32 Ytab[];
extern const uint32 *Rtab;
extern const uint32 *Gtab;
extern const uint32 *Btab;

void gfx_conv_YCbCr420p_RGB32_c(AVFrame *in, AVFrame *out, int width, int height)
{
	uint32 poutInc = 2 * out->linesize[0];
	uint32 *poutEven = (uint32 *)out->data[0];
	uint32 *poutOdd = (uint32 *)(out->linesize[0] + (uint8 *)poutEven);
	
	uint32 pi1Inc = in->linesize[0];
	uint32 pi1Inc2 = 2 * pi1Inc;
	uint32 pi2Inc = in->linesize[1];
	uint32 pi3Inc = in->linesize[2];
	
	uint8 *pi1Base = (uint8 *)in->data[0];
	uint8 *pi2Base = (uint8 *)in->data[1];
	uint8 *pi3Base = (uint8 *)in->data[2];

	uint32 runs = height / 2;
	for (uint32 i = 0; i < runs; i++) {

		uint16 *pi1Even = (uint16 *) (i * pi1Inc2 + pi1Base);
		uint16 *pi1Odd = (uint16 *) (pi1Inc + (uint8 *)pi1Even);
		uint8 *pi2 = i * pi2Inc + pi2Base;
		uint8 *pi3 = i *pi3Inc + pi3Base;

		for (uint32 j = 0; j < (uint32)width; j+= 2) {

			int32 Cr_R, Cr_G, Cb_G, Cb_B;
			int32 Y0, Y1, R, G, B;
			uint32 temp;

			temp = *(pi2++);
			Cb_G = Cb_Gtab[temp];
			Cb_B = Cb_Btab[temp];
			temp = *(pi3++);
			Cr_R = Cr_Rtab[temp];
			Cr_G = Cr_Gtab[temp];

			temp = *(pi1Even++);
			Y0 = Ytab[temp & 0xff];
			Y1 = Ytab[temp >> 8];

			R = (Y0 + Cr_R) >> 15;
			G = (Y0 + Cr_G + Cb_G) >> 15;
			B = (Y0 + Cb_B) >> 15;
			poutEven[j] = Rtab[R] | Gtab[G] | Btab[B];

			R = (Y1 + Cr_R) >> 15;
			G = (Y1 + Cr_G + Cb_G) >> 15;
			B = (Y1 + Cb_B) >> 15;
			poutEven[j + 1] = Rtab[R] | Gtab[G] | Btab[B];

			temp = *(pi1Odd++);
			Y0 = Ytab[temp & 0xff];
			Y1 = Ytab[temp >> 8];

			R = (Y0 + Cr_R) >> 15;
			G = (Y0 + Cr_G + Cb_G) >> 15;
			B = (Y0 + Cb_B) >> 15;
			poutOdd[j] = Rtab[R] | Gtab[G] | Btab[B];

			R = (Y1 + Cr_R) >> 15;
			G = (Y1 + Cr_G + Cb_G) >> 15;
			B = (Y1 + Cb_B) >> 15;
			poutOdd[j + 1] = Rtab[R] | Gtab[G] | Btab[B];

		}
		poutEven = (uint32 *)(poutInc + (uint8 *)poutEven);
		poutOdd = (uint32 *)(poutInc + (uint8 *)poutOdd);
	}
	if (height & 1) {
		// XXX special case for last line if height not multiple of 2 goes here
		memset((height - 1) * out->linesize[0] + (uint8 *)out->data[0], 0, width * 4);
	}
}

const int32 Cb_Gtab[256] = {
	1644167, 1631322, 1618477, 1605632, 1592787, 1579942, 1567097, 1554252, 1541407, 1528562, 
	1515717, 1502872, 1490026, 1477181, 1464336, 1451491, 1438646, 1425801, 1412956, 1400111, 
	1387266, 1374421, 1361576, 1348731, 1335886, 1323041, 1310196, 1297351, 1284506, 1271661, 
	1258815, 1245970, 1233125, 1220280, 1207435, 1194590, 1181745, 1168900, 1156055, 1143210, 
	1130365, 1117520, 1104675, 1091830, 1078985, 1066140, 1053295, 1040450, 1027604, 1014759, 
	1001914, 989069, 976224, 963379, 950534, 937689, 924844, 911999, 899154, 886309, 
	873464, 860619, 847774, 834929, 822084, 809239, 796393, 783548, 770703, 757858, 
	745013, 732168, 719323, 706478, 693633, 680788, 667943, 655098, 642253, 629408, 
	616563, 603718, 590873, 578028, 565182, 552337, 539492, 526647, 513802, 500957, 
	488112, 475267, 462422, 449577, 436732, 423887, 411042, 398197, 385352, 372507, 
	359662, 346817, 333971, 321126, 308281, 295436, 282591, 269746, 256901, 244056, 
	231211, 218366, 205521, 192676, 179831, 166986, 154141, 141296, 128451, 115606, 
	102760, 89915, 77070, 64225, 51380, 38535, 25690, 12845, 0, -12844, 
	-25689, -38534, -51379, -64224, -77069, -89914, -102759, -115605, -128450, -141295, 
	-154140, -166985, -179830, -192675, -205520, -218365, -231210, -244055, -256900, -269745, 
	-282590, -295435, -308280, -321125, -333970, -346816, -359661, -372506, -385351, -398196, 
	-411041, -423886, -436731, -449576, -462421, -475266, -488111, -500956, -513801, -526646, 
	-539491, -552336, -565181, -578027, -590872, -603717, -616562, -629407, -642252, -655097, 
	-667942, -680787, -693632, -706477, -719322, -732167, -745012, -757857, -770702, -783547, 
	-796392, -809238, -822083, -834928, -847773, -860618, -873463, -886308, -899153, -911998, 
	-924843, -937688, -950533, -963378, -976223, -989068, -1001913, -1014758, -1027603, -1040449, 
	-1053294, -1066139, -1078984, -1091829, -1104674, -1117519, -1130364, -1143209, -1156054, -1168899, 
	-1181744, -1194589, -1207434, -1220279, -1233124, -1245969, -1258814, -1271660, -1284505, -1297350, 
	-1310195, -1323040, -1335885, -1348730, -1361575, -1374420, -1387265, -1400110, -1412955, -1425800, 
	-1438645, -1451490, -1464335, -1477180, -1490025, -1502871, -1515716, -1528561, -1541406, -1554251, 
	-1567096, -1579941, -1592786, -1605631, -1618476, -1631321, 
};

const int32 Cb_Btab[256] = {
	-8459910, -8393817, -8327724, -8261631, -8195538, -8129445, -8063352, -7997259, -7931166, -7865073, 
	-7798980, -7732887, -7666793, -7600700, -7534607, -7468514, -7402421, -7336328, -7270235, -7204142, 
	-7138049, -7071956, -7005863, -6939770, -6873677, -6807584, -6741491, -6675398, -6609305, -6543212, 
	-6477118, -6411025, -6344932, -6278839, -6212746, -6146653, -6080560, -6014467, -5948374, -5882281, 
	-5816188, -5750095, -5684002, -5617909, -5551816, -5485723, -5419630, -5353537, -5287443, -5221350, 
	-5155257, -5089164, -5023071, -4956978, -4890885, -4824792, -4758699, -4692606, -4626513, -4560420, 
	-4494327, -4428234, -4362141, -4296048, -4229955, -4163862, -4097768, -4031675, -3965582, -3899489, 
	-3833396, -3767303, -3701210, -3635117, -3569024, -3502931, -3436838, -3370745, -3304652, -3238559, 
	-3172466, -3106373, -3040280, -2974187, -2908093, -2842000, -2775907, -2709814, -2643721, -2577628, 
	-2511535, -2445442, -2379349, -2313256, -2247163, -2181070, -2114977, -2048884, -1982791, -1916698, 
	-1850605, -1784512, -1718418, -1652325, -1586232, -1520139, -1454046, -1387953, -1321860, -1255767, 
	-1189674, -1123581, -1057488, -991395, -925302, -859209, -793116, -727023, -660930, -594837, 
	-528743, -462650, -396557, -330464, -264371, -198278, -132185, -66092, 0, 66093, 
	132186, 198279, 264372, 330465, 396558, 462651, 528744, 594838, 660931, 727024, 
	793117, 859210, 925303, 991396, 1057489, 1123582, 1189675, 1255768, 1321861, 1387954, 
	1454047, 1520140, 1586233, 1652326, 1718419, 1784513, 1850606, 1916699, 1982792, 2048885, 
	2114978, 2181071, 2247164, 2313257, 2379350, 2445443, 2511536, 2577629, 2643722, 2709815, 
	2775908, 2842001, 2908094, 2974188, 3040281, 3106374, 3172467, 3238560, 3304653, 3370746, 
	3436839, 3502932, 3569025, 3635118, 3701211, 3767304, 3833397, 3899490, 3965583, 4031676, 
	4097769, 4163863, 4229956, 4296049, 4362142, 4428235, 4494328, 4560421, 4626514, 4692607, 
	4758700, 4824793, 4890886, 4956979, 5023072, 5089165, 5155258, 5221351, 5287444, 5353538, 
	5419631, 5485724, 5551817, 5617910, 5684003, 5750096, 5816189, 5882282, 5948375, 6014468, 
	6080561, 6146654, 6212747, 6278840, 6344933, 6411026, 6477119, 6543213, 6609306, 6675399, 
	6741492, 6807585, 6873678, 6939771, 7005864, 7071957, 7138050, 7204143, 7270236, 7336329, 
	7402422, 7468515, 7534608, 7600701, 7666794, 7732888, 7798981, 7865074, 7931167, 7997260, 
	8063353, 8129446, 8195539, 8261632, 8327725, 8393818, 
};

const int32 Cr_Rtab[256] = {
	-6694108, -6641810, -6589513, -6537215, -6484917, -6432620, -6380322, -6328024, -6275726, -6223429, 
	-6171131, -6118833, -6066535, -6014238, -5961940, -5909642, -5857345, -5805047, -5752749, -5700451, 
	-5648154, -5595856, -5543558, -5491260, -5438963, -5386665, -5334367, -5282070, -5229772, -5177474, 
	-5125176, -5072879, -5020581, -4968283, -4915985, -4863688, -4811390, -4759092, -4706795, -4654497, 
	-4602199, -4549901, -4497604, -4445306, -4393008, -4340710, -4288413, -4236115, -4183817, -4131520, 
	-4079222, -4026924, -3974626, -3922329, -3870031, -3817733, -3765435, -3713138, -3660840, -3608542, 
	-3556245, -3503947, -3451649, -3399351, -3347054, -3294756, -3242458, -3190160, -3137863, -3085565, 
	-3033267, -2980969, -2928672, -2876374, -2824076, -2771779, -2719481, -2667183, -2614885, -2562588, 
	-2510290, -2457992, -2405694, -2353397, -2301099, -2248801, -2196504, -2144206, -2091908, -2039610, 
	-1987313, -1935015, -1882717, -1830419, -1778122, -1725824, -1673526, -1621229, -1568931, -1516633, 
	-1464335, -1412038, -1359740, -1307442, -1255144, -1202847, -1150549, -1098251, -1045954, -993656, 
	-941358, -889060, -836763, -784465, -732167, -679869, -627572, -575274, -522976, -470679, 
	-418381, -366083, -313785, -261488, -209190, -156892, -104594, -52297, 0, 52298, 
	104595, 156893, 209191, 261489, 313786, 366084, 418382, 470680, 522977, 575275, 
	627573, 679870, 732168, 784466, 836764, 889061, 941359, 993657, 1045955, 1098252, 
	1150550, 1202848, 1255145, 1307443, 1359741, 1412039, 1464336, 1516634, 1568932, 1621230, 
	1673527, 1725825, 1778123, 1830420, 1882718, 1935016, 1987314, 2039611, 2091909, 2144207, 
	2196505, 2248802, 2301100, 2353398, 2405695, 2457993, 2510291, 2562589, 2614886, 2667184, 
	2719482, 2771780, 2824077, 2876375, 2928673, 2980970, 3033268, 3085566, 3137864, 3190161, 
	3242459, 3294757, 3347055, 3399352, 3451650, 3503948, 3556246, 3608543, 3660841, 3713139, 
	3765436, 3817734, 3870032, 3922330, 3974627, 4026925, 4079223, 4131521, 4183818, 4236116, 
	4288414, 4340711, 4393009, 4445307, 4497605, 4549902, 4602200, 4654498, 4706796, 4759093, 
	4811391, 4863689, 4915986, 4968284, 5020582, 5072880, 5125177, 5177475, 5229773, 5282071, 
	5334368, 5386666, 5438964, 5491261, 5543559, 5595857, 5648155, 5700452, 5752750, 5805048, 
	5857346, 5909643, 5961941, 6014239, 6066536, 6118834, 6171132, 6223430, 6275727, 6328025, 
	6380323, 6432621, 6484918, 6537216, 6589514, 6641811, 
};

const int32 Cr_Gtab[256] = {
	3409969, 3383329, 3356688, 3330048, 3303408, 3276767, 3250127, 3223486, 3196846, 3170206, 
	3143565, 3116925, 3090285, 3063644, 3037004, 3010363, 2983723, 2957083, 2930442, 2903802, 
	2877161, 2850521, 2823881, 2797240, 2770600, 2743960, 2717319, 2690679, 2664038, 2637398, 
	2610758, 2584117, 2557477, 2530836, 2504196, 2477556, 2450915, 2424275, 2397635, 2370994, 
	2344354, 2317713, 2291073, 2264433, 2237792, 2211152, 2184511, 2157871, 2131231, 2104590, 
	2077950, 2051310, 2024669, 1998029, 1971388, 1944748, 1918108, 1891467, 1864827, 1838186, 
	1811546, 1784906, 1758265, 1731625, 1704985, 1678344, 1651704, 1625063, 1598423, 1571783, 
	1545142, 1518502, 1491862, 1465221, 1438581, 1411940, 1385300, 1358660, 1332019, 1305379, 
	1278738, 1252098, 1225458, 1198817, 1172177, 1145537, 1118896, 1092256, 1065615, 1038975, 
	1012335, 985694, 959054, 932413, 905773, 879133, 852492, 825852, 799212, 772571, 
	745931, 719290, 692650, 666010, 639369, 612729, 586088, 559448, 532808, 506167, 
	479527, 452887, 426246, 399606, 372965, 346325, 319685, 293044, 266404, 239763, 
	213123, 186483, 159842, 133202, 106562, 79921, 53281, 26640, 0, -26639, 
	-53280, -79920, -106561, -133201, -159841, -186482, -213122, -239762, -266403, -293043, 
	-319684, -346324, -372964, -399605, -426245, -452886, -479526, -506166, -532807, -559447, 
	-586087, -612728, -639368, -666009, -692649, -719289, -745930, -772570, -799211, -825851, 
	-852491, -879132, -905772, -932412, -959053, -985693, -1012334, -1038974, -1065614, -1092255, 
	-1118895, -1145536, -1172176, -1198816, -1225457, -1252097, -1278737, -1305378, -1332018, -1358659, 
	-1385299, -1411939, -1438580, -1465220, -1491861, -1518501, -1545141, -1571782, -1598422, -1625062, 
	-1651703, -1678343, -1704984, -1731624, -1758264, -1784905, -1811545, -1838185, -1864826, -1891466, 
	-1918107, -1944747, -1971387, -1998028, -2024668, -2051309, -2077949, -2104589, -2131230, -2157870, 
	-2184510, -2211151, -2237791, -2264432, -2291072, -2317712, -2344353, -2370993, -2397634, -2424274, 
	-2450914, -2477555, -2504195, -2530835, -2557476, -2584116, -2610757, -2637397, -2664037, -2690678, 
	-2717318, -2743959, -2770599, -2797239, -2823880, -2850520, -2877160, -2903801, -2930441, -2957082, 
	-2983722, -3010362, -3037003, -3063643, -3090284, -3116924, -3143564, -3170205, -3196845, -3223485, 
	-3250126, -3276766, -3303407, -3330047, -3356687, -3383328, 
};

const int32 Ytab[256] = {
	-610270, -572128, -533986, -495844, -457702, -419560, -381419, -343277, -305135, -266993, 
	-228851, -190709, -152567, -114425, -76283, -38141, 0, 38142, 76284, 114426, 
	152568, 190710, 228852, 266994, 305136, 343278, 381420, 419561, 457703, 495845, 
	533987, 572129, 610271, 648413, 686555, 724697, 762839, 800981, 839123, 877265, 
	915407, 953549, 991691, 1029833, 1067975, 1106117, 1144259, 1182401, 1220542, 1258684, 
	1296826, 1334968, 1373110, 1411252, 1449394, 1487536, 1525678, 1563820, 1601962, 1640104, 
	1678246, 1716388, 1754530, 1792672, 1830814, 1868956, 1907098, 1945240, 1983382, 2021523, 
	2059665, 2097807, 2135949, 2174091, 2212233, 2250375, 2288517, 2326659, 2364801, 2402943, 
	2441085, 2479227, 2517369, 2555511, 2593653, 2631795, 2669937, 2708079, 2746221, 2784362, 
	2822504, 2860646, 2898788, 2936930, 2975072, 3013214, 3051356, 3089498, 3127640, 3165782, 
	3203924, 3242066, 3280208, 3318350, 3356492, 3394634, 3432776, 3470918, 3509060, 3547202, 
	3585343, 3623485, 3661627, 3699769, 3737911, 3776053, 3814195, 3852337, 3890479, 3928621, 
	3966763, 4004905, 4043047, 4081189, 4119331, 4157473, 4195615, 4233757, 4271899, 4310041, 
	4348183, 4386324, 4424466, 4462608, 4500750, 4538892, 4577034, 4615176, 4653318, 4691460, 
	4729602, 4767744, 4805886, 4844028, 4882170, 4920312, 4958454, 4996596, 5034738, 5072880, 
	5111022, 5149164, 5187305, 5225447, 5263589, 5301731, 5339873, 5378015, 5416157, 5454299, 
	5492441, 5530583, 5568725, 5606867, 5645009, 5683151, 5721293, 5759435, 5797577, 5835719, 
	5873861, 5912003, 5950145, 5988286, 6026428, 6064570, 6102712, 6140854, 6178996, 6217138, 
	6255280, 6293422, 6331564, 6369706, 6407848, 6445990, 6484132, 6522274, 6560416, 6598558, 
	6636700, 6674842, 6712984, 6751126, 6789267, 6827409, 6865551, 6903693, 6941835, 6979977, 
	7018119, 7056261, 7094403, 7132545, 7170687, 7208829, 7246971, 7285113, 7323255, 7361397, 
	7399539, 7437681, 7475823, 7513965, 7552106, 7590248, 7628390, 7666532, 7704674, 7742816, 
	7780958, 7819100, 7857242, 7895384, 7933526, 7971668, 8009810, 8047952, 8086094, 8124236, 
	8162378, 8200520, 8238662, 8276804, 8314946, 8353087, 8391229, 8429371, 8467513, 8505655, 
	8543797, 8581939, 8620081, 8658223, 8696365, 8734507, 8772649, 8810791, 8848933, 8887075, 
	8925217, 8963359, 9001501, 9039643, 9077785, 9115927, 
};

const uint32 Rsat[704] = {
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 
	0x000000, 0x010000, 0x020000, 0x030000, 0x040000, 0x050000, 0x060000, 0x070000, 0x080000, 0x090000, 
	0x0a0000, 0x0b0000, 0x0c0000, 0x0d0000, 0x0e0000, 0x0f0000, 0x100000, 0x110000, 0x120000, 0x130000, 
	0x140000, 0x150000, 0x160000, 0x170000, 0x180000, 0x190000, 0x1a0000, 0x1b0000, 0x1c0000, 0x1d0000, 
	0x1e0000, 0x1f0000, 0x200000, 0x210000, 0x220000, 0x230000, 0x240000, 0x250000, 0x260000, 0x270000, 
	0x280000, 0x290000, 0x2a0000, 0x2b0000, 0x2c0000, 0x2d0000, 0x2e0000, 0x2f0000, 0x300000, 0x310000, 
	0x320000, 0x330000, 0x340000, 0x350000, 0x360000, 0x370000, 0x380000, 0x390000, 0x3a0000, 0x3b0000, 
	0x3c0000, 0x3d0000, 0x3e0000, 0x3f0000, 0x400000, 0x410000, 0x420000, 0x430000, 0x440000, 0x450000, 
	0x460000, 0x470000, 0x480000, 0x490000, 0x4a0000, 0x4b0000, 0x4c0000, 0x4d0000, 0x4e0000, 0x4f0000, 
	0x500000, 0x510000, 0x520000, 0x530000, 0x540000, 0x550000, 0x560000, 0x570000, 0x580000, 0x590000, 
	0x5a0000, 0x5b0000, 0x5c0000, 0x5d0000, 0x5e0000, 0x5f0000, 0x600000, 0x610000, 0x620000, 0x630000, 
	0x640000, 0x650000, 0x660000, 0x670000, 0x680000, 0x690000, 0x6a0000, 0x6b0000, 0x6c0000, 0x6d0000, 
	0x6e0000, 0x6f0000, 0x700000, 0x710000, 0x720000, 0x730000, 0x740000, 0x750000, 0x760000, 0x770000, 
	0x780000, 0x790000, 0x7a0000, 0x7b0000, 0x7c0000, 0x7d0000, 0x7e0000, 0x7f0000, 0x800000, 0x810000, 
	0x820000, 0x830000, 0x840000, 0x850000, 0x860000, 0x870000, 0x880000, 0x890000, 0x8a0000, 0x8b0000, 
	0x8c0000, 0x8d0000, 0x8e0000, 0x8f0000, 0x900000, 0x910000, 0x920000, 0x930000, 0x940000, 0x950000, 
	0x960000, 0x970000, 0x980000, 0x990000, 0x9a0000, 0x9b0000, 0x9c0000, 0x9d0000, 0x9e0000, 0x9f0000, 
	0xa00000, 0xa10000, 0xa20000, 0xa30000, 0xa40000, 0xa50000, 0xa60000, 0xa70000, 0xa80000, 0xa90000, 
	0xaa0000, 0xab0000, 0xac0000, 0xad0000, 0xae0000, 0xaf0000, 0xb00000, 0xb10000, 0xb20000, 0xb30000, 
	0xb40000, 0xb50000, 0xb60000, 0xb70000, 0xb80000, 0xb90000, 0xba0000, 0xbb0000, 0xbc0000, 0xbd0000, 
	0xbe0000, 0xbf0000, 0xc00000, 0xc10000, 0xc20000, 0xc30000, 0xc40000, 0xc50000, 0xc60000, 0xc70000, 
	0xc80000, 0xc90000, 0xca0000, 0xcb0000, 0xcc0000, 0xcd0000, 0xce0000, 0xcf0000, 0xd00000, 0xd10000, 
	0xd20000, 0xd30000, 0xd40000, 0xd50000, 0xd60000, 0xd70000, 0xd80000, 0xd90000, 0xda0000, 0xdb0000, 
	0xdc0000, 0xdd0000, 0xde0000, 0xdf0000, 0xe00000, 0xe10000, 0xe20000, 0xe30000, 0xe40000, 0xe50000, 
	0xe60000, 0xe70000, 0xe80000, 0xe90000, 0xea0000, 0xeb0000, 0xec0000, 0xed0000, 0xee0000, 0xef0000, 
	0xf00000, 0xf10000, 0xf20000, 0xf30000, 0xf40000, 0xf50000, 0xf60000, 0xf70000, 0xf80000, 0xf90000, 
	0xfa0000, 0xfb0000, 0xfc0000, 0xfd0000, 0xfe0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
	0xff0000, 0xff0000, 0xff0000, 0xff0000, 0xff0000, 
};
const uint32 *Rtab = &Rsat[223];

const uint32 Gsat[605] = {
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 
	0x000000, 0x000100, 0x000200, 0x000300, 0x000400, 0x000500, 0x000600, 0x000700, 0x000800, 0x000900, 
	0x000a00, 0x000b00, 0x000c00, 0x000d00, 0x000e00, 0x000f00, 0x001000, 0x001100, 0x001200, 0x001300, 
	0x001400, 0x001500, 0x001600, 0x001700, 0x001800, 0x001900, 0x001a00, 0x001b00, 0x001c00, 0x001d00, 
	0x001e00, 0x001f00, 0x002000, 0x002100, 0x002200, 0x002300, 0x002400, 0x002500, 0x002600, 0x002700, 
	0x002800, 0x002900, 0x002a00, 0x002b00, 0x002c00, 0x002d00, 0x002e00, 0x002f00, 0x003000, 0x003100, 
	0x003200, 0x003300, 0x003400, 0x003500, 0x003600, 0x003700, 0x003800, 0x003900, 0x003a00, 0x003b00, 
	0x003c00, 0x003d00, 0x003e00, 0x003f00, 0x004000, 0x004100, 0x004200, 0x004300, 0x004400, 0x004500, 
	0x004600, 0x004700, 0x004800, 0x004900, 0x004a00, 0x004b00, 0x004c00, 0x004d00, 0x004e00, 0x004f00, 
	0x005000, 0x005100, 0x005200, 0x005300, 0x005400, 0x005500, 0x005600, 0x005700, 0x005800, 0x005900, 
	0x005a00, 0x005b00, 0x005c00, 0x005d00, 0x005e00, 0x005f00, 0x006000, 0x006100, 0x006200, 0x006300, 
	0x006400, 0x006500, 0x006600, 0x006700, 0x006800, 0x006900, 0x006a00, 0x006b00, 0x006c00, 0x006d00, 
	0x006e00, 0x006f00, 0x007000, 0x007100, 0x007200, 0x007300, 0x007400, 0x007500, 0x007600, 0x007700, 
	0x007800, 0x007900, 0x007a00, 0x007b00, 0x007c00, 0x007d00, 0x007e00, 0x007f00, 0x008000, 0x008100, 
	0x008200, 0x008300, 0x008400, 0x008500, 0x008600, 0x008700, 0x008800, 0x008900, 0x008a00, 0x008b00, 
	0x008c00, 0x008d00, 0x008e00, 0x008f00, 0x009000, 0x009100, 0x009200, 0x009300, 0x009400, 0x009500, 
	0x009600, 0x009700, 0x009800, 0x009900, 0x009a00, 0x009b00, 0x009c00, 0x009d00, 0x009e00, 0x009f00, 
	0x00a000, 0x00a100, 0x00a200, 0x00a300, 0x00a400, 0x00a500, 0x00a600, 0x00a700, 0x00a800, 0x00a900, 
	0x00aa00, 0x00ab00, 0x00ac00, 0x00ad00, 0x00ae00, 0x00af00, 0x00b000, 0x00b100, 0x00b200, 0x00b300, 
	0x00b400, 0x00b500, 0x00b600, 0x00b700, 0x00b800, 0x00b900, 0x00ba00, 0x00bb00, 0x00bc00, 0x00bd00, 
	0x00be00, 0x00bf00, 0x00c000, 0x00c100, 0x00c200, 0x00c300, 0x00c400, 0x00c500, 0x00c600, 0x00c700, 
	0x00c800, 0x00c900, 0x00ca00, 0x00cb00, 0x00cc00, 0x00cd00, 0x00ce00, 0x00cf00, 0x00d000, 0x00d100, 
	0x00d200, 0x00d300, 0x00d400, 0x00d500, 0x00d600, 0x00d700, 0x00d800, 0x00d900, 0x00da00, 0x00db00, 
	0x00dc00, 0x00dd00, 0x00de00, 0x00df00, 0x00e000, 0x00e100, 0x00e200, 0x00e300, 0x00e400, 0x00e500, 
	0x00e600, 0x00e700, 0x00e800, 0x00e900, 0x00ea00, 0x00eb00, 0x00ec00, 0x00ed00, 0x00ee00, 0x00ef00, 
	0x00f000, 0x00f100, 0x00f200, 0x00f300, 0x00f400, 0x00f500, 0x00f600, 0x00f700, 0x00f800, 0x00f900, 
	0x00fa00, 0x00fb00, 0x00fc00, 0x00fd00, 0x00fe00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
	0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 0x00ff00, 
};
const uint32 *Gtab = &Gsat[172];

const uint32 Bsat[812] = {
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 
	0x000000, 0x000001, 0x000002, 0x000003, 0x000004, 0x000005, 0x000006, 0x000007, 0x000008, 0x000009, 
	0x00000a, 0x00000b, 0x00000c, 0x00000d, 0x00000e, 0x00000f, 0x000010, 0x000011, 0x000012, 0x000013, 
	0x000014, 0x000015, 0x000016, 0x000017, 0x000018, 0x000019, 0x00001a, 0x00001b, 0x00001c, 0x00001d, 
	0x00001e, 0x00001f, 0x000020, 0x000021, 0x000022, 0x000023, 0x000024, 0x000025, 0x000026, 0x000027, 
	0x000028, 0x000029, 0x00002a, 0x00002b, 0x00002c, 0x00002d, 0x00002e, 0x00002f, 0x000030, 0x000031, 
	0x000032, 0x000033, 0x000034, 0x000035, 0x000036, 0x000037, 0x000038, 0x000039, 0x00003a, 0x00003b, 
	0x00003c, 0x00003d, 0x00003e, 0x00003f, 0x000040, 0x000041, 0x000042, 0x000043, 0x000044, 0x000045, 
	0x000046, 0x000047, 0x000048, 0x000049, 0x00004a, 0x00004b, 0x00004c, 0x00004d, 0x00004e, 0x00004f, 
	0x000050, 0x000051, 0x000052, 0x000053, 0x000054, 0x000055, 0x000056, 0x000057, 0x000058, 0x000059, 
	0x00005a, 0x00005b, 0x00005c, 0x00005d, 0x00005e, 0x00005f, 0x000060, 0x000061, 0x000062, 0x000063, 
	0x000064, 0x000065, 0x000066, 0x000067, 0x000068, 0x000069, 0x00006a, 0x00006b, 0x00006c, 0x00006d, 
	0x00006e, 0x00006f, 0x000070, 0x000071, 0x000072, 0x000073, 0x000074, 0x000075, 0x000076, 0x000077, 
	0x000078, 0x000079, 0x00007a, 0x00007b, 0x00007c, 0x00007d, 0x00007e, 0x00007f, 0x000080, 0x000081, 
	0x000082, 0x000083, 0x000084, 0x000085, 0x000086, 0x000087, 0x000088, 0x000089, 0x00008a, 0x00008b, 
	0x00008c, 0x00008d, 0x00008e, 0x00008f, 0x000090, 0x000091, 0x000092, 0x000093, 0x000094, 0x000095, 
	0x000096, 0x000097, 0x000098, 0x000099, 0x00009a, 0x00009b, 0x00009c, 0x00009d, 0x00009e, 0x00009f, 
	0x0000a0, 0x0000a1, 0x0000a2, 0x0000a3, 0x0000a4, 0x0000a5, 0x0000a6, 0x0000a7, 0x0000a8, 0x0000a9, 
	0x0000aa, 0x0000ab, 0x0000ac, 0x0000ad, 0x0000ae, 0x0000af, 0x0000b0, 0x0000b1, 0x0000b2, 0x0000b3, 
	0x0000b4, 0x0000b5, 0x0000b6, 0x0000b7, 0x0000b8, 0x0000b9, 0x0000ba, 0x0000bb, 0x0000bc, 0x0000bd, 
	0x0000be, 0x0000bf, 0x0000c0, 0x0000c1, 0x0000c2, 0x0000c3, 0x0000c4, 0x0000c5, 0x0000c6, 0x0000c7, 
	0x0000c8, 0x0000c9, 0x0000ca, 0x0000cb, 0x0000cc, 0x0000cd, 0x0000ce, 0x0000cf, 0x0000d0, 0x0000d1, 
	0x0000d2, 0x0000d3, 0x0000d4, 0x0000d5, 0x0000d6, 0x0000d7, 0x0000d8, 0x0000d9, 0x0000da, 0x0000db, 
	0x0000dc, 0x0000dd, 0x0000de, 0x0000df, 0x0000e0, 0x0000e1, 0x0000e2, 0x0000e3, 0x0000e4, 0x0000e5, 
	0x0000e6, 0x0000e7, 0x0000e8, 0x0000e9, 0x0000ea, 0x0000eb, 0x0000ec, 0x0000ed, 0x0000ee, 0x0000ef, 
	0x0000f0, 0x0000f1, 0x0000f2, 0x0000f3, 0x0000f4, 0x0000f5, 0x0000f6, 0x0000f7, 0x0000f8, 0x0000f9, 
	0x0000fa, 0x0000fb, 0x0000fc, 0x0000fd, 0x0000fe, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
	0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 0x0000ff, 
};
const uint32 *Btab = &Bsat[277];
