/* 
 * Copyright 2002, Marcus Overhagen. All rights reserved.
 * Distributed under the terms of the MIT License.
 */

#ifndef _NOTIFICATIONS_H
#define _NOTIFICATIONS_H

namespace BPrivate {
namespace media {
namespace notifications {

	/*
	 *	This are the possible notifications that can be watched.
	 *	The notifications marked with "N" are only send when the
	 *	media_node specific BMediaRoster::StartWatching() is used
	 *	and the notification belongs to the watched node.
	 *	
	 *	In addition, anyone watching a specific node will also receive
	 *	error notifications generated by BMediaNode::ReportError()
 	 *
	 *	B_MEDIA_WILDCARD 					used to match any notification in Start/StopWatching
	 *	B_MEDIA_NODE_CREATED 				"media_node_id" (multiple items)
	 *	B_MEDIA_NODE_DELETED				"media_node_id" (multiple items)
	 *	B_MEDIA_CONNECTION_MADE				"output", "input", "format"
	 *	B_MEDIA_CONNECTION_BROKEN			"source", "destination"
	 *	B_MEDIA_BUFFER_CREATED				"clone_info" -- handled by BMediaRoster
	 *	B_MEDIA_BUFFER_DELETED				"media_buffer_id" -- handled by BMediaRoster
	 *	B_MEDIA_TRANSPORT_STATE				"state", "location", "realtime"
	 *	B_MEDIA_PARAMETER_CHANGED		N 	"node", "parameter"
	 *	B_MEDIA_FORMAT_CHANGED			N 	"source", "destination", "format"
	 *	B_MEDIA_WEB_CHANGED				N 	"node"
	 *	B_MEDIA_DEFAULT_CHANGED				"default", "node" -- handled by BMediaRoster
	 *	B_MEDIA_NEW_PARAMETER_VALUE		N 	"node", "parameter", "when", "value"
	 *	B_MEDIA_NODE_STOPPED			N 	"node", "when"
	 *	B_MEDIA_FLAVORS_CHANGED				"be:addon_id", "be:new_count", "be:gone_count"
	 */
	 
	// used for BMediaRoster::StartWatching() parameter validation
	bool IsValidNotificationRequest(bool node_specific, int32 notification);

	// called by BMediaRoster::StartWatching()
	status_t Register(const BMessenger &notifyHandler, const media_node &node, int32 notification);
	
	// called by BMediaRoster::StopWatching()
	status_t Unregister(const BMessenger &notifyHandler, const media_node &node, int32 notification);

	// called by BMediaNode::ReportError()
	status_t ReportError(const media_node &node, BMediaNode::node_error what, const BMessage * info);

	void NodesCreated(const media_node_id *ids, int32 count);
	void NodesDeleted(const media_node_id *ids, int32 count);
	void ConnectionMade(const media_input &input, const media_output &output, const media_format &format);
	void ConnectionBroken(const media_source &source, const media_destination &destination);
	void BuffersCreated(area_info *areas, int32 count);
	void BuffersDeleted(const media_buffer_id *ids, int32 count);
	
	// called by BMediaNode::NodeStopped()
	void NodeStopped(const media_node &node, bigtime_t when);

	// called by BControllable::BroadcastChangedParameter()
	status_t ParameterChanged(const media_node &node, int32 parameterid);
	
	// called by BControllable::SetParameterWeb()
	void WebChanged(const media_node &node);
	
	// called by BControllable::BroadcastNewParameterValue()
	status_t NewParameterValue(const media_node &node, int32 parameterid, bigtime_t when, const void *param, size_t paramsize);
	
	// called by the media_addon_server AFTER a flavor change has been 
	// handled. NOT CALLED by BMediaAddOn::NotifyFlavorChange()
	void FlavorsChanged(media_addon_id addonid, int32 newcount, int32 gonecount);

	void FormatChanged(const media_source &source, const media_destination &destination, const media_format &format);

	// XXX missing: B_MEDIA_TRANSPORT_STATE		/* "state", "location", "realtime" */
	// XXX missing: B_MEDIA_DEFAULT_CHANGED		/* "default", "node"  */

}; // namespace notifications
}; // namespace media
}; // namespace BPrivate

#define NOTIFICATION_PARAM_WHAT "be:media:internal:what"
#define NOTIFICATION_PARAM_TEAM "be:media:internal:team"
#define NOTIFICATION_PARAM_MESSENGER "be:media:internal:messenger"

#endif // _NOTIFICATIONS_H
