#!/usr/bin/env python3

# Copyright 2021 Collabora, Ltd.
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice (including the
# next paragraph) shall be included in all copies or substantial
# portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

# Helper script that reads drm_fourcc.h and writes a static table with the
# simpler format token modifiers

import sys
import re

filename = sys.argv[1]
towrite = sys.argv[2]

fm_re = {
    'intel': r'^#define I915_FORMAT_MOD_(\w+)',
    'others': r'^#define DRM_FORMAT_MOD_((?:ARM|SAMSUNG|QCOM|VIVANTE|NVIDIA|BROADCOM|ALLWINNER)\w+)\s',
    'vendors': r'^#define DRM_FORMAT_MOD_VENDOR_(\w+)'
}

def print_fm_intel(f, f_mod):
    f.write('    {{ DRM_MODIFIER_INTEL({}, {}) }},\n'.format(f_mod, f_mod))

# generic write func
def print_fm(f, vendor, mod, f_name):
    f.write('    {{ DRM_MODIFIER({}, {}, {}) }},\n'.format(vendor, mod, f_name))

with open(filename, "r") as f:
    data = f.read()
    for k, v in fm_re.items():
        fm_re[k] = re.findall(v, data, flags=re.M)

with open(towrite, "w") as f:
    f.write('''\
/* AUTOMATICALLY GENERATED by gen_table_fourcc.py. You should modify
   that script instead of adding here entries manually! */
static const struct drmFormatModifierInfo drm_format_modifier_table[] = {
''')
    f.write('    { DRM_MODIFIER_INVALID(NONE, INVALID_MODIFIER) },\n')
    f.write('    { DRM_MODIFIER_LINEAR(NONE, LINEAR) },\n')

    for entry in fm_re['intel']:
        print_fm_intel(f, entry)

    for entry in fm_re['others']:
        (vendor, mod) = entry.split('_', 1)
        if vendor == 'ARM' and (mod == 'TYPE_AFBC' or mod == 'TYPE_MISC'):
            continue
        print_fm(f, vendor, mod, mod)

    f.write('''\
};
''')

    f.write('''\
static const struct drmFormatModifierVendorInfo drm_format_modifier_vendor_table[] = {
''')

    for entry in fm_re['vendors']:
        f.write("    {{ DRM_FORMAT_MOD_VENDOR_{}, \"{}\" }},\n".format(entry, entry))

    f.write('''\
};
''')
