﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/athena/AthenaRequest.h>
#include <aws/athena/Athena_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Athena {
namespace Model {

/**
 */
class UpdatePreparedStatementRequest : public AthenaRequest {
 public:
  AWS_ATHENA_API UpdatePreparedStatementRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdatePreparedStatement"; }

  AWS_ATHENA_API Aws::String SerializePayload() const override;

  AWS_ATHENA_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the prepared statement.</p>
   */
  inline const Aws::String& GetStatementName() const { return m_statementName; }
  inline bool StatementNameHasBeenSet() const { return m_statementNameHasBeenSet; }
  template <typename StatementNameT = Aws::String>
  void SetStatementName(StatementNameT&& value) {
    m_statementNameHasBeenSet = true;
    m_statementName = std::forward<StatementNameT>(value);
  }
  template <typename StatementNameT = Aws::String>
  UpdatePreparedStatementRequest& WithStatementName(StatementNameT&& value) {
    SetStatementName(std::forward<StatementNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The workgroup for the prepared statement.</p>
   */
  inline const Aws::String& GetWorkGroup() const { return m_workGroup; }
  inline bool WorkGroupHasBeenSet() const { return m_workGroupHasBeenSet; }
  template <typename WorkGroupT = Aws::String>
  void SetWorkGroup(WorkGroupT&& value) {
    m_workGroupHasBeenSet = true;
    m_workGroup = std::forward<WorkGroupT>(value);
  }
  template <typename WorkGroupT = Aws::String>
  UpdatePreparedStatementRequest& WithWorkGroup(WorkGroupT&& value) {
    SetWorkGroup(std::forward<WorkGroupT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The query string for the prepared statement.</p>
   */
  inline const Aws::String& GetQueryStatement() const { return m_queryStatement; }
  inline bool QueryStatementHasBeenSet() const { return m_queryStatementHasBeenSet; }
  template <typename QueryStatementT = Aws::String>
  void SetQueryStatement(QueryStatementT&& value) {
    m_queryStatementHasBeenSet = true;
    m_queryStatement = std::forward<QueryStatementT>(value);
  }
  template <typename QueryStatementT = Aws::String>
  UpdatePreparedStatementRequest& WithQueryStatement(QueryStatementT&& value) {
    SetQueryStatement(std::forward<QueryStatementT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the prepared statement.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdatePreparedStatementRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_statementName;
  bool m_statementNameHasBeenSet = false;

  Aws::String m_workGroup;
  bool m_workGroupHasBeenSet = false;

  Aws::String m_queryStatement;
  bool m_queryStatementHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;
};

}  // namespace Model
}  // namespace Athena
}  // namespace Aws
