// Copyright 2024 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package webrisk provides access to the Web Risk API.
//
// For product documentation, see: https://cloud.google.com/web-risk/
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/webrisk/v1"
//	...
//	ctx := context.Background()
//	webriskService, err := webrisk.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	webriskService, err := webrisk.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	webriskService, err := webrisk.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package webrisk // import "google.golang.org/api/webrisk/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "webrisk:v1"
const apiName = "webrisk"
const apiVersion = "v1"
const basePath = "https://webrisk.googleapis.com/"
const basePathTemplate = "https://webrisk.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://webrisk.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// See, edit, configure, and delete your Google Cloud data and see the email
	// address for your Google Account.
	CloudPlatformScope = "https://www.googleapis.com/auth/cloud-platform"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/cloud-platform",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.Hashes = NewHashesService(s)
	s.Projects = NewProjectsService(s)
	s.ThreatLists = NewThreatListsService(s)
	s.Uris = NewUrisService(s)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.Background(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Hashes *HashesService

	Projects *ProjectsService

	ThreatLists *ThreatListsService

	Uris *UrisService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewHashesService(s *Service) *HashesService {
	rs := &HashesService{s: s}
	return rs
}

type HashesService struct {
	s *Service
}

func NewProjectsService(s *Service) *ProjectsService {
	rs := &ProjectsService{s: s}
	rs.Operations = NewProjectsOperationsService(s)
	rs.Submissions = NewProjectsSubmissionsService(s)
	return rs
}

type ProjectsService struct {
	s *Service

	Operations *ProjectsOperationsService

	Submissions *ProjectsSubmissionsService
}

func NewProjectsOperationsService(s *Service) *ProjectsOperationsService {
	rs := &ProjectsOperationsService{s: s}
	return rs
}

type ProjectsOperationsService struct {
	s *Service
}

func NewProjectsSubmissionsService(s *Service) *ProjectsSubmissionsService {
	rs := &ProjectsSubmissionsService{s: s}
	return rs
}

type ProjectsSubmissionsService struct {
	s *Service
}

func NewThreatListsService(s *Service) *ThreatListsService {
	rs := &ThreatListsService{s: s}
	return rs
}

type ThreatListsService struct {
	s *Service
}

func NewUrisService(s *Service) *UrisService {
	rs := &UrisService{s: s}
	return rs
}

type UrisService struct {
	s *Service
}

type GoogleCloudWebriskV1ComputeThreatListDiffResponse struct {
	// Additions: A set of entries to add to a local threat type's list.
	Additions *GoogleCloudWebriskV1ThreatEntryAdditions `json:"additions,omitempty"`
	// Checksum: The expected SHA256 hash of the client state; that is, of the
	// sorted list of all hashes present in the database after applying the
	// provided diff. If the client state doesn't match the expected state, the
	// client must discard this diff and retry later.
	Checksum *GoogleCloudWebriskV1ComputeThreatListDiffResponseChecksum `json:"checksum,omitempty"`
	// NewVersionToken: The new opaque client version token. This should be
	// retained by the client and passed into the next call of
	// ComputeThreatListDiff as 'version_token'. A separate version token should be
	// stored and used for each threatList.
	NewVersionToken string `json:"newVersionToken,omitempty"`
	// RecommendedNextDiff: The soonest the client should wait before issuing any
	// diff request. Querying sooner is unlikely to produce a meaningful diff.
	// Waiting longer is acceptable considering the use case. If this field is not
	// set clients may update as soon as they want.
	RecommendedNextDiff string `json:"recommendedNextDiff,omitempty"`
	// Removals: A set of entries to remove from a local threat type's list. This
	// field may be empty.
	Removals *GoogleCloudWebriskV1ThreatEntryRemovals `json:"removals,omitempty"`
	// ResponseType: The type of response. This may indicate that an action must be
	// taken by the client when the response is received.
	//
	// Possible values:
	//   "RESPONSE_TYPE_UNSPECIFIED" - Unknown.
	//   "DIFF" - Partial updates are applied to the client's existing local
	// database.
	//   "RESET" - Full updates resets the client's entire local database. This
	// means that either the client had no state, was seriously out-of-date, or the
	// client is believed to be corrupt.
	ResponseType string `json:"responseType,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Additions") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Additions") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleCloudWebriskV1ComputeThreatListDiffResponse) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleCloudWebriskV1ComputeThreatListDiffResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleCloudWebriskV1ComputeThreatListDiffResponseChecksum: The expected
// state of a client's local database.
type GoogleCloudWebriskV1ComputeThreatListDiffResponseChecksum struct {
	// Sha256: The SHA256 hash of the client state; that is, of the sorted list of
	// all hashes present in the database.
	Sha256 string `json:"sha256,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Sha256") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Sha256") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleCloudWebriskV1ComputeThreatListDiffResponseChecksum) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleCloudWebriskV1ComputeThreatListDiffResponseChecksum
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleCloudWebriskV1RawHashes: The uncompressed threat entries in hash
// format. Hashes can be anywhere from 4 to 32 bytes in size. A large majority
// are 4 bytes, but some hashes are lengthened if they collide with the hash of
// a popular URI. Used for sending ThreatEntryAdditons to clients that do not
// support compression, or when sending non-4-byte hashes to clients that do
// support compression.
type GoogleCloudWebriskV1RawHashes struct {
	// PrefixSize: The number of bytes for each prefix encoded below. This field
	// can be anywhere from 4 (shortest prefix) to 32 (full SHA256 hash). In
	// practice this is almost always 4, except in exceptional circumstances.
	PrefixSize int64 `json:"prefixSize,omitempty"`
	// RawHashes: The hashes, in binary format, concatenated into one long string.
	// Hashes are sorted in lexicographic order. For JSON API users, hashes are
	// base64-encoded.
	RawHashes string `json:"rawHashes,omitempty"`
	// ForceSendFields is a list of field names (e.g. "PrefixSize") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "PrefixSize") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleCloudWebriskV1RawHashes) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleCloudWebriskV1RawHashes
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleCloudWebriskV1RawIndices: A set of raw indices to remove from a local
// list.
type GoogleCloudWebriskV1RawIndices struct {
	// Indices: The indices to remove from a lexicographically-sorted local list.
	Indices []int64 `json:"indices,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Indices") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Indices") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleCloudWebriskV1RawIndices) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleCloudWebriskV1RawIndices
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleCloudWebriskV1RiceDeltaEncoding: The Rice-Golomb encoded data. Used
// for sending compressed 4-byte hashes or compressed removal indices.
type GoogleCloudWebriskV1RiceDeltaEncoding struct {
	// EncodedData: The encoded deltas that are encoded using the Golomb-Rice
	// coder.
	EncodedData string `json:"encodedData,omitempty"`
	// EntryCount: The number of entries that are delta encoded in the encoded
	// data. If only a single integer was encoded, this will be zero and the single
	// value will be stored in `first_value`.
	EntryCount int64 `json:"entryCount,omitempty"`
	// FirstValue: The offset of the first entry in the encoded data, or, if only a
	// single integer was encoded, that single integer's value. If the field is
	// empty or missing, assume zero.
	FirstValue int64 `json:"firstValue,omitempty,string"`
	// RiceParameter: The Golomb-Rice parameter, which is a number between 2 and
	// 28. This field is missing (that is, zero) if `num_entries` is zero.
	RiceParameter int64 `json:"riceParameter,omitempty"`
	// ForceSendFields is a list of field names (e.g. "EncodedData") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "EncodedData") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleCloudWebriskV1RiceDeltaEncoding) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleCloudWebriskV1RiceDeltaEncoding
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type GoogleCloudWebriskV1SearchHashesResponse struct {
	// NegativeExpireTime: For requested entities that did not match the threat
	// list, how long to cache the response until.
	NegativeExpireTime string `json:"negativeExpireTime,omitempty"`
	// Threats: The full hashes that matched the requested prefixes. The hash will
	// be populated in the key.
	Threats []*GoogleCloudWebriskV1SearchHashesResponseThreatHash `json:"threats,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NegativeExpireTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NegativeExpireTime") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleCloudWebriskV1SearchHashesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleCloudWebriskV1SearchHashesResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleCloudWebriskV1SearchHashesResponseThreatHash: Contains threat
// information on a matching hash.
type GoogleCloudWebriskV1SearchHashesResponseThreatHash struct {
	// ExpireTime: The cache lifetime for the returned match. Clients must not
	// cache this response past this timestamp to avoid false positives.
	ExpireTime string `json:"expireTime,omitempty"`
	// Hash: A 32 byte SHA256 hash. This field is in binary format. For JSON
	// requests, hashes are base64-encoded.
	Hash string `json:"hash,omitempty"`
	// ThreatTypes: The ThreatList this threat belongs to. This must contain at
	// least one entry.
	//
	// Possible values:
	//   "THREAT_TYPE_UNSPECIFIED" - No entries should match this threat type. This
	// threat type is unused.
	//   "MALWARE" - Malware targeting any platform.
	//   "SOCIAL_ENGINEERING" - Social engineering targeting any platform.
	//   "UNWANTED_SOFTWARE" - Unwanted software targeting any platform.
	//   "SOCIAL_ENGINEERING_EXTENDED_COVERAGE" - A list of extended coverage
	// social engineering URIs targeting any platform.
	ThreatTypes []string `json:"threatTypes,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ExpireTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ExpireTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleCloudWebriskV1SearchHashesResponseThreatHash) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleCloudWebriskV1SearchHashesResponseThreatHash
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type GoogleCloudWebriskV1SearchUrisResponse struct {
	// Threat: The threat list matches. This might be empty if the URI is on no
	// list.
	Threat *GoogleCloudWebriskV1SearchUrisResponseThreatUri `json:"threat,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Threat") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Threat") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleCloudWebriskV1SearchUrisResponse) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleCloudWebriskV1SearchUrisResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleCloudWebriskV1SearchUrisResponseThreatUri: Contains threat information
// on a matching uri.
type GoogleCloudWebriskV1SearchUrisResponseThreatUri struct {
	// ExpireTime: The cache lifetime for the returned match. Clients must not
	// cache this response past this timestamp to avoid false positives.
	ExpireTime string `json:"expireTime,omitempty"`
	// ThreatTypes: The ThreatList this threat belongs to.
	//
	// Possible values:
	//   "THREAT_TYPE_UNSPECIFIED" - No entries should match this threat type. This
	// threat type is unused.
	//   "MALWARE" - Malware targeting any platform.
	//   "SOCIAL_ENGINEERING" - Social engineering targeting any platform.
	//   "UNWANTED_SOFTWARE" - Unwanted software targeting any platform.
	//   "SOCIAL_ENGINEERING_EXTENDED_COVERAGE" - A list of extended coverage
	// social engineering URIs targeting any platform.
	ThreatTypes []string `json:"threatTypes,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ExpireTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ExpireTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleCloudWebriskV1SearchUrisResponseThreatUri) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleCloudWebriskV1SearchUrisResponseThreatUri
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleCloudWebriskV1Submission: Wraps a URI that might be displaying
// malicious content.
type GoogleCloudWebriskV1Submission struct {
	// Uri: Required. The URI that is being reported for malicious content to be
	// analyzed.
	Uri string `json:"uri,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Uri") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Uri") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleCloudWebriskV1Submission) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleCloudWebriskV1Submission
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleCloudWebriskV1ThreatEntryAdditions: Contains the set of entries to add
// to a local database. May contain a combination of compressed and raw data in
// a single response.
type GoogleCloudWebriskV1ThreatEntryAdditions struct {
	// RawHashes: The raw SHA256-formatted entries. Repeated to allow returning
	// sets of hashes with different prefix sizes.
	RawHashes []*GoogleCloudWebriskV1RawHashes `json:"rawHashes,omitempty"`
	// RiceHashes: The encoded 4-byte prefixes of SHA256-formatted entries, using a
	// Golomb-Rice encoding. The hashes are converted to uint32, sorted in
	// ascending order, then delta encoded and stored as encoded_data.
	RiceHashes *GoogleCloudWebriskV1RiceDeltaEncoding `json:"riceHashes,omitempty"`
	// ForceSendFields is a list of field names (e.g. "RawHashes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RawHashes") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleCloudWebriskV1ThreatEntryAdditions) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleCloudWebriskV1ThreatEntryAdditions
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleCloudWebriskV1ThreatEntryRemovals: Contains the set of entries to
// remove from a local database.
type GoogleCloudWebriskV1ThreatEntryRemovals struct {
	// RawIndices: The raw removal indices for a local list.
	RawIndices *GoogleCloudWebriskV1RawIndices `json:"rawIndices,omitempty"`
	// RiceIndices: The encoded local, lexicographically-sorted list indices, using
	// a Golomb-Rice encoding. Used for sending compressed removal indices. The
	// removal indices (uint32) are sorted in ascending order, then delta encoded
	// and stored as encoded_data.
	RiceIndices *GoogleCloudWebriskV1RiceDeltaEncoding `json:"riceIndices,omitempty"`
	// ForceSendFields is a list of field names (e.g. "RawIndices") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RawIndices") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleCloudWebriskV1ThreatEntryRemovals) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleCloudWebriskV1ThreatEntryRemovals
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleLongrunningCancelOperationRequest: The request message for
// Operations.CancelOperation.
type GoogleLongrunningCancelOperationRequest struct {
}

// GoogleLongrunningListOperationsResponse: The response message for
// Operations.ListOperations.
type GoogleLongrunningListOperationsResponse struct {
	// NextPageToken: The standard List next-page token.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Operations: A list of operations that matches the specified filter in the
	// request.
	Operations []*GoogleLongrunningOperation `json:"operations,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleLongrunningListOperationsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleLongrunningListOperationsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleLongrunningOperation: This resource represents a long-running
// operation that is the result of a network API call.
type GoogleLongrunningOperation struct {
	// Done: If the value is `false`, it means the operation is still in progress.
	// If `true`, the operation is completed, and either `error` or `response` is
	// available.
	Done bool `json:"done,omitempty"`
	// Error: The error result of the operation in case of failure or cancellation.
	Error *GoogleRpcStatus `json:"error,omitempty"`
	// Metadata: Contains a `SubmitUriMetadata` object.
	Metadata googleapi.RawMessage `json:"metadata,omitempty"`
	// Name: Matches the `/v1/{project-name}/operations/{operation-id}` pattern.
	Name string `json:"name,omitempty"`
	// Response: The normal, successful response of the operation. If the original
	// method returns no data on success, such as `Delete`, the response is
	// `google.protobuf.Empty`. If the original method is standard
	// `Get`/`Create`/`Update`, the response should be the resource. For other
	// methods, the response should have the type `XxxResponse`, where `Xxx` is the
	// original method name. For example, if the original method name is
	// `TakeSnapshot()`, the inferred response type is `TakeSnapshotResponse`.
	Response googleapi.RawMessage `json:"response,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Done") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Done") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleLongrunningOperation) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleLongrunningOperation
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleProtobufEmpty: A generic empty message that you can re-use to avoid
// defining duplicated empty messages in your APIs. A typical example is to use
// it as the request or the response type of an API method. For instance:
// service Foo { rpc Bar(google.protobuf.Empty) returns
// (google.protobuf.Empty); }
type GoogleProtobufEmpty struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// GoogleRpcStatus: The `Status` type defines a logical error model that is
// suitable for different programming environments, including REST APIs and RPC
// APIs. It is used by gRPC (https://github.com/grpc). Each `Status` message
// contains three pieces of data: error code, error message, and error details.
// You can find out more about this error model and how to work with it in the
// API Design Guide (https://cloud.google.com/apis/design/errors).
type GoogleRpcStatus struct {
	// Code: The status code, which should be an enum value of google.rpc.Code.
	Code int64 `json:"code,omitempty"`
	// Details: A list of messages that carry the error details. There is a common
	// set of message types for APIs to use.
	Details []googleapi.RawMessage `json:"details,omitempty"`
	// Message: A developer-facing error message, which should be in English. Any
	// user-facing error message should be localized and sent in the
	// google.rpc.Status.details field, or localized by the client.
	Message string `json:"message,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Code") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Code") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleRpcStatus) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleRpcStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type HashesSearchCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Search: Gets the full hashes that match the requested hash prefix. This is
// used after a hash prefix is looked up in a threatList and there is a match.
// The client side threatList only holds partial hashes so the client must
// query this method to determine if there is a full hash match of a threat.
func (r *HashesService) Search() *HashesSearchCall {
	c := &HashesSearchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// HashPrefix sets the optional parameter "hashPrefix": A hash prefix,
// consisting of the most significant 4-32 bytes of a SHA256 hash. For JSON
// requests, this field is base64-encoded. Note that if this parameter is
// provided by a URI, it must be encoded using the web safe base64 variant (RFC
// 4648).
func (c *HashesSearchCall) HashPrefix(hashPrefix string) *HashesSearchCall {
	c.urlParams_.Set("hashPrefix", hashPrefix)
	return c
}

// ThreatTypes sets the optional parameter "threatTypes": Required. The
// ThreatLists to search in. Multiple ThreatLists may be specified.
//
// Possible values:
//
//	"THREAT_TYPE_UNSPECIFIED" - No entries should match this threat type. This
//
// threat type is unused.
//
//	"MALWARE" - Malware targeting any platform.
//	"SOCIAL_ENGINEERING" - Social engineering targeting any platform.
//	"UNWANTED_SOFTWARE" - Unwanted software targeting any platform.
//	"SOCIAL_ENGINEERING_EXTENDED_COVERAGE" - A list of extended coverage
//
// social engineering URIs targeting any platform.
func (c *HashesSearchCall) ThreatTypes(threatTypes ...string) *HashesSearchCall {
	c.urlParams_.SetMulti("threatTypes", append([]string{}, threatTypes...))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *HashesSearchCall) Fields(s ...googleapi.Field) *HashesSearchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *HashesSearchCall) IfNoneMatch(entityTag string) *HashesSearchCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *HashesSearchCall) Context(ctx context.Context) *HashesSearchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *HashesSearchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *HashesSearchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/hashes:search")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webrisk.hashes.search", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webrisk.hashes.search" call.
// Any non-2xx status code is an error. Response headers are in either
// *GoogleCloudWebriskV1SearchHashesResponse.ServerResponse.Header or (if a
// response was returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *HashesSearchCall) Do(opts ...googleapi.CallOption) (*GoogleCloudWebriskV1SearchHashesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &GoogleCloudWebriskV1SearchHashesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webrisk.hashes.search", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsOperationsCancelCall struct {
	s                                       *Service
	name                                    string
	googlelongrunningcanceloperationrequest *GoogleLongrunningCancelOperationRequest
	urlParams_                              gensupport.URLParams
	ctx_                                    context.Context
	header_                                 http.Header
}

// Cancel: Starts asynchronous cancellation on a long-running operation. The
// server makes a best effort to cancel the operation, but success is not
// guaranteed. If the server doesn't support this method, it returns
// `google.rpc.Code.UNIMPLEMENTED`. Clients can use Operations.GetOperation or
// other methods to check whether the cancellation succeeded or whether the
// operation completed despite cancellation. On successful cancellation, the
// operation is not deleted; instead, it becomes an operation with an
// Operation.error value with a google.rpc.Status.code of `1`, corresponding to
// `Code.CANCELLED`.
//
// - name: The name of the operation resource to be cancelled.
func (r *ProjectsOperationsService) Cancel(name string, googlelongrunningcanceloperationrequest *GoogleLongrunningCancelOperationRequest) *ProjectsOperationsCancelCall {
	c := &ProjectsOperationsCancelCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.googlelongrunningcanceloperationrequest = googlelongrunningcanceloperationrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsOperationsCancelCall) Fields(s ...googleapi.Field) *ProjectsOperationsCancelCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsOperationsCancelCall) Context(ctx context.Context) *ProjectsOperationsCancelCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsOperationsCancelCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsOperationsCancelCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.googlelongrunningcanceloperationrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:cancel")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webrisk.projects.operations.cancel", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webrisk.projects.operations.cancel" call.
// Any non-2xx status code is an error. Response headers are in either
// *GoogleProtobufEmpty.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ProjectsOperationsCancelCall) Do(opts ...googleapi.CallOption) (*GoogleProtobufEmpty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &GoogleProtobufEmpty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webrisk.projects.operations.cancel", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsOperationsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a long-running operation. This method indicates that the
// client is no longer interested in the operation result. It does not cancel
// the operation. If the server doesn't support this method, it returns
// `google.rpc.Code.UNIMPLEMENTED`.
//
// - name: The name of the operation resource to be deleted.
func (r *ProjectsOperationsService) Delete(name string) *ProjectsOperationsDeleteCall {
	c := &ProjectsOperationsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsOperationsDeleteCall) Fields(s ...googleapi.Field) *ProjectsOperationsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsOperationsDeleteCall) Context(ctx context.Context) *ProjectsOperationsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsOperationsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsOperationsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webrisk.projects.operations.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webrisk.projects.operations.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *GoogleProtobufEmpty.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ProjectsOperationsDeleteCall) Do(opts ...googleapi.CallOption) (*GoogleProtobufEmpty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &GoogleProtobufEmpty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webrisk.projects.operations.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsOperationsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the latest state of a long-running operation. Clients can use this
// method to poll the operation result at intervals as recommended by the API
// service.
//
// - name: The name of the operation resource.
func (r *ProjectsOperationsService) Get(name string) *ProjectsOperationsGetCall {
	c := &ProjectsOperationsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsOperationsGetCall) Fields(s ...googleapi.Field) *ProjectsOperationsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsOperationsGetCall) IfNoneMatch(entityTag string) *ProjectsOperationsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsOperationsGetCall) Context(ctx context.Context) *ProjectsOperationsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsOperationsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsOperationsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webrisk.projects.operations.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webrisk.projects.operations.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *GoogleLongrunningOperation.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ProjectsOperationsGetCall) Do(opts ...googleapi.CallOption) (*GoogleLongrunningOperation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &GoogleLongrunningOperation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webrisk.projects.operations.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsOperationsListCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists operations that match the specified filter in the request. If
// the server doesn't support this method, it returns `UNIMPLEMENTED`.
//
// - name: The name of the operation's parent resource.
func (r *ProjectsOperationsService) List(name string) *ProjectsOperationsListCall {
	c := &ProjectsOperationsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Filter sets the optional parameter "filter": The standard list filter.
func (c *ProjectsOperationsListCall) Filter(filter string) *ProjectsOperationsListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": The standard list page
// size.
func (c *ProjectsOperationsListCall) PageSize(pageSize int64) *ProjectsOperationsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": The standard list page
// token.
func (c *ProjectsOperationsListCall) PageToken(pageToken string) *ProjectsOperationsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsOperationsListCall) Fields(s ...googleapi.Field) *ProjectsOperationsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsOperationsListCall) IfNoneMatch(entityTag string) *ProjectsOperationsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsOperationsListCall) Context(ctx context.Context) *ProjectsOperationsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsOperationsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsOperationsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}/operations")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webrisk.projects.operations.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webrisk.projects.operations.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *GoogleLongrunningListOperationsResponse.ServerResponse.Header or (if a
// response was returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ProjectsOperationsListCall) Do(opts ...googleapi.CallOption) (*GoogleLongrunningListOperationsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &GoogleLongrunningListOperationsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webrisk.projects.operations.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ProjectsOperationsListCall) Pages(ctx context.Context, f func(*GoogleLongrunningListOperationsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type ProjectsSubmissionsCreateCall struct {
	s                              *Service
	parent                         string
	googlecloudwebriskv1submission *GoogleCloudWebriskV1Submission
	urlParams_                     gensupport.URLParams
	ctx_                           context.Context
	header_                        http.Header
}

// Create: Creates a Submission of a URI suspected of containing phishing
// content to be reviewed. If the result verifies the existence of malicious
// phishing content, the site will be added to the Google's Social Engineering
// lists (https://support.google.com/webmasters/answer/6350487/) in order to
// protect users that could get exposed to this threat in the future. Only
// allowlisted projects can use this method during Early Access. Please reach
// out to Sales or your customer engineer to obtain access.
//
//   - parent: The name of the project that is making the submission. This string
//     is in the format "projects/{project_number}".
func (r *ProjectsSubmissionsService) Create(parent string, googlecloudwebriskv1submission *GoogleCloudWebriskV1Submission) *ProjectsSubmissionsCreateCall {
	c := &ProjectsSubmissionsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.googlecloudwebriskv1submission = googlecloudwebriskv1submission
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsSubmissionsCreateCall) Fields(s ...googleapi.Field) *ProjectsSubmissionsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsSubmissionsCreateCall) Context(ctx context.Context) *ProjectsSubmissionsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsSubmissionsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsSubmissionsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.googlecloudwebriskv1submission)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/submissions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webrisk.projects.submissions.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webrisk.projects.submissions.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *GoogleCloudWebriskV1Submission.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ProjectsSubmissionsCreateCall) Do(opts ...googleapi.CallOption) (*GoogleCloudWebriskV1Submission, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &GoogleCloudWebriskV1Submission{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webrisk.projects.submissions.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ThreatListsComputeDiffCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// ComputeDiff: Gets the most recent threat list diffs. These diffs should be
// applied to a local database of hashes to keep it up-to-date. If the local
// database is empty or excessively out-of-date, a complete snapshot of the
// database will be returned. This Method only updates a single ThreatList at a
// time. To update multiple ThreatList databases, this method needs to be
// called once for each list.
func (r *ThreatListsService) ComputeDiff() *ThreatListsComputeDiffCall {
	c := &ThreatListsComputeDiffCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// ConstraintsMaxDatabaseEntries sets the optional parameter
// "constraints.maxDatabaseEntries": Sets the maximum number of entries that
// the client is willing to have in the local database. This should be a power
// of 2 between 2**10 and 2**20. If zero, no database size limit is set.
func (c *ThreatListsComputeDiffCall) ConstraintsMaxDatabaseEntries(constraintsMaxDatabaseEntries int64) *ThreatListsComputeDiffCall {
	c.urlParams_.Set("constraints.maxDatabaseEntries", fmt.Sprint(constraintsMaxDatabaseEntries))
	return c
}

// ConstraintsMaxDiffEntries sets the optional parameter
// "constraints.maxDiffEntries": The maximum size in number of entries. The
// diff will not contain more entries than this value. This should be a power
// of 2 between 2**10 and 2**20. If zero, no diff size limit is set.
func (c *ThreatListsComputeDiffCall) ConstraintsMaxDiffEntries(constraintsMaxDiffEntries int64) *ThreatListsComputeDiffCall {
	c.urlParams_.Set("constraints.maxDiffEntries", fmt.Sprint(constraintsMaxDiffEntries))
	return c
}

// ConstraintsSupportedCompressions sets the optional parameter
// "constraints.supportedCompressions": The compression types supported by the
// client.
//
// Possible values:
//
//	"COMPRESSION_TYPE_UNSPECIFIED" - Unknown.
//	"RAW" - Raw, uncompressed data.
//	"RICE" - Rice-Golomb encoded data.
func (c *ThreatListsComputeDiffCall) ConstraintsSupportedCompressions(constraintsSupportedCompressions ...string) *ThreatListsComputeDiffCall {
	c.urlParams_.SetMulti("constraints.supportedCompressions", append([]string{}, constraintsSupportedCompressions...))
	return c
}

// ThreatType sets the optional parameter "threatType": Required. The threat
// list to update. Only a single ThreatType should be specified per request. If
// you want to handle multiple ThreatTypes, you must make one request per
// ThreatType.
//
// Possible values:
//
//	"THREAT_TYPE_UNSPECIFIED" - No entries should match this threat type. This
//
// threat type is unused.
//
//	"MALWARE" - Malware targeting any platform.
//	"SOCIAL_ENGINEERING" - Social engineering targeting any platform.
//	"UNWANTED_SOFTWARE" - Unwanted software targeting any platform.
//	"SOCIAL_ENGINEERING_EXTENDED_COVERAGE" - A list of extended coverage
//
// social engineering URIs targeting any platform.
func (c *ThreatListsComputeDiffCall) ThreatType(threatType string) *ThreatListsComputeDiffCall {
	c.urlParams_.Set("threatType", threatType)
	return c
}

// VersionToken sets the optional parameter "versionToken": The current version
// token of the client for the requested list (the client version that was
// received from the last successful diff). If the client does not have a
// version token (this is the first time calling ComputeThreatListDiff), this
// may be left empty and a full database snapshot will be returned.
func (c *ThreatListsComputeDiffCall) VersionToken(versionToken string) *ThreatListsComputeDiffCall {
	c.urlParams_.Set("versionToken", versionToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ThreatListsComputeDiffCall) Fields(s ...googleapi.Field) *ThreatListsComputeDiffCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ThreatListsComputeDiffCall) IfNoneMatch(entityTag string) *ThreatListsComputeDiffCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ThreatListsComputeDiffCall) Context(ctx context.Context) *ThreatListsComputeDiffCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ThreatListsComputeDiffCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ThreatListsComputeDiffCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/threatLists:computeDiff")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webrisk.threatLists.computeDiff", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webrisk.threatLists.computeDiff" call.
// Any non-2xx status code is an error. Response headers are in either
// *GoogleCloudWebriskV1ComputeThreatListDiffResponse.ServerResponse.Header or
// (if a response was returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ThreatListsComputeDiffCall) Do(opts ...googleapi.CallOption) (*GoogleCloudWebriskV1ComputeThreatListDiffResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &GoogleCloudWebriskV1ComputeThreatListDiffResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webrisk.threatLists.computeDiff", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type UrisSearchCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Search: This method is used to check whether a URI is on a given threatList.
// Multiple threatLists may be searched in a single query. The response will
// list all requested threatLists the URI was found to match. If the URI is not
// found on any of the requested ThreatList an empty response will be returned.
func (r *UrisService) Search() *UrisSearchCall {
	c := &UrisSearchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// ThreatTypes sets the optional parameter "threatTypes": Required. The
// ThreatLists to search in. Multiple ThreatLists may be specified.
//
// Possible values:
//
//	"THREAT_TYPE_UNSPECIFIED" - No entries should match this threat type. This
//
// threat type is unused.
//
//	"MALWARE" - Malware targeting any platform.
//	"SOCIAL_ENGINEERING" - Social engineering targeting any platform.
//	"UNWANTED_SOFTWARE" - Unwanted software targeting any platform.
//	"SOCIAL_ENGINEERING_EXTENDED_COVERAGE" - A list of extended coverage
//
// social engineering URIs targeting any platform.
func (c *UrisSearchCall) ThreatTypes(threatTypes ...string) *UrisSearchCall {
	c.urlParams_.SetMulti("threatTypes", append([]string{}, threatTypes...))
	return c
}

// Uri sets the optional parameter "uri": Required. The URI to be checked for
// matches.
func (c *UrisSearchCall) Uri(uri string) *UrisSearchCall {
	c.urlParams_.Set("uri", uri)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *UrisSearchCall) Fields(s ...googleapi.Field) *UrisSearchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *UrisSearchCall) IfNoneMatch(entityTag string) *UrisSearchCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *UrisSearchCall) Context(ctx context.Context) *UrisSearchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *UrisSearchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *UrisSearchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/uris:search")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webrisk.uris.search", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webrisk.uris.search" call.
// Any non-2xx status code is an error. Response headers are in either
// *GoogleCloudWebriskV1SearchUrisResponse.ServerResponse.Header or (if a
// response was returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *UrisSearchCall) Do(opts ...googleapi.CallOption) (*GoogleCloudWebriskV1SearchUrisResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &GoogleCloudWebriskV1SearchUrisResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webrisk.uris.search", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}
