/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isSafeCast = require( './../../../../base/assert/is-safe-data-type-cast' );
var isSameKindCast = require( './../../../../base/assert/is-same-kind-data-type-cast' );


// MAIN //

/**
* Returns a boolean indicating if a provided ndarray data type can be cast to another ndarray data type according to a specified casting mode.
*
* @param {string} from - ndarray data type
* @param {string} to - ndarray data type
* @param {string} casting - ndarray casting mode
* @returns {boolean} boolean indicating if a data type can be cast to another data type
*
* @example
* var bool = isAllowedCast( 'float32', 'float64', 'safe' );
* // returns true
*
* bool = isAllowedCast( 'float64', 'int32', 'safe' );
* // returns false
*/
function isAllowedCast( from, to, casting ) {
	// Anything goes for "unsafe" casting...
	if ( casting === 'unsafe' ) {
		return true;
	}
	// "Casting" to the same data type is always allowed, regardless of the casting mode...
	if ( from === to ) {
		return true;
	}
	// No casts between different data types are allowed in "none" or "equiv" casting modes...
	if ( casting === 'none' || casting === 'equiv' ) {
		return false;
	}
	// In "safe" casting mode, only casts which preserve values are allowed...
	if ( casting === 'safe' ) {
		return isSafeCast( from, to );
	}
	// In "same-kind" casting mode, in addition to "safe" casts, casts within a "kind" (e.g., between signed integers or between floating-point numbers) are allowed...
	return isSameKindCast( from, to );
}


// EXPORTS //

module.exports = isAllowedCast;
