/*

    File: ntfs.c

    Copyright (C) 1998-2004 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
/*define NTFS_DEBUG 1 */

#define NTFS_GETU8(p)      (*(const uint8_t*)(p))
#define NTFS_GETU16(p)     (*(const uint16_t*)(p))
#define NTFS_GETU32(p)     (*(const uint32_t*)(p))
#define NTFS_GETU64(p)     (*(const uint64_t*)(p))
 
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#include <stdio.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <ctype.h>
#include "types.h"
#include "common.h"
#include "intrf.h"
#include "ntfs.h"
#include "fnctdsk.h"
#include "lang.h"
static int testdisk_ffs(int x);
static int create_ntfs_boot_sector(t_param_disk *disk_car,const t_partition *partition, const int interface, const unsigned int cluster_size, const uint64_t mft_lcn, const uint64_t mftmirr_lcn, const int32_t mft_record_size, const int32_t index_block_size);
static int dump_ntfs_info(const struct ntfs_boot_sector *ntfs_header);
static int dump_2ntfs_info(const struct ntfs_boot_sector *nh1, const struct ntfs_boot_sector *nh2);

static int ntfs_read_MFT(t_param_disk *disk_car, t_partition *partition, const struct ntfs_boot_sector*ntfs_header,const int my_type,const int debug, const int dump_ind);

static int ntfs_get_attr(const char *mft_record, const int my_type, t_partition *partition, const char *end, const int debug, const int dump_ind, const char*file_name_to_find);
static int ntfs_get_attr_aux(const char *attr_record, const int my_type, t_partition *partition, const char *end, const int debug, const int dump_ind, const char*file_name_to_find);
unsigned int ntfs_sector_size(const struct ntfs_boot_sector *ntfs_header);

unsigned int ntfs_sector_size(const struct ntfs_boot_sector *ntfs_header)
{ return (ntfs_header->sector_size[1]<<8)+ntfs_header->sector_size[0]; }

int check_NTFS(t_param_disk *disk_car,t_partition *partition,const int debug,const int dump_ind)
{
  unsigned char buffer[DEFAULT_SECTOR_SIZE];
/*  ecrit_rapport("check_NTFS part_offset=%llu\n",(long long unsigned)partition->part_offset); */
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset)!=0)
  { return 1; }
  if(test_NTFS(disk_car,(struct ntfs_boot_sector*)&buffer,partition,debug,dump_ind)!=0)
    return 1;
  set_NTFS_info(disk_car,(struct ntfs_boot_sector*)&buffer,partition,debug,dump_ind);
  return 0;
}

int recover_NTFS(t_param_disk *disk_car, const struct ntfs_boot_sector*ntfs_header,t_partition *partition,const int debug, const int dump_ind, const int backup)
{
  if(test_NTFS(disk_car,ntfs_header,partition,debug,dump_ind)!=0)
    return 1;
  if(debug>0)
  {
    dump_ntfs_rapport(ntfs_header);
  }
  partition->part_size=(uint64_t)(le64(ntfs_header->sectors_nbr)+1)*ntfs_sector_size(ntfs_header);
  partition->part_type=(unsigned char)P_NTFS;
  if(backup)
  {
    if(debug>1)
      ecrit_rapport("NTFS part_offset=%llu, part_size=%llu, sector_size=%u\n",
	  (long long unsigned)partition->part_offset, (long long unsigned)partition->part_size,
	  disk_car->sector_size);
    partition->boot_sector=partition->part_size/disk_car->sector_size-1;
    partition->part_offset=partition->part_offset+disk_car->sector_size-partition->part_size;
    if(debug>1)
      ecrit_rapport("part_offset=%llu\n",(long long unsigned)partition->part_offset);
  }
  set_NTFS_info(disk_car,ntfs_header,partition,debug,dump_ind);
  return 0;
}

int set_NTFS_info(t_param_disk *disk_car, const struct ntfs_boot_sector*ntfs_header,t_partition *partition,const int debug, const int dump_ind)
{
  partition->name[0]='\0';
  snprintf(partition->info,sizeof(partition->info),"NTFS");
  return ntfs_read_MFT(disk_car, partition, ntfs_header,0x60,debug,dump_ind);
}

int test_NTFS(const t_param_disk *disk_car,const struct ntfs_boot_sector*ntfs_header, t_partition *partition,const int debug, const int dump_ind)
{
  const char*buffer=(const char*)ntfs_header;
  if(le16(ntfs_header->marker)==0xAA55)
  {
    if(memcmp(ntfs_header->system_id,"NTFS",4)==0)
    {
      if((ntfs_header->reserved>0) || (ntfs_header->fats>0) ||
	  (ntfs_header->dir_entries[0]!=0) || (ntfs_header->dir_entries[1]!=0) ||
	  (ntfs_header->sectors[0]!=0) || (ntfs_header->sectors[1]!=0) ||
	  (ntfs_header->fat_length!=0) || (ntfs_header->total_sect!=0) || (ntfs_header->sectors_per_cluster==0))
	return 1;
      if((debug>0) || (dump_ind!=0))
      {
	ecrit_rapport("NTFS at %u/%u/%u\n", offset2cylinder(disk_car,partition->part_offset),offset2head(disk_car,partition->part_offset),offset2sector(disk_car,partition->part_offset));
	if(dump_ind!=0)
	  dump(stdscr,buffer,DEFAULT_SECTOR_SIZE);
      }
      if(ntfs_sector_size(ntfs_header)!=disk_car->sector_size)
      {
	aff_buffer(BUFFER_ADD,"test_NTFS: Incorrect number of bytes per sector %u (NTFS) != %u (HD)\n",ntfs_sector_size(ntfs_header),disk_car->sector_size);
	ecrit_rapport("test_NTFS: Incorrect number of bytes per sector %u (NTFS) != %u (HD)\n",ntfs_sector_size(ntfs_header),disk_car->sector_size);
	return 1;
      }
      if(le16(ntfs_header->secs_track)!=disk_car->CHS.sector)
      {
	aff_buffer(BUFFER_ADD,"check_NTFS: Incorrect number of sectors per track %u (NTFS) != %u (HD)\n",le16(ntfs_header->secs_track),disk_car->CHS.sector);
	ecrit_rapport("sect/track %u (NTFS) != %u (HD)\n",le16(ntfs_header->secs_track),disk_car->CHS.sector);
      }
      if(le16(ntfs_header->heads)!=disk_car->CHS.head+1)
      {
	aff_buffer(BUFFER_ADD,"check_NTFS: Incorrect number of heads/cylinder %u (NTFS) != %u (HD)\n",le16(ntfs_header->heads),disk_car->CHS.head+1);
	ecrit_rapport("heads/cylinder %u (NTFS) != %u (HD)\n",le16(ntfs_header->heads),disk_car->CHS.head+1);
      }

      if(partition->part_size>0)
      {
	uint64_t part_size;
	part_size=le64(ntfs_header->sectors_nbr)+1;

	if(part_size*ntfs_sector_size(ntfs_header)>partition->part_size)
	{
	  ecrit_rapport("test_NTFS size boot_sector %lu > partition %lu\n",(long unsigned)part_size,(long unsigned)(partition->part_size/disk_car->sector_size));
	  aff_buffer(BUFFER_ADD,"test_NTFS size boot_sector %lu > partition %lu\n",(long unsigned)part_size,(long unsigned)(partition->part_size/disk_car->sector_size));
	  return 1;
	}
	if((debug>0) && (part_size!=partition->part_size/disk_car->sector_size))
	{
	    ecrit_rapport("test_NTFS size boot_sector %lu, partition %lu\n",(long unsigned)part_size,(long unsigned)(partition->part_size/disk_car->sector_size));
	}
      }
      partition->upart_type=UP_NTFS;
      return 0;
    }
  }     /* fin marqueur de fin :)) */
  return 1;
}

/* */
static int ntfs_get_attr(const char *mft_record, const int my_type, t_partition *partition, const char *end, const int debug, const int dump_ind, const char*file_name_to_find)
{
  const char *attr_record;
  /* Only check for magic DWORD here, fixup should have happened before */
  if(memcmp(mft_record,"FILE",4)) return 2;	/* NTFS_RECORD_TYPES == magic_FILE ?*/
  if(NTFS_GETU16(mft_record + 0x14)%8!=0)
    return 2;
  if(NTFS_GETU16(mft_record + 0x14)<42)		/* sizeof(MFT_RECORD)>=42 */
    return 2;
  /*	aff_buffer(BUFFER_ADD,"FILE\n"); */
  /*	aff_buffer(BUFFER_ADD,"seq nbr %lu ",NTFS_GETU16(mft_record+0x10)); */
  /*	aff_buffer(BUFFER_ADD,"main MFT record %lu ",NTFS_GETU64(mft_record+0x20)); */
  /* location of first attribute */
  attr_record= mft_record + NTFS_GETU16(mft_record + 0x14);
  return ntfs_get_attr_aux(attr_record, my_type, partition, end, debug, dump_ind, file_name_to_find);
}

static int ntfs_get_attr_aux(const char *attr_record, const int my_type, t_partition *partition, const char *end, const int debug, const int dump_ind, const char*file_name_to_find)
{
  int attr_type;
  while(1)
  {
    /* Resident attributes attr_len>=24(0x18), non resident is bigger */
    unsigned int attr_len;
    if(attr_record+0x18>=end)
    {
      ecrit_rapport("ntfs_get_attr attr_record+0x18>=end\n");
      return 2;
    }
    attr_type=NTFS_GETU32(attr_record);
    if(attr_type==-1) /* attribute list end with type -1 */
      return 0;
    if(debug>1)
    {
      ecrit_rapport("attr_type=%x %s\n",attr_type,(NTFS_GETU8(attr_record+8)==0?"resident":"non resident"));
    }
    attr_len=NTFS_GETU16(attr_record+4);
    if((attr_len%8!=0)||(attr_len<0x18))
    {
      ecrit_rapport("ntfs_get_attr attr_type=%x attr_len=%u (attr_len%%8!0)||(attr_len<0x18)\n",attr_type,attr_len);
      return 2;
    }
    if(dump_ind)
    {
      WINDOW *window=newwin(0,0,0,0);	/* full screen */
      keypad(window, TRUE); /* Need it to get arrow key */
      aff_copy(window);
      dump(window,attr_record,attr_len);
      delwin(window);
#ifdef DJGPP
      wredrawln(stdscr,0,stdscr->_maxy);	/* redrawwin def is boggus in pdcur24 */
#else
      redrawwin(stdscr);	/* stdscr has been corrupted by window */
#endif
    }
    if(NTFS_GETU8(attr_record+8)==0)	/* attribute is resident */
    {
      unsigned int attr_value_length=NTFS_GETU16(attr_record+0x10);
      unsigned int attr_value_offset=NTFS_GETU16(attr_record+0x14);
      const char *attr_list_entry=attr_record+attr_value_offset;
      if(attr_value_offset%8!=0)
      {
	ecrit_rapport("ntfs_get_attr attr_value_offset=%u (%%8!=0)\n",attr_value_offset);
	return 2;
      }
      if(attr_list_entry+26>=end)
      {
	ecrit_rapport("ntfs_get_attr attr_list_entry+26=%p, end=%p\n",attr_list_entry+26,end);
	return 2;
      }
      /* We found the attribute type. Is the name correct, too? */
      if((attr_value_offset+attr_value_length>attr_len) || (attr_list_entry+attr_len >= end))
      {
	ecrit_rapport("ntfs_get_attr ");
	return 2;
      }
      if((attr_type==my_type)&&(attr_value_offset!=0))
      {
	switch(attr_type)
	{
	  case 0x30:	/* AT_FILE_NAME */
	    {
	      const char *file_name_attr=attr_list_entry;
	      unsigned int file_name_length;
	      const char *name_it;
	      if(file_name_attr+0x42>=end)
		return 2;
	      file_name_length=NTFS_GETU8(file_name_attr+0x40);	/* size in unicode char */
	      if(file_name_attr+0x42+2*file_name_length>=end)
		return 2;
	      {
		char file_name[file_name_length+1];
		unsigned int i;
		/*		aff_buffer(BUFFER_ADD,"MFT record nbr %lu ",NTFS_GETU64(file_name_attr)); */
		for(name_it=file_name_attr+0x42,i=0;i<file_name_length; name_it+=2,i++)
		  file_name[i]=*name_it;
		file_name[i]='\0';
		if(debug>1)
		{
		  ecrit_rapport("file_name=%s\n",file_name);
		}
		if(file_name_to_find!=NULL)
		{
		  if(attr_type==my_type)
		  {
		    if(strcmp(file_name_to_find,file_name)==0)
		      return 1;
		    else
		      return 2;
		  }
		} else
		  aff_buffer(BUFFER_ADD,"%s\n",file_name);
	      }
	    }
	    break;
	  case 0x60:	/* AT_VOLUME_NAME */
	    {
	      unsigned int volume_name_length=attr_value_length;
	      const char *name_it;
	      char *dest=partition->name;
	      volume_name_length/=2;	/* Unicode */
	      if(volume_name_length>sizeof(partition->name)-1)
		volume_name_length=sizeof(partition->name)-1;
	      for(name_it=attr_list_entry;(volume_name_length>0) && (*name_it!='\0') && (name_it[1]=='\0'); name_it+=2,volume_name_length--)
		*dest++=*name_it;
	      *dest++='\0'; /* 27 january 2003: Correct a bug found by Andreas du Plessis-Denz */
	    }
	    return 1;
	  case 0x90:	/* AT_INDEX_ROOT */
	    return NTFS_GETU32(attr_list_entry+8);	/* index_block_size */
	}
      }
    }
    else
    {	/* attribute is not resident */
      if(attr_type==my_type)
      {
	switch(attr_type)
	{
	  case 0x80:	/* AT_DATA */
	    {
	      const uint8_t *buf;
	      uint8_t b;                   /* Current byte offset in buf. */
	      int64_t deltaxcn;           /* Change in [vl]cn. */
	      uint16_t mapping_pairs_offset;
	      const uint8_t *attr_end;     /* End of attribute. */
	      long lcn;
	      mapping_pairs_offset=NTFS_GETU16(attr_record+32);
	      buf=attr_record+mapping_pairs_offset;
	      attr_end = (const uint8_t*)attr_record + attr_len;
	      lcn = 0;
	      /* return first element of the run_list */
	      {
		b = *buf & 0xf;
		if (b) {
		  if (buf + b > attr_end)
		  {
		    ecrit_rapport("Attribut AT_DATA: bad size\n");
		    return 0;
		  }
		  for (deltaxcn = (int8_t)buf[b--]; b; b--)
		    deltaxcn = (deltaxcn << 8) + buf[b];
		} else { /* The length entry is compulsory. */
		  ecrit_rapport("Missing length entry in mapping pairs array.");
		  deltaxcn = (int64_t)-1;
		}
		if (deltaxcn < 0) {
		  ecrit_rapport("Invalid length in mapping pairs array.");
		  return 0;
		}
		if (!(*buf & 0xf0))
		{
		  ecrit_rapport("LCN_HOLE\n");
		  return 0;
		}
		else
		{
		  /* Get the lcn change which really can be negative. */
		  uint8_t b2 = *buf & 0xf;
		  b = b2 + ((*buf >> 4) & 0xf);
		  if (buf + b > attr_end)
		  {
		    ecrit_rapport("Attribut AT_DATA: bad size\n");
		    return 0;
		  }
		  for (deltaxcn = (int8_t)buf[b--]; b > b2; b--)
		    deltaxcn = (deltaxcn << 8) + buf[b];
		  /* Change the current lcn to it's new value. */
		  lcn += deltaxcn;
		  /* Check lcn is not below -1. */
		  if (lcn < -1) {
		    ecrit_rapport("Invalid LCN < -1 in mapping pairs array.");
		    return 0;
		  }
		  if(debug>1)
		  {
		    ecrit_rapport("LCN %ld\n",lcn);
		  }
		  return lcn;
		}
	      }
	    }
	    break;
	}
      }
    }
    attr_record+=attr_len;
  }
}

static int ntfs_read_MFT(t_param_disk *disk_car, t_partition *partition, const struct ntfs_boot_sector*ntfs_header,const int my_type,const int debug, const int dump_ind)
{
  unsigned char *buffer;
  unsigned char *attr;
  uint64_t mft_pos;
  unsigned int mft_recordsize;
  mft_pos=partition->part_offset+(uint64_t)(ntfs_header->reserved+ntfs_header->mft_lcn*ntfs_header->sectors_per_cluster)*DEFAULT_SECTOR_SIZE;
  if(ntfs_header->clusters_per_index_record>0)
    mft_recordsize=ntfs_header->sectors_per_cluster*ntfs_header->clusters_per_index_record;
  else
    mft_recordsize=1<<(-ntfs_header->clusters_per_index_record);
  buffer=(unsigned char *)MALLOC(mft_recordsize*DEFAULT_SECTOR_SIZE);
#ifdef NTFS_DEBUG
  ecrit_rapport("NTFS cluster size = %u\n",ntfs_header->sectors_per_cluster);
  ecrit_rapport("NTFS MFT_record_size = %u\n",mft_recordsize);
  ecrit_rapport("NTFS MFT cluster = %lu\n",ntfs_header->mft_lcn);
#endif
  if(disk_car->read(disk_car,mft_recordsize*DEFAULT_SECTOR_SIZE, buffer, mft_pos)!=0)
  {
    ecrit_rapport(msg_ROOT_CLUSTER_RERR);
    FREE(buffer);
    return 1;
  }
  attr=buffer;
  while(attr+0x30<=(buffer+mft_recordsize*DEFAULT_SECTOR_SIZE))
  {
    int res=ntfs_get_attr(attr,my_type,partition,buffer+mft_recordsize*DEFAULT_SECTOR_SIZE,debug,dump_ind,NULL);
    if((res>0)|| (NTFS_GETU32(attr + 0x1C)<0x30))
    {
      FREE(buffer);
      return res;
    }
    attr+= NTFS_GETU32(attr + 0x1C);
  }
  FREE(buffer);
  return 0;
}

int is_ntfs(const int part_type)
{
  switch(part_type)
  {
    case P_NTFS:
    case P_NTFSH:
      return 1;
  }
  return 0;
}

static int testdisk_ffs(int x)
{
  int r = 1;

  if (!x)
    return 0;
  if (!(x & 0xffff)) {
    x >>= 16;
    r += 16;
  }
  if (!(x & 0xff)) {
    x >>= 8;
    r += 8;
  }
  if (!(x & 0xf)) {
    x >>= 4;
    r += 4;
  }
  if (!(x & 3)) {
    x >>= 2;
    r += 2;
  }
  if (!(x & 1)) {
    x >>= 1;
    r += 1;
  }
  return r;
}

static int create_ntfs_boot_sector(t_param_disk *disk_car,const t_partition *partition, const int interface, const unsigned int cluster_size, const uint64_t mft_lcn, const uint64_t mftmirr_lcn, const int32_t mft_record_size, const int32_t index_block_size)
{
  unsigned char orgboot[DEFAULT_SECTOR_SIZE];
  unsigned char newboot[DEFAULT_SECTOR_SIZE];
  struct ntfs_boot_sector *org_ntfs_header=(struct ntfs_boot_sector *)&orgboot;
  struct ntfs_boot_sector *ntfs_header=(struct ntfs_boot_sector *)&newboot;
  int error=0;
  if(disk_car->read(disk_car,sizeof(orgboot), &orgboot, partition->part_offset)!=0)
  {
    ecrit_rapport(msg_CHKFAT_RERR); return 1;
  }
  if(cluster_size==0)
  {
    error=1;
  }
  if(error)
  {
    ecrit_rapport("Bad extrapolation.\n");
    return 1;
  }
  memcpy(&newboot,&orgboot,sizeof(newboot));
  memcpy(ntfs_header->system_id,"NTFS    ",8);
  ntfs_header->sector_size[0]=disk_car->sector_size & 0xFF;
  ntfs_header->sector_size[1]=disk_car->sector_size>>8;
  ntfs_header->sectors_per_cluster=cluster_size/disk_car->sector_size;
  ntfs_header->reserved=0;
  ntfs_header->fats=0;
  ntfs_header->dir_entries[0]=0;
  ntfs_header->dir_entries[1]=0;
  ntfs_header->sectors[0]=0;
  ntfs_header->sectors[1]=0;
  ntfs_header->media=0xF8;
  ntfs_header->fat_length=0;
  ntfs_header->secs_track=le16(disk_car->CHS.sector);
  ntfs_header->heads=le16(disk_car->CHS.head+1);
  if(partition->status==STATUS_LOG)
    ntfs_header->hidden=63;
  else
    ntfs_header->hidden=partition->part_offset/disk_car->sector_size;
  ntfs_header->total_sect=0;
  ntfs_header->sectors_nbr=le64(partition->part_size/disk_car->sector_size-1);
  ntfs_header->mft_lcn=mft_lcn;
  ntfs_header->mftmirr_lcn=mftmirr_lcn;
  ntfs_header->clusters_per_mft_record=(mft_record_size >= cluster_size?mft_record_size / cluster_size:
      -(testdisk_ffs(mft_record_size) - 1));
  ntfs_header->clusters_per_index_record =(index_block_size >= cluster_size?index_block_size / cluster_size:
      -(testdisk_ffs(index_block_size) - 1));
  ntfs_header->reserved0[0]=0;
  ntfs_header->reserved0[1]=0;
  ntfs_header->reserved0[2]=0;
  ntfs_header->reserved1[0]=0;
  ntfs_header->reserved1[1]=0;
  ntfs_header->reserved1[2]=0;
  /*
  {
    uint32_t *u;
    uint32_t checksum;
    for (checksum = 0,u=(uint32_t*)ntfs_header; u < (uint32_t*)(&ntfs_header->checksum); u++)
      checksum += NTFS_GETU32(u);
    ntfs_header->checksum=checksum;
  }
  */
  ntfs_header->checksum=0;
  ntfs_header->marker=0xAA55;
  if(memcmp(newboot,orgboot,sizeof(newboot)))
  {
    ecrit_rapport("             New / Current boot sector");
    dump_2ntfs_rapport(ntfs_header,org_ntfs_header);
    ecrit_rapport("Extrapolated boot sector and current boot sector are different.\n");
  }
  else
  {
    ecrit_rapport("Extrapolated boot sector and current boot sector are identical.\n");
  }
  /* */
  if(interface)
  {
    struct MenuItem menuSaveBoot[]=
    {
      { 'D', "Dump", "Dump sector" },
      { 'Q',"Quit","Quit this section"},
      { 'W', "Write","Write boot"},
      { 0, NULL, NULL }
    };
    const char *options="DQ";
    int do_write=0;
    aff_copy(stdscr);
    wmove(stdscr,4,0);
    wdoprintf(stdscr,"%s",disk_car->description(disk_car));
    mvwaddstr(stdscr,5,0,msg_PART_HEADER_LONG);
    wmove(stdscr,6,0);
    aff_part(stdscr,AFF_PART_ORDER,disk_car,partition);
    wmove(stdscr,8,0);
    if(memcmp(newboot,orgboot,sizeof(newboot)))	/* Only compare the first sector */
    {
      options="DWQ";
      dump_2ntfs_info(ntfs_header, org_ntfs_header);
      dump_2ntfs_rapport(ntfs_header, org_ntfs_header);
      wdoprintf(stdscr,"Extrapolated boot sector and current boot sector are different.\n");
      if(error)
	ecrit_rapport("Warning: Extrapolated boot sector have incorrect values.\n");
    }
    else
    {
      dump_ntfs_info(ntfs_header);
      dump_ntfs_rapport(ntfs_header);
      wdoprintf(stdscr,"Extrapolated boot sector and current boot sector are identical.\n");
    }
    switch(toupper(wmenuSelect(stdscr,INTER_DUMP_Y, INTER_DUMP_X, menuSaveBoot,8,options,MENU_HORIZ | MENU_BUTTON, 0)))
    {
      case 'W':
	if(strchr(options,'W')!=NULL)
	  do_write=1;
	break;
      case 'D':
	if(strchr(options,'D')!=NULL)
	{
	  dump2(stdscr,newboot,orgboot, sizeof(newboot));
	}
	break;
    }
    if(do_write!=0 && ask_confirmation("Write new NTFS boot sector, confirm ? (Y/N)")!=0)
    {
      ecrit_rapport("Write new boot!\n");
      /* Write boot sector and backup boot sector */
      if(disk_car->write(disk_car,sizeof(newboot), &newboot, partition->part_offset))
      {
	display_message("Write error: Can't write new NTFS boot sector\n");
      }
      if(disk_car->write(disk_car,sizeof(newboot), &newboot, partition->part_offset+partition->part_size-disk_car->sector_size)!=0)
      {
	display_message("Write error: Can't write new NTFS backup boot sector\n");
      }
    }
    else
      ecrit_rapport("Don't write new boot!\n");
  }
  return 0;
}

static int dump_2ntfs_info(const struct ntfs_boot_sector *nh1, const struct ntfs_boot_sector *nh2)
{
  wdoprintf(stdscr,"sectors_per_cluster       %u %u\n",nh1->sectors_per_cluster,nh2->sectors_per_cluster);
  wdoprintf(stdscr,"mft_lcn                   %lu %lu\n",(long unsigned int)nh1->mft_lcn,(long unsigned int)nh2->mft_lcn);
  wdoprintf(stdscr,"mftmirr_lcn               %lu %lu\n",(long unsigned int)nh1->mftmirr_lcn,(long unsigned int)nh2->mftmirr_lcn);
  wdoprintf(stdscr,"clusters_per_mft_record   %d %d\n",nh1->clusters_per_mft_record,nh2->clusters_per_mft_record);
  wdoprintf(stdscr,"clusters_per_index_record %d %d\n",nh1->clusters_per_index_record,nh2->clusters_per_index_record);
  return 0;
}

int dump_2ntfs_rapport(const struct ntfs_boot_sector *nh1, const struct ntfs_boot_sector *nh2)
{
  ecrit_rapport("sectors_per_cluster       %u %u\n",nh1->sectors_per_cluster,nh2->sectors_per_cluster);
  ecrit_rapport("mft_lcn                   %lu %lu\n",(long unsigned int)nh1->mft_lcn,(long unsigned int)nh2->mft_lcn);
  ecrit_rapport("mftmirr_lcn               %lu %lu\n",(long unsigned int)nh1->mftmirr_lcn,(long unsigned int)nh2->mftmirr_lcn);
  ecrit_rapport("clusters_per_mft_record   %d %d\n",nh1->clusters_per_mft_record,nh2->clusters_per_mft_record);
  ecrit_rapport("clusters_per_index_record %d %d\n",nh1->clusters_per_index_record,nh2->clusters_per_index_record);
  return 0;
}

static int dump_ntfs_info(const struct ntfs_boot_sector *ntfs_header)
{
  wdoprintf(stdscr,"sectors_per_cluster       %u\n",ntfs_header->sectors_per_cluster);
  wdoprintf(stdscr,"mft_lcn                   %lu\n",(long unsigned int)ntfs_header->mft_lcn);
  wdoprintf(stdscr,"mftmirr_lcn               %lu\n",(long unsigned int)ntfs_header->mftmirr_lcn);
  wdoprintf(stdscr,"clusters_per_mft_record   %d\n",ntfs_header->clusters_per_mft_record);
  wdoprintf(stdscr,"clusters_per_index_record %d\n",ntfs_header->clusters_per_index_record);
  return 0;
}

int dump_ntfs_rapport(const struct ntfs_boot_sector *ntfs_header)
{
  ecrit_rapport("sectors_per_cluster       %u\n",ntfs_header->sectors_per_cluster);
  ecrit_rapport("mft_lcn                   %lu\n",(long unsigned int)ntfs_header->mft_lcn);
  ecrit_rapport("mftmirr_lcn               %lu\n",(long unsigned int)ntfs_header->mftmirr_lcn);
  ecrit_rapport("clusters_per_mft_record   %d\n",ntfs_header->clusters_per_mft_record);
  ecrit_rapport("clusters_per_index_record %d\n",ntfs_header->clusters_per_index_record);
  return 0;
}


int rebuild_NTFS_BS(t_param_disk *disk_car, t_partition *partition, const int debug, const int dump_ind,const int interface)
{
  uint64_t sector;
  unsigned char buffer[8*DEFAULT_SECTOR_SIZE];
  int ind_stop=FALSE;
  int mft_sector=0;
  if(interface)
  {
    aff_copy(stdscr);
    wmove(stdscr,4,0);
    wdoprintf(stdscr,"%s",disk_car->description(disk_car));
    mvwaddstr(stdscr,5,0,msg_PART_HEADER_LONG);
    wmove(stdscr,6,0);
    aff_part(stdscr,AFF_PART_ORDER,disk_car,partition);
  }
  /* try to find MFT Backup first */
  for(sector=(partition->part_size/disk_car->sector_size/2-20>0?partition->part_size/disk_car->sector_size/2-20:1);(sector<partition->part_size/disk_car->sector_size)&&(sector<=partition->part_size/disk_car->sector_size/2+20)&&(ind_stop==FALSE);sector++)
  {
    if(disk_car->read(disk_car,2*DEFAULT_SECTOR_SIZE, &buffer, partition->part_offset+sector*(uint64_t)disk_car->sector_size)==0)
    {
      if(memcmp(buffer,"FILE",4)==0 && (NTFS_GETU16(buffer+ 0x14)%8==0) && (NTFS_GETU16(buffer+ 0x14)>=42)
	  &&(NTFS_GETU16(buffer+22)==1))	/* MFT_RECORD_IN_USE */
      {
	int res;
	res=ntfs_get_attr(buffer,0x30,partition,buffer+2*DEFAULT_SECTOR_SIZE,debug,0,"$MFT");
	if(res==1)
	{
	  mft_sector=sector;
	  ecrit_rapport("mft at %lu, seq=%u, main=%u res=%d\n",(long unsigned)sector,NTFS_GETU8(buffer+0x10),(unsigned int)NTFS_GETU32(buffer+0x20),res);
	  ind_stop=TRUE;
	}
      }
    }
  }
  for(sector=1;(sector<partition->part_size/disk_car->sector_size)&&(ind_stop==FALSE);sector++)
  {
    if((interface!=0) &&(sector%200)==0)
    {
      wmove(stdscr,9,0);
      wclrtoeol(stdscr);
      wdoprintf(stdscr,"Search mft %10lu/%lu",(long unsigned)sector,(long unsigned)(partition->part_size/disk_car->sector_size));
      wrefresh(stdscr);
      switch(wgetch_nodelay(stdscr))
      {
	case KEY_ENTER:
#ifdef PADENTER
      case PADENTER:
#endif
	case '\n':
	case '\r':
	case 's':
	case 'S':
	  ind_stop=1;
	  break;
      }
    }
    if(disk_car->read(disk_car,2*DEFAULT_SECTOR_SIZE, &buffer, partition->part_offset+sector*(uint64_t)disk_car->sector_size)==0)
    {
      if(memcmp(buffer,"FILE",4)==0 && (NTFS_GETU16(buffer+ 0x14)%8==0) && (NTFS_GETU16(buffer+ 0x14)>=42))
      {
	int res;
	res=ntfs_get_attr(buffer,0x30,partition,buffer+2*DEFAULT_SECTOR_SIZE,debug,0,"$MFT");
	if(res==1)
	{
	  mft_sector=sector;
	  ecrit_rapport("mft at %lu, seq=%u, main=%u res=%d\n",(long unsigned)sector,NTFS_GETU8(buffer+0x10),(unsigned int)NTFS_GETU32(buffer+0x20),res);
	  ind_stop=TRUE;
	}
      }
    }
  }
  if(mft_sector>0)
  {
    uint64_t mft_lcn;
    uint64_t mftmirr_lcn;
    unsigned int sectors_per_cluster=0;
    const char *attr=buffer;
    unsigned int index_block_size=4096;
    unsigned int mft_record_size=1024;
    if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset+(uint64_t)mft_sector*disk_car->sector_size)!=0)
    {
      ecrit_rapport("NTFS: Can't read mft_sector\n");
      return 1;
    }
    mft_lcn=ntfs_get_attr(attr,0x80,partition,buffer+8*DEFAULT_SECTOR_SIZE,debug,0,NULL);
    mft_record_size=NTFS_GETU32(attr + 0x1C);
    attr+= NTFS_GETU32(attr + 0x1C);
    mftmirr_lcn=ntfs_get_attr(attr,0x80,partition,buffer+8*DEFAULT_SECTOR_SIZE,debug,0,NULL);
    if(mft_lcn>0 && mft_sector%mft_lcn==0)
    {
      sectors_per_cluster=mft_sector/mft_lcn;
    }
    if(mftmirr_lcn>0 && mft_sector%mftmirr_lcn==0)
    {
      sectors_per_cluster=mft_sector/mftmirr_lcn;
    }
    if(sectors_per_cluster>0 && mft_record_size>0)
    {
      /* Read "root directory" in MFT */
      if(disk_car->read(disk_car,mft_record_size, &buffer, partition->part_offset+(uint64_t)mft_lcn*sectors_per_cluster*disk_car->sector_size+5*(uint64_t)mft_record_size)!=0)
      {
	ecrit_rapport("Can't read \"root directory\" in MFT\n");
	return 1;
      }
      index_block_size=ntfs_get_attr(buffer,0x90,partition,buffer+mft_record_size,debug,0,NULL);
      if(index_block_size%512!=0)
	index_block_size=4096;
    }
    ecrit_rapport("ntfs_find_mft: sectors_per_cluster %u\n",sectors_per_cluster);
    ecrit_rapport("ntfs_find_mft: mft_lcn             %lu\n",(long unsigned int)mft_lcn);
    ecrit_rapport("ntfs_find_mft: mftmirr_lcn         %lu\n",(long unsigned int)mftmirr_lcn);
    ecrit_rapport("ntfs_find_mft: mft_record_size     %u\n",mft_record_size);
    ecrit_rapport("ntfs_find_mft: index_block_size    %u\n",index_block_size);
    create_ntfs_boot_sector(disk_car,partition, interface, sectors_per_cluster*disk_car->sector_size, mft_lcn, mftmirr_lcn, mft_record_size, index_block_size);
  }
  return 0;
}
