// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


// mathops.cc
//
// Vector3 and SolidAngle class definitions, and a couple random functions

#include "vector3.h"

// For the Vector3 class ----------------------------------------

// Vector3 math ops: +-*/, magnitude, unit vector, dot and cross products.
//  Should be self-explanatory.
Vector3 &Vector3::operator+=(Vector3 v)
{ x += v.x; y += v.y; z += v.z; return *this; }

Vector3 &Vector3::operator-=(Vector3 v)
{ x -= v.x; y -= v.y; z -= v.z; return *this; }

Vector3 Vector3::operator+(Vector3 v) const
{ return Vector3(x+v.x, y+v.y, z+v.z); }

Vector3 Vector3::operator-(Vector3 v) const
{ return Vector3(x-v.x, y-v.y, z-v.z); }

Vector3 Vector3::operator*(double c) const
{ return Vector3(x*c, y*c, z*c); }

Vector3 operator*(double c, Vector3 v)
{ return v*c; }

Vector3 Vector3::operator/(double c) const
{ return Vector3(x/c, y/c, z/c); }

double Vector3::magnitude() const
{ return sqrt( x*x + y*y + z*z ); }

// If object calling this is the zero vector, return another zero vector.
//  Else produce the expected result: a unit vector in the same direction.
Vector3 Vector3::unitvector() const
{
  double radius = magnitude();
  if (radius > 0.0)
    return (*this)/radius;
  else
    return Vector3(0,0,0);
}

double Vector3::dot(Vector3 v) const
{ return x*v.x + y*v.y + x*v.z; }

Vector3 Vector3::cross(Vector3 v) const
{ return Vector3(y*v.z - z*v.y, z*v.x - x*v.z, x*v.y - y*v.x); }


// toSpherical(): returns the solid angle {theta,phi} defining the direction
//  of the current vector.  If object calling this is the zero vector,
//  return some random solid angle for definiteness (specifically, theta = 0,
//  phi = 0).  Else return the angular coordinates of the direction the
//  vector is pointing.  (If the vector is polar, return longitude phi = 0
//  for definiteness.)

SolidAngle Vector3::toSpherical() const
{
  double theta = 0.0, phi = 0.0;
  Vector3 unit = unitvector();

  if (magnitude() == 0.0)
    return SolidAngle(0.0, 0.0); // for neatness' sake

  if (unit.z >= 1.0)
    theta = M_PI_2;
  else if (unit.z <= -1.0)
    theta = -M_PI_2;
  else {
    theta = asin(unit.z);
    if (!(unit.x == 0.0 && unit.y == 0.0))
      phi = atan2(unit.y, unit.x);
    if (phi < 0.0) // put phi in the range 0 to 2pi
      phi += (2 * M_PI);
  }
  return SolidAngle(phi, theta);
}


// toGalactic(), toCelestial(): return the conversions of the current vector
//  from celestial to galactic coordinates and vice versa, respectively.
//  We assume that the basis vectors are oriented as follows:
//
// Basis vector:     Celestial coords:      Galactic coords:
// -------------     -----------------      ----------------
//      +x             RA 0h, Dec 0        Lat 0, Lon 0 (Sgr A)
//      +y             RA 6h, Dec 0         Lat 0, Lon 90 deg
//      +z          North Celestial Pole   North Galactic Pole
//
// Note there is no safeguard to prevent applying (e.g.) toCelestial() to
// a vector which is already in celestial coordinates.  Preventing this is
// the responsibility of the main program.

Vector3 Vector3::toGalactic() const
{
  Vector3 galactic;
  galactic.x =
    x * (COS_A * SIN_D * SIN_T - SIN_A * COS_T)
    + y * (SIN_A * SIN_D * SIN_T + COS_A * COS_T)
    - z * (COS_D * SIN_T);
  galactic.y =
    -x * (COS_A * SIN_D * COS_T + SIN_A * SIN_T)
    + y * (-SIN_A * SIN_D * COS_T + COS_A * SIN_T)
    + z * (COS_D * COS_T);
  galactic.z = 
    x * COS_A * COS_D + y * SIN_A * COS_D + z * SIN_D;
  return galactic;
}

Vector3 Vector3::toCelestial() const
{
  Vector3 celestial;
  celestial.x =
    x * (COS_A * SIN_D * SIN_T - SIN_A * COS_T)
    - y * (COS_A * SIN_D * COS_T + SIN_A * SIN_T)
    + z * (COS_A * COS_D);
  celestial.y =
    x * (SIN_A * SIN_D * SIN_T + COS_A * COS_T)
    + y * (-SIN_A * SIN_D * COS_T + COS_A * SIN_T)
    + z * (SIN_A * COS_D);
  celestial.z = 
    -x * COS_D * SIN_T + y * COS_D * COS_T + z * SIN_D;
  return celestial;
}


// For the SolidAngle class ------------------------------------

// toCartesian(): returns a vector pointing in the same direction as the
//  current solid angle, with a magnitude specified by radius.

Vector3 SolidAngle::toCartesian(double radius) const
{
  double x, y, z;

  x = cos(theta) * cos(phi);
  y = cos(theta) * sin(phi);
  z = sin(theta);

  return radius * Vector3(x, y, z);
}


// toGalactic(), toCelestial(): return conversions of the current solid angle
//  from celestial to galactic coordinates and vice versa, respectively.
//  These just call the appropriate vector conversion functions in Cartesian
//  coords, where such a transformation is much easier.  See warning
//  in comments above the vector conversion functions.

SolidAngle SolidAngle::toGalactic() const
{ return toCartesian(1.0).toGalactic().toSpherical(); }

SolidAngle SolidAngle::toCelestial() const
{ return toCartesian(1.0).toCelestial().toSpherical(); }

