/* ping - ping/pong demo of GRAS features                                   */

/* Copyright (c) 2006, 2007, 2009, 2010. The SimGrid Team.
 * All rights reserved.                                                     */

/* This program is free software; you can redistribute it and/or modify it
 * under the terms of the license (GNU LGPL) which comes with this package. */

#include "ping.h"
XBT_LOG_EXTERNAL_DEFAULT_CATEGORY(Ping);

static xbt_socket_t try_gras_socket_client(const char *host, int port)
{
  volatile xbt_socket_t sock = NULL;
  xbt_ex_t e;
  TRY {
    sock = gras_socket_client(host, port);
  }
  CATCH(e) {
    if (e.category != system_error)
      /* dunno what happened, let the exception go through */
      RETHROWF("Unable to connect to the server: %s");
    xbt_ex_free(e);
  }
  return sock;
}

int client(int argc, char *argv[])
{
  xbt_socket_t toserver = NULL;        /* peer */

  xbt_socket_t from;
  int ping, pong;

  const char *host = "127.0.0.1";
  int port = 4000;

  /* 1. Init the GRAS's infrastructure */
  gras_init(&argc, argv);

  /* 2. Get the server's address. The command line override defaults when specified */
  if (argc == 3) {
    host = argv[1];
    port = atoi(argv[2]);
  }

  XBT_INFO("Launch client (server on %s:%d)", host, port);

  /* 3. Create a socket to speak to the server */
  while (!(toserver = try_gras_socket_client(host, port)))
    gras_os_sleep(0.05);

  XBT_INFO("Connected to %s:%d.", host, port);

  /* 4. Register the messages.
     See, it doesn't have to be done completely at the beginning, only before use */
  ping_register_messages();

  /* 5. Keep the user informed of what's going on */
  XBT_INFO(">>>>>>>> Connected to server which is on %s:%d <<<<<<<<",
        xbt_socket_peer_name(toserver), xbt_socket_peer_port(toserver));

  /* 6. Prepare and send the ping message to the server */
  ping = 1234;
  TRY {
    gras_msg_send(toserver, "ping", &ping);
  }
  CATCH_ANONYMOUS {
    gras_socket_close(toserver);
    RETHROWF("Failed to send PING to server: %s");
  }
  XBT_INFO(">>>>>>>> Message PING(%d) sent to %s:%d <<<<<<<<",
        ping,
        xbt_socket_peer_name(toserver), xbt_socket_peer_port(toserver));

  /* 7. Wait for the answer from the server, and deal with issues */
  TRY {
    gras_msg_wait(6000, "pong", &from, &pong);
  }
  CATCH_ANONYMOUS {
    gras_socket_close(toserver);
    RETHROWF("Why can't I get my PONG message like everyone else: %s");
  }

  /* 8. Keep the user informed of what's going on, again */
  XBT_INFO(">>>>>>>> Got PONG(%d) from %s:%d <<<<<<<<",
        pong, xbt_socket_peer_name(from), xbt_socket_peer_port(from));

  /* 9. Free the allocated resources, and shut GRAS down */
  gras_socket_close(toserver);
  XBT_INFO("Done.");
  gras_exit();
  return 0;
}                               /* end_of_client */
