# -*- encoding: utf-8 -*-
require_relative 'test_helper'

# Test to ensure that existing representations in database do not break on
# migrating to new versions of this gem. This ensures that future versions of
# this gem will retain backwards compatibility with data generated by earlier
# versions.
class LegacyCompatibilityTest < Minitest::Test
  class LegacyNonmarshallingPet < ActiveRecord::Base
    PET_NICKNAME_SALT = Digest::SHA256.hexdigest('my-really-really-secret-pet-nickname-salt')
    PET_NICKNAME_KEY = 'my-really-really-secret-pet-nickname-key'
    PET_BIRTHDATE_SALT = Digest::SHA256.hexdigest('my-really-really-secret-pet-birthdate-salt')
    PET_BIRTHDATE_KEY = 'my-really-really-secret-pet-birthdate-key'

    self.attr_encrypted_options[:insecure_mode] = true
    self.attr_encrypted_options[:algorithm] = 'aes-256-cbc'
    self.attr_encrypted_options[:mode] = :single_iv_and_salt

    attr_encrypted :nickname,
      :key => proc { Encryptor.encrypt(:value => PET_NICKNAME_SALT, :key => PET_NICKNAME_KEY, insecure_mode: true, algorithm: 'aes-256-cbc') }
    attr_encrypted :birthdate,
      :key => proc { Encryptor.encrypt(:value => PET_BIRTHDATE_SALT, :key => PET_BIRTHDATE_KEY, insecure_mode: true, algorithm: 'aes-256-cbc') }
  end

  class LegacyMarshallingPet < ActiveRecord::Base
    PET_NICKNAME_SALT = Digest::SHA256.hexdigest('my-really-really-secret-pet-nickname-salt')
    PET_NICKNAME_KEY = 'my-really-really-secret-pet-nickname-key'
    PET_BIRTHDATE_SALT = Digest::SHA256.hexdigest('my-really-really-secret-pet-birthdate-salt')
    PET_BIRTHDATE_KEY = 'my-really-really-secret-pet-birthdate-key'

    self.attr_encrypted_options[:insecure_mode] = true
    self.attr_encrypted_options[:algorithm] = 'aes-256-cbc'
    self.attr_encrypted_options[:mode] = :single_iv_and_salt

    attr_encrypted :nickname,
                   :key => proc { Encryptor.encrypt(:value => PET_NICKNAME_SALT, :key => PET_NICKNAME_KEY, insecure_mode: true, algorithm: 'aes-256-cbc') },
                   :marshal => true
    attr_encrypted :birthdate,
                   :key => proc { Encryptor.encrypt(:value => PET_BIRTHDATE_SALT, :key => PET_BIRTHDATE_KEY, insecure_mode: true, algorithm: 'aes-256-cbc') },
                   :marshal => true
  end

  def setup
    ActiveRecord::Base.connection.tables.each { |table| ActiveRecord::Base.connection.drop_table(table) }
    create_tables
  end

  def test_nonmarshalling_backwards_compatibility
    pet = LegacyNonmarshallingPet.create!(
      :name => 'Fido',
      :encrypted_nickname => 'uSUB6KGzta87yxesyVc3DA==',
      :encrypted_birthdate => 'I3d691B2PtFXLx15kO067g=='
    )

    assert_equal 'Fido', pet.name
    assert_equal 'Fido the Dog', pet.nickname
    assert_equal '2011-07-09', pet.birthdate
  end

  def test_marshalling_backwards_compatibility
    pet = LegacyMarshallingPet.create!(
      :name => 'Fido',
      :encrypted_nickname => '7RwoT64in4H+fGVBPYtRcN0K4RtriIy1EP4nDojUa8g=',
      :encrypted_birthdate => 'bSp9sJhXQSp2QlNZHiujtcK4lRVBE8HQhn1y7moQ63bGJR20hvRSZ73ePAmm+wc5'
    )

    assert_equal 'Fido', pet.name
    assert_equal 'Mummy\'s little helper', pet.nickname

    assert_equal Date.new(2011, 7, 9), pet.birthdate
  end

  private

  def create_tables
    silence_stream(STDOUT) do
      ActiveRecord::Schema.define(:version => 1) do
        create_table :legacy_nonmarshalling_pets do |t|
          t.string :name
          t.string :encrypted_nickname
          t.string :encrypted_birthdate
          t.string :salt
        end
        create_table :legacy_marshalling_pets do |t|
          t.string :name
          t.string :encrypted_nickname
          t.string :encrypted_birthdate
          t.string :salt
        end
      end
    end
  end
end

ActiveRecord::Base.establish_connection :adapter => 'sqlite3', :database => ':memory:'
