# A debhelper build system class for building PHP PEAR based
# projects.
#
# Copyright: © 2011 Mathieu Parent
# License: GPL-2+

package Debian::Debhelper::Buildsystem::phppear;

use strict;
use warnings;
use Cwd ();
use Debian::Debhelper::Dh_Lib qw(error complex_doit);
use base "Debian::Debhelper::Buildsystem";

sub DESCRIPTION {
	"PHP PEAR (package.xml)"
}

sub new {
	my $class=shift;
	my $this=$class->SUPER::new(@_);
	# Out of source tree building is prefered.
	$this->prefer_out_of_source_building(@_);
	return $this;
}

sub check_auto_buildable {
	my $this=shift;
	return -e $this->get_sourcepath("package.xml") ? 1 : 0;
}

# Local functions
sub _shell_exec {
	my $command=shift;
	my $output = `$command`;
	if ($? == -1) {
		error("$command failed to execute: $!");
	}
	elsif ($? & 127) {
		error("$command died with signal ".($? & 127));
	}
	elsif ($? != 0) {
		error("$command returned exit code ".($? >> 8));
	}
	return $output;
}

# Get peardir (relative to sourcedir)
sub _get_peardir {
	my $this=shift;
	return $this->get_sourcedir()."/".$this->{phppkginfo_package}."-".$this->{phppkginfo_version};
}

sub _get_mainpackage {
	my @packages = split("\n", _shell_exec('dh_listpackages'));
	if (scalar(@packages) != 1) {
		error("only one binary package supported, got ".scalar(@packages));
	}
	return $packages[0];
}

sub _pear {
	my $this=shift;
	my $command=shift;
	my $destdir=shift;
	my $params=@_;
	$this->doit_in_sourcedir("/usr/bin/pear",
		"-c", "debian/pearrc", # Allows local override
		"-d", "download_dir=/tmp",
		"-d", "include_path=/usr/share/php",
		"-d", "php_bin=/usr/bin/php",
		"-d", "bin_dir=/usr/bin",
		"-d", "php_dir=/usr/share/php",
		"-d", "data_dir=/usr/share/php/data",
		"-d", "doc_dir=/usr/share/doc/".$this->_get_mainpackage(),
		"-d", "test_dir=/usr/share/php/tests",
		"$command",
		"--offline",
		"--nodeps",
		"-P", $destdir,
		$this->_get_peardir()."/package.xml");


}

sub pre_building_step {
	my $this=shift;
	my ($step)=@_;
	if (!$this->{phppkginfo_package}) {
		$this->{phppkginfo_package} = _shell_exec('/usr/share/pkg-php-tools/scripts/phppkginfo -d package .');
	}
	if (!$this->{phppkginfo_version}) {
		$this->{phppkginfo_version} = _shell_exec('/usr/share/pkg-php-tools/scripts/phppkginfo -d version .');
	}
}

sub install {
	my $this=shift;
	my $destdir=shift;

	$this->doit_in_sourcedir("cp", "package.xml", $this->_get_peardir()."/package.xml");
	$this->doit_in_sourcedir('sed', '-i', '-e', 's/md5sum="[^"]*"//', $this->_get_peardir()."/package.xml");
	$this->mkdir_builddir();
	$this->_pear("install", $destdir);
	# remove unwanted files
	$this->doit_in_sourcedir("rm", "-f", $destdir."/usr/share/php/.filemap");
	$this->doit_in_sourcedir("rm", "-f", $destdir."/usr/share/php/.lock");
	$this->doit_in_sourcedir("rm", "-rf", $destdir."/usr/share/php/.channels");
	$this->doit_in_sourcedir("rm", "-rf", $destdir."/usr/share/php/.depdblock");
	$this->doit_in_sourcedir("rm", "-rf", $destdir."/usr/share/php/.depdb");
	$this->doit_in_sourcedir("rm", "-rf", $destdir."/usr/share/php/.registry/.channel.pecl.php.net");
	$this->doit_in_sourcedir("rm", "-rf", $destdir."/usr/share/php/.registry/.channel.doc.php.net");
	$this->doit_in_sourcedir("rm", "-rf", $destdir."/usr/share/php/.registry/.channel.__uri");
	$this->doit_in_sourcedir("rm", "-f", $destdir."/usr/share/doc/".$this->_get_mainpackage()."/".$this->{phppkginfo_package}."/COPYING");
	if (-d $destdir."/usr/share/doc/") {
		$this->doit_in_sourcedir("find", $destdir."/usr/share/doc/", "-type", "d", "-empty", "-delete");
	}

	# add package.xml and changelog to doc dir
	$this->doit_in_sourcedir("mkdir", "-p", $destdir."/usr/share/doc/".$this->_get_mainpackage());
	$this->doit_in_sourcedir("cp", "package.xml", $destdir."/usr/share/doc/".$this->_get_mainpackage());
	complex_doit("/usr/share/pkg-php-tools/scripts/phppkginfo -d changelog ".$this->get_sourcedir()." > ".$destdir."/usr/share/doc/".$this->_get_mainpackage()."/changelog");
}

sub clean {
	my $this=shift;
	$this->doit_in_sourcedir("rm", "-f", $this->_get_peardir()."/package.xml");
	$this->rmdir_builddir();
}

1
