/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "support.h"

#include <libsyncml/syncml.h>
#include <libsyncml/sml_error_internals.h>

START_TEST (error_create)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	smlErrorSet(&error, SML_ERROR_GENERIC, "test%i", 1);
	fail_unless(error != NULL, NULL);
	fail_unless(error->type == SML_ERROR_GENERIC, NULL);
	fail_unless(!strcmp(error->message, "test1"), NULL);
	fail_unless(smlErrorIsSet(&error), NULL);
	
	smlErrorDeref(&error);
	fail_unless(error == NULL, NULL);
}
END_TEST

START_TEST (error_create_null)
{
	setup_testbed(NULL);
	
	smlErrorSet(NULL, SML_ERROR_GENERIC, "test%i", 1);
}
END_TEST

START_TEST (error_free_null)
{
	setup_testbed(NULL);
	
	smlErrorDeref(NULL);
}
END_TEST

START_TEST (error_free_null2)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	smlErrorDeref(&error);
}
END_TEST

START_TEST (error_free)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	smlErrorSet(&error, SML_ERROR_GENERIC, "test");
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	fail_unless(error == NULL, NULL);
}
END_TEST

START_TEST (error_check_null)
{
	setup_testbed(NULL);
	
	fail_unless(smlErrorIsSet(NULL) == FALSE, NULL);
}
END_TEST

START_TEST (error_check_null2)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	fail_unless(smlErrorIsSet(&error) == FALSE, NULL);
}
END_TEST

START_TEST (error_check)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	smlErrorSet(&error, SML_ERROR_GENERIC, "test");
	fail_unless(smlErrorIsSet(&error) == TRUE, NULL);
}
END_TEST

START_TEST (error_check2)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	smlErrorSet(&error, SML_NO_ERROR, NULL);
	fail_unless(smlErrorIsSet(&error) == FALSE, NULL);
}
END_TEST

START_TEST (error_update_null)
{
	setup_testbed(NULL);
	
	smlErrorUpdate(NULL, NULL);
}
END_TEST

START_TEST (error_update_null2)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	smlErrorUpdate(&error, NULL);
}
END_TEST

START_TEST (error_update)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	smlErrorSet(&error, SML_ERROR_GENERIC, "test");
	smlErrorUpdate(&error, "test2%i", 1);
	fail_unless(!strcmp(error->message, "test21"), NULL);
}
END_TEST

START_TEST (error_update2)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	smlErrorSet(&error, SML_ERROR_GENERIC, "test");
	smlErrorUpdate(&error, "test2%s", error->message);
	fail_unless(!strcmp(error->message, "test2test"), NULL);
}
END_TEST

START_TEST (error_set_null)
{
	setup_testbed(NULL);
	
	smlErrorSet(NULL, SML_NO_ERROR, NULL);

}
END_TEST

START_TEST (error_set_null2)
{
	
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	smlErrorUpdate(&error, NULL);
}
END_TEST

START_TEST (error_duplicate_null)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	smlErrorSet(&error, SML_ERROR_GENERIC, "asd");
	smlErrorDuplicate(NULL, &error);
}
END_TEST

START_TEST (error_class)
{
	setup_testbed(NULL);
	
	fail_unless(smlErrorGetClass(NULL) == SML_ERRORCLASS_SUCCESS, NULL);
	
	SmlError *error = NULL;
	fail_unless(smlErrorGetClass(&error) == SML_ERRORCLASS_SUCCESS, NULL);
	
	smlErrorSet(&error, SML_NO_ERROR, "test%i", 1);
	fail_unless(smlErrorGetClass(&error) == SML_ERRORCLASS_SUCCESS, NULL);
	smlErrorDeref(&error);
	
	smlErrorSet(&error, SML_ERROR_RETRY, "test%i", 1);
	fail_unless(smlErrorGetClass(&error) == SML_ERRORCLASS_RETRY, NULL);
	smlErrorDeref(&error);
	
	smlErrorSet(&error, SML_ERROR_GENERIC, "test%i", 1);
	fail_unless(smlErrorGetClass(&error) == SML_ERRORCLASS_FATAL, NULL);
	smlErrorDeref(&error);
	
	fail_unless(error == NULL, NULL);
}
END_TEST

Suite *error_suite(void)
{
	Suite *s = suite_create("Error");
	//Suite *s2 = suite_create("Error");
	
	create_case(s, "error_create", error_create);
	create_case(s, "error_create_null", error_create_null);
	create_case(s, "error_free_null", error_free_null);
	create_case(s, "error_free_null2", error_free_null2);
	create_case(s, "error_free", error_free);
	create_case(s, "error_check_null", error_check_null);
	create_case(s, "error_check_null2", error_check_null2);
	create_case(s, "error_check", error_check);
	create_case(s, "error_check2", error_check2);
	create_case(s, "error_update_null", error_update_null);
	create_case(s, "error_update_null2", error_update_null2);
	create_case(s, "error_update", error_update);
	create_case(s, "error_update2", error_update2);
	create_case(s, "error_set_null", error_set_null);
	create_case(s, "error_set_null2", error_set_null2);
	create_case(s, "error_duplicate_null", error_duplicate_null);
	create_case(s, "error_class", error_class);
	
	return s;
}

int main(void)
{
	int nf;
	
	Suite *s = error_suite();
	
	SRunner *sr;
	sr = srunner_create(s);

	srunner_run_all(sr, CK_VERBOSE);
	nf = srunner_ntests_failed(sr);
	srunner_free(sr);
	return (nf == 0) ? 0 : 1;
}
