		/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package javax.management.monitor;

import javax.management.ObjectName;
import javax.management.NotificationBroadcasterSupport;
import javax.management.MBeanRegistration;
import javax.management.MBeanServer;
import javax.management.MBeanInfo;
import javax.management.MBeanAttributeInfo;
import java.util.Timer;
import java.util.TimerTask;
import java.util.ArrayList;

import mx4j.log.Logger;
import mx4j.log.Log;

/**
 * Class monitor. Parent class of all Monitoring classes
 *
 * In JMX 1.2 monitors can observe multiple objects.  This makes the
 * monitor execution a bit more complex, and synchronization is of
 * particular concern.  I haven't paid much attention to synchronization so far,
 * but plan to address it in the test cases applied to this class.  I believe
 * the only piece that needs synchronization is the objectNames list of monitored
 * objects -markmcbride
 * @see MonitorMBean
 * @author <a href="mailto:tibu@users.sourceforge.net">Carlos Quiroz</a>
 * @author <a href="mailto:markmcbride@users.sourceforge.net">Mark McBride</a>
 * @version $Revision: 1.11 $
 */
public abstract class Monitor extends NotificationBroadcasterSupport
implements MonitorMBean, MBeanRegistration {
		/** @deprecated */
		protected int alreadyNotified;
		protected int alreadyNotifieds[];
		protected static final int capacityIncrement = 16;
		/** @deprecated */
		protected String dbgTag;
		protected int elementCount;
		protected static final int OBSERVED_ATTRIBUTE_ERROR_NOTIFIED = 2;
		protected static final int OBSERVED_ATTRIBUTE_TYPE_ERROR_NOTIFIED = 4;
		protected static final int OBSERVED_OBJECT_ERROR_NOTIFIED = 1;
		protected static final int RESET_FLAGS_ALREADY_NOTIFIED = 0;
		protected static final int RUNTIME_ERROR_NOTIFIED = 8;
		protected MBeanServer server;

		/** Object names of the monitored MBeans */
		ArrayList objectNames = new ArrayList();

		/** Monitored attribute */
		String attribute;

		/** Granularity of the monitoring in milliseconds */
		long granularity = 10000;

		/** Indicates whether the Monitor is active */
		boolean isActive = false;

		transient boolean errorNotified = false, mBeanFound = true;

		static Timer notificationTimer = new Timer();

		static long notificationID = 0;

		MonitorTask monitorTask = null;

		static synchronized long createNotificationID() {
				return ++notificationID;
		}

		abstract void executeMonitor(ObjectName objectName,Object attribute);

		Logger getLogger() {
				return Log.getLogger(getClass().getName());
		}

		private class MonitorTask extends TimerTask {
				public void run() {
						if (isActive()) {
								// if no objects are set to be monitored then send a jmx.monitor.error.mbean
								// notification to the listeners
								if(objectNames.size() == 0){
										getLogger().warn(new StringBuffer("Monitor ").append(Monitor.this.toString()).append(" no objects specified for monitoring"));
										if(!errorNotified){
												notifyListeners(MonitorNotification.OBSERVED_OBJECT_ERROR,null);
										}
										errorNotified = true;
								}
								// loop over all monitored objects in JMX 1.2
								// note that on an error we continue to the next object
								// rather than just returning as we did in JMX 1.1
								for(int i = 0; i < objectNames.size(); i++){
										ObjectName objectName = (ObjectName)objectNames.get(i);
										try {
												if (objectName == null) {
														getLogger().warn(new StringBuffer("Monitor ").append(Monitor.this.toString()).append(" object name ").append(objectName).append(" not found").toString());
														if (!errorNotified) {
																notifyListeners(MonitorNotification.OBSERVED_OBJECT_ERROR, objectName);
														}
														errorNotified = true;
														continue;
												}
												getLogger().info("Execute monitor " + Monitor.this.toString());
												if (!server.isRegistered(objectName)) {
														getLogger().warn(new StringBuffer("Monitor ").append(Monitor.this.toString()).append(" object name ").append(objectName).append(" not found").toString());
														if (mBeanFound) {
																mBeanFound = false;
																notifyListeners(MonitorNotification.OBSERVED_OBJECT_ERROR, objectName);
														}
														continue;
												}
												mBeanFound = true;
												if (errorNotified) {
														continue;
												}
												MBeanInfo info = server.getMBeanInfo(objectName);
												MBeanAttributeInfo[] attributes =info.getAttributes();
												boolean found = false;
												if (attributes != null) {
														for (int j=0;j<attributes.length;j++) {
																if (attributes[j].getName().equals(attribute)) {
																		found = true;
																}
														}
												}
												if (!found) {
														getLogger().warn(new StringBuffer("Monitor ").append(Monitor.this.toString()).append(" attribute ").append(attribute).append(" not found").toString());
														errorNotified = true;
														notifyListeners(MonitorNotification.OBSERVED_ATTRIBUTE_ERROR, objectName, attribute);
														continue;
												}
												Object attributeValue = server.getAttribute(objectName, attribute);
												if (attributeValue == null) {
														getLogger().warn(new StringBuffer("Monitor ").append(Monitor.this.toString()).append(" attribute ").append(attribute).append(" is null").toString());
														errorNotified = true;
														notifyListeners(MonitorNotification.OBSERVED_ATTRIBUTE_TYPE_ERROR, objectName, attribute);
														continue;
												}
												executeMonitor(objectName,attributeValue);
										} catch (Exception e) {
												getLogger().warn(new StringBuffer("Monitor ").append(Monitor.this.toString()).append(" object name ").append(objectName).append(" not found").toString());
												errorNotified = true;
												notifyListeners(MonitorNotification.RUNTIME_ERROR, objectName, attribute, e);
										}
								}
						}
				}
		}

		public abstract void start();

		synchronized void doStart() {
				if (!isActive) {
						getLogger().info(new StringBuffer("Starting monitor ").append(this).append(" every ").append(granularity).append(" milliseconds").toString());
						this.isActive = true;
						errorNotified = false;
						// updated to create a new MonitorTask every time
						monitorTask = new MonitorTask();
						notificationTimer.scheduleAtFixedRate(monitorTask, 0, granularity);
				}
		}

		public abstract void stop();

		synchronized void doStop() {
				if (isActive) {
						getLogger().info("Stopping monitor " + toString());
						this.isActive = false;
						monitorTask.cancel();
				}
		}

		/**
		 * @deprecated
		 */
		public synchronized ObjectName getObservedObject() {
				if(objectNames.size() == 0) return null;
				return (ObjectName)objectNames.get(0);
		}

		public synchronized ObjectName[]  getObservedObjects() {
				if(objectNames.size() == 0) return null;
				ObjectName [] names = new ObjectName[objectNames.size()];
				for(int i = 0; i < names.length; i++){
						names[i] = (ObjectName)objectNames.get(i);
				}
				return names;
		}

		public synchronized void addObservedObject(ObjectName objectName)  throws java.lang.IllegalArgumentException {
				if(objectName == null){
						throw new IllegalArgumentException("The observed object name cannot be null");
				}
				if(objectNames.contains(objectName)){
                                  // changed behaviour as JMX 1.2 No exception is thrown
						//throw new IllegalArgumentException(objectName.getCanonicalName() + " is already being monitored");
                                                return;
				}
				errorNotified = false;



				objectNames.add(objectNames.size(),objectName);
				//objectNames.add(objectName);
		}

		/**
		 * @deprecated
		 */
		public synchronized void setObservedObject(ObjectName objectName)  throws java.lang.IllegalArgumentException {
				addObservedObject(objectName);
		}

		public String getObservedAttribute() {
				return attribute;
		}

		public synchronized void setObservedAttribute(String attribute) throws java.lang.IllegalArgumentException {
				if (attribute == null) {
						throw new IllegalArgumentException("The observed attribute cannot be null");
				}
				errorNotified = true;
				this.attribute = attribute;
		}

		public long getGranularityPeriod() {
				return granularity;
		}

		public void setGranularityPeriod(long period) throws java.lang.IllegalArgumentException {
				if (period <=0) {
						throw new IllegalArgumentException("The monitoring period can't be negative or zero");
				}
				this.granularity = period;
		}

		public synchronized boolean isActive() {
				return isActive;
		}

		public ObjectName preRegister(MBeanServer server, ObjectName name) throws Exception {
				getLogger().info("Pre register of monitor " + toString());
				this.server = server;
				errorNotified = false;
				return name;
		}

		public void postRegister(Boolean registrationDone) {}

		public void preDeregister() throws Exception {
				getLogger().info("Pre deregister of monitor " + toString());
				this.stop();
		}

		public void postDeregister() {}

		void notifyListeners(String type, ObjectName objectName) {
				MonitorNotification not = new MonitorNotification(type, this, createNotificationID(), System.currentTimeMillis(), "", objectName, null, null, null);
				sendNotification(not);
		}

		void notifyListeners(String type, ObjectName objectName, String attribute) {
				MonitorNotification not = new MonitorNotification(type, this, createNotificationID()
				, System.currentTimeMillis(), "", objectName, attribute, null, null);
				sendNotification(not);
		}

		void notifyListeners(String type, ObjectName objectName, String attribute, Throwable e) {
				MonitorNotification not = new MonitorNotification(type, this, createNotificationID()
				, System.currentTimeMillis(), e.toString(), objectName, attribute, null, null);
				sendNotification(not);
		}

		public synchronized String toString() {
				StringBuffer sb = new StringBuffer();
				if(objectNames == null || objectNames.size() == 0){
						sb.append("?");
				}else{
						for(int i = 0; i < objectNames.size(); i++){
								sb.append(objectNames.get(i).toString()).append(",");
						}
						sb.setLength(sb.length() - 1);
				}
				sb.append(" attribute ").append(attribute);
				return sb.toString();
		}

		public synchronized boolean containsObservedObject(ObjectName objectName){
				return objectNames.contains(objectName);
		}

		public synchronized void removeObservedObject(ObjectName objectName){
                  // The behaviour changed in JMX 1.2. No exception is thrown if the object is not found
				if(objectNames.contains(objectName)){
                                  objectNames.remove(objectName);
				}

		}
}
