/*
Copyright (c) 2008, Dennis M. Sosnoski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.
 * Neither the name of JiBX nor the names of its contributors may be used
   to endorse or promote products derived from this software without specific
   prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package org.jibx.schema.support;

import java.util.ArrayList;

import org.jibx.runtime.IListItemDeserializer;
import org.jibx.runtime.IMarshallingContext;
import org.jibx.runtime.IUnmarshallingContext;
import org.jibx.runtime.IXMLWriter;
import org.jibx.runtime.JiBXException;
import org.jibx.runtime.QName;
import org.jibx.runtime.Utility;
import org.jibx.runtime.impl.MarshallingContext;
import org.jibx.runtime.impl.UnmarshallingContext;
import org.jibx.schema.elements.SchemaElement;

/**
 * Qualified name serializer/deserializer for use in schema definitions. This
 * uses special handling for values using the default namespace, checking if the
 * schema being processed is a no-namespace schema being included into a schema
 * with a namespace. If it is, the including namespace is used as the default.
 * 
 * @author Dennis M. Sosnoski
 */
public class QNameConverter
{
    /**
     * Qualified name deserializer method for use within schema definitions.
     * 
     * @param text value text
     * @param ictx unmarshalling context
     * @return created class instance
     * @throws JiBXException on error in unmarshalling
     */
    public static QName deserialize(String text, IUnmarshallingContext ictx)
        throws JiBXException {
        if (text == null) {
            return null;
        } else {
            
            // check for prefix used in text representation
            int split = text.indexOf(':');
            if (split > 0) {
                
                // strip off prefix
                String prefix = text.substring(0, split);
                text = text.substring(split+1);
                
                // look up the namespace URI associated with the prefix
                String uri =
                    ((UnmarshallingContext)ictx).getNamespaceUri(prefix);
                if (uri == null) {
                    throw new JiBXException("Undefined prefix " + prefix);
                } else {
                    
                    // create an instance of class to hold all components
                    return new QName(uri, prefix, text);
                    
                }
                
            } else {
                
                // check for inside no-namespace schema imported into namespace
                String uri = ((UnmarshallingContext)ictx).getNamespaceUri(null);
                if (uri == null) {
                    int depth = ictx.getStackDepth();
                    for (int i = 0; i < depth; i++) {
                        Object obj = ictx.getStackObject(i);
                        if (obj instanceof SchemaElement) {
                            SchemaElement schema = (SchemaElement)obj;
                            String ens = schema.getEffectiveNamespace();
                            if (ens != schema.getTargetNamespace()) {
                                uri = ens;
                            }
                            break;
                        }
                    }
                }
                
                // create it using the default namespace URI
                if (uri != null && uri.length() == 0) {
                    uri = null;
                }
                return new QName(uri, "", text);
            }
        }
    }
    
    /**
     * Qualified name serializer method for use within schema definitions.
     * 
     * @param qname value to be serialized
     * @param ictx unmarshalling context
     * @return created class instance
     * @throws JiBXException on error in marshalling
     */
    public static String serialize(QName qname, IMarshallingContext ictx)
        throws JiBXException {
        if (qname == null) {
            return null;
        } else {
            
            // check for specified prefix
            IXMLWriter ixw = ((MarshallingContext)ictx).getXmlWriter();
            int index = -1;
            String uri = qname.getUri();
            if (uri == null) {
                uri = "";
            }
            String prefix = qname.getPrefix();
            if (prefix != null) {
                
                // see if prefix already defined in document with correct URI
                int tryidx = ixw.getPrefixIndex(prefix);
                if (tryidx >= 0 &&
                    uri.equals(ixw.getNamespaceUri(tryidx))) {
                    index = tryidx;
                } else if ("".equals(prefix)) {
                    
                    // check for no-namespace schema imported into namespace
                    int depth = ictx.getStackDepth();
                    for (int i = 0; i < depth; i++) {
                        Object obj = ictx.getStackObject(i);
                        if (obj instanceof SchemaElement) {
                            SchemaElement schema = (SchemaElement)obj;
                            String ens = schema.getEffectiveNamespace();
                            if (ens != schema.getTargetNamespace()) {
                                return qname.getName();
                            } else {
                                break;
                            }
                        }
                    }
                    
                }
            }
            
            // check if need to lookup prefix for namespace
            if (index < 0) {
                
                // prefix not defined, find the namespace index in binding
                if (uri == null) {
                    uri = "";
                }
                String[] nss = ixw.getNamespaces();
                for (int i = 0; i < nss.length; i++) {
                    if (nss[i].equals(uri)) {
                        index = i;
                        break;
                    }
                }
                if (index < 0) {
                    
                    // namespace not in binding, check extensions
                    String[][] nsss = ixw.getExtensionNamespaces();
                    if (nsss != null) {
                        int base = nss.length;
                        outer: for (int i = 0; i < nsss.length; i++) {
                            nss = nsss[i];
                            for (int j = 0; j < nss.length; j++) {
                                if (nss[j].equals(uri)) {
                                    index = base + j;
                                    break outer;
                                }
                            }
                            base += nss.length;
                        }
                    }
                }
            }
            
            // check if prefix is alread defined in document with correct URI
            if (index >= 0) {
                
                // get prefix defined for namespace
                prefix = ixw.getNamespacePrefix(index);
                if (prefix == null) {
                    throw new JiBXException("Namespace URI " + qname.getUri() +
                        " cannot be used since it is not active");
                } else if (prefix.length() > 0) {
                    return prefix + ':' + qname.getName();
                } else {
                    return qname.getName();
                }
                
            } else {
                throw new JiBXException("Unknown namespace URI " + qname.getUri());
            }
        }
    }

    /**
     * Qualified name list deserializer method for use within schema
     * definitions.
     * 
     * @param text value text
     * @param ictx unmarshalling context
     * @return array of instances
     * @throws JiBXException on error in marshalling
     */
    public static QName[] deserializeList(String text,
        final IUnmarshallingContext ictx) throws JiBXException {
        
        // use basic qualified name deserializer to handle items
        IListItemDeserializer ldser = new IListItemDeserializer() {
            public Object deserialize(String text) throws JiBXException {
                return QNameConverter.deserialize(text, ictx);
            }
        };
        ArrayList list = Utility.deserializeList(text, ldser);
        if (list == null) {
            return null;
        } else {
            return (QName[])list.toArray(new QName[list.size()]);
        }
    }
    
    /**
     * Qualified name list serializer method for use within schema definitions.
     * 
     * @param qnames array of names to be serialized
     * @param ictx unmarshalling context
     * @return generated text
     * @throws JiBXException on error in marshalling
     */
    public static String serializeList(QName[] qnames, IMarshallingContext ictx)
        throws JiBXException {
        StringBuffer buff = new StringBuffer();
        for (int i = 0; i < qnames.length; i++) {
            QName qname = qnames[i];
            if (qname != null) {
                if (buff.length() > 0) {
                    buff.append(' ');
                }
                buff.append(serialize(qname, ictx));
            }
        }
        return buff.toString();
    }
}