/**
 * GUI Commands
 * Copyright 2004 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: AbstractSaveAsCommand.java,v 1.6 2007/01/11 08:28:40 pietschy Exp $
 */
package org.pietschy.command.file;

import org.pietschy.command.CommandManager;

import javax.swing.*;
import javax.swing.filechooser.FileFilter;
import java.awt.*;
import java.io.File;

/**
 * This command provides generic behaviour for the standard "save as" operation.  On execution, the
 * command will display a {@link JFileChooser} and if successful {@link #performSave} will be
 * invoked.  If the file selected by the user already exists the command will prompt the user to
 * confirm before {@link #performSave} is invoked.  The message displayed to the user can be
 * configured by overriding {@link #getFileExistsMessage} and {@link #getFileExistsTitle}.
 * <p/>
 * If the filters specified are instances of {@link ExtensionFileFilter}, the command will also
 * ensure the extension of the file is correct.
 *
 * @author andrewp
 * @version $Revision: 1.6 $
 */
public abstract class
AbstractSaveAsCommand
extends AbstractFileCommand
{
   private static final String _ID_ = "$Id: AbstractSaveAsCommand.java,v 1.6 2007/01/11 08:28:40 pietschy Exp $";


   /**
    * Creates new instance with the specified id and {@link FileFilter}.  This command is
    * bound to the default command manager.
    *
    * @param id     the command id.
    * @param filter the {@link FileFilter} for the {@link JFileChooser} to use.  If it is an
    *               instance of {@link ExtensionFileFilter} the command will ensure the selected file has
    *               the correct extension.
    */
   public AbstractSaveAsCommand(String id, FileFilter filter)
   {
      this(id, new FileFilter[]{filter});
   }

   /**
    * Creates new instance with the specified id and {@link FileFilter} list.  This command is
    * bound to the default command manager.
    *
    * @param id      the command id.
    * @param filters the {@link FileFilter} list for the {@link JFileChooser} to use.  If they are
    *                instances of {@link ExtensionFileFilter} the command will ensure the selected file has
    *                the correct extension.
    */
   public AbstractSaveAsCommand(String id, FileFilter[] filters)
   {
      this(CommandManager.defaultInstance(), id, filters);
   }


   /**
    * Creates new instance with the specified id and {@link FileFilter}.
    *
    * @param id     the commands id.
    * @param filter the {@link FileFilter} for the {@link JFileChooser} to use.  If it is an
    *               instance of {@link ExtensionFileFilter} the command will ensure the selected file has
    *               the correct extension.
    */
   public AbstractSaveAsCommand(CommandManager manager, String id, FileFilter filter)
   {
      this(manager, id, new FileFilter[]{filter});
   }

   /**
    * Creates new instance with the specified id and {@link FileFilter FileFilters}.
    *
    * @param id      the commands id.
    * @param filters the {@link FileFilter} list for the {@link JFileChooser} to use.  If they are
    *                instances of {@link ExtensionFileFilter} the command will ensure the selected file has
    *                the correct extension.
    */
   public AbstractSaveAsCommand(CommandManager manager, String id, FileFilter[] filters)
   {
      super(manager, id, filters);
   }


   protected int
   showChooserDialog(JFileChooser chooser, Window invoker)
   {
      chooser.setMultiSelectionEnabled(false);
      return chooser.showSaveDialog(invoker);
   }

   protected void
   performFileAction(File[] selectedFiles, JFileChooser chooser, Window invoker)
   {
      File file = verifyFileExtension(selectedFiles[0], chooser.getFileFilter());

      if (!file.exists() || confirmOverwrite(invoker, file))
      {
         performSave(file);
      }
   }

   /**
    * Invoked to verify the extension of the file selected by the user.  This is provided to catch
    * the case where the user has typed the file name but has ommited the extension.
    * <p/>
    * This implementation checks if the {@link FileFilter} is an instance of
    * {@link ExtensionFileFilter}.  If it is, it invokes
    * {@link ExtensionFileFilter#checkAndAddExtension} on the file, otherwise the file is returned
    * as is.
    *
    * @param file           the file the users has selected, that may or may not have an extension.
    * @param selectedFilter the {@link FileFilter} selected in the chooser.
    * @return a filename with the correct extension.
    * @see ExtensionFileFilter
    */
   protected File
   verifyFileExtension(File file, FileFilter selectedFilter)
   {
      if (selectedFilter instanceof ExtensionFileFilter)
         return ((ExtensionFileFilter) selectedFilter).checkAndAddExtension(file);
      else
         return file;
   }

   /**
    * This method is called to confirm the save if the selected file already exists.  By default
    * this method invokes {@link javax.swing.JOptionPane#showConfirmDialog(java.awt.Component, Object)}
    * with {@link #getFileExistsTitle()} and {@link #getFileExistsMessage(java.io.File)}.
    * <p/>
    * Subclasses can override for complete control over the confirmation process.
    *
    * @param invoker the invoker window.
    * @param file    the file that is to be overwritten.
    * @return <code>true</code> to overwrite the file, <code>false</code> to cancel the operation.
    */
   protected boolean confirmOverwrite(Window invoker, File file)
   {
      boolean continueWithSave;
      int response = JOptionPane.showConfirmDialog(invoker,
                                                   getFileExistsMessage(file),
                                                   getFileExistsTitle(),
                                                   JOptionPane.YES_NO_CANCEL_OPTION);

      continueWithSave = (response == JOptionPane.YES_OPTION);
      return continueWithSave;
   }

   /**
    * Gets the title to display in the dialog that confirms file overwrite.  This defaults
    * to the string <tt>"Confirm file overwrite"</tt>.
    *
    * @return the string <tt>"Confirm file overwrite"</tt>.
    */
   private String
   getFileExistsTitle()
   {
      return "Confirm file overwrite";
   }

   /**
    * Gets the text to display in the dialog that confirms file overwrite.  This defaults
    * to the string <tt>"File '" + file.getName() + "' already exists.  Overwrite?"</tt>
    *
    * @return the string <tt>"File '" + file.getName() + "' already exists.  Overwrite?"</tt>.
    */
   protected String
   getFileExistsMessage(File file)
   {
      return "File '" + file.getName() + "' already exists.  Overwrite?";
   }

   /**
    * Called to perform the save operation.  Subclasses must implement this method to perform
    * the required save behaviour.
    *
    * @param file the file to save.
    */
   protected abstract void
   performSave(File file);

}

