/*

    DSP operations
    Copyright (C) 1998-2001 Jussi Laako

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#ifndef FREEBSD
    #ifndef _ISOC9X_SOURCE
        #define _ISOC9X_SOURCE
    #endif
#endif


#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <math.h>
#include <float.h>

#include "dsp/DSPOp.hh"


inline int FloatCompare (const void *vpValue1, const void *vpValue2)
{
    if (*(float *)vpValue1 == *(float *)vpValue2)
    {
        return 0;
    }
    if (*(float *)vpValue1 < *(float *)vpValue2)
    {
        return -1;
    }
    else
    {
        return 1;
    }
}


inline int DoubleCompare (const void *vpValue1, const void *vpValue2)
{
    if (*(double *)vpValue1 == *(double *)vpValue2)
    {
        return 0;
    }
    if (*(double *)vpValue1 < *(double *)vpValue2)
    {
        return -1;
    }
    else
    {
        return 1;
    }
}


signed long clDSPOp::Round (float fValue)
{
    signed long slTemp;

    // nearbyintf() seems to be buggy in some glibc versions
    //#ifndef _ISOC9X_SOURCE
        slTemp = (fValue >= 0.0F) ?
            ((signed long) (fValue + 0.5F)) : ((signed long) (fValue - 0.5F));
    /*#else
        slTemp = (signed long) nearbyintf(fValue);
    #endif*/
    return slTemp;
}


signed long clDSPOp::Round (double dValue)
{
    signed long slTemp;

    // nearbyint() seems to be buggy in some glibc versions
    //#ifndef _ISOC9X_SOURCE
        slTemp = (dValue >= 0.0) ?
            ((signed long) (dValue + 0.5)) : ((signed long) (dValue - 0.5));
    /*#else
        slTemp = (signed long) nearbyint(dValue);
    #endif*/
    return slTemp;
}


inline void clDSPOp::Cart2Polar (float *fpMagn, float *fpPhase,
    float fReal, float fImag)
{
    #ifndef _ISOC9X_SOURCE
        *fpMagn = sqrtf(fReal * fReal + fImag * fImag);
    #else
        *fpMagn = hypotf(fReal, fImag);
    #endif
    *fpPhase = atan2f(fImag, fReal);
}


inline void clDSPOp::Cart2Polar (double *dpMagn, double *dpPhase,
    double dReal, double dImag)
{
    #ifndef _ISOC9X_SOURCE
        *dpMagn = sqrt(dReal * dReal + dImag * dImag);
    #else
        *dpMagn = hypot(dReal, dImag);
    #endif
    *dpPhase = atan2(dImag, dReal);
}


inline void clDSPOp::Cart2Polar (float *fpMagn, float *fpPhase,
    const stpSCplx spCplx)
{
    #ifndef _ISOC9X_SOURCE
        *fpMagn = sqrtf(spCplx->R * spCplx->R + spCplx->I * spCplx->I);
    #else
        *fpMagn = hypotf(spCplx->R, spCplx->I);
    #endif
    *fpPhase = atan2f(spCplx->I, spCplx->R);
}


inline void clDSPOp::Cart2Polar (double *dpMagn, double *dpPhase,
    const stpDCplx spCplx)
{
    #ifndef _ISOC9X_SOURCE
        *dpMagn = sqrt(spCplx->R * spCplx->R + spCplx->I * spCplx->I);
    #else
        *dpMagn = hypot(spCplx->R, spCplx->I);
    #endif
    *dpPhase = atan2(spCplx->I, spCplx->R);
}


inline void clDSPOp::Cart2Polar (stpSPolar spPolar, stpSCplx spCplx)
{
    #ifndef _ISOC9X_SOURCE
        spPolar->M = sqrtf(spCplx->R * spCplx->R + spCplx->I * spCplx->I);
    #else
        spPolar->M = hypotf(spCplx->R, spCplx->I);
    #endif
    spPolar->P = atan2f(spCplx->I, spCplx->R);
}


inline void clDSPOp::Cart2Polar (stpDPolar spPolar, stpDCplx spCplx)
{
    #ifndef _ISOC9X_SOURCE
        spPolar->M = sqrt(spCplx->R * spCplx->R + spCplx->I * spCplx->I);
    #else
        spPolar->M = hypot(spCplx->R, spCplx->I);
    #endif
    spPolar->P = atan2(spCplx->I, spCplx->R);
}


inline void clDSPOp::Cart2Polar (utpSCoord upCoord)
{
    #ifndef _ISOC9X_SOURCE
        upCoord->P.M = sqrtf(upCoord->C.R * upCoord->C.R +
            upCoord->C.I * upCoord->C.I);
    #else
        upCoord->P.M = hypotf(upCoord->C.R, upCoord->C.I);
    #endif
    upCoord->P.P = atan2f(upCoord->C.I, upCoord->C.R);
}


inline void clDSPOp::Cart2Polar (utpDCoord upCoord)
{
    #ifndef _ISOC9X_SOURCE
        upCoord->P.M = sqrt(upCoord->C.R * upCoord->C.R +
            upCoord->C.I * upCoord->C.I);
    #else
        upCoord->P.M = hypot(upCoord->C.R, upCoord->C.I);
    #endif
    upCoord->P.P = atan2(upCoord->C.I, upCoord->C.R);
}


inline void clDSPOp::Polar2Cart (float *fpReal, float *fpImag,
    float fMagn, float fPhase)
{
    *fpReal = fMagn * cosf(fPhase);
    *fpImag = fMagn * sinf(fPhase);
}


inline void clDSPOp::Polar2Cart (double *dpReal, double *dpImag,
    double dMagn, double dPhase)
{
    *dpReal = dMagn * cos(dPhase);
    *dpImag = dMagn * sin(dPhase);
}


inline void clDSPOp::Polar2Cart (stpSCplx spCplx, float fMagn, float fPhase)
{
    spCplx->R = fMagn * cosf(fPhase);
    spCplx->I = fMagn * sinf(fPhase);
}


inline void clDSPOp::Polar2Cart (stpDCplx spCplx, double dMagn, double dPhase)
{
    spCplx->R = dMagn * cos(dPhase);
    spCplx->I = dMagn * sin(dPhase);
}


inline void clDSPOp::Polar2Cart (stpSCplx spCplx, stpSPolar spPolar)
{
    spCplx->R = spPolar->M * cosf(spPolar->P);
    spCplx->I = spPolar->M * sinf(spPolar->P);
}


inline void clDSPOp::Polar2Cart (stpDCplx spCplx, stpDPolar spPolar)
{
    spCplx->R = spPolar->M * cos(spPolar->P);
    spCplx->I = spPolar->M * sin(spPolar->P);
}


inline void clDSPOp::Polar2Cart (utpSCoord upCoord)
{
    upCoord->C.R = upCoord->P.M * cosf(upCoord->P.P);
    upCoord->C.I = upCoord->P.M * sinf(upCoord->P.P);
}


inline void clDSPOp::Polar2Cart (utpDCoord upCoord)
{
    upCoord->C.R = upCoord->P.M * cos(upCoord->P.P);
    upCoord->C.I = upCoord->P.M * sin(upCoord->P.P);
}


inline void clDSPOp::CplxAdd (stpSCplx spCplxDest, const stpSCplx spCplxSrc)
{
    spCplxDest->R += spCplxSrc->R;
    spCplxDest->I += spCplxSrc->I;
}


inline void clDSPOp::CplxAdd (stpDCplx spCplxDest, const stpDCplx spCplxSrc)
{
    spCplxDest->R += spCplxSrc->R;
    spCplxDest->I += spCplxSrc->I;
}


inline void clDSPOp::CplxAdd (stpSCplx spCplxDest, const stpSCplx spCplxSrc1,
    const stpSCplx spCplxSrc2)
{
    spCplxDest->R = spCplxSrc1->R + spCplxSrc2->R;
    spCplxDest->I = spCplxSrc1->I + spCplxSrc2->I;
}


inline void clDSPOp::CplxAdd (stpDCplx spCplxDest, const stpDCplx spCplxSrc1,
    const stpDCplx spCplxSrc2)
{
    spCplxDest->R = spCplxSrc1->R + spCplxSrc2->R;
    spCplxDest->I = spCplxSrc1->I + spCplxSrc2->I;
}


inline void clDSPOp::CplxSub (stpSCplx spCplxDest, const stpSCplx spCplxSrc)
{
    spCplxDest->R -= spCplxSrc->R;
    spCplxDest->I -= spCplxSrc->I;
}


inline void clDSPOp::CplxSub (stpDCplx spCplxDest, const stpDCplx spCplxSrc)
{
    spCplxDest->R -= spCplxSrc->R;
    spCplxDest->I -= spCplxSrc->I;
}


inline void clDSPOp::CplxSub (stpSCplx spCplxDest, const stpSCplx spCplxSrc1,
    const stpSCplx spCplxSrc2)
{
    spCplxDest->R = spCplxSrc1->R - spCplxSrc2->R;
    spCplxDest->I = spCplxSrc1->I - spCplxSrc2->I;
}


inline void clDSPOp::CplxSub (stpDCplx spCplxDest, const stpDCplx spCplxSrc1,
    const stpDCplx spCplxSrc2)
{
    spCplxDest->R = spCplxSrc1->R - spCplxSrc2->R;
    spCplxDest->I = spCplxSrc1->I - spCplxSrc2->I;
}


inline void clDSPOp::CplxMul (stpSCplx spCplxDest, float fSrc)
{
    spCplxDest->R *= fSrc;
    spCplxDest->I *= fSrc;
}


inline void clDSPOp::CplxMul (stpDCplx spCplxDest, double dSrc)
{
    spCplxDest->R *= dSrc;
    spCplxDest->I *= dSrc;
}


inline void clDSPOp::CplxMul (stpSCplx spCplxDest, const stpSCplx spCplxSrc)
{
    float fReal;
    float fImag;

    fReal = spCplxDest->R * spCplxSrc->R - spCplxDest->I * spCplxSrc->I;
    fImag = spCplxDest->R * spCplxSrc->I + spCplxDest->I * spCplxSrc->R;
    spCplxDest->R = fReal;
    spCplxDest->I = fImag;
}


inline void clDSPOp::CplxMul (stpDCplx spCplxDest, const stpDCplx spCplxSrc)
{
    double dReal;
    double dImag;

    dReal = spCplxDest->R * spCplxSrc->R - spCplxDest->I * spCplxSrc->I;
    dImag = spCplxDest->R * spCplxSrc->I + spCplxDest->I * spCplxSrc->R;
    spCplxDest->R = dReal;
    spCplxDest->I = dImag;
}


inline void clDSPOp::CplxMul (stpSCplx spCplxDest, const stpSCplx spCplxSrc1,
    const stpSCplx spCplxSrc2)
{
    spCplxDest->R = 
        spCplxSrc1->R * spCplxSrc2->R - spCplxSrc1->I * spCplxSrc2->I;
    spCplxDest->I =
        spCplxSrc1->R * spCplxSrc2->I + spCplxSrc1->I * spCplxSrc2->R;
}


inline void clDSPOp::CplxMul (stpDCplx spCplxDest, const stpDCplx spCplxSrc1,
    const stpDCplx spCplxSrc2)
{
    spCplxDest->R =
        spCplxSrc1->R * spCplxSrc2->R - spCplxSrc1->I * spCplxSrc2->I;
    spCplxDest->I =
        spCplxSrc1->R * spCplxSrc2->I + spCplxSrc1->I * spCplxSrc2->R;
}


inline void clDSPOp::CplxMulC (stpSCplx spCplxDest, const stpSCplx spCplxSrc)
{
    float fReal;
    float fImag;

    fReal = spCplxDest->R * spCplxSrc->R - spCplxDest->I * (-spCplxSrc->I);
    fImag = spCplxDest->R * (-spCplxSrc->I) + spCplxDest->I * spCplxSrc->R;
    spCplxDest->R = fReal;
    spCplxDest->I = fImag;
}


inline void clDSPOp::CplxMulC (stpDCplx spCplxDest, const stpDCplx spCplxSrc)
{
    double dReal;
    double dImag;

    dReal = spCplxDest->R * spCplxSrc->R - spCplxDest->I * (-spCplxSrc->I);
    dImag = spCplxDest->R * (-spCplxSrc->I) + spCplxDest->I * spCplxSrc->R;
    spCplxDest->R = dReal;
    spCplxDest->I = dImag;
}


inline void clDSPOp::CplxMulC (stpSCplx spCplxDest, const stpSCplx spCplxSrc1,
    const stpSCplx spCplxSrc2)
{
    spCplxDest->R =
        spCplxSrc1->R * spCplxSrc2->R - spCplxSrc1->I * (-spCplxSrc2->I);
    spCplxDest->I =
        spCplxSrc1->R * (-spCplxSrc2->I) + spCplxSrc1->I * spCplxSrc2->R;
}


inline void clDSPOp::CplxMulC (stpDCplx spCplxDest, const stpDCplx spCplxSrc1,
    const stpDCplx spCplxSrc2)
{
    spCplxDest->R =
        spCplxSrc1->R * spCplxSrc2->R - spCplxSrc1->I * (-spCplxSrc2->I);
    spCplxDest->I =
        spCplxSrc1->R * (-spCplxSrc2->I) + spCplxSrc1->I * spCplxSrc2->R;
}


inline void clDSPOp::CplxDiv (stpSCplx spCplxDest, const stpSCplx spCplxSrc)
{
    float fReal;
    float fImag;

    fReal = (spCplxDest->R * spCplxSrc->R + spCplxDest->I * spCplxSrc->I) /
        (spCplxSrc->R * spCplxSrc->R + spCplxSrc->I * spCplxSrc->I);
    fImag = (spCplxDest->I * spCplxSrc->R - spCplxDest->R * spCplxSrc->I) /
        (spCplxSrc->R * spCplxSrc->R + spCplxSrc->I * spCplxSrc->I);
    spCplxDest->R = fReal;
    spCplxDest->I = fImag;
}


inline void clDSPOp::CplxDiv (stpDCplx spCplxDest, const stpDCplx spCplxSrc)
{
    double dReal;
    double dImag;

    dReal = (spCplxDest->R * spCplxSrc->R + spCplxDest->I * spCplxSrc->I) /
        (spCplxSrc->R * spCplxSrc->R + spCplxSrc->I * spCplxSrc->I);
    dImag = (spCplxDest->I * spCplxSrc->R - spCplxDest->R * spCplxSrc->I) /
        (spCplxSrc->R * spCplxSrc->R + spCplxSrc->I * spCplxSrc->I);
    spCplxDest->R = dReal;
    spCplxDest->I = dImag;
}


inline void clDSPOp::CplxDiv (stpSCplx spCplxDest, const stpSCplx spCplxSrc1,
    const stpSCplx spCplxSrc2)
{
    spCplxDest->R =
        (spCplxSrc1->R * spCplxSrc2->R + spCplxSrc1->I * spCplxSrc2->I) /
        (spCplxSrc2->R * spCplxSrc2->R + spCplxSrc2->I * spCplxSrc2->I);
    spCplxDest->I =
        (spCplxSrc1->I * spCplxSrc2->R - spCplxSrc1->R * spCplxSrc2->I) /
        (spCplxSrc2->R * spCplxSrc2->R + spCplxSrc2->I * spCplxSrc2->I);
}


inline void clDSPOp::CplxDiv (stpDCplx spCplxDest, const stpDCplx spCplxSrc1,
    const stpDCplx spCplxSrc2)
{
    spCplxDest->R =
        (spCplxSrc1->R * spCplxSrc2->R + spCplxSrc1->I * spCplxSrc2->I) /
        (spCplxSrc2->R * spCplxSrc2->R + spCplxSrc2->I * spCplxSrc2->I);
    spCplxDest->I =
        (spCplxSrc1->I * spCplxSrc2->R - spCplxSrc1->R * spCplxSrc2->I) /
        (spCplxSrc2->R * spCplxSrc2->R + spCplxSrc2->I * spCplxSrc2->I);
}


inline void clDSPOp::CplxExp (stpSCplx spCplxDest, const stpSCplx spCplxSrc)
{
    spCplxDest->R = expf(spCplxSrc->R);
    spCplxDest->I = expf(spCplxSrc->I);
}


inline void clDSPOp::CplxExp (stpDCplx spCplxDest, const stpDCplx spCplxSrc)
{
    spCplxDest->R = exp(spCplxSrc->R);
    spCplxDest->I = exp(spCplxSrc->I);
}


inline void clDSPOp::CplxLog (stpSCplx spCplxDest, const stpSCplx spCplxSrc)
{
    spCplxDest->R = log10f(spCplxSrc->R);
    spCplxDest->I = log10f(spCplxSrc->I);
}


inline void clDSPOp::CplxLog (stpDCplx spCplxDest, const stpDCplx spCplxSrc)
{
    spCplxDest->R = log10(spCplxSrc->R);
    spCplxDest->I = log10(spCplxSrc->I);
}


inline void clDSPOp::CplxPow (stpSCplx spCplxDest, const stpSCplx spCplxSrc,
    float fPower)
{
    spCplxDest->R = powf(spCplxSrc->R, fPower);
    spCplxDest->I = powf(spCplxSrc->I, fPower);
}


inline void clDSPOp::CplxPow (stpDCplx spCplxDest, const stpDCplx spCplxSrc,
    double dPower)
{
    spCplxDest->R = pow(spCplxSrc->R, dPower);
    spCplxDest->I = pow(spCplxSrc->I, dPower);
}


inline void clDSPOp::CplxRoot (stpSCplx spCplxDest, const stpSCplx spCplxSrc,
    float fRoot)
{
    spCplxDest->R = powf(spCplxSrc->R, 1.0F / fRoot);
    spCplxDest->I = powf(spCplxSrc->I, 1.0F / fRoot);
}


inline void clDSPOp::CplxRoot (stpDCplx spCplxDest, const stpDCplx spCplxSrc,
    double dRoot)
{
    spCplxDest->R = pow(spCplxSrc->R, 1.0 / dRoot);
    spCplxDest->I = pow(spCplxSrc->I, 1.0 / dRoot);
}


/*inline void clDSPOp::CplxConj (stpSCplx spCplx)
{
    spCplx->I = -(spCplx->I);
}


inline void clDSPOp::CplxConj (stpDCplx spCplx)
{
    spCplx->I = -(spCplx->I);
}*/


inline void clDSPOp::CplxConj (stpSCplx spCplxDest, const stpSCplx spCplxSrc)
{
    spCplxDest->R = spCplxSrc->R;
    spCplxDest->I = -(spCplxSrc->I);
}


inline void clDSPOp::CplxConj (stpDCplx spCplxDest, const stpDCplx spCplxSrc)
{
    spCplxDest->R = spCplxSrc->R;
    spCplxDest->I = -(spCplxSrc->I);
}


inline double clDSPOp::Multiple (long lValue)
{
    long lLoopCntr;
    double dMult = 1.0;

    for (lLoopCntr = 1L; lLoopCntr <= lValue; lLoopCntr++)
    {
        dMult *= (double) lLoopCntr;
    }
    return dMult;
}


inline float clDSPOp::ModZeroBessel (float fValue)
{
    long lLoopCntr;
    float fMZBessel = 0.0F;
    float fHalfValue;

    fHalfValue = fValue / 2.0F;
    for (lLoopCntr = 0L; lLoopCntr <= DSP_MAXBESSEL; lLoopCntr++)
    {
        fMZBessel += (float) pow(
            pow(fHalfValue, lLoopCntr) / Multiple(lLoopCntr), 2.0);
    }
    return fMZBessel;
}


inline double clDSPOp::ModZeroBessel (double dValue)
{
    long lLoopCntr;
    double dMZBessel = 0.0;
    double dHalfValue;

    dHalfValue = dValue / 2.0;
    for (lLoopCntr = 0L; lLoopCntr <= DSP_MAXBESSEL; lLoopCntr++)
    {
        dMZBessel += pow(
            pow(dHalfValue, lLoopCntr) / Multiple(lLoopCntr), 2.0);
    }
    return dMZBessel;
}


clDSPOp::clDSPOp()
{
    lPrevSrcCount = 0;
    lPrevDestCount = 0;
    // This is to get maximum precision for PI
    /*fPI = (float) acosl(-1.0);
    dPI = (double) acosl(-1.0);*/
    fPI = (float) acos(-1.0);
    dPI = acos(-1.0);
    bFFTInitialized = false;
    lpSBitRevWork = NULL;
    lpDBitRevWork = NULL;
    fpCosSinTable = NULL;
    dpCosSinTable = NULL;
}


clDSPOp::~clDSPOp()
{
    if (bFFTInitialized)
    {
        FFTUninitialize();
    }
}


void clDSPOp::Add (float *fpDest, float fSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] += fSrc;
    }
}


void clDSPOp::Add (double *dpDest, double dSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] += dSrc;
    }
}


void clDSPOp::Add (stpSCplx spCplxDest, stSCplx sCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxAdd(&spCplxDest[lLoopCntr], &sCplxSrc);
    }
}


void clDSPOp::Add (stpDCplx spCplxDest, stDCplx sCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxAdd(&spCplxDest[lLoopCntr], &sCplxSrc);
    }
}


void clDSPOp::Add (float *fpDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] += fpSrc[lLoopCntr];
    }
}


void clDSPOp::Add (double *dpDest, const double *dpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] += dpSrc[lLoopCntr];
    }
}


void clDSPOp::Add (stpSCplx spCplxDest, const stpSCplx spCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxAdd(&spCplxDest[lLoopCntr], &spCplxSrc[lLoopCntr]);
    }
}


void clDSPOp::Add (stpDCplx spCplxDest, const stpDCplx spCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxAdd(&spCplxDest[lLoopCntr], &spCplxSrc[lLoopCntr]);
    }
}


void clDSPOp::Add (float *fpDest, const float *fpSrc1, const float *fpSrc2, 
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = fpSrc1[lLoopCntr] + fpSrc2[lLoopCntr];
    }
}


void clDSPOp::Add (double *dpDest, const double *dpSrc1, const double *dpSrc2,
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = dpSrc1[lLoopCntr] + dpSrc2[lLoopCntr];
    }
}


void clDSPOp::Add (stpSCplx spCplxDest, const stpSCplx spCplxSrc1,
    const stpSCplx spCplxSrc2, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxAdd(&spCplxDest[lLoopCntr], &spCplxSrc1[lLoopCntr],
            &spCplxSrc2[lLoopCntr]);
    }
}


void clDSPOp::Add (stpDCplx spCplxDest, const stpDCplx spCplxSrc1,
    const stpDCplx spCplxSrc2, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxAdd(&spCplxDest[lLoopCntr], &spCplxSrc1[lLoopCntr],
            &spCplxSrc2[lLoopCntr]);
    }
}


void clDSPOp::Sub (float *fpDest, float fSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] -= fSrc;
    }
}


void clDSPOp::Sub (double *dpDest, double dSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] -= dSrc;
    }
}


void clDSPOp::Sub (stpSCplx spCplxDest, stSCplx sCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxSub(&spCplxDest[lLoopCntr], &sCplxSrc);
    }
}


void clDSPOp::Sub (stpDCplx spCplxDest, stDCplx sCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxSub(&spCplxDest[lLoopCntr], &sCplxSrc);
    }
}


void clDSPOp::Sub (float *fpDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] -= fpSrc[lLoopCntr];
    }
}


void clDSPOp::Sub (double *dpDest, const double *dpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] -= dpSrc[lLoopCntr];
    }
}


void clDSPOp::Sub (stpSCplx spCplxDest, const stpSCplx spCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxSub(&spCplxDest[lLoopCntr], &spCplxSrc[lLoopCntr]);
    }
}


void clDSPOp::Sub (stpDCplx spCplxDest, const stpDCplx spCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxSub(&spCplxDest[lLoopCntr], &spCplxSrc[lLoopCntr]);
    }
}


void clDSPOp::Sub (float *fpDest, const float *fpSrc1, const float *fpSrc2, 
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = fpSrc1[lLoopCntr] - fpSrc2[lLoopCntr];
    }
}


void clDSPOp::Sub (double *dpDest, const double *dpSrc1, const double *dpSrc2,
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = dpSrc1[lLoopCntr] - dpSrc2[lLoopCntr];
    }
}


void clDSPOp::Sub (stpSCplx spCplxDest, const stpSCplx spCplxSrc1,
    const stpSCplx spCplxSrc2, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxSub(&spCplxDest[lLoopCntr], &spCplxSrc1[lLoopCntr],
            &spCplxSrc2[lLoopCntr]);
    }
}


void clDSPOp::Sub (stpDCplx spCplxDest, const stpDCplx spCplxSrc1,
    const stpDCplx spCplxSrc2, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxSub(&spCplxDest[lLoopCntr], &spCplxSrc1[lLoopCntr],
            &spCplxSrc2[lLoopCntr]);
    }
}



void clDSPOp::Mul (float *fpVect, float fSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpVect[lLoopCntr] *= fSrc;
    }
}


void clDSPOp::Mul (double *dpVect, double dSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpVect[lLoopCntr] *= dSrc;
    }
}


void clDSPOp::Mul (stpSCplx spCplxDest, float fSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxMul(&spCplxDest[lLoopCntr], fSrc);
    }
}


void clDSPOp::Mul (stpDCplx spCplxDest, double dSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxMul(&spCplxDest[lLoopCntr], dSrc);
    }
}


void clDSPOp::Mul (stpSCplx spCplxDest, stSCplx sCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxMul(&spCplxDest[lLoopCntr], &sCplxSrc);
    }
}


void clDSPOp::Mul (stpDCplx spCplxDest, stDCplx sCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxMul(&spCplxDest[lLoopCntr], &sCplxSrc);
    }
}


void clDSPOp::Mul (float *fpDest, const float *fpSrc1, float fSrc2, 
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = fpSrc1[lLoopCntr] * fSrc2;
    }
}


void clDSPOp::Mul (double *dpDest, const double *dpSrc1, double dSrc2,
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = dpSrc1[lLoopCntr] * dSrc2;
    }
}


void clDSPOp::Mul (float *fpDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] *= fpSrc[lLoopCntr];
    }
}


void clDSPOp::Mul (double *dpDest, const double *dpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] *= dpSrc[lLoopCntr];
    }
}


void clDSPOp::Mul (stpSCplx spCplxDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxMul(&spCplxDest[lLoopCntr], fpSrc[lLoopCntr]);
    }
}


void clDSPOp::Mul (stpDCplx spCplxDest, const double *dpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxMul(&spCplxDest[lLoopCntr], dpSrc[lLoopCntr]);
    }
}


void clDSPOp::Mul (stpSCplx spCplxDest, const stpSCplx spCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxMul(&spCplxDest[lLoopCntr], &spCplxSrc[lLoopCntr]);
    }
}


void clDSPOp::Mul (stpDCplx spCplxDest, const stpDCplx spCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxMul(&spCplxDest[lLoopCntr], &spCplxSrc[lLoopCntr]);
    }
}


void clDSPOp::Mul (float *fpDest, const float *fpSrc1, 
    const float *fpSrc2, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = fpSrc1[lLoopCntr] * fpSrc2[lLoopCntr];
    }
}


void clDSPOp::Mul (double *dpDest, const double *dpSrc1,
    const double *dpSrc2, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = dpSrc1[lLoopCntr] * dpSrc2[lLoopCntr];
    }
}


void clDSPOp::Mul (stpSCplx spCplxDest, const stpSCplx spCplxSrc1,
    const stpSCplx spCplxSrc2, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxMul(&spCplxDest[lLoopCntr], &spCplxSrc1[lLoopCntr],
            &spCplxSrc2[lLoopCntr]);
    }
}


void clDSPOp::Mul (stpDCplx spCplxDest, const stpDCplx spCplxSrc1,
    const stpDCplx spCplxSrc2, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxMul(&spCplxDest[lLoopCntr], &spCplxSrc1[lLoopCntr],
            &spCplxSrc2[lLoopCntr]);
    }
}


void clDSPOp::MulC (stpSCplx spCplxDest, const stpSCplx spCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxMulC(&spCplxDest[lLoopCntr], &spCplxSrc[lLoopCntr]);
    }
}


void clDSPOp::MulC (stpDCplx spCplxDest, const stpDCplx spCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxMulC(&spCplxDest[lLoopCntr], &spCplxSrc[lLoopCntr]);
    }
}


void clDSPOp::MulC (stpSCplx spCplxDest, const stpSCplx spCplxSrc1,
    const stpSCplx spCplxSrc2, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxMulC(&spCplxDest[lLoopCntr], &spCplxSrc1[lLoopCntr],
            &spCplxSrc2[lLoopCntr]);
    }
}


void clDSPOp::MulC (stpDCplx spCplxDest, const stpDCplx spCplxSrc1,
    const stpDCplx spCplxSrc2, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxMulC(&spCplxDest[lLoopCntr], &spCplxSrc1[lLoopCntr],
            &spCplxSrc2[lLoopCntr]);
    }
}


void clDSPOp::Mul2 (float *fpDst1, float *fpDst2, const float *fpSrc, 
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDst1[lLoopCntr] *= fpSrc[lLoopCntr];
        fpDst2[lLoopCntr] *= fpSrc[lLoopCntr];
    }
}


void clDSPOp::Mul2 (double *dpDst1, double *dpDst2, const double *dpSrc, 
    long lCount)
{
   long lLoopCntr;

   for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
   {
      dpDst1[lLoopCntr] *= dpSrc[lLoopCntr];
      dpDst2[lLoopCntr] *= dpSrc[lLoopCntr];
   }
}


void clDSPOp::Mul2 (float *fpDst1, float *fpDst2, const float *fpSrc1,
    const float *fpSrc2, const float *fpMul, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDst1[lLoopCntr] = fpSrc1[lLoopCntr] * fpMul[lLoopCntr];
        fpDst2[lLoopCntr] = fpSrc2[lLoopCntr] * fpMul[lLoopCntr];
    }
}


void clDSPOp::Mul2 (double *dpDst1, double *dpDst2, const double *dpSrc1,
   const double *dpSrc2, const double *dpMul, long lCount)
{
   long lLoopCntr;

   for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
   {
      dpDst1[lLoopCntr] = dpSrc1[lLoopCntr] * dpMul[lLoopCntr];
      dpDst2[lLoopCntr] = dpSrc2[lLoopCntr] * dpMul[lLoopCntr];
   }
}


void clDSPOp::Div (float *fpVect, float fSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpVect[lLoopCntr] /= fSrc;
    }
}


void clDSPOp::Div (double *dpVect, double dSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpVect[lLoopCntr] /= dSrc;
    }
}


void clDSPOp::Div (stpSCplx spCplxDest, stSCplx sCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxDiv(&spCplxDest[lLoopCntr], &sCplxSrc);
    }
}


void clDSPOp::Div (stpDCplx spCplxDest, stDCplx sCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxDiv(&spCplxDest[lLoopCntr], &sCplxSrc);
    }
}


void clDSPOp::Div (float *fpDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] /= fpSrc[lLoopCntr];
    }
}


void clDSPOp::Div (double *dpDest, const double *dpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] /= dpSrc[lLoopCntr];
    }
}


void clDSPOp::Div (stpSCplx spCplxDest, const stpSCplx spCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxDiv(&spCplxDest[lLoopCntr], &spCplxSrc[lLoopCntr]);
    }
}


void clDSPOp::Div (stpDCplx spCplxDest, const stpDCplx spCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxDiv(&spCplxDest[lLoopCntr], &spCplxSrc[lLoopCntr]);
    }
}


void clDSPOp::Div (float *fpDest, const float *fpSrc1, const float *fpSrc2, 
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = fpSrc1[lLoopCntr] / fpSrc2[lLoopCntr];
    }
}


void clDSPOp::Div (double *dpDest, const double *dpSrc1, const double *dpSrc2,
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = dpSrc1[lLoopCntr] / dpSrc2[lLoopCntr];
    }
}


void clDSPOp::Div (stpSCplx spCplxDest, const stpSCplx spCplxSrc1,
    const stpSCplx spCplxSrc2, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxDiv(&spCplxDest[lLoopCntr], &spCplxSrc1[lLoopCntr],
            &spCplxSrc2[lLoopCntr]);
    }
}


void clDSPOp::Div (stpDCplx spCplxDest, const stpDCplx spCplxSrc1,
    const stpDCplx spCplxSrc2, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        CplxDiv(&spCplxDest[lLoopCntr], &spCplxSrc1[lLoopCntr],
            &spCplxSrc2[lLoopCntr]);
    }
}


void clDSPOp::Div1x (float *fpVect, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpVect[lLoopCntr] = 1.0F / fpVect[lLoopCntr];
    }
}


void clDSPOp::Div1x (double *dpVect, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpVect[lLoopCntr] = 1.0 / dpVect[lLoopCntr];
    }
}


void clDSPOp::Div1x (float *fpDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = 1.0F / fpSrc[lLoopCntr];
    }
}


void clDSPOp::Div1x (double *dpDest, const double *dpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = 1.0 / dpSrc[lLoopCntr];
    }
}


void clDSPOp::Abs (float *fpVect, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpVect[lLoopCntr] = fabsf(fpVect[lLoopCntr]);
    }
}


void clDSPOp::Abs (double *dpVect, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpVect[lLoopCntr] = fabs(dpVect[lLoopCntr]);
    }
}


void clDSPOp::Abs (float *fpDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = fabsf(fpSrc[lLoopCntr]);
    }
}


void clDSPOp::Abs (double *dpDest, const double *dpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = fabs(dpSrc[lLoopCntr]);
    }
}


void clDSPOp::Sqrt (float *fpVect, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpVect[lLoopCntr] = sqrtf(fpVect[lLoopCntr]);
    }
}


void clDSPOp::Sqrt (double *dpVect, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpVect[lLoopCntr] = sqrt(dpVect[lLoopCntr]);
    }
}


void clDSPOp::Sqrt (float *fpDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = sqrtf(fpSrc[lLoopCntr]);
    }
}


void clDSPOp::Sqrt (double *dpDest, const double *dpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = sqrt(dpSrc[lLoopCntr]);
    }
}


void clDSPOp::Zero (float *fpDest, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = 0.0F;
    }
}


void clDSPOp::Zero (double *dpDest, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = 0.0;
    }
}


void clDSPOp::Zero (stpSCplx spCplxDest, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        spCplxDest[lLoopCntr].R = 0.0F;
        spCplxDest[lLoopCntr].I = 0.0F;
    }
}


void clDSPOp::Zero (stpDCplx spCplxDest, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        spCplxDest[lLoopCntr].R = 0.0;
        spCplxDest[lLoopCntr].I = 0.0;
    }
}


void clDSPOp::Set (float *fpDest, float fSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = fSrc;
    }
}


void clDSPOp::Set (double *dpDest, double dSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = dSrc;
    }
}


void clDSPOp::Set (stpSCplx spCplxDest, stSCplx sCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        spCplxDest[lLoopCntr].R = sCplxSrc.R;
        spCplxDest[lLoopCntr].I = sCplxSrc.I;
    }
}


void clDSPOp::Set (stpDCplx spCplxDest, stDCplx sCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        spCplxDest[lLoopCntr].R = sCplxSrc.R;
        spCplxDest[lLoopCntr].I = sCplxSrc.I;
    }
}


void clDSPOp::Set (float *fpDest, float fSrc, long lStart, long lCount,
    long lLength)
{
    long lLoopCntr;
    long lEnd;

    lEnd = ((lStart + lCount) > lLength) ? lLength : (lStart + lCount);
    for (lLoopCntr = 0L; lLoopCntr < lEnd; lLoopCntr++)
    {
        fpDest[lLoopCntr] = fSrc;
    }
}


void clDSPOp::Set (double *dpDest, double dSrc, long lStart, long lCount,
    long lLength)
{
    long lLoopCntr;
    long lEnd;

    lEnd = ((lStart + lCount) > lLength) ? lLength : (lStart + lCount);
    for (lLoopCntr = 0L; lLoopCntr < lEnd; lLoopCntr++)
    {
        dpDest[lLoopCntr] = dSrc;
    }
}


void clDSPOp::Set (stpSCplx spCplxDest, stSCplx sCplxSrc, long lStart, 
    long lCount, long lLength)
{
    long lLoopCntr;
    long lEnd;

    lEnd = ((lStart + lCount) > lLength) ? lLength : (lStart + lCount);
    for (lLoopCntr = 0L; lLoopCntr < lEnd; lLoopCntr++)
    {
        spCplxDest[lLoopCntr].R = sCplxSrc.R;
        spCplxDest[lLoopCntr].I = sCplxSrc.I;
    }
}


void clDSPOp::Set (stpDCplx spCplxDest, stDCplx sCplxSrc, long lStart,
    long lCount, long lLength)
{
    long lLoopCntr;
    long lEnd;

    lEnd = ((lStart + lCount) > lLength) ? lLength : (lStart + lCount);
    for (lLoopCntr = 0L; lLoopCntr < lEnd; lLoopCntr++)
    {
        spCplxDest[lLoopCntr].R = sCplxSrc.R;
        spCplxDest[lLoopCntr].I = sCplxSrc.I;
    }
}


void clDSPOp::Clip (float *fpVect, float fValue, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            if (fpVect[lLoopCntr] > fValue)
            {
                fpVect[lLoopCntr] = fValue;
            }
        #else
            if (isgreater(fpVect[lLoopCntr], fValue))
            {
                fpVect[lLoopCntr] = fValue;
            }
        #endif
    }
}


void clDSPOp::Clip (double *dpVect, double dValue, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            if (dpVect[lLoopCntr] > dValue)
            {
                dpVect[lLoopCntr] = dValue;
            }
        #else
            if (isgreater(dpVect[lLoopCntr], dValue))
            {
                dpVect[lLoopCntr] = dValue;
            }
        #endif
    }
}


void clDSPOp::Clip (float *fpDest, const float *fpSrc, float fValue, 
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            fpDest[lLoopCntr] = (fpSrc[lLoopCntr] > fValue) ? 
                fValue : fpSrc[lLoopCntr];
        #else
            fpDest[lLoopCntr] = (isgreater(fpSrc[lLoopCntr], fValue)) ?
                fValue : fpSrc[lLoopCntr];
        #endif
    }
}


void clDSPOp::Clip (double *dpDest, const double *dpSrc, double dValue,
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            dpDest[lLoopCntr] = (dpSrc[lLoopCntr] > dValue) ?
                dValue : dpSrc[lLoopCntr];
        #else
            dpDest[lLoopCntr] = (isgreater(dpSrc[lLoopCntr], dValue)) ?
                dValue : dpSrc[lLoopCntr];
        #endif
    }
}


void clDSPOp::Clip (float *fpVect, float fMin, float fMax, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            if (fpVect[lLoopCntr] < fMin)
            {
                fpVect[lLoopCntr] = fMin;
            }
            else if (fpVect[lLoopCntr] > fMax)
            {
                fpVect[lLoopCntr] = fMax;
            }
        #else
            if (isless(fpVect[lLoopCntr], fMin))
            {
                fpVect[lLoopCntr] = fMin;
            }
            else if (isgreater(fpVect[lLoopCntr], fMax))
            {
                fpVect[lLoopCntr] = fMax;
            }
        #endif
    }
}


void clDSPOp::Clip (double *dpVect, double dMin, double dMax, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            if (dpVect[lLoopCntr] < dMin)
            {
                dpVect[lLoopCntr] = dMin;
            }
            else if (dpVect[lLoopCntr] > dMax)
            {
                dpVect[lLoopCntr] = dMax;
            }
        #else
            if (isless(dpVect[lLoopCntr], dMin))
            {
                dpVect[lLoopCntr] = dMin;
            }
            else if (isgreater(dpVect[lLoopCntr], dMax))
            {
                dpVect[lLoopCntr] = dMax;
            }
        #endif
    }
}


void clDSPOp::Clip (float *fpDest, const float *fpSrc, float fMin, 
    float fMax, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            if (fpSrc[lLoopCntr] < fMin)
            {
                fpDest[lLoopCntr] = fMin;
            }
            else if (fpSrc[lLoopCntr] > fMax)
            {
                fpDest[lLoopCntr] = fMax;
            }
            else
            {
                fpDest[lLoopCntr] = fpSrc[lLoopCntr];
            }
        #else
            if (isless(fpSrc[lLoopCntr], fMin))
            {
                fpDest[lLoopCntr] = fMin;
            }
            else if (isgreater(fpSrc[lLoopCntr], fMax))
            {
                fpDest[lLoopCntr] = fMax;
            }
            else
            {
                fpDest[lLoopCntr] = fpSrc[lLoopCntr];
            }
        #endif
    }
}


void clDSPOp::Clip (double *dpDest, const double *dpSrc, double dMin, 
    double dMax, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            if (dpSrc[lLoopCntr] < dMin)
            {
                dpDest[lLoopCntr] = dMin;
            }
            else if (dpSrc[lLoopCntr] > dMax)
            {
                dpDest[lLoopCntr] = dMax;
            }
            else
            {
                dpDest[lLoopCntr] = dpSrc[lLoopCntr];
            }
        #else
            if (isless(dpSrc[lLoopCntr], dMin))
            {
                dpDest[lLoopCntr] = dMin;
            }
            else if (isgreater(dpSrc[lLoopCntr], dMax))
            {
                dpDest[lLoopCntr] = dMax;
            }
            else
            {
                dpDest[lLoopCntr] = dpSrc[lLoopCntr];
            }
        #endif
    }
}


void clDSPOp::ClipZero (float *fpVect, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            if (fpVect[lLoopCntr] < 0.0F)
            {
                fpVect[lLoopCntr] = 0.0F;
            }
        #else
            if (isless(fpVect[lLoopCntr], 0.0F))
            {
                fpVect[lLoopCntr] = 0.0F;
            }
        #endif
    }
}


void clDSPOp::ClipZero (double *dpVect, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            if (dpVect[lLoopCntr] < 0.0)
            {
                dpVect[lLoopCntr] = 0.0;
            }
        #else
            if (isless(dpVect[lLoopCntr], 0.0))
            {
                dpVect[lLoopCntr] = 0.0;
            }
        #endif
    }
}


void clDSPOp::ClipZero (float *fpDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            fpDest[lLoopCntr] = (fpSrc[lLoopCntr] < 0.0F) ? 
                0.0F : fpSrc[lLoopCntr];
        #else
            fpDest[lLoopCntr] = (isless(fpSrc[lLoopCntr], 0.0F)) ?
                0.0F : fpSrc[lLoopCntr];
        #endif
    }
}


void clDSPOp::ClipZero (double *dpDest, const double *dpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            dpDest[lLoopCntr] = (dpSrc[lLoopCntr] < 0.0) ?
                0.0 : dpSrc[lLoopCntr];
        #else
            dpDest[lLoopCntr] = (isless(dpSrc[lLoopCntr], 0.0)) ?
                0.0 : dpSrc[lLoopCntr];
        #endif
    }
}


void clDSPOp::Copy (float *fpDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = fpSrc[lLoopCntr];
    }
}


void clDSPOp::Copy (double *dpDest, const double *dpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = dpSrc[lLoopCntr];
    }
}


void clDSPOp::Copy (stpSCplx spCplxDest, const stpSCplx spCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        spCplxDest[lLoopCntr].R = spCplxSrc[lLoopCntr].R;
        spCplxDest[lLoopCntr].I = spCplxSrc[lLoopCntr].I;
    }
}


void clDSPOp::Copy (stpDCplx spCplxDest, const stpDCplx spCplxSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        spCplxDest[lLoopCntr].R = spCplxSrc[lLoopCntr].R;
        spCplxDest[lLoopCntr].I = spCplxSrc[lLoopCntr].I;
    }
}


void clDSPOp::Copy (float *fpDest1, float *fpDest2, const float *fpSrc,
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest1[lLoopCntr] = fpDest2[lLoopCntr] = fpSrc[lLoopCntr];
    }
}


void clDSPOp::Copy (double *dpDest1, double *dpDest2, const double *dpSrc,
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest1[lLoopCntr] = dpDest2[lLoopCntr] = dpSrc[lLoopCntr];
    }
}


float clDSPOp::Convolve (const float *fpSrc1, const float *fpSrc2, 
    long lCount)
{
    long lLoopCntr;
    long lMax;
    float fConv = 0.0F;

    lMax = lCount - 1L;
    for (lLoopCntr = 0L; lLoopCntr <= lMax; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            fConv += fpSrc1[lLoopCntr] * fpSrc2[lMax - lLoopCntr];
        #else
            fConv = fmaf(fpSrc1[lLoopCntr], fpSrc2[lMax - lLoopCntr],
                fConv);
        #endif
    }
    return fConv;
}


double clDSPOp::Convolve (const double *dpSrc1, const double *dpSrc2,
    long lCount)
{
    long lLoopCntr;
    long lMax;
    double dConv = 0.0;

    lMax = lCount - 1L;
    for (lLoopCntr = 0L; lLoopCntr <= lMax; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            dConv += dpSrc1[lLoopCntr] * dpSrc2[lMax - lLoopCntr];
        #else
            dConv = fma(dpSrc1[lLoopCntr], dpSrc2[lMax - lLoopCntr],
                dConv);
        #endif
    }
    return dConv;
}


void clDSPOp::Convolve (float *fpDest, const float *fpSrc1, 
    const float *fpSrc2, long lCount)
{
    long lLoopDest;
    long lLoopConv;
    long lIdx;
    long lMax;
    float fConv;

    lMax = lCount - 1L;
    for (lLoopDest = 0L; lLoopDest < lCount; lLoopDest++)
    {
        fConv = 0.0F;
        for (lLoopConv = 0L; lLoopConv <= lMax; lLoopConv++)
        {
            lIdx = ((lLoopConv - lLoopDest) < 0L) ?
                (lLoopConv - lLoopDest + lMax) : (lLoopConv - lLoopDest);
            #ifndef _ISOC9X_SOURCE
                fConv += fpSrc1[lMax - lIdx] * fpSrc2[lIdx];
            #else
                fConv = fmaf(fpSrc1[lMax - lIdx], fpSrc2[lIdx], fConv);
            #endif
        }
        fpDest[lLoopDest] = fConv;
    }
}


void clDSPOp::Convolve (double *dpDest, const double *dpSrc1,
    const double *dpSrc2, long lCount)
{
    long lLoopDest;
    long lLoopConv;
    long lIdx;
    long lMax;
    double dConv;

    lMax = lCount - 1L;
    for (lLoopDest = 0L; lLoopDest < lCount; lLoopDest++)
    {
        dConv = 0.0;
        for (lLoopConv = 0L; lLoopConv <= lMax; lLoopConv++)
        {
            lIdx = ((lLoopConv - lLoopDest) < 0L) ?
                (lLoopConv - lLoopDest + lMax) : (lLoopConv - lLoopDest);
            #ifndef _ISOC9X_SOURCE
                dConv += dpSrc1[lMax - lIdx] * dpSrc2[lIdx];
            #else
                dConv = fma(dpSrc1[lMax - lIdx], dpSrc2[lIdx], dConv);
            #endif
        }
        dpDest[lLoopDest] = dConv;
    }
}


float clDSPOp::Correlate (const float *fpSrc1, const float *fpSrc2, 
    long lCount)
{
    long lLoopCntr;
    float fCorr = 0.0F;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            fCorr += fpSrc1[lLoopCntr] * fpSrc2[lLoopCntr];
        #else
            fCorr = fmaf(fpSrc1[lLoopCntr], fpSrc2[lLoopCntr], fCorr);
        #endif
    }
    fCorr /= (float) lCount;
    return fCorr;
}


double clDSPOp::Correlate (const double *dpSrc1, const double *dpSrc2,
    long lCount)
{
    long lLoopCntr;
    double dCorr = 0.0;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            dCorr += dpSrc1[lLoopCntr] * dpSrc2[lLoopCntr];
        #else
            dCorr = fma(dpSrc1[lLoopCntr], dpSrc2[lLoopCntr], dCorr);
        #endif
    }
    dCorr /= (double) lCount;
    return dCorr;
}


void clDSPOp::Correlate (float *fpDest, const float *fpSrc1, 
    const float *fpSrc2, long lCount)
{
    long lLoopDest;
    long lLoopCorr;
    long lMax;
    long lIdx;
    float fCorr;

    lMax = lCount - 1L;
    for (lLoopDest = 0L; lLoopDest <= lMax; lLoopDest++)
    {
        fCorr = 0.0F;
        for (lLoopCorr = 0L; lLoopCorr <= lMax; lLoopCorr++)
        {
            lIdx = ((lLoopCorr + lLoopDest) > lMax) ?
                (lLoopCorr + lLoopDest - lMax) : (lLoopCorr + lLoopDest);
            #ifndef _ISOC9X_SOURCE
                fCorr += fpSrc1[lLoopCorr] * fpSrc2[lIdx];
            #else
                fCorr = fmaf(fpSrc1[lLoopCorr], fpSrc2[lIdx], fCorr);
            #endif
        }
        fpDest[lLoopDest] = fCorr / (float) lCount;
    }
}


void clDSPOp::Correlate (double *dpDest, const double *dpSrc1,
    const double *dpSrc2, long lCount)
{
    long lLoopDest;
    long lLoopCorr;
    long lMax;
    long lIdx;
    double dCorr;

    lMax = lCount - 1L;
    for (lLoopDest = 0L; lLoopDest <= lMax; lLoopDest++)
    {
        dCorr = 0.0;
        for (lLoopCorr = 0L; lLoopCorr <= lMax; lLoopCorr++)
        {
            lIdx = ((lLoopCorr + lLoopDest) > lMax) ?
                (lLoopCorr + lLoopDest - lMax) : (lLoopCorr + lLoopDest);
            #ifndef _ISOC9X_SOURCE
                dCorr += dpSrc1[lLoopCorr] * dpSrc2[lIdx];
            #else
                dCorr = fma(dpSrc1[lLoopCorr], dpSrc2[lIdx], dCorr);
            #endif
        }
        dpDest[lLoopDest] = dCorr / (double) lCount;
    }
}


float clDSPOp::AutoCorrelate (const float *fpSrc, long lCount)
{
    long lLoopCntr;
    float fAutoCorr = 0.0F;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            fAutoCorr += fpSrc[lLoopCntr] * fpSrc[lLoopCntr];
        #else
            fAutoCorr = fmaf(fpSrc[lLoopCntr], fpSrc[lLoopCntr], fAutoCorr);
        #endif
    }
    fAutoCorr /= (float) lCount;
    return fAutoCorr;
}


double clDSPOp::AutoCorrelate (const double *dpSrc, long lCount)
{
    long lLoopCntr;
    double dAutoCorr = 0.0;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            dAutoCorr += dpSrc[lLoopCntr] * dpSrc[lLoopCntr];
        #else
            dAutoCorr = fma(dpSrc[lLoopCntr], dpSrc[lLoopCntr], dAutoCorr);
        #endif
    }
    dAutoCorr /= (double) lCount;
    return dAutoCorr;
}


void clDSPOp::AutoCorrelate (float *fpDest, const float *fpSrc, long lCount)
{
    long lLoopDest;
    long lLoopCorr;
    long lMax;
    long lIdx;
    float fAutoCorr;

    lMax = lCount - 1L;
    for (lLoopDest = 0L; lLoopDest <= lMax; lLoopDest++)
    {
        fAutoCorr = 0.0F;
        for (lLoopCorr = 0L; lLoopCorr <= lMax; lLoopCorr++)
        {
            lIdx = ((lLoopCorr + lLoopDest) > lMax) ?
                (lLoopCorr + lLoopDest - lMax) : (lLoopCorr + lLoopDest);
            #ifndef _ISOC9X_SOURCE
                fAutoCorr += fpSrc[lLoopCorr] * fpSrc[lIdx];
            #else
                fAutoCorr = fmaf(fpSrc[lLoopCorr], fpSrc[lIdx], fAutoCorr);
            #endif
        }
        fpDest[lLoopDest] = fAutoCorr / (float) lCount;
    }
}


void clDSPOp::AutoCorrelate (double *dpDest, const double *dpSrc, long lCount)
{
    long lLoopDest;
    long lLoopCorr;
    long lMax;
    long lIdx;
    double dAutoCorr;

    lMax = lCount - 1L;
    for (lLoopDest = 0L; lLoopDest <= lMax; lLoopDest++)
    {
        dAutoCorr = 0.0;
        for (lLoopCorr = 0L; lLoopCorr <= lMax; lLoopCorr++)
        {
            lIdx = ((lLoopCorr + lLoopDest) > lMax) ?
                (lLoopCorr + lLoopDest - lMax) : (lLoopCorr + lLoopDest);
            #ifndef _ISOC9X_SOURCE
                dAutoCorr += dpSrc[lLoopCorr] * dpSrc[lIdx];
            #else
                dAutoCorr = fma(dpSrc[lLoopCorr], dpSrc[lIdx], dAutoCorr);
            #endif
        }
        dpDest[lLoopDest] = dAutoCorr / (double) lCount;
    }
}


float clDSPOp::DotProduct (const float *fpSrc1, const float *fpSrc2,
    long lCount)
{
    long lLoopCntr;
    float fDotProd = 0.0F;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            fDotProd += fpSrc1[lLoopCntr] * fpSrc2[lLoopCntr];
        #else
            fDotProd = fmaf(fpSrc1[lLoopCntr], fpSrc2[lLoopCntr], fDotProd);
        #endif
    }
    return fDotProd;
}


double clDSPOp::DotProduct (const double *dpSrc1, const double *dpSrc2,
    long lCount)
{
    long lLoopCntr;
    double dDotProd = 0.0;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            dDotProd += dpSrc1[lLoopCntr] * dpSrc2[lLoopCntr];
        #else
            dDotProd = fma(dpSrc1[lLoopCntr], dpSrc2[lLoopCntr], dDotProd);
        #endif
    }
    return dDotProd;
}


void clDSPOp::MinMax (float *fpMin, float *fpMax, const float *fpSrc,
    long lCount)
{
    long lLoopCntr;
    float fTempVal;
    float fTempMin = FLT_MAX;
    float fTempMax = -FLT_MAX;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fTempVal = fpSrc[lLoopCntr];
        #ifndef _ISOC9X_SOURCE
            if (fTempVal < fTempMin)
            {
                fTempMin = fTempVal;
            }
            if (fTempVal > fTempMax)
            {
                fTempMax = fTempVal;
            }
        #else
            fTempMin = fminf(fTempVal, fTempMin);
            fTempMax = fmaxf(fTempVal, fTempMax);
        #endif
    }
    *fpMin = fTempMin;
    *fpMax = fTempMax;
}


void clDSPOp::MinMax (double *dpMin, double *dpMax, const double *dpSrc,
    long lCount)
{
    long lLoopCntr;
    double dTempVal;
    double dTempMin = DBL_MAX;
    double dTempMax = -DBL_MAX;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dTempVal = dpSrc[lLoopCntr];
        #ifndef _ISOC9X_SOURCE
            if (dTempVal < dTempMin)
            {
                dTempMin = dTempVal;
            }
            if (dTempVal > dTempMax)
            {
                dTempMax = dTempVal;
            }
        #else
            dTempMin = fmin(dTempVal, dTempMin);
            dTempMax = fmax(dTempVal, dTempMax);
        #endif
    }
    *dpMin = dTempMin;
    *dpMax = dTempMax;
}


float clDSPOp::Mean (const float *fpSrc, long lCount)
{
    long lLoopCntr;
    float fMean = 0.0F;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fMean += fpSrc[lLoopCntr];
    }
    fMean /= (float) lCount;
    return fMean;
}


double clDSPOp::Mean (const double *dpSrc, long lCount)
{
    long lLoopCntr;
    double dMean = 0.0;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dMean += dpSrc[lLoopCntr];
    }
    dMean /= (double) lCount;
    return dMean;
}


float clDSPOp::Median (const float *fpSrc, long lCount)
{
    long lMax;
    float fMedian = 0.0F;
    #ifdef __GNUG__
        float fpTemp[lCount];
    #else
        float *fpTemp;
        clDSPAlloc Temp(lCount * sizeof(float));
    #endif

    #ifndef __GNUG__
        fpTemp = Temp;
    #endif
    lMax = lCount - 1L;
    if (fpTemp != NULL)
    {
        Copy(fpTemp, fpSrc, lCount);
        Sort(fpTemp, lCount);
        fMedian = ((lCount % 2L) != 0L) ?
            fpTemp[lMax / 2L] :
            (0.5F * (fpTemp[lCount / 2L - 1L] + fpTemp[lCount / 2L]));
    }
    return fMedian;
}


double clDSPOp::Median (const double *dpSrc, long lCount)
{
    long lMax;
    double dMedian = 0.0;
    #ifdef __GNUG__
        double dpTemp[lCount];
    #else
        double *dpTemp;
        clDSPAlloc Temp(lCount * sizeof(double));
    #endif

    #ifndef __GNUG__
        dpTemp = Temp;
    #endif
    lMax = lCount - 1L;
    if (dpTemp != NULL)
    {
        Copy(dpTemp, dpSrc, lCount);
        Sort(dpTemp, lCount);
        dMedian = ((lCount % 2L) != 0L) ?
            dpTemp[lMax / 2L] :
            (0.5 * (dpTemp[lCount / 2L - 1L] + dpTemp[lCount / 2L]));
    }
    return dMedian;
}


void clDSPOp::Negate (float *fpVect, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpVect[lLoopCntr] = -(fpVect[lLoopCntr]);
    }
}


void clDSPOp::Negate (double *dpVect, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpVect[lLoopCntr] = -(dpVect[lLoopCntr]);
    }
}


void clDSPOp::Normalize (float *fpVect, long lCount)
{
    long lLoopCntr;
    float fMean;
    float fStdDev;

    StdDev(&fStdDev, &fMean, fpVect, lCount);
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpVect[lLoopCntr] = (fpVect[lLoopCntr] - fMean) / fStdDev;
    }
}


void clDSPOp::Normalize (double *dpVect, long lCount)
{
    long lLoopCntr;
    double dMean;
    double dStdDev;

    StdDev(&dStdDev, &dMean, dpVect, lCount);
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpVect[lLoopCntr] = (dpVect[lLoopCntr] - dMean) / dStdDev;
    }
}


float clDSPOp::Product (const float *fpSrc, long lCount)
{
    long lLoopCntr;
    float fProd = 1.0F;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fProd *= fpSrc[lLoopCntr];
    }
    return fProd;
}


double clDSPOp::Product (const double *dpSrc, long lCount)
{
    long lLoopCntr;
    double dProd = 1.0;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dProd *= dpSrc[lLoopCntr];
    }
    return dProd;
}


void clDSPOp::Reverse (float *fpDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;
    long lMax;

    lMax = lCount - 1L;
    for (lLoopCntr = 0L; lLoopCntr <= lMax; lLoopCntr++)
    {
        fpDest[lLoopCntr] = fpSrc[lMax - lLoopCntr];
    }
}


void clDSPOp::Reverse (double *dpDest, const double *dpSrc, long lCount)
{
    long lLoopCntr;
    long lMax;

    lMax = lCount - 1L;
    for (lLoopCntr = 0L; lLoopCntr <= lMax; lLoopCntr++)
    {
        dpDest[lLoopCntr] = dpSrc[lMax - lLoopCntr];
    }
}


void clDSPOp::Scale (float *fpVect, long lCount)
{
    long lLoopCntr;
    float fMin;
    float fMax;
    float fScale;

    MinMax(&fMin, &fMax, fpVect, lCount);
    fScale = 2.0F / (fMax - fMin);
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            fpVect[lLoopCntr] = (fpVect[lLoopCntr] - fMin) * fScale - 1.0F;
        #else
            fpVect[lLoopCntr] = 
                fmaf((fpVect[lLoopCntr] - fMin), fScale, -1.0F);
        #endif
    }
}


void clDSPOp::Scale (double *dpVect, long lCount)
{
    long lLoopCntr;
    double dMin;
    double dMax;
    double dScale;

    MinMax(&dMin, &dMax, dpVect, lCount);
    dScale = 2.0 / (dMax - dMin);
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            dpVect[lLoopCntr] = (dpVect[lLoopCntr] - dMin) * dScale - 1.0;
        #else
            dpVect[lLoopCntr] = 
                fma((dpVect[lLoopCntr] - dMin), dScale, -1.0);
        #endif
    }
}


void clDSPOp::Scale (float *fpDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;
    float fMin;
    float fMax;
    float fScale;

    MinMax(&fMin, &fMax, fpSrc, lCount);
    fScale = 2.0F / (fMax - fMin);
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            fpDest[lLoopCntr] = (fpSrc[lLoopCntr] - fMin) * fScale - 1.0F;
        #else
            fpDest[lLoopCntr] =
                fmaf((fpSrc[lLoopCntr] - fMin), fScale, -1.0F);
        #endif
    }
}


void clDSPOp::Scale (double *dpDest, const double *dpSrc, long lCount)
{
    long lLoopCntr;
    double dMin;
    double dMax;
    double dScale;

    MinMax(&dMin, &dMax, dpSrc, lCount);
    dScale = 2.0 / (dMax - dMin);
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            dpDest[lLoopCntr] = (dpSrc[lLoopCntr] - dMin) * dScale - 1.0;
        #else
            dpDest[lLoopCntr] =
                fma((dpSrc[lLoopCntr] - dMin), dScale, -1.0);
        #endif
    }
}


void clDSPOp::Scale01 (float *fpVect, long lCount)
{
    long lLoopCntr;
    float fMin;
    float fMax;
    float fScale;

    MinMax(&fMin, &fMax, fpVect, lCount);
    fScale = 1.0F / (fMax - fMin);
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpVect[lLoopCntr] = (fpVect[lLoopCntr] - fMin) * fScale;
    }
}


void clDSPOp::Scale01 (double *dpVect, long lCount)
{
    long lLoopCntr;
    double dMin;
    double dMax;
    double dScale;

    MinMax(&dMin, &dMax, dpVect, lCount);
    dScale = 1.0 / (dMax - dMin);
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpVect[lLoopCntr] = (dpVect[lLoopCntr] - dMin) * dScale;
    }
}


void clDSPOp::Scale01 (float *fpDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;
    float fMin;
    float fMax;
    float fScale;

    MinMax(&fMin, &fMax, fpSrc, lCount);
    fScale = 1.0F / (fMax - fMin);
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = (fpSrc[lLoopCntr] - fMin) * fScale;
    }
}


void clDSPOp::Scale01 (double *dpDest, const double *dpSrc, long lCount)
{
    long lLoopCntr;
    double dMin;
    double dMax;
    double dScale;

    MinMax(&dMin, &dMax, dpSrc, lCount);
    dScale = 1.0 / (dMax - dMin);
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = (dpSrc[lLoopCntr] - dMin) * dScale;
    }
}


void clDSPOp::Sort (float *fpVect, long lCount)
{
    qsort(fpVect, (size_t) lCount, sizeof(float), FloatCompare);
}


void clDSPOp::Sort (double *dpVect, long lCount)
{
    qsort(dpVect, (size_t) lCount, sizeof(double), DoubleCompare);
}


void clDSPOp::StdDev (float *fpStdDev, float *fpMean, const float *fpSrc,
    long lCount)
{
    long lLoopCntr;
    float fMean = 0.0F;
    float fTempVal;
    float fTemp = 0.0F;
    float fStdDev;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fMean += fpSrc[lLoopCntr];
    }
    fMean /= (float) lCount;
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fTempVal = fpSrc[lLoopCntr] - fMean;
        #ifndef _ISOC9X_SOURCE
            fTemp += fTempVal * fTempVal;
        #else
            fTemp = fmaf(fTempVal, fTempVal, fTemp);
        #endif
    }
    fStdDev = sqrtf(fTemp / (float) lCount);
    *fpStdDev = fStdDev;
    *fpMean = fMean;
}


void clDSPOp::StdDev (double *dpStdDev, double *dpMean, const double *dpSrc,
    long lCount)
{
    long lLoopCntr;
    double dMean = 0.0;
    double dTempVal;
    double dTemp = 0.0;
    double dStdDev;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dMean += dpSrc[lLoopCntr];
    }
    dMean /= (double) lCount;
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dTempVal = dpSrc[lLoopCntr] - dMean;
        #ifndef _ISOC9X_SOURCE
            dTemp += dTempVal * dTempVal;
        #else
            dTemp = fma(dTempVal, dTempVal, dTemp);
        #endif
    }
    dStdDev = sqrt(dTemp / (double) lCount);
    *dpStdDev = dStdDev;
    *dpMean = dMean;
}


float clDSPOp::Sum (const float *fpSrc, long lCount)
{
    long lLoopCntr;
    float fSum = 0.0F;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fSum += fpSrc[lLoopCntr];
    }
    return fSum;
}


double clDSPOp::Sum (const double *dpSrc, long lCount)
{
    long lLoopCntr;
    double dSum = 0.0;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dSum += dpSrc[lLoopCntr];
    }
    return dSum;
}


void clDSPOp::Square (float *fpVect, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpVect[lLoopCntr] *= fpVect[lLoopCntr];
    }
}


void clDSPOp::Square (double *dpVect, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpVect[lLoopCntr] *= dpVect[lLoopCntr];
    }
}


void clDSPOp::Square (float *fpDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = fpSrc[lLoopCntr] * fpSrc[lLoopCntr];
    }
}


void clDSPOp::Square (double *dpDest, const double *dpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = dpSrc[lLoopCntr] * dpSrc[lLoopCntr];
    }
}


void clDSPOp::Convert (float *fpDest, const unsigned char *ucpSrc, long lCount)
{
    long lLoopCntr;
    float fMult;

    fMult = 1.0F / (float) UCHAR_MAX;
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = (ucpSrc[lLoopCntr] * fMult - 0.5F) * 2.0F;
    }
}


void clDSPOp::Convert (float *fpDest, const signed short *sspSrc, long lCount,
    bool b12bit)
{
    long lLoopCntr;
    float fMult;

    fMult = (b12bit) ? (1.0F / (float) 0x1000) : (1.0F / (float) SHRT_MAX);
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = sspSrc[lLoopCntr] * fMult;
    }
}


void clDSPOp::Convert (float *fpDest, const signed int *sipSrc, long lCount,
    bool b24bit)
{
    long lLoopCntr;
    float fMult;

    fMult = (b24bit) ? 
        (1.0F / (float) 0x7fffff00) : 
        (1.0F / (float) INT_MAX);
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = sipSrc[lLoopCntr] * fMult;
    }
}


void clDSPOp::Convert (float *fpDest, const double *dpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = (float) dpSrc[lLoopCntr];
    }
}


void clDSPOp::Convert (double *dpDest, const unsigned char *ucpSrc, 
    long lCount)
{
    long lLoopCntr;
    double dMult;

    dMult = 1.0 / (double) UCHAR_MAX;
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = (ucpSrc[lLoopCntr] * dMult - 0.5) * 2.0;
    }
}


void clDSPOp::Convert (double *dpDest, const signed short *sspSrc, long lCount,
    bool b12bit)
{
    long lLoopCntr;
    double dMult;

    dMult = (b12bit) ? (1.0 / (double) 0x1000) : (1.0 / (double) SHRT_MAX);
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = sspSrc[lLoopCntr] * dMult;
    }
}


void clDSPOp::Convert (double *dpDest, const signed int *sipSrc, long lCount,
    bool b24bit)
{
    long lLoopCntr;
    double dMult;

    dMult = (b24bit) ? 
        (1.0 / (double) 0x7fffff00) : 
        (1.0 / (double) INT_MAX);
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = sipSrc[lLoopCntr] * dMult;
    }
}


void clDSPOp::Convert (double *dpDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = (double) fpSrc[lLoopCntr];
    }
}


void clDSPOp::Convert (unsigned char *ucpDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        ucpDest[lLoopCntr] = (unsigned char)
            Round((fpSrc[lLoopCntr] + 1.0F) / 2.0F * (float) UCHAR_MAX);
    }
}


void clDSPOp::Convert (unsigned char *ucpDest, const double *dpSrc, 
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        ucpDest[lLoopCntr] = (unsigned char)
            Round((dpSrc[lLoopCntr] + 1.0) / 2.0 * (double) UCHAR_MAX);
    }
}


void clDSPOp::Convert (signed short *sspDest, const float *fpSrc, long lCount,
    bool b12bit)
{
    long lLoopCntr;
    float fMult;

    fMult = (b12bit) ? (float) 0x1000 : (float) SHRT_MAX;
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        sspDest[lLoopCntr] = (signed short)
            Round(fpSrc[lLoopCntr] * fMult);
    }
}


void clDSPOp::Convert (signed short *sspDest, const double *dpSrc, long lCount,
    bool b12bit)
{
    long lLoopCntr;
    double dMult;

    dMult = (b12bit) ? (double) 0x1000 : (double) SHRT_MAX;
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        sspDest[lLoopCntr] = (signed short) 
            Round(dpSrc[lLoopCntr] * dMult);
    }
}


void clDSPOp::Convert (signed int *sipDest, const float *fpSrc, long lCount,
    bool b24bit)
{
    long lLoopCntr;
    float fMult;

    fMult = (b24bit) ? ((float) 0x7fffff00) : ((float) INT_MAX);
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        sipDest[lLoopCntr] = (signed int)
            Round(fpSrc[lLoopCntr] * fMult);
    }
}


void clDSPOp::Convert (signed int *sipDest, const double *dpSrc, long lCount,
    bool b24bit)
{
    long lLoopCntr;
    double dMult;

    dMult = (b24bit) ? ((double) 0x7fffff00) : ((double) INT_MAX);
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        sipDest[lLoopCntr] = (signed int)
            Round(dpSrc[lLoopCntr] * dMult);
    }
}


void clDSPOp::CartToPolar (float *fpMagn, float *fpPhase, 
    const float *fpReal, const float *fpImag, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        Cart2Polar(&fpMagn[lLoopCntr], &fpPhase[lLoopCntr], 
            fpReal[lLoopCntr], fpImag[lLoopCntr]);
    }
}


void clDSPOp::CartToPolar (double *dpMagn, double *dpPhase, 
    const double *dpReal, const double *dpImag, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        Cart2Polar(&dpMagn[lLoopCntr], &dpPhase[lLoopCntr],
            dpReal[lLoopCntr], dpImag[lLoopCntr]);
    }
}


void clDSPOp::CartToPolar (float *fpMagn, float *fpPhase,
    const stpSCplx spCart, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        Cart2Polar(&fpMagn[lLoopCntr], &fpPhase[lLoopCntr],
            &spCart[lLoopCntr]);
    }
}


void clDSPOp::CartToPolar (double *dpMagn, double *dpPhase,
    const stpDCplx spCart, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        Cart2Polar(&dpMagn[lLoopCntr], &dpPhase[lLoopCntr],
            &spCart[lLoopCntr]);
    }
}


void clDSPOp::CartToPolar (stpSPolar spPolar, const stpSCplx spCart,
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        Cart2Polar(&spPolar[lLoopCntr], &spCart[lLoopCntr]);
    }
}


void clDSPOp::CartToPolar (stpDPolar spPolar, const stpDCplx spCart,
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        Cart2Polar(&spPolar[lLoopCntr], &spCart[lLoopCntr]);
    }
}


void clDSPOp::CartToPolar (utpSCoord upCoord, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        Cart2Polar(&upCoord[lLoopCntr]);
    }
}


void clDSPOp::CartToPolar (utpDCoord upCoord, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        Cart2Polar(&upCoord[lLoopCntr]);
    }
}


void clDSPOp::PolarToCart (float *fpReal, float *fpImag, 
    const float *fpMagn, const float *fpPhase, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        Polar2Cart(&fpReal[lLoopCntr], &fpImag[lLoopCntr],
            fpMagn[lLoopCntr], fpPhase[lLoopCntr]);
    }
}


void clDSPOp::PolarToCart (double *dpReal, double *dpImag,
    const double *dpMagn, const double *dpPhase, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        Polar2Cart(&dpReal[lLoopCntr], &dpImag[lLoopCntr],
            dpMagn[lLoopCntr], dpPhase[lLoopCntr]);
    }
}


void clDSPOp::PolarToCart (stpSCplx spCart, const float *fpMagn,
    const float *fpPhase, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        Polar2Cart(&spCart[lLoopCntr], fpMagn[lLoopCntr], fpPhase[lLoopCntr]);
    }
}


void clDSPOp::PolarToCart (stpDCplx spCart, const double *dpMagn,
    const double *dpPhase, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        Polar2Cart(&spCart[lLoopCntr], dpMagn[lLoopCntr], dpPhase[lLoopCntr]);
    }
}


void clDSPOp::PolarToCart (stpSCplx spCart, const stpSPolar spPolar,
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        Polar2Cart(&spCart[lLoopCntr], &spPolar[lLoopCntr]);
    }
}


void clDSPOp::PolarToCart (stpDCplx spCart, const stpDPolar spPolar,
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        Polar2Cart(&spCart[lLoopCntr], &spPolar[lLoopCntr]);
    }
}


void clDSPOp::PolarToCart (utpSCoord upCoord, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        Polar2Cart(&upCoord[lLoopCntr]);
    }
}


void clDSPOp::PolarToCart (utpDCoord upCoord, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        Polar2Cart(&upCoord[lLoopCntr]);
    }
}


float clDSPOp::CrossCorr (const float *fpSrc1, const float *fpSrc2, 
    long lCount)
{
    long lLoopCntr;
    #ifndef DSP_EXTPREC
        float fScale;
        float fNormFact;
        float fProdSum = 0.0F;
        float fSqSum1 = 0.0F;
        float fSqSum2 = 0.0F;
    #else
        double fScale;
        double fNormFact;
        double fProdSum = 0.0;
        double fSqSum1 = 0.0;
        double fSqSum2 = 0.0;
    #endif

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            fProdSum += fpSrc1[lLoopCntr] * fpSrc2[lLoopCntr];
            fSqSum1 += fpSrc1[lLoopCntr] * fpSrc1[lLoopCntr];
            fSqSum2 += fpSrc2[lLoopCntr] * fpSrc2[lLoopCntr];
        #else
            #ifndef DSP_EXTPREC
                fProdSum = fmaf(fpSrc1[lLoopCntr], fpSrc2[lLoopCntr], 
                    fProdSum);
                fSqSum1 = fmaf(fpSrc1[lLoopCntr], fpSrc1[lLoopCntr],
                    fSqSum1);
                fSqSum2 = fmaf(fpSrc2[lLoopCntr], fpSrc2[lLoopCntr],
                    fSqSum2);
            #else
                fProdSum = fma(fpSrc1[lLoopCntr], fpSrc2[lLoopCntr],
                    fProdSum);
                fSqSum1 = fma(fpSrc1[lLoopCntr], fpSrc1[lLoopCntr],
                    fSqSum1);
                fSqSum2 = fma(fpSrc2[lLoopCntr], fpSrc2[lLoopCntr],
                    fSqSum2);
            #endif
        #endif
    }
    #ifndef DSP_EXTPREC
        fScale = 1.0F / lCount;
        fNormFact = sqrtf(fSqSum1 * fSqSum2) * fScale;
    #else
        fScale = 1.0 / lCount;
        fNormFact = sqrt(fSqSum1 * fSqSum2) * fScale;
    #endif
    return ((fProdSum * fScale) / fNormFact);
}


double clDSPOp::CrossCorr (const double *dpSrc1, const double *dpSrc2, 
    long lCount)
{
    long lLoopCntr;
    #ifndef DSP_EXTPREC
        double dScale;
        double dNormFact;
        double dProdSum = 0.0;
        double dSqSum1 = 0.0;
        double dSqSum2 = 0.0;
    #else
        long double dScale;
        long double dNormFact;
        long double dProdSum = 0.0L;
        long double dSqSum1 = 0.0L;
        long double dSqSum2 = 0.0L;
    #endif
   
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            dProdSum += dpSrc1[lLoopCntr] * dpSrc2[lLoopCntr];
            dSqSum1 += dpSrc1[lLoopCntr] * dpSrc1[lLoopCntr];
            dSqSum2 += dpSrc2[lLoopCntr] * dpSrc2[lLoopCntr];
        #else
            #ifndef DSP_EXTPREC
                dProdSum = fma(dpSrc1[lLoopCntr], dpSrc2[lLoopCntr],
                    dProdSum);
                dSqSum1 = fma(dpSrc1[lLoopCntr], dpSrc1[lLoopCntr],
                    dSqSum1);
                dSqSum2 = fma(dpSrc2[lLoopCntr], dpSrc2[lLoopCntr],
                    dSqSum2);
            #else
                dProdSum = fmal(dpSrc1[lLoopCntr], dpSrc2[lLoopCntr],
                    dProdSum);
                dSqSum1 = fmal(dpSrc1[lLoopCntr], dpSrc1[lLoopCntr],
                    dSqSum1);
                dSqSum2 = fmal(dpSrc2[lLoopCntr], dpSrc2[lLoopCntr],
                    dSqSum2);
            #endif
        #endif
    }
    #ifndef DSP_EXTPREC
        dScale = 1.0 / lCount;
        dNormFact = sqrt(dSqSum1 * dSqSum2) * dScale;
    #else
        dScale = 1.0L / lCount;
        dNormFact = sqrtl(dSqSum1 * dSqSum2) * dScale;
    #endif
    return ((dProdSum * dScale) / dNormFact);
}


float clDSPOp::DelCrossCorr (const float *fpSrc1, const float *fpSrc2,
    long lDelay, long lCount)
{
    long lLoopCntr;
    long lMax;
    long lIdx2;
    float fScale;
    float fNormFact;
    float fProdSum = 0.0F;
    float fSqSum1 = 0.0F;
    float fSqSum2 = 0.0F;

    lMax = lCount - lDelay;
    for (lLoopCntr = 0L; lLoopCntr < lMax; lLoopCntr++)
    {
        lIdx2 = lLoopCntr + lDelay;
        #ifndef _ISOC9X_SOURCE
            fProdSum += fpSrc1[lLoopCntr] * fpSrc2[lIdx2];
            fSqSum1 += fpSrc1[lLoopCntr] * fpSrc1[lLoopCntr];
            fSqSum2 += fpSrc2[lIdx2] * fpSrc2[lIdx2];
        #else
            fProdSum = fmaf(fpSrc1[lLoopCntr], fpSrc2[lIdx2], fProdSum);
            fSqSum1 = fmaf(fpSrc1[lLoopCntr], fpSrc1[lLoopCntr], fSqSum1);
            fSqSum2 = fmaf(fpSrc2[lIdx2], fpSrc2[lIdx2], fSqSum2);
        #endif
    }
    fScale = 1.0F / (float) lCount;
    fNormFact = sqrtf(fSqSum1 * fSqSum2) * fScale;
    return ((fProdSum * fScale) / fNormFact);
}


double clDSPOp::DelCrossCorr (const double *dpSrc1, const double *dpSrc2,
    long lDelay, long lCount)
{
    long lLoopCntr;
    long lMax;
    long lIdx2;
    double dScale;
    double dNormFact;
    double dProdSum = 0.0;
    double dSqSum1 = 0.0;
    double dSqSum2 = 0.0;

    lMax = lCount - lDelay;
    for (lLoopCntr = 0L; lLoopCntr < lMax; lLoopCntr++)
    {
        lIdx2 = lLoopCntr + lDelay;
        #ifndef _ISOC9X_SOURCE
            dProdSum += dpSrc1[lLoopCntr] * dpSrc2[lIdx2];
            dSqSum1 += dpSrc1[lLoopCntr] * dpSrc1[lLoopCntr];
            dSqSum2 += dpSrc2[lIdx2] * dpSrc2[lIdx2];
        #else
            dProdSum = fma(dpSrc1[lLoopCntr], dpSrc2[lIdx2], dProdSum);
            dSqSum1 = fma(dpSrc1[lLoopCntr], dpSrc1[lLoopCntr], dSqSum1);
            dSqSum2 = fma(dpSrc2[lIdx2], dpSrc2[lIdx2], dSqSum2);
        #endif
    }
    dScale = 1.0 / (double) lCount;
    dNormFact = sqrt(dSqSum1 * dSqSum2) * dScale;
    return ((dProdSum * dScale) / dNormFact);
}


void clDSPOp::DelCrossCorr (float *fpDest, const float *fpSrc1, 
    const float *fpSrc2, long lCount, const long *lpDelay, long lDelayCount)
{
    long lLoopDelay;
    long lLoopCorr;
    long lDelay;
    long lMax;
    long lIdx2;
    float fScale;
    float fNormFact;
    float fProdSum;
    float fSqSum1;
    float fSqSum2;

    for (lLoopDelay = 0L; lLoopDelay < lDelayCount; lLoopDelay++)
    {
        fProdSum = 0.0F;
        fSqSum1 = 0.0F;
        fSqSum2 = 0.0F;
        lDelay = lpDelay[lLoopDelay];
        lMax = lCount - lDelay;
        for (lLoopCorr = 0L; lLoopCorr < lMax; lLoopCorr++)
        {
            lIdx2 = lLoopCorr + lDelay;
            #ifndef _ISOC9X_SOURCE
                fProdSum += fpSrc1[lLoopCorr] * fpSrc2[lIdx2];
                fSqSum1 += fpSrc1[lLoopCorr] * fpSrc1[lLoopCorr];
                fSqSum2 += fpSrc2[lIdx2] * fpSrc2[lIdx2];
            #else
                fProdSum = fmaf(fpSrc1[lLoopCorr], fpSrc2[lIdx2], fProdSum);
                fSqSum1 = fmaf(fpSrc1[lLoopCorr], fpSrc1[lLoopCorr], fSqSum1);
                fSqSum2 = fmaf(fpSrc2[lIdx2], fpSrc2[lIdx2], fSqSum2);
            #endif
        }
        fScale = 1.0F / (float) lCount;
        fNormFact = sqrtf(fSqSum1 * fSqSum2) * fScale;
        fpDest[lLoopDelay] = (fProdSum * fScale) / fNormFact;
    }
}


void clDSPOp::DelCrossCorr (double *dpDest, const double *dpSrc1,
    const double *dpSrc2, long lCount, const long *lpDelay, long lDelayCount)
{
    long lLoopDelay;
    long lLoopCorr;
    long lDelay;
    long lMax;
    long lIdx2;
    double dScale;
    double dNormFact;
    double dProdSum;
    double dSqSum1;
    double dSqSum2;

    for (lLoopDelay = 0L; lLoopDelay < lDelayCount; lLoopDelay++)
    {
        dProdSum = 0.0;
        dSqSum1 = 0.0;
        dSqSum2 = 0.0;
        lDelay = lpDelay[lLoopDelay];
        lMax = lCount - lDelay;
        for (lLoopCorr = 0L; lLoopCorr < lMax; lLoopCorr++)
        {
            lIdx2 = lLoopCorr + lDelay;
            #ifndef _ISOC9X_SOURCE
                dProdSum += dpSrc1[lLoopCorr] * dpSrc2[lIdx2];
                dSqSum1 += dpSrc1[lLoopCorr] * dpSrc1[lLoopCorr];
                dSqSum2 += dpSrc2[lIdx2] * dpSrc2[lIdx2];
            #else
                dProdSum = fma(dpSrc1[lLoopCorr], dpSrc2[lIdx2], dProdSum);
                dSqSum1 = fma(dpSrc1[lLoopCorr], dpSrc1[lLoopCorr], dSqSum1);
                dSqSum2 = fma(dpSrc2[lIdx2], dpSrc2[lIdx2], dSqSum2);
            #endif
        }
        dScale = 1.0 / (double) lCount;
        dNormFact = sqrt(dSqSum1 * dSqSum2) * dScale;
        dpDest[lLoopDelay] = (dProdSum * dScale) / dNormFact;
    }
}


float clDSPOp::Energy (const float *fpSrc, long lCount)
{
    long lLoopCntr;
    float fEnergy = 0.0F;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            fEnergy += fpSrc[lLoopCntr] * fpSrc[lLoopCntr];
        #else
            fEnergy = fmaf(fpSrc[lLoopCntr], fpSrc[lLoopCntr], fEnergy);
        #endif
    }
    return fEnergy;
}


double clDSPOp::Energy (const double *dpSrc, long lCount)
{
    long lLoopCntr;
    double dEnergy = 0.0;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            dEnergy += dpSrc[lLoopCntr] * dpSrc[lLoopCntr];
        #else
            dEnergy = fma(dpSrc[lLoopCntr], dpSrc[lLoopCntr], dEnergy);
        #endif
    }
    return dEnergy;
}


void clDSPOp::Magnitude (float *fpMagn, const stpSCplx spCplx, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            fpMagn[lLoopCntr] = sqrtf(
                spCplx[lLoopCntr].R * spCplx[lLoopCntr].R +
                spCplx[lLoopCntr].I * spCplx[lLoopCntr].I);
        #else
            fpMagn[lLoopCntr] = 
                hypotf(spCplx[lLoopCntr].R, spCplx[lLoopCntr].I);
        #endif
    }
}


void clDSPOp::Magnitude (double *dpMagn, const stpDCplx spCplx, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            dpMagn[lLoopCntr] = sqrt(
                spCplx[lLoopCntr].R * spCplx[lLoopCntr].R +
                spCplx[lLoopCntr].I * spCplx[lLoopCntr].I);
        #else
            dpMagn[lLoopCntr] =
                hypot(spCplx[lLoopCntr].R, spCplx[lLoopCntr].I);
        #endif
    }
}


void clDSPOp::Power (float *fpPower, const stpSCplx spCplx, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            fpPower[lLoopCntr] = 20.0F * log10f(sqrtf(
                spCplx[lLoopCntr].R * spCplx[lLoopCntr].R +
                spCplx[lLoopCntr].I * spCplx[lLoopCntr].I));
        #else
            fpPower[lLoopCntr] = 20.0F * log10f(
                hypotf(spCplx[lLoopCntr].R, spCplx[lLoopCntr].I));
        #endif
    }
}


void clDSPOp::Power (double *dpPower, const stpDCplx spCplx, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            dpPower[lLoopCntr] = 20.0 * log10(sqrt(
                spCplx[lLoopCntr].R * spCplx[lLoopCntr].R +
                spCplx[lLoopCntr].I * spCplx[lLoopCntr].I));
        #else
            dpPower[lLoopCntr] = 20.0 * log10(
                hypot(spCplx[lLoopCntr].R, spCplx[lLoopCntr].I));
        #endif
    }
}


void clDSPOp::Phase (float *fpPhase, const stpSCplx spCplx, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpPhase[lLoopCntr] =
            atan2f(spCplx[lLoopCntr].I, spCplx[lLoopCntr].R);
    }
}


void clDSPOp::Phase (double *dpPhase, const stpDCplx spCplx, long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpPhase[lLoopCntr] = 
            atan2(spCplx[lLoopCntr].I, spCplx[lLoopCntr].R);
    }
}



void clDSPOp::PowerPhase (float *fpPower, float *fpPhase, 
    const stpSCplx spCplx, long lCount)
{
    long lLoopCntr;
    float fReal;
    float fImag;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fReal = spCplx[lLoopCntr].R;
        fImag = spCplx[lLoopCntr].I;
        #ifndef _ISOC9X_SOURCE
            fpPower[lLoopCntr] = 
                20.0F * log10f(sqrtf(fReal * fReal + fImag * fImag));
        #else
            fpPower[lLoopCntr] =
                20.0F * log10f(hypotf(fReal, fImag));
        #endif
        fpPhase[lLoopCntr] = atan2f(fImag, fReal);
    }
}


void clDSPOp::PowerPhase (double *dpPower, double *dpPhase,
    const stpDCplx spCplx, long lCount)
{
    long lLoopCntr;
    double dReal;
    double dImag;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dReal = spCplx[lLoopCntr].R;
        dImag = spCplx[lLoopCntr].I;
        #ifndef _ISOC9X_SOURCE
            dpPower[lLoopCntr] = 
                20.0 * log10(sqrt(dReal * dReal + dImag * dImag));
        #else
            dpPower[lLoopCntr] =
                20.0 * log10(hypot(dReal, dImag));
        #endif
        dpPhase[lLoopCntr] = atan2(dImag, dReal);
    }
}


void clDSPOp::Decimate (float *fpDest, const float *fpSrc, long lFactor,
    long lCount)
{
    long lLoopCntr;
    long lMax;

    lMax = lCount / lFactor;
    for (lLoopCntr = 0L; lLoopCntr < lMax; lLoopCntr++)
    {
        fpDest[lLoopCntr] = fpSrc[lLoopCntr * lFactor];
    }
}


void clDSPOp::Decimate (double *dpDest, const double *dpSrc, long lFactor,
    long lCount)
{
    long lLoopCntr;
    long lMax;

    lMax = lCount / lFactor;
    for (lLoopCntr = 0L; lLoopCntr < lMax; lLoopCntr++)
    {
        dpDest[lLoopCntr] = dpSrc[lLoopCntr * lFactor];
    }
}


void clDSPOp::DecimateAvg (float *fpDest, const float *fpSrc, long lFactor,
    long lCount)
{
    long lLoopDest;
    long lLoopAvg;
    long lMax;
    float fAvg;

    lMax = lCount / lFactor;
    for (lLoopDest = 0L; lLoopDest < lMax; lLoopDest++)
    {
        fAvg = 0.0F;
        for (lLoopAvg = 0L; lLoopAvg < lFactor; lLoopAvg++)
        {
            fAvg += fpSrc[lLoopDest * lFactor + lLoopAvg];
        }
        fpDest[lLoopDest] = fAvg / (float) lFactor;
    }
}


void clDSPOp::DecimateAvg (double *dpDest, const double *dpSrc, long lFactor,
    long lCount)
{
    long lLoopDest;
    long lLoopAvg;
    long lMax;
    double dAvg;

    lMax = lCount / lFactor;
    for (lLoopDest = 0L; lLoopDest < lMax; lLoopDest++)
    {
        dAvg = 0.0;
        for (lLoopAvg = 0L; lLoopAvg < lFactor; lLoopAvg++)
        {
            dAvg += dpSrc[lLoopDest * lFactor + lLoopAvg];
        }
        dpDest[lLoopDest] = dAvg / (double) lFactor;
    }
}


void clDSPOp::Interpolate (float *fpDest, const float *fpSrc, long lFactor,
    long lCount)
{
    long lLoopSrc;
    long lLoopInt;

    for (lLoopSrc = 0L; lLoopSrc < lCount; lLoopSrc++)
    {
        fpDest[lLoopSrc * lFactor] = fpSrc[lLoopSrc];
        for (lLoopInt = 1L; lLoopInt < lFactor; lLoopInt++)
        {
            fpDest[lLoopSrc * lFactor + lLoopInt] = 0.0F;
        }
    }
}


void clDSPOp::Interpolate (double *dpDest, const double *dpSrc, long lFactor,
    long lCount)
{
    long lLoopSrc;
    long lLoopInt;

    for (lLoopSrc = 0L; lLoopSrc < lCount; lLoopSrc++)
    {
        dpDest[lLoopSrc * lFactor] = dpSrc[lLoopSrc];
        for (lLoopInt = 1L; lLoopInt < lFactor; lLoopInt++)
        {
            dpDest[lLoopSrc * lFactor + lLoopInt] = 0.0;
        }
    }
}


void clDSPOp::InterpolateAvg (float *fpDest, const float *fpSrc, 
    long lFactor, long lCount)
{
    long lSrcCntr;
    long lIntCntr;
    long lX;
    long lX0;
    long lX1;
    float fF0;
    float fF1;
    float fL0;
    float fL1;
    float fP1;

    for (lSrcCntr = 0L; lSrcCntr < lCount; lSrcCntr++)
    {
        lX0 = lSrcCntr * lFactor;
        lX1 = (lSrcCntr + 1L) * lFactor;
        if (lSrcCntr != (lCount - 1L))
        {
            fF0 = fpSrc[lSrcCntr];
            fF1 = fpSrc[lSrcCntr + 1L];
        }
        else
        {
            fF0 = fpSrc[lSrcCntr];
            fF1 = fpSrc[lSrcCntr] - (fpSrc[lSrcCntr - 1L] - fpSrc[lSrcCntr]);
        }
        fpDest[lSrcCntr * lFactor] = fF0;
        for (lIntCntr = 1L; lIntCntr < lFactor; lIntCntr++)
        {
            lX = lSrcCntr * lFactor + lIntCntr;
            fL0 = (float) (lX - lX1) / (float) (lX0 - lX1);
            fL1 = (float) (lX - lX0) / (float) (lX1 - lX0);
            fP1 = fL0 * fF0 + fL1 * fF1;
            fpDest[lX] = fP1;
        }
    }
}


void clDSPOp::InterpolateAvg (double *dpDest, const double *dpSrc,
    long lFactor, long lCount)
{
    long lSrcCntr;
    long lIntCntr;
    long lX;
    long lX0;
    long lX1;
    double dF0;
    double dF1;
    double dL0;
    double dL1;
    double dP1;

    for (lSrcCntr = 0L; lSrcCntr < lCount; lSrcCntr++)
    {
        lX0 = lSrcCntr * lFactor;
        lX1 = (lSrcCntr + 1L) * lFactor;
        if (lSrcCntr != (lCount - 1L))
        {
            dF0 = dpSrc[lSrcCntr];
            dF1 = dpSrc[lSrcCntr + 1L];
        }
        else
        {
            dF0 = dpSrc[lSrcCntr];
            dF1 = dpSrc[lSrcCntr] - (dpSrc[lSrcCntr - 1L] - dpSrc[lSrcCntr]);
        }
        dpDest[lSrcCntr * lFactor] = dF0;
        for (lIntCntr = 1L; lIntCntr < lFactor; lIntCntr++)
        {
            lX = lSrcCntr * lFactor + lIntCntr;
            dL0 = (double) (lX - lX1) / (double) (lX0 - lX1);
            dL1 = (double) (lX - lX0) / (double) (lX1 - lX0);
            dP1 = dL0 * dF0 + dL1 * dF1;
            dpDest[lX] = dP1;
        }
    }
}


float clDSPOp::RMS (const float *fpSrc, long lCount)
{
    long lLoopCntr;
    float fSqSum = 0.0F;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            fSqSum += fpSrc[lLoopCntr] * fpSrc[lLoopCntr];
        #else
            fSqSum = fmaf(fpSrc[lLoopCntr], fpSrc[lLoopCntr], fSqSum);
        #endif
    }
    return ((float) sqrtf(fSqSum / (float) lCount));
}


double clDSPOp::RMS (const double *dpSrc, long lCount)
{
    long lLoopCntr;
    double dSqSum = 0.0;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        #ifndef _ISOC9X_SOURCE
            dSqSum += dpSrc[lLoopCntr] * dpSrc[lLoopCntr];
        #else
            dSqSum = fma(dpSrc[lLoopCntr], dpSrc[lLoopCntr], dSqSum);
        #endif
    }
    return (sqrt(dSqSum / (double) lCount));
}


float clDSPOp::Variance (float *fpVariance, float *fpMean, 
    const float *fpSrc, long lCount)
{
    long lLoopCntr;
    float fMean = 0.0F;
    float fVariance = 0.0F;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fMean += fpSrc[lLoopCntr];
    }
    fMean /= (float) lCount;
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fVariance += (float) powf(fpSrc[lLoopCntr] - fMean, 2.0F);
    }
    fVariance /= (float) lCount;
    if (fpVariance != NULL)
    {
        *fpVariance = fVariance;
    }
    if (fpMean != NULL)
    {
        *fpMean = fMean;
    }
    return fVariance;
}


double clDSPOp::Variance (double *dpVariance, double *dpMean,
    const double *dpSrc, long lCount)
{
    long lLoopCntr;
    double dMean = 0.0;
    double dVariance = 0.0;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dMean += dpSrc[lLoopCntr];
    }
    dMean /= (double) lCount;
    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dVariance += (double) pow(dpSrc[lLoopCntr] - dMean, 2.0);
    }
    dVariance /= (double) lCount;
    if (dpVariance != NULL)
    {
        *dpVariance = dVariance;
    }
    if (dpMean != NULL)
    {
        *dpMean = dMean;
    }
    return dVariance;
}


float clDSPOp::PeakLevel (const float *fpSrc, long lSize)
{
    float fMin;
    float fMax;
    float fLinMax;

    MinMax(&fMin, &fMax, fpSrc, lSize);
    fMin = fabsf(fMin);
    fMax = fabsf(fMax);
    #ifndef _ISOC9X_SOURCE
        fLinMax = (fMax >= fMin) ? fMax : fMin;
    #else
        fLinMax = fmaxf(fMin, fMax);
    #endif
    return (20.0F * log10f(fLinMax));
}


double clDSPOp::PeakLevel (const double *dpSrc, long lSize)
{
    double dMin;
    double dMax;
    double dLinMax;

    MinMax(&dMin, &dMax, dpSrc, lSize);
    dMin = fabs(dMin);
    dMax = fabs(dMax);
    #ifndef _ISOC9X_SOURCE
        dLinMax = (dMax >= dMin) ? dMax : dMin;
    #else
        dLinMax = fmax(dMin, dMax);
    #endif
    return (20.0 * log10(dLinMax));
}


void clDSPOp::WinBartlett (float *fpDest, long lSize)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr <= ((lSize - 1L) / 2L); lLoopCntr++)
    {
        fpDest[lLoopCntr] = 2.0F * (float) lLoopCntr / (float) (lSize - 1L);
    }
    for (lLoopCntr = ((lSize - 1L) / 2L + 1L); lLoopCntr < lSize; lLoopCntr++)
    {
        fpDest[lLoopCntr] = 
            2.0F - 2.0F * (float) lLoopCntr / (float) (lSize - 1L);
    }
}


void clDSPOp::WinBartlett (double *dpDest, long lSize)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr <= ((lSize - 1L) / 2L); lLoopCntr++)
    {
        dpDest[lLoopCntr] = 2.0 * (double) lLoopCntr / (double) (lSize - 1L);
    }
    for (lLoopCntr = ((lSize - 1L) / 2L + 1L); lLoopCntr < lSize; lLoopCntr++)
    {
        dpDest[lLoopCntr] =
            2.0 - 2.0 * (double) lLoopCntr / (double) (lSize - 1L);
    }
}


void clDSPOp::WinBlackman (float *fpDest, long lSize)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        fpDest[lLoopCntr] = 0.42F - 
            0.5F * cos((2.0F * fPI * (float) lLoopCntr) / (float) lSize) +
            0.08F * cos((4.0F * fPI * (float) lLoopCntr) / (float) lSize);
    }
}


void clDSPOp::WinBlackman (double *dpDest, long lSize)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        dpDest[lLoopCntr] = 0.42 - 
            0.5 * cos((2.0 * dPI * (double) lLoopCntr) / (double) lSize) +
            0.08 * cos((4.0 * dPI * (double) lLoopCntr) / (double) lSize);
    }
}


void clDSPOp::WinBlackmanHarris (float *fpDest, long lSize)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        fpDest[lLoopCntr] = 0.42323F -
            0.49855F * 
            cos((2.0F * fPI * (float) lLoopCntr) / (float) lSize) +
            0.07922F * 
            cos((4.0F * fPI * (float) lLoopCntr) / (float) lSize);
    }
}


void clDSPOp::WinBlackmanHarris (double *dpDest, long lSize)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        dpDest[lLoopCntr] = 0.42323 -
            0.49855 *
            cos((2.0 * dPI * (double) lLoopCntr) / (double) lSize) +
            0.07922 *
            cos((4.0 * dPI * (double) lLoopCntr) / (double) lSize);
    }
}


void clDSPOp::WinCosTapered (float *fpDest, long lSize)
{
    long lLoopCntr;
    long lM;

    lM = Round((float) lSize / 10.0F);
    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        if ((lLoopCntr < lM) || (lLoopCntr > (lSize - lM - 1L)))
        {
            fpDest[lLoopCntr] = 0.5F * (1.0F -
                cos((2.0F * fPI * (float) lLoopCntr) / (float) lSize));
        }
        else
        {
            fpDest[lLoopCntr] = 1.0F;
        }
    }
}


void clDSPOp::WinCosTapered (double *dpDest, long lSize)
{
    long lLoopCntr;
    long lM;

    lM = Round((double) lSize / 10.0);
    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        if ((lLoopCntr < lM) || (lLoopCntr > (lSize - lM - 1L)))
        {
            dpDest[lLoopCntr] = 0.5 * (1.0 - 
                cos((2.0 * dPI * (double) lLoopCntr) / (double) lSize));
        }
        else
        {
            dpDest[lLoopCntr] = 1.0;
        }
    }
}


void clDSPOp::WinCosTaperedA (float *fpVect, long lSize)
{
    long lLoopCntr;
    long lM;

    lM = Round((float) lSize / 10.0F);
    for (lLoopCntr = 0L; lLoopCntr < lM; lLoopCntr++)
    {
        fpVect[lLoopCntr] = 0.5F * fpVect[lLoopCntr] *
            (1.0F - cos((2.0F * fPI * (float) lLoopCntr) / (float) lSize));
    }
    for (lLoopCntr = (lSize - lM); lLoopCntr < lSize; lLoopCntr++)
    {
        fpVect[lLoopCntr] = 0.5F * fpVect[lLoopCntr] *
            (1.0F - cos ((2.0F * fPI * (float) lLoopCntr) / (float) lSize));
    }
}


void clDSPOp::WinCosTaperedA (double *dpVect, long lSize)
{
    long lLoopCntr;
    long lM;

    lM = Round((double) lSize / 10.0);
    for (lLoopCntr = 0L; lLoopCntr < lM; lLoopCntr++)
    {
        dpVect[lLoopCntr] = 0.5 * dpVect[lLoopCntr] *
            (1.0 - cos((2.0 * dPI * (double) lLoopCntr) / (double) lSize));
    }
    for (lLoopCntr = (lSize - lM); lLoopCntr < lSize; lLoopCntr++)
    {
        dpVect[lLoopCntr] = 0.5 * dpVect[lLoopCntr] *
            (1.0 - cos ((2.0 * dPI * (double) lLoopCntr) / (double) lSize));
    }
}


void clDSPOp::WinCosTaperedA (float *fpDest, const float *fpSrc, long lSize)
{
    long lLoopCntr;
    long lM;

    lM = Round((float) lSize / 10.0F);
    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        if ((lLoopCntr < lM) || (lLoopCntr > (lSize - lM)))
        {
            fpDest[lLoopCntr] = 0.5F * fpSrc[lLoopCntr] * (1.0F - 
                cos((2.0F * fPI * (float) lLoopCntr) / (float) lSize));
        }
        else
        {
            fpDest[lLoopCntr] = fpSrc[lLoopCntr];
        }
    }
}


void clDSPOp::WinCosTaperedA (double *dpDest, const double *dpSrc, long lSize)
{
    long lLoopCntr;
    long lM;

    lM = Round((double) lSize / 10.0);
    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        if ((lLoopCntr < lM) || (lLoopCntr > (lSize - lM)))
        {
            dpDest[lLoopCntr] = 0.5 * dpSrc[lLoopCntr] * (1.0 -
                cos((2.0 * dPI * (double) lLoopCntr) / (double) lSize));
        }
        else
        {
            dpDest[lLoopCntr] = dpSrc[lLoopCntr];
        }
    }
}


void clDSPOp::WinExactBlackman (float *fpDest, long lSize)
{
    long lLoopCntr;
    float fA0;
    float fA1;
    float fA2;

    fA0 = 7938.0F / 18608.0F;
    fA1 = 9240.0F / 18608.0F;
    fA2 = 1430.0F / 18608.0F;
    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        fpDest[lLoopCntr] = fA0 - 
            fA1 * cos((2.0F * fPI * (float) lLoopCntr) / (float) lSize) +
            fA2 * cos((4.0F * fPI * (float) lLoopCntr) / (float) lSize);
    }
}


void clDSPOp::WinExactBlackman (double *dpDest, long lSize)
{
    long lLoopCntr;
    double dA0;
    double dA1;
    double dA2;

    dA0 = 7938.0 / 18608.0;
    dA1 = 9240.0 / 18608.0;
    dA2 = 1430.0 / 18608.0;
    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        dpDest[lLoopCntr] = dA0 -
            dA1 * cos((2.0 * dPI * (double) lLoopCntr) / (double) lSize) +
            dA2 * cos((4.0 * dPI * (double) lLoopCntr) / (double) lSize);
    }
}


void clDSPOp::WinExp (float *fpDest, float fFinal, long lSize)
{
    long lLoopCntr;
    float fA;

    fA = log(fFinal + 1.0F) / ((float) lSize / 2.0F);
    for (lLoopCntr = 0L; lLoopCntr < (lSize / 2L + 1L); lLoopCntr++)
    {
        fpDest[lLoopCntr] = fpDest[lSize - lLoopCntr - 1L] = 
            exp(fA * (float) lLoopCntr) - 1.0F;
    }
}


void clDSPOp::WinExp (double *dpDest, double dFinal, long lSize)
{
    long lLoopCntr;
    double dA;

    dA = log(dFinal + 1.0) / ((double) lSize / 2.0);
    for (lLoopCntr = 0L; lLoopCntr < (lSize / 2L + 1L); lLoopCntr++)
    {
        dpDest[lLoopCntr] = dpDest[lSize - lLoopCntr - 1L] = 
            exp(dA * (double) lLoopCntr) - 1.0;
    }
}


void clDSPOp::WinFlatTop (float *fpDest, long lSize)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        fpDest[lLoopCntr] = 0.2810639F - 
            0.5208972F * 
            cos((2.0F * fPI * (float) lLoopCntr) / (float) lSize) +
            0.1980399F * 
            cos((4.0F * fPI * (float) lLoopCntr) / (float) lSize);
    }
}


void clDSPOp::WinFlatTop (double *dpDest, long lSize)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        dpDest[lLoopCntr] = 0.2810639 -
            0.5208972 *
            cos((2.0 * dPI * (double) lLoopCntr) / (double) lSize) +
            0.1980399 *
            cos((4.0 * dPI * (double) lLoopCntr) / (double) lSize);
    }
}


void clDSPOp::WinGenericCos (float *fpDest, long lSize, 
    const float *fpCoeff, long lCoeffCount)
{
    long lLoopWin;
    long lLoopCoeff;
    float fWinCoeff;

    for (lLoopWin = 0L; lLoopWin < lSize; lLoopWin++)
    {
        fWinCoeff = 0.0F;
        for (lLoopCoeff = 0L; lLoopCoeff < lCoeffCount; lLoopCoeff++)
        {
            fWinCoeff += pow(-1.0F, (float) lLoopCoeff) * 
                fpCoeff[lLoopCoeff] * cos(
                (2.0F * fPI * (float) lLoopCoeff * (float) lLoopWin) /
                (float) lSize);
        }
        fpDest[lLoopWin] = fWinCoeff;
    }
}


void clDSPOp::WinGenericCos (double *dpDest, long lSize,
    const double *dpCoeff, long lCoeffCount)
{
    long lLoopWin;
    long lLoopCoeff;
    double dWinCoeff;

    for (lLoopWin = 0L; lLoopWin < lSize; lLoopWin++)
    {
        dWinCoeff = 0.0;
        for (lLoopCoeff = 0L; lLoopCoeff < lCoeffCount; lLoopCoeff++)
        {
            dWinCoeff += pow(-1.0, (double) lLoopCoeff) *
                dpCoeff[lLoopCoeff] * cos(
                (2.0 * dPI * (double) lLoopCoeff * (double) lLoopWin) /
                (double) lSize);
        }
        dpDest[lLoopWin] = dWinCoeff;
    }
}


void clDSPOp::WinHamming (float *fpDest, long lSize)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        fpDest[lLoopCntr] = 0.54F -
            0.46F * cos((2.0F * fPI * (float) lLoopCntr) / (float) lSize);
    }
}


void clDSPOp::WinHamming (double *dpDest, long lSize)
{
    long lLoopCntr;
   
    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        dpDest[lLoopCntr] = 0.54 - 
            0.46 * cos((2.0 * dPI * (double) lLoopCntr) / (double) lSize);
    }
}


void clDSPOp::WinHanning (float *fpDest, long lSize)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        fpDest[lLoopCntr] = 0.5F -
            0.5F * cos((2.0F * fPI * (float) lLoopCntr) / (float) lSize);
    }
}


void clDSPOp::WinHanning (double *dpDest, long lSize)
{
    long lLoopCntr;
   
    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        dpDest[lLoopCntr] = 0.5 - 
            0.5 * cos((2.0 * dPI * (double) lLoopCntr) / (double) lSize);
    }
}


void clDSPOp::WinKaiser (float *fpDest, float fBeta, long lSize)
{
    long lLoopCntr;
    float fA;

    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        fA = fabs(1.0F - (2.0F * (float) lLoopCntr) / (float) lSize);
        fpDest[lLoopCntr] =
            ModZeroBessel(fBeta * sqrt(1.0F - fA * fA)) /
            ModZeroBessel(fBeta);
    }
}


void clDSPOp::WinKaiser (double *dpDest, double dBeta, long lSize)
{
    long lLoopCntr;
    double dA;

    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        dA = fabs(1.0 - (2.0 * (double) lLoopCntr) / (double) lSize);
        dpDest[lLoopCntr] =
            ModZeroBessel(dBeta * sqrt(1.0 - dA * dA)) /
            ModZeroBessel(dBeta);
    }
}


void clDSPOp::WinKaiserBessel (float *fpDest, float fAlpha, long lSize)
{
    long lLoopCntr;
    float fHalfN;
    float fPiAlpha;

    fHalfN = (float) lSize / 2.0F;
    fPiAlpha = fPI * fAlpha;
    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        fpDest[lLoopCntr] =
            ModZeroBessel(fPiAlpha * 
            sqrt(1.0F - pow(((float) lLoopCntr - fHalfN) / fHalfN, 2.0F))) /
            ModZeroBessel(fPiAlpha);
    }
}


void clDSPOp::WinKaiserBessel (double *dpDest, double dAlpha, long lSize)
{
    long lLoopCntr;
    double dHalfN;
    double dPiAlpha;

    dHalfN = (double) lSize / 2.0;
    dPiAlpha = dPI * dAlpha;
    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        dpDest[lLoopCntr] =
            ModZeroBessel(dPiAlpha *
            sqrt(1.0 - pow(((double) lLoopCntr - dHalfN) / dHalfN, 2.0))) /
            ModZeroBessel(dPiAlpha);
    }
}


void clDSPOp::WinTukey (float *fpDest, long lSize)
{
    long lLoopCntr;
    long lHalfSize;

    lHalfSize = lSize / 2L;
    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        fpDest[lLoopCntr] = 0.5F * 
            (1.0F + cos((float) (lLoopCntr - lHalfSize) * fPI / lHalfSize));
    }
}


void clDSPOp::WinTukey (double *dpDest, long lSize)
{
    long lLoopCntr;
    long lHalfSize;

    lHalfSize = lSize / 2L;
    for (lLoopCntr = 0L; lLoopCntr < lSize; lLoopCntr++)
    {
        dpDest[lLoopCntr] = 0.5 *
            (1.0 + cos((float) (lLoopCntr - lHalfSize) * dPI / lHalfSize));
    }
}


void clDSPOp::Mix (float *fpDest, const float *fpSrc, long lCount)
{
    long lLoopCntr;
    long lSrcIdx;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        lSrcIdx = lLoopCntr << 1;
        fpDest[lLoopCntr] = (fpSrc[lSrcIdx] + fpSrc[lSrcIdx + 1L]) * 0.5F;
    }
}


void clDSPOp::Mix (double *dpDest, const double *dpSrc, long lCount)
{
    long lLoopCntr;
    long lSrcIdx;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        lSrcIdx = lLoopCntr << 1;
        dpDest[lLoopCntr] = (dpSrc[lSrcIdx] + dpSrc[lSrcIdx + 1L]) * 0.5;
    }
}


void clDSPOp::Mix (float *fpDest, const float *fpSrc1, const float *fpSrc2,
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fpDest[lLoopCntr] = (fpSrc1[lLoopCntr] + fpSrc2[lLoopCntr]) * 0.5F;
    }
}


void clDSPOp::Mix (double *dpDest, const double *dpSrc1, const double *dpSrc2,
    long lCount)
{
    long lLoopCntr;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dpDest[lLoopCntr] = (dpSrc1[lLoopCntr] + dpSrc2[lLoopCntr]) * 0.5;
    }
}


void clDSPOp::Mix (float *fpDest, const float *fpSrc, long lChCount,
    long lDestCount)
{
    long lDestCntr;
    long lSrcCntr;
    long lSrcMax;
    float fMix;
    float fScaler;

    fScaler = 1.0F / (float) lChCount;
    for (lDestCntr = 0L; lDestCntr < lDestCount; lDestCntr++)
    {
        fMix = 0.0F;
        lSrcMax = lDestCntr * lChCount + lChCount;
        for (lSrcCntr = lDestCntr * lChCount; lSrcCntr < lSrcMax; lSrcCntr++)
        {
            fMix += fpSrc[lSrcCntr];
        }
        fpDest[lDestCntr] = fMix * fScaler;
    }
}


void clDSPOp::Mix (double *dpDest, const double *dpSrc, long lChCount,
    long lDestCount)
{
    long lDestCntr;
    long lSrcCntr;
    long lSrcMax;
    double dMix;
    double dScaler;

    dScaler = 1.0 / (double) lChCount;
    for (lDestCntr = 0L; lDestCntr < lDestCount; lDestCntr++)
    {
        dMix = 0.0;
        lSrcMax = lDestCntr * lChCount + lChCount;
        for (lSrcCntr = lDestCntr * lChCount; lSrcCntr < lSrcMax; lSrcCntr++)
        {
            dMix += dpSrc[lSrcCntr];
        }
        dpDest[lDestCntr] = dMix * dScaler;
    }
}


void clDSPOp::Spatialize (float *fpDest1, float *fpDest2, 
    const float *fpSrc, long lCount)
{
    long lLoopCntr;
    float fTemp;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        fTemp = fpSrc[lLoopCntr];
        fpDest1[lLoopCntr] = fTemp;
        fpDest2[lLoopCntr] = -fTemp;
    }
}


void clDSPOp::Spatialize (double *dpDest1, double *dpDest2,
    const double *dpSrc, long lCount)
{
    long lLoopCntr;
    double dTemp;

    for (lLoopCntr = 0L; lLoopCntr < lCount; lLoopCntr++)
    {
        dTemp = dpSrc[lLoopCntr];
        dpDest1[lLoopCntr] = dTemp;
        dpDest2[lLoopCntr] = -dTemp;
    }
}


void clDSPOp::Spatialize (float *fpDest, const float *fpSrc, long lCount)
{
    long lSrcCntr;
    long lDestIdx;
    float fTemp;

    for (lSrcCntr = 0L; lSrcCntr < lCount; lSrcCntr++)
    {
        fTemp = fpSrc[lSrcCntr];
        lDestIdx = lSrcCntr << 1;
        fpDest[lDestIdx] = fTemp;
        fpDest[lDestIdx + 1L] = -fTemp;
    }
}


void clDSPOp::Spatialize (double *dpDest, const double *dpSrc, long lCount)
{
    long lSrcCntr;
    long lDestIdx;
    double dTemp;

    for (lSrcCntr = 0L; lSrcCntr < lCount; lSrcCntr++)
    {
        dTemp = dpSrc[lSrcCntr];
        lDestIdx = lSrcCntr << 1;
        dpDest[lDestIdx] = dTemp;
        dpDest[lDestIdx + 1L] = -dTemp;
    }
}


void clDSPOp::Extract (float *fpDest, const float *fpSrc, long lChannel,
    long lChCount, long lSrcLength)
{
    long lLoopCntr;
    long lMax;

    lMax = lSrcLength / lChCount;
    for (lLoopCntr = 0; lLoopCntr < lMax; lLoopCntr++)
    {
        fpDest[lLoopCntr] = fpSrc[lLoopCntr * lChCount + lChannel];
    }
}


void clDSPOp::Extract (double *dpDest, const double *dpSrc, long lChannel,
    long lChCount, long lSrcLength)
{
    long lLoopCntr;
    long lMax;

    lMax = lSrcLength / lChCount;
    for (lLoopCntr = 0; lLoopCntr < lMax; lLoopCntr++)
    {
        dpDest[lLoopCntr] = dpSrc[lLoopCntr * lChCount + lChannel];
    }
}


void clDSPOp::Pack (float *fpDest, const float *fpSrc, long lChannel,
    long lChCount, long lSrcLength)
{
    long lLoopCntr;

    for (lLoopCntr = 0; lLoopCntr < lSrcLength; lLoopCntr++)
    {
        fpDest[lLoopCntr * lChCount + lChannel] = fpSrc[lLoopCntr];
    }
}


void clDSPOp::Pack (double *dpDest, const double *dpSrc, long lChannel,
    long lChCount, long lSrcLength)
{
    long lLoopCntr;

    for (lLoopCntr = 0; lLoopCntr < lSrcLength; lLoopCntr++)
    {
        dpDest[lLoopCntr * lChCount + lChannel] = dpSrc[lLoopCntr];
    }
}


long clDSPOp::ReBuffer (float *fpDest, const float *fpSrc, long lDestSize,
    long lSrcSize)
{
    // Fast implementation, but requires size modulo 0
    /*if (lDestSize < lSrcSize)
    {
        Copy(fpDest, &fpSrc[lPrevSrcCount], lDestSize);
        lPrevSrcCount += lDestSize;
        if (lPrevSrcCount == lSrcSize)
        {
            lPrevSrcCount = 0L;
            return 1;
        }
        return 2;
    }
    else if (lDestSize > lSrcSize)
    {
        Copy(&fpDest[lPrevDestCount], fpSrc, lSrcSize);
        lPrevDestCount += lSrcSize;
        if (lPrevDestCount == lDestSize)
        {
            lPrevDestCount = 0L;
            return 1;
        }
        return 0;
    }
    else
    {
        Copy(fpDest, fpSrc, lDestSize);
        return 1;
    }*/

    // Slower, but supports any size
    long lDestCount;
    long lSrcCount;
    long lCopyCount;

    lDestCount = lDestSize - lPrevDestCount;
    lSrcCount = lSrcSize - lPrevSrcCount;
    lCopyCount = (lDestCount < lSrcCount) ? lDestCount : lSrcCount;
    Copy(&fpDest[lPrevDestCount], &fpSrc[lPrevSrcCount], lCopyCount);
    lPrevDestCount += lCopyCount;
    lPrevSrcCount += lCopyCount;
    if ((lPrevDestCount == lDestSize) && (lPrevSrcCount == lSrcSize))
    {
        lPrevDestCount = 0L;
        lPrevSrcCount = 0L;
        return 1;
    }
    else if (lPrevDestCount == lDestSize)
    {
        lPrevDestCount = 0L;
        return 2;
    }
    else if (lPrevSrcCount == lSrcSize)
    {
        lPrevSrcCount = 0L;
        return 0;
    }
    else
    {
        fprintf(stderr, "clDSPOp::ReBuffer(): Fatal error; bug found\n");
    }
    return 0;
}


long clDSPOp::ReBuffer (double *dpDest, const double *dpSrc, long lDestSize,
    long lSrcSize)
{
    // Fast implementation, but requires size modulo 0
    /*if (lDestSize < lSrcSize)
    {
        Copy(dpDest, &dpSrc[lPrevSrcCount], lDestSize);
        lPrevSrcCount += lDestSize;
        if (lPrevSrcCount == lSrcSize)
        {
            lPrevSrcCount = 0L;
            return 1;
        }
        return 2;
    }
    else if (lDestSize > lSrcSize)
    {
        Copy(&dpDest[lPrevDestCount], dpSrc, lSrcSize);
        lPrevDestCount += lSrcSize;
        if (lPrevDestCount == lDestSize)
        {
            lPrevDestCount = 0L;
            return 1;
        }
        return 0;
    }
    else
    {
        Copy(dpDest, dpSrc, lDestSize);
        return 1;
    }*/

    // Slower, but supports any size
    long lDestCount;
    long lSrcCount;
    long lCopyCount;

    lDestCount = lDestSize - lPrevDestCount;
    lSrcCount = lSrcSize - lPrevSrcCount;
    lCopyCount = (lDestCount < lSrcCount) ? lDestCount : lSrcCount;
    Copy(&dpDest[lPrevDestCount], &dpSrc[lPrevSrcCount], lCopyCount);
    lPrevDestCount += lCopyCount;
    lPrevSrcCount += lCopyCount;
    if (lPrevDestCount == lDestSize && lPrevSrcCount == lSrcSize)
    {
        lPrevDestCount = 0L;
        lPrevSrcCount = 0L;
        return 1;
    }
    else if (lPrevDestCount == lDestSize)
    {
        lPrevDestCount = 0L;
        return 2;
    }
    else if (lPrevSrcCount == lSrcSize)
    {
        lPrevSrcCount = 0L;
        return 0;
    }
    else
    {
        fprintf(stderr, "clDSPOp::ReBuffer(): Fatal error; bug found\n");
    }
    return 0;
}


/* I know, here we actually play with one unneccessary value in FIRBuf,
   but it makes code a lot cleaner, small price for one extra value! */

void clDSPOp::FIRAllocate (const float *fpCoeff, long lCount)
{
    lFIRLength = lCount;
    FIRCoeff.Size(lCount * sizeof(float));
    FIRBuf.Size(lCount * sizeof(float));
    Copy((float *) FIRCoeff, fpCoeff, lCount);
    Zero((float *) FIRBuf, lCount);
}


void clDSPOp::FIRAllocate (const double *dpCoeff, long lCount)
{
    lFIRLength = lCount;
    FIRCoeff.Size(lCount * sizeof(double));
    FIRBuf.Size(lCount * sizeof(double));
    Copy((double *) FIRCoeff, dpCoeff, lCount);
    Zero((double *) FIRBuf, lCount);
}


void clDSPOp::FIRFilter (float *fpVect, long lCount)
{
    long lSrcCntr;
    long lConvCntr;
    long lDestCntr;
    long lMax;
    float fTempVal;
    float *fpFIRCoeff = FIRCoeff;
    float *fpFIRBuf = FIRBuf;
    #ifdef __GNUG__
        float fpTempData[lCount + lFIRLength];
    #else
        float *fpTempData;
        clDSPAlloc TempData((lCount + lFIRLength) * sizeof(float));
    #endif

    #ifndef __GNUG__
        fpTempData = TempData;
    #endif
    Copy(fpTempData, fpFIRBuf, lFIRLength);
    Copy(&fpTempData[lFIRLength], fpVect, lCount);
    lDestCntr = 0L;
    lMax = lCount + lFIRLength;
    for (lSrcCntr = lFIRLength; lSrcCntr < lMax; lSrcCntr++)
    {
        fTempVal = 0.0F;
        for (lConvCntr = 0L; lConvCntr < lFIRLength; lConvCntr++)
        {
            #ifndef _ISOC9X_SOURCE
                fTempVal += fpFIRCoeff[lConvCntr] * 
                    fpTempData[lSrcCntr - lConvCntr];
            #else
                fTempVal = fmaf(fpFIRCoeff[lConvCntr], 
                    fpTempData[lSrcCntr - lConvCntr], fTempVal);
            #endif
        }
        fpVect[lDestCntr++] = fTempVal;
    }
    Copy(fpFIRBuf, &fpTempData[lMax - lFIRLength], lFIRLength);
}


void clDSPOp::FIRFilter (double *dpVect, long lCount)
{
    long lSrcCntr;
    long lConvCntr;
    long lDestCntr;
    long lMax;
    double dTempVal;
    double *dpFIRCoeff = FIRCoeff;
    double *dpFIRBuf = FIRBuf;
    #ifdef __GNUG__
        double dpTempData[lCount + lFIRLength];
    #else
        double *dpTempData;
        clDSPAlloc TempData((lCount + lFIRLength) * sizeof(double));
    #endif

    #ifndef __GNUG__
        dpTempData = TempData;
    #endif
    Copy(dpTempData, dpFIRBuf, lFIRLength);
    Copy(&dpTempData[lFIRLength], dpVect, lCount);
    lDestCntr = 0L;
    lMax = lCount + lFIRLength;
    for (lSrcCntr = lFIRLength; lSrcCntr < lMax; lSrcCntr++)
    {
        dTempVal = 0.0;
        for (lConvCntr = 0L; lConvCntr < lFIRLength; lConvCntr++)
        {
            #ifndef _ISOC9X_SOURCE
                dTempVal += dpFIRCoeff[lConvCntr] *
                    dpTempData[lSrcCntr - lConvCntr];
            #else
                dTempVal = fma(dpFIRCoeff[lConvCntr],
                    dpTempData[lSrcCntr - lConvCntr], dTempVal);
            #endif
        }
        dpVect[lDestCntr++] = dTempVal;
    }
    Copy(dpFIRBuf, &dpTempData[lMax - lFIRLength], lFIRLength);
}


void clDSPOp::FIRFilter (float *fpDest, const float *fpSrc, long lCount)
{
    long lSrcCntr;
    long lConvCntr;
    long lDestCntr;
    long lMax;
    float fTempVal;
    float *fpFIRCoeff = FIRCoeff;
    float *fpFIRBuf = FIRBuf;
    #ifdef __GNUG__
        float fpTempData[lCount + lFIRLength];
    #else
        float *fpTempData;
        clDSPAlloc TempData((lCount + lFIRLength) * sizeof(float));
    #endif

    #ifndef __GNUG__
        fpTempData = TempData;
    #endif
    Copy(fpTempData, fpFIRBuf, lFIRLength);
    Copy(&fpTempData[lFIRLength], fpSrc, lCount);
    lDestCntr = 0L;
    lMax = lCount + lFIRLength;
    for (lSrcCntr = lFIRLength; lSrcCntr < lMax; lSrcCntr++)
    {
        fTempVal = 0.0F;
        for (lConvCntr = 0L; lConvCntr < lFIRLength; lConvCntr++)
        {
            #ifndef _ISOC9X_SOURCE
                fTempVal += fpFIRCoeff[lConvCntr] *
                    fpTempData[lSrcCntr - lConvCntr];
            #else
                fTempVal = fmaf(fpFIRCoeff[lConvCntr],
                    fpTempData[lSrcCntr - lConvCntr], fTempVal);
            #endif
        }
        fpDest[lDestCntr++] = fTempVal;
    }
    Copy(fpFIRBuf, &fpTempData[lMax - lFIRLength], lFIRLength);
}


void clDSPOp::FIRFilter (double *dpDest, const double *dpSrc, long lCount)
{
    long lSrcCntr;
    long lConvCntr;
    long lDestCntr;
    long lMax;
    double dTempVal;
    double *dpFIRCoeff = FIRCoeff;
    double *dpFIRBuf = FIRBuf;
    #ifdef __GNUG__
        double dpTempData[lCount + lFIRLength];
    #else
        double *dpTempData;
        clDSPAlloc TempData((lCount + lFIRLength) * sizeof(double));
    #endif

    #ifndef __GNUG__
        dpTempData = TempData;
    #endif
    Copy(dpTempData, dpFIRBuf, lFIRLength);
    Copy(&dpTempData[lFIRLength], dpSrc, lCount);
    lDestCntr = 0L;
    lMax = lCount + lFIRLength;
    for (lSrcCntr = lFIRLength; lSrcCntr < lMax; lSrcCntr++)
    {
        dTempVal = 0.0;
        for (lConvCntr = 0L; lConvCntr < lFIRLength; lConvCntr++)
        {
            #ifndef _ISOC9X_SOURCE
                dTempVal += dpFIRCoeff[lConvCntr] *
                    dpTempData[lSrcCntr - lConvCntr];
            #else
                dTempVal = fma(dpFIRCoeff[lConvCntr],
                    dpTempData[lSrcCntr - lConvCntr], dTempVal);
            #endif
        }
        dpDest[lDestCntr++] = dTempVal;
    }
    Copy(dpFIRBuf, &dpTempData[lMax - lFIRLength], lFIRLength);
}


void clDSPOp::FIRFilterF (float *fpDest, float *fpSrc, long lCount)
{
    long lSrcCntr;
    long lConvCntr;
    long lDestCntr;
    long lMax;
    float fTempVal;
    float *fpFIRCoeff = FIRCoeff;

    lDestCntr = 0L;
    lMax = lCount + lFIRLength;
    for (lSrcCntr = lFIRLength; lSrcCntr < lMax; lSrcCntr++)
    {
        fTempVal = 0.0F;
        for (lConvCntr = 0L; lConvCntr < lFIRLength; lConvCntr++)
        {
            #ifndef _ISOC9X_SOURCE
                fTempVal += fpFIRCoeff[lConvCntr] *
                    fpSrc[lSrcCntr - lConvCntr];
            #else
                fTempVal = fmaf(fpFIRCoeff[lConvCntr],
                    fpSrc[lSrcCntr - lConvCntr], fTempVal);
            #endif
        }
        fpDest[lDestCntr++] = fTempVal;
    }
    Copy(fpSrc, &fpSrc[lMax - lFIRLength], lFIRLength);
}


void clDSPOp::FIRFilterF (double *dpDest, double *dpSrc, long lCount)
{
    long lSrcCntr;
    long lConvCntr;
    long lDestCntr;
    long lMax;
    double dTempVal;
    double *dpFIRCoeff = FIRCoeff;

    lDestCntr = 0L;
    lMax = lCount + lFIRLength;
    for (lSrcCntr = lFIRLength; lSrcCntr < lMax; lSrcCntr++)
    {
        dTempVal = 0.0;
        for (lConvCntr = 0L; lConvCntr < lFIRLength; lConvCntr++)
        {
            #ifndef _ISOC9X_SOURCE
                dTempVal += dpFIRCoeff[lConvCntr] *
                    dpSrc[lSrcCntr - lConvCntr];
            #else
                dTempVal = fma(dpFIRCoeff[lConvCntr],
                    dpSrc[lSrcCntr - lConvCntr], dTempVal);
            #endif
        }
        dpDest[lDestCntr++] = dTempVal;
    }
    Copy(dpSrc, &dpSrc[lMax - lFIRLength], lFIRLength);
}


void clDSPOp::FIRFree ()
{
    FIRCoeff.Free();
    FIRBuf.Free();
}


void clDSPOp::FFTInitialize(long lSize, bool bIsReal)
{
    #ifdef __GNUG__
        float fpFFTBuf[lSize * 2];
        double dpFFTBuf[lSize * 2];
    #else
        float *fpFFTBuf;
        double *dpFFTBuf;
        clDSPAlloc FFTBufS(lSize * 2 * sizeof(float));
        clDSPAlloc FFTBufD(lSize * 2 * sizeof(double));
    #endif

    #ifndef __GNUG__
    fpFFTBuf = FFTBufS;
    dpFFTBuf = FFTBufD;
    #endif
    bFFTInitialized = true;
    bRealTransform = bIsReal;
    lFFTLength = lSize;
    if (bIsReal)
    {
        fFFTScale = 2.0F / (float) lSize;
        dFFTScale = 2.0 / (double) lSize;
    }
    else
    {
        fFFTScale = 1.0F / (float) lSize;
        dFFTScale = 1.0 / (double) lSize;
    }
    lpSBitRevWork = (long *)
        SBitRevWork.Size(((size_t) ceil(2.0 + sqrt(lSize))) * sizeof(long));
    lpDBitRevWork = (long *)
        DBitRevWork.Size(((size_t) ceil(2.0 + sqrt(lSize))) * sizeof(long));
    fpCosSinTable = (float *) 
        SCosSinTable.Size((lSize / 2 + 1) * sizeof(float));
    dpCosSinTable = (double *) 
        DCosSinTable.Size((lSize / 2 + 1) * sizeof(double));
    lpSBitRevWork[0] = 0;
    lpSBitRevWork[1] = 0;
    lpDBitRevWork[0] = 0;
    lpDBitRevWork[1] = 0;
    if (bIsReal)
    {
        Tfrm.rdft(lSize, 1, fpFFTBuf, lpSBitRevWork, fpCosSinTable);
        Tfrm.rdft(lSize, 1, dpFFTBuf, lpDBitRevWork, dpCosSinTable);
    }
    else
    {
        Tfrm.cdft(lSize * 2, 1, fpFFTBuf, lpSBitRevWork, fpCosSinTable);
        Tfrm.cdft(lSize * 2, 1, dpFFTBuf, lpDBitRevWork, dpCosSinTable);
    }
    #ifdef DSP_HAVE_FFTW
        FILE *pWisdomFile;

        pWisdomFile = fopen(DSP_WISDOM_FILE, "rt");
        if (pWisdomFile != NULL)
        {
            fftw_import_wisdom_from_file(pWisdomFile);
            fclose(pWisdomFile);
        }
        fftwrScale = (fftw_real) 1.0 / (fftw_real) lSize;
        if (bIsReal)
        {
            rfftwpRPlan = rfftw_create_plan(lSize, FFTW_FORWARD, 
                FFTW_MEASURE|FFTW_USE_WISDOM);
            rfftwpIRPlan = rfftw_create_plan(lSize, FFTW_BACKWARD, 
                FFTW_MEASURE|FFTW_USE_WISDOM);
        }
        else
        {
            fftwpPlan = fftw_create_plan(lSize, FFTW_FORWARD, 
                FFTW_MEASURE|FFTW_USE_WISDOM);
            fftwpIPlan = fftw_create_plan(lSize, FFTW_BACKWARD, 
                FFTW_MEASURE|FFTW_USE_WISDOM);
        }
    #endif
}


void clDSPOp::FFTUninitialize()
{
    #ifdef DSP_HAVE_FFTW
        if (bFFTInitialized)
        {
            FILE *pWisdomFile;

            pWisdomFile = fopen(DSP_WISDOM_FILE, "w+t");
            if (pWisdomFile != NULL)
            {
                fftw_export_wisdom_to_file(pWisdomFile);
                fclose(pWisdomFile);
            }
            fftw_forget_wisdom();
            if (bRealTransform)
            {
                rfftw_destroy_plan(rfftwpRPlan);
                rfftw_destroy_plan(rfftwpIRPlan);
            }
            else
            {
                rfftw_destroy_plan(fftwpPlan);
                rfftw_destroy_plan(fftwpIPlan);
            }
        }
    #endif
    SBitRevWork.Free();
    DBitRevWork.Free();
    SCosSinTable.Free();
    DCosSinTable.Free();
    bFFTInitialized = false;
}


void clDSPOp::FFTi(stpSCplx spDest, float *fpSrc)
{
    long lLoopCntr;
    long lMax;
    
    Mul(fpSrc, fFFTScale, lFFTLength);
    Tfrm.rdft(lFFTLength, 1, fpSrc, lpSBitRevWork, fpCosSinTable);
    lMax = lFFTLength / 2 - 1;
    spDest[0].R = fpSrc[0];
    spDest[0].I = 0.0F;
    for (lLoopCntr = 1; lLoopCntr <= lMax; lLoopCntr++)
    {
        spDest[lLoopCntr].R = fpSrc[lLoopCntr * 2];
        spDest[lLoopCntr].I = fpSrc[lLoopCntr * 2 + 1];
    }
    spDest[lMax + 1].R = fpSrc[1];
    spDest[lMax + 1].I = 0.0F;
}


void clDSPOp::FFTi(stpDCplx spDest, double *dpSrc)
{
    long lLoopCntr;
    long lMax;

    Mul(dpSrc, dFFTScale, lFFTLength);
    Tfrm.rdft(lFFTLength, 1, dpSrc, lpDBitRevWork, dpCosSinTable);
    lMax = lFFTLength / 2 - 1;
    spDest[0].R = dpSrc[0];
    spDest[0].I = 0.0;
    for (lLoopCntr = 1; lLoopCntr <= lMax; lLoopCntr++)
    {
        spDest[lLoopCntr].R = dpSrc[lLoopCntr * 2];
        spDest[lLoopCntr].I = dpSrc[lLoopCntr * 2 + 1];
    }
    spDest[lMax + 1].R = dpSrc[1];
    spDest[lMax + 1].I = 0.0;
}


void clDSPOp::FFTo(stpSCplx spDest, const float *fpSrc)
{
    #ifndef DSP_HAVE_FFTW
        long lLoopCntr;
        long lMax;
        #ifdef __GNUG__
            float fpFFTBuf[lFFTLength];
        #else
            float *fpFFTBuf;
            clDSPAlloc FFTBuf(lFFTLength * sizeof(float));
        #endif

        #ifndef __GNUG__
            fpFFTBuf = FFTBuf;
        #endif
        Mul(fpFFTBuf, fpSrc, fFFTScale, lFFTLength);
        Tfrm.rdft(lFFTLength, 1, fpFFTBuf, lpSBitRevWork, fpCosSinTable);
        lMax = lFFTLength / 2 - 1;
        spDest[0].R = fpFFTBuf[0];
        spDest[0].I = 0.0F;
        for (lLoopCntr = 1; lLoopCntr <= lMax; lLoopCntr++)
        {
            spDest[lLoopCntr].R = fpFFTBuf[lLoopCntr * 2];
            spDest[lLoopCntr].I = fpFFTBuf[lLoopCntr * 2 + 1];
        }
        spDest[lMax + 1].R = fpFFTBuf[1];
        spDest[lMax + 1].I = 0.0F;
    #else
        long lLoopCntr;
        #ifndef __GNUG__
            fftw_real fftwrIn[lFFTLength];
            fftw_real fftwrOut[lFFTLength];
        #else
            fftw_real *fftwrIn;
            fftw_real *fftwrOut;
            clDSPAlloc In.Size(lFFTLength * sizeof(fftw_real));
            clDSPAlloc Out.Size(lFFTLength * sizeof(fftw_real));
        #endif

        #ifndef __GNUG__
            fftwrIn = (fftw_real *) In.GetPtr();
            fftwrOut = (fftw_real *) Out.GetPtr();
        #endif
        if (sizeof(fftw_real) != sizeof(float))
        {
            for (lLoopCntr = 0; lLoopCntr < lFFTLength; lLoopCntr++)
            {
                fftwrIn[lLoopCntr] = 
                    (fftw_real) fpSrc[lLoopCntr] * fftwrScale;
            }
            rfftw_one(rfftwpRPlan, fftwrIn, fftwrOut);
        }
        else
        {
            Mul((float *) fftwrIn, fpSrc, (float) fftwrScale, lFFTLength);
            rfftw_one(rfftwpRPlan, fftwrIn, fftwrOut);
        }
        FFTWConvert(spDest, fftwrOut, lFFTLength);
    #endif
}


void clDSPOp::FFTo(stpDCplx spDest, const double *dpSrc)
{
    #ifndef DSP_HAVE_FFTW
        long lLoopCntr;
        long lMax;
        #ifdef __GNUG__
            double dpFFTBuf[lFFTLength];
        #else
            double *dpFFTBuf;
            clDSPAlloc FFTBuf(lFFTLength * sizeof(double));
        #endif

        #ifndef __GNUG__
            dpFFTBuf = FFTBuf;
        #endif
        Mul(dpFFTBuf, dpSrc, dFFTScale, lFFTLength);
        Tfrm.rdft(lFFTLength, 1, dpFFTBuf, lpDBitRevWork, dpCosSinTable);
        lMax = lFFTLength / 2 - 1;
        spDest[0].R = dpFFTBuf[0];
        spDest[0].I = 0.0;
        for (lLoopCntr = 1; lLoopCntr <= lMax; lLoopCntr++)
        {
            spDest[lLoopCntr].R = dpFFTBuf[lLoopCntr * 2];
            spDest[lLoopCntr].I = dpFFTBuf[lLoopCntr * 2 + 1];
        }
        spDest[lMax + 1].R = dpFFTBuf[1];
        spDest[lMax + 1].I = 0.0;
    #else
        long lLoopCntr;
        #ifdef __GNUG__
            fftw_real fftwrIn[lFFTLength];
            fftw_real fftwrOut[lFFTLength];
        #else
            fftw_real *fftwrIn;
            fftw_real *fftwrOut;
            clDSPAlloc In(lFFTLength * sizeof(fftw_real));
            clDSPAlloc Out(lFFTLength * sizeof(fftw_real));
        #endif

        #ifndef __GNUG__
            fftwrIn = (fftw_real *) In.GetPtr();
            fftwrOut = (fftw_real *) Out.GetPtr();
        #endif
        if (sizeof(fftw_real) != sizeof(double))
        {
            for (lLoopCntr = 0; lLoopCntr < lFFTLength; lLoopCntr++)
            {
                fftwrIn[lLoopCntr] = 
                    (fftw_real) dpSrc[lLoopCntr] * fftwrScale;
            }
            rfftw_one(rfftwpRPlan, fftwrIn, fftwrOut);
        }
        else
        {
            Mul((double *) fftwrIn, dpSrc, (double) fftwrScale, lFFTLength);
            rfftw_one(rfftwpRPlan, fftwrIn, fftwrOut);
        }
        FFTWConvert(spDest, fftwrOut, lFFTLength);
    #endif
}


void clDSPOp::FFTo(stpSCplx spDest, const stpSCplx spSrc)
{
    long lLoopCntr;
    #ifdef __GNUG__
        float fpFFTBuf[lFFTLength * 2];
    #else
        float *fpFFTBuf;
        clDSPAlloc FFTBuf(lFFTLength * 2 * sizeof(float));
    #endif

    #ifndef __GNUG__
        fpFFTBuf = FFTBuf;
    #endif
    for (lLoopCntr = 0; lLoopCntr < lFFTLength; lLoopCntr++)
    {
        fpFFTBuf[lLoopCntr * 2] = spSrc[lLoopCntr].R;
        fpFFTBuf[lLoopCntr * 2 + 1] = spSrc[lLoopCntr].I;
    }
    Tfrm.cdft(lFFTLength * 2, 1, fpFFTBuf, lpSBitRevWork, fpCosSinTable);
    for (lLoopCntr = 0; lLoopCntr < lFFTLength; lLoopCntr++)
    {
        spDest[lLoopCntr].R = fpFFTBuf[lLoopCntr * 2];
        spDest[lLoopCntr].I = fpFFTBuf[lLoopCntr * 2 + 1];
    }
}


void clDSPOp::FFTo(stpDCplx spDest, const stpDCplx spSrc)
{
    long lLoopCntr;
    #ifdef __GNUG__
        double dpFFTBuf[lFFTLength * 2];
    #else
        double *dpFFTBuf;
        clDSPAlloc FFTBuf(lFFTLength * 2 * sizeof(double));
    #endif

    #ifndef __GNUG__
        dpFFTBuf = FFTBuf;
    #endif
    for (lLoopCntr = 0; lLoopCntr < lFFTLength; lLoopCntr++)
    {
        dpFFTBuf[lLoopCntr * 2] = spSrc[lLoopCntr].R;
        dpFFTBuf[lLoopCntr * 2 + 1] = spSrc[lLoopCntr].I;
    }
    Tfrm.cdft(lFFTLength * 2, 1, dpFFTBuf, lpDBitRevWork, dpCosSinTable);
    for (lLoopCntr = 0; lLoopCntr < lFFTLength; lLoopCntr++)
    {
        spDest[lLoopCntr].R = dpFFTBuf[lLoopCntr * 2];
        spDest[lLoopCntr].I = dpFFTBuf[lLoopCntr * 2 + 1];
    }
}


void clDSPOp::IFFTo(float *fpDest, stpSCplx spSrc)
{
    #ifndef DSP_HAVE_FFTW
        long lLoopCntr;
        long lMax;

        lMax = lFFTLength / 2 - 1;
        fpDest[0] = spSrc[0].R;
        for (lLoopCntr = 1; lLoopCntr <= lMax; lLoopCntr++)
        {
            fpDest[lLoopCntr * 2] = spSrc[lLoopCntr].R;
            fpDest[lLoopCntr * 2 + 1] = spSrc[lLoopCntr].I;
        }
        fpDest[1] = spSrc[lMax + 1].R;
        Tfrm.rdft(lFFTLength, -1, fpDest, lpSBitRevWork, fpCosSinTable);
    #else
        long lLoopCntr;
        #ifdef __GNUG__
            fftw_real fftwrIn[lFFTLength];
            fftw_real fftwrOut[lFFTLength];
        #else
            fftw_real *fftwrIn;
            fftw_real *fftwrOut;
            clDSPAlloc In(lFFTLength * sizeof(fftw_real));
            clDSPAlloc Out(lFFTLength * sizeof(fftw_real));
        #endif

        #ifndef __GNUG__
            fftwrIn = (fftw_real *) In.GetPtr();
            fftwrOut = (fftw_real *) Out.GetPtr();
        #endif
        FFTWConvert(fftwrIn, spSrc, lFFTLength);
        rfftw_one(rfftwpIRPlan, fftwrIn, fftwrOut);
        for (lLoopCntr = 0; lLoopCntr < lFFTLength; lLoopCntr++)
        {
            fpDest[lLoopCntr] = fftwrOut[lLoopCntr] * 0.5F;
        }
    #endif
}


void clDSPOp::IFFTo(double *dpDest, stpDCplx spSrc)
{
    #ifndef DSP_HAVE_FFTW
        long lLoopCntr;
        long lMax;

        lMax = lFFTLength / 2 - 1;
        dpDest[0] = spSrc[0].R;
        for (lLoopCntr = 1; lLoopCntr <= lMax; lLoopCntr++)
        {
            dpDest[lLoopCntr * 2] = spSrc[lLoopCntr].R;
            dpDest[lLoopCntr * 2 + 1] = spSrc[lLoopCntr].I;
        }
        dpDest[1] = spSrc[lMax + 1].R;
        Tfrm.rdft(lFFTLength, -1, dpDest, lpDBitRevWork, dpCosSinTable);
    #else
        long lLoopCntr;
        #ifdef __GNUG__
            fftw_real fftwrIn[lFFTLength];
            fftw_real fftwrOut[lFFTLength];
        #else
            fftw_real *fftwrIn;
            fftw_real *fftwrOut;
            clDSPAlloc In(lFFTLength * sizeof(fftw_real));
            clDSPAlloc Out(lFFTLength * sizeof(fftw_real));
        #endif

        #ifndef __GNUG__
            fftwrIn = (fftw_real *) In.GetPtr();
            fftwrOut = (fftw_real *) Out.GetPtr();
        #endif
        FFTWConvert(fftwrIn, spSrc, lFFTLength);
        rfftw_one(rfftwpIRPlan, fftwrIn, fftwrOut);
        for (lLoopCntr = 0; lLoopCntr < lFFTLength; lLoopCntr++)
        {
            dpDest[lLoopCntr] = fftwrOut[lLoopCntr] * 0.5;
        }
    #endif
}


void clDSPOp::IFFTo(stpSCplx spDest, const stpSCplx spSrc)
{
    long lLoopCntr;
    float fScale;
    #ifdef __GNUG__
        float fpFFTBuf[lFFTLength * 2];
    #else
        float *fpFFTBuf;
        clDSPAlloc FFTBuf(lFFTLength * 2 * sizeof(float));
    #endif

    #ifndef __GNUG__
        fpFFTBuf = FFTBuf;
    #endif
    for (lLoopCntr = 0; lLoopCntr < lFFTLength; lLoopCntr++)
    {
        fpFFTBuf[lLoopCntr * 2] = spSrc[lLoopCntr].R;
        fpFFTBuf[lLoopCntr * 2 + 1] = spSrc[lLoopCntr].I;
    }
    Tfrm.cdft(lFFTLength * 2, -1, fpFFTBuf, lpSBitRevWork, fpCosSinTable);
    fScale = 1.0F / (float) lFFTLength;
    for (lLoopCntr = 0; lLoopCntr < lFFTLength; lLoopCntr++)
    {
        spDest[lLoopCntr].R = fpFFTBuf[lLoopCntr * 2] * fScale;
        spDest[lLoopCntr].I = fpFFTBuf[lLoopCntr * 2 + 1] * fScale;
    }
}


void clDSPOp::IFFTo(stpDCplx spDest, const stpDCplx spSrc)
{
    long lLoopCntr;
    double dScale;
    #ifdef __GNUG__
        double dpFFTBuf[lFFTLength * 2];
    #else
        double *dpFFTBuf;
        clDSPAlloc FFTBuf(lFFTLength * 2 * sizeof(double));
    #endif

    #ifndef __GNUG__
        dpFFTBuf = FFTBuf;
    #endif
    for (lLoopCntr = 0; lLoopCntr < lFFTLength; lLoopCntr++)
    {
        dpFFTBuf[lLoopCntr * 2] = spSrc[lLoopCntr].R;
        dpFFTBuf[lLoopCntr * 2 + 1] = spSrc[lLoopCntr].I;
    }
    Tfrm.cdft(lFFTLength * 2, -1, dpFFTBuf, lpDBitRevWork, dpCosSinTable);
    dScale = 1.0 / (double) lFFTLength;
    for (lLoopCntr = 0; lLoopCntr < lFFTLength; lLoopCntr++)
    {
        spDest[lLoopCntr].R = dpFFTBuf[lLoopCntr * 2] * dScale;
        spDest[lLoopCntr].I = dpFFTBuf[lLoopCntr * 2 + 1] * dScale;
    }
}


void clDSPOp::FFTWConvert(stpSCplx spDest, const float *fpSrc, long lLength)
{
    long lLoopCntr;
    long lMax;

    lMax = lLength / 2;
    spDest[0].R = fpSrc[0];
    spDest[0].I = 0.0F;
    for (lLoopCntr = 1; lLoopCntr < lMax; lLoopCntr++)
    {
        spDest[lLoopCntr].R = fpSrc[lLoopCntr];
        spDest[lLoopCntr].I = fpSrc[lLength - lLoopCntr];
    }
    spDest[lMax].R = fpSrc[lMax];
    spDest[lMax].I = 0.0F;
}


void clDSPOp::FFTWConvert(stpDCplx spDest, const float *fpSrc, long lLength)
{
    long lLoopCntr;
    long lMax;

    lMax = lLength / 2;
    spDest[0].R = fpSrc[0];
    spDest[0].I = 0.0;
    for (lLoopCntr = 1; lLoopCntr < lMax; lLoopCntr++)
    {
        spDest[lLoopCntr].R = fpSrc[lLoopCntr];
        spDest[lLoopCntr].I = fpSrc[lLength - lLoopCntr];
    }
    spDest[lMax].R = fpSrc[lMax];
    spDest[lMax].I = 0.0;
}


void clDSPOp::FFTWConvert(stpSCplx spDest, const double *dpSrc, long lLength)
{
    long lLoopCntr;
    long lMax;

    lMax = lLength / 2;
    spDest[0].R = dpSrc[0];
    spDest[0].I = 0.0F;
    for (lLoopCntr = 1; lLoopCntr < lMax; lLoopCntr++)
    {
        spDest[lLoopCntr].R = dpSrc[lLoopCntr];
        spDest[lLoopCntr].I = dpSrc[lLength - lLoopCntr];
    }
    spDest[lMax].R = dpSrc[lMax];
    spDest[lMax].I = 0.0F;
}


void clDSPOp::FFTWConvert(stpDCplx spDest, const double *dpSrc, long lLength)
{
    long lLoopCntr;
    long lMax;

    lMax = lLength / 2;
    spDest[0].R = dpSrc[0];
    spDest[0].I = 0.0;
    for (lLoopCntr = 1; lLoopCntr < lMax; lLoopCntr++)
    {
        spDest[lLoopCntr].R = dpSrc[lLoopCntr];
        spDest[lLoopCntr].I = dpSrc[lLength - lLoopCntr];
    }
    spDest[lMax].R = dpSrc[lMax];
    spDest[lMax].I = 0.0;
}


void clDSPOp::FFTWConvert(float *fpDest, const stpSCplx spSrc, long lLength)
{
    long lLoopCntr;
    long lMax;

    lMax = lLength / 2;
    fpDest[0] = spSrc[0].R;
    for (lLoopCntr = 1; lLoopCntr < lMax; lLoopCntr++)
    {
        fpDest[lLoopCntr] = spSrc[lLoopCntr].R;
        fpDest[lLength - lLoopCntr] = spSrc[lLoopCntr].I;
    }
    fpDest[lMax] = spSrc[lMax].R;
}


void clDSPOp::FFTWConvert(float *fpDest, const stpDCplx spSrc, long lLength)
{
    long lLoopCntr;
    long lMax;

    lMax = lLength / 2;
    fpDest[0] = spSrc[0].R;
    for (lLoopCntr = 1; lLoopCntr < lMax; lLoopCntr++)
    {
        fpDest[lLoopCntr] = spSrc[lLoopCntr].R;
        fpDest[lLength - lLoopCntr] = spSrc[lLoopCntr].I;
    }
    fpDest[lMax] = spSrc[lMax].R;
}


void clDSPOp::FFTWConvert(double *dpDest, const stpSCplx spSrc, long lLength)
{
    long lLoopCntr;
    long lMax;

    lMax = lLength / 2;
    dpDest[0] = spSrc[0].R;
    for (lLoopCntr = 1; lLoopCntr < lMax; lLoopCntr++)
    {
        dpDest[lLoopCntr] = spSrc[lLoopCntr].R;
        dpDest[lLength - lLoopCntr] = spSrc[lLoopCntr].I;
    }
    dpDest[lMax] = spSrc[lMax].R;
}


void clDSPOp::FFTWConvert(double *dpDest, const stpDCplx spSrc, long lLength)
{
    long lLoopCntr;
    long lMax;

    lMax = lLength / 2;
    dpDest[0] = spSrc[0].R;
    for (lLoopCntr = 1; lLoopCntr < lMax; lLoopCntr++)
    {
        dpDest[lLoopCntr] = spSrc[lLoopCntr].R;
        dpDest[lLength - lLoopCntr] = spSrc[lLoopCntr].I;
    }
    dpDest[lMax] = spSrc[lMax].R;
}


/*inline double DegToRad (double dSrc)
{
   return (M_PI / 180.0) * dSrc;
}*/


/*inline double RadToDeg (double dSrc)
{
   return (180.0 / M_PI) * dSrc;
}*/

