/***************************************************************************
                          csearchmanager.cpp  -  description
                             -------------------
    begin                : Thu May 27 2004
    copyright            : (C) 2004 by Mathias Kster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>

#ifndef WIN32
#include <unistd.h>
#else
#include <io.h>
#endif

#include <dclib/dcos.h>
#include <dclib/dclib.h>
#include <dclib/dcobject.h>
#include <dclib/cconfig.h>
#include <dclib/cclient.h>
#include <dclib/core/cmanager.h>
#include <dclib/cmessagehandler.h>
#include <dclib/cconnectionmanager.h>
#include <dclib/cfilemanager.h>

#include "csearchmanager.h"

#define SEARCH_TIMEOUT				60
#define SEARCH_CLIENT_TIMEOUT			60

// no longer defined because we are using a configured value
//define SEARCH_SPAM_TIMEOUT			60

#define SEARCH_PASSIVE_CLIENT_REMOVE_TIMEOUT	5
#define SEARCH_ACTIVE_CLIENT_REMOVE_TIMEOUT	60

/** */
CSearchManager::CSearchManager()
{
	SearchState(esNONE);

	m_eSearchType  = estyNONE;
	m_eClientMode  = ecmNONE;

	m_pClientList  = 0;
	m_pHubList     = 0;
	m_nMaxClients  = 0;
	m_sCurrentHub  = 0;
	m_pParentCallback = 0;

	m_bEnableTag   = FALSE;
	m_bHandleUserList = FALSE;

	// set manager callback
	m_pCallback = new CCallback<CSearchManager>( this, &CSearchManager::CallBackManager );
	CManager::Instance()->Add( m_pCallback );
	
	// set udp socket callback
	m_SearchSocket.SetCallBackFunction( new CCallback<CSearchManager>( this, &CSearchManager::CallBackSearchSocket ) );
	
	SetInstance(this);
}

/** */
CSearchManager::~CSearchManager()
{
	SetInstance(0);

	// disconnect listen udp socket
	m_SearchSocket.Disconnect(TRUE);

	if ( CManager::Instance() )
		CManager::Instance()->Remove( m_pCallback );
	delete m_pCallback;
	m_pCallback = 0;
	if ( m_pParentCallback )
		delete m_pParentCallback;
}

/** */
long CSearchManager::HubCount()
{
	long res = 0;

	m_Mutex.Lock();
	
	if ( m_pHubList )
	{
		res = m_pHubList->Count();
	}
	
	m_Mutex.UnLock();

	return res;	
}

/** */
eSearchError CSearchManager::StartSearch( eSearchMode mode, eSearchType type, CList<CObject> * querylist, CStringList * serverlist )
{
	if ( SearchState() != esNONE )
	{
		return eseALREADYRUN;
	}

	if ( !querylist )
	{
		DPRINTF("empty query list\n");
		return eseNONE;
	}

	// check hubs
	if ( (mode != esmCONNECTEDSINGLE) &&
	     (mode != esmCONNECTEDALL) )
	{
		// get hublist
		if ( !serverlist )
		{
			if ( mode == esmPUBLIC )
			{
				// get public hublist
				serverlist = CConfig::Instance()->GetPublicHubServerList();
			}
			else if ( mode == esmBOOKMARK )
			{
				// get bookmark hublist
				serverlist = CConfig::Instance()->GetBookmarkHubServerList();
			}
		}
		
		// check hublist
		if ( !serverlist )
		{
			DPRINTF("empty hublist\n");
			return eseNONE;
		}
		
		if ( serverlist->Count() == 0 )
		{
			DPRINTF("empty hublist 0\n");
			return eseNONE;
		}
	}
	else
	{
		// get connected hub count
		if ( CConnectionManager::Instance()->GetConnectedHubCount() == 0 )
		{
			DPRINTF("no connected hub\n");
			return eseNONE;
		}

		if ( mode == esmCONNECTEDSINGLE )
		{
			if ( !serverlist || (serverlist->Count() == 0) )
			{
				DPRINTF("empty hublist\n");
				return eseNONE;
			}
		}
	}

	// start listen
	if ( CConfig::Instance()->GetMode() != ecmPASSIVE )
	{
		m_eClientMode = ecmACTIVE;

		if (  m_SearchSocket.Connect( "", CConfig::Instance()->GetUDPListenPort(), estUDP ) != 0 )
		{
			return eseNONE;
		}
	}
	else
	{
		m_eClientMode = ecmPASSIVE;
	}

	// now we can start the search
	m_eSearchType  = type;
	m_eSearchMode  = mode;
	m_nHubIndex    = 0;
	m_nHubError    = 0;

	m_pCurrentSearchObject = 0;
	m_pSearchList          = querylist;
	m_pHubList             = serverlist;

	// set start time
	m_tStartTime = time(0);
	// reset timeout
	m_tTimeoutTime = 0;
	
	SearchState(esSEARCH);
	
	DPRINTF("start search\n");
	
	DPRINTF("M: %d T: %d QC: %ld\n",
		mode,
		type,
		querylist->Count());

	return eseNONE;
}

/** */
void CSearchManager::StopSearch()
{
	if ( SearchState() != esNONE )
	{
		SearchState(esSTOP);
		// disconnect listen udp socket
		m_SearchSocket.Disconnect(TRUE);
	}
}

/** */
int CSearchManager::CallBackManager( CObject *, CObject * )
{
	// no search running
	if ( SearchState() == esNONE )
	{
		return 0;
	}
	
	m_SearchSocket.Thread(0);

	// check for search timeout
	if ( SearchState() == esTIMEOUT )
	{
//		DPRINTF("timeout\n");
		if ( (time(0)-m_tTimeoutTime) >= SEARCH_TIMEOUT )
		{
			StopSearch();
		}
	}

	// check for stop search
	if ( SearchState() == esSTOP )
	{
		DPRINTF("stop\n");
		// disconnect all clients
		DisconnectClients();
		
		// remove all clients
		if ( RemoveClients() == TRUE )
		{
			if ( m_pClientList )
			{
				delete m_pClientList;
				m_pClientList = 0;
			}
			
			DPRINTF("end\n");
			// no more clients, reset search
			SearchState(esNONE);
			// disconnect listen udp socket
			m_SearchSocket.Disconnect(TRUE);
			
			m_eSearchType = estyNONE;
			
			return 0;
		}
	}

	// update all clients
	UpdateClients();

	if ( SearchState() == esSEARCH )
	{
		switch (m_eSearchMode)
		{
			case esmCONNECTEDSINGLE:
			case esmCONNECTEDALL:
				if ( (time(0) - m_tTimeoutTime) >= (CConfig::Instance()->GetAutoSearchInterval()) )
				{
					if ( DoSearch(0) == FALSE )
					{
						SearchState(esTIMEOUT);
					}
					
					m_tTimeoutTime = time(0);
				}
				break;

			case esmPUBLIC:
			case esmBOOKMARK:
				bool b1,b2;
				// add new clients
				b1 = AddClients();
				// remove disconnected clients
				b2 = RemoveClients();
				
				if ( (b1 == FALSE) && (b2 == TRUE) )
				{
					// search done
					SearchState(esTIMEOUT);
					m_tTimeoutTime = time(0);
				}
				break;
			default:
				break;
		}
	}

	return 0;
}

/** */
bool CSearchManager::AddClients()
{
	if ( m_pHubList == 0 )
	{
		return FALSE;
	}

	if ( m_pHubList->Count() == m_nHubIndex )
	{
		return FALSE;
	}
	
	m_Mutex.Lock();
	
	if ( m_pClientList == 0 )
	{
		m_pClientList = new CList<CSearchClient>();
	}
	
	while( m_pClientList->Count() < m_nMaxClients )
	{
		if ( AddClient() == FALSE )
		{
			break;
		}
	}
	
	m_Mutex.UnLock();
	
	return TRUE;
}

/** */
bool CSearchManager::AddClient()
{
	CSearchClient * client;

	// check for hublist done
	if ( m_pHubList->Next((CObject*&)m_sCurrentHub) == 0 )
	{
		DPRINTF("no new hubserver\n");
		return FALSE;
	}
	
	m_nHubIndex++;
	
	DPRINTF("add client: '%s'\n",m_sCurrentHub->Data());

	if (m_sCurrentHub != 0)
	{
		char timestamp[23];
		struct tm * t;
		time_t ti;
		
		ti = time(0);
		t  = localtime(&ti);
		strftime(timestamp, sizeof(timestamp), "[%F.%H:%M:%S] ", t);
		
		CMessageLog * msglog = new CMessageLog();
		msglog->sMessage = CString(timestamp) + "Add client on " + *m_sCurrentHub;
		//DPRINTF("%s\n", msglog->sMessage.Data());
		SendObject(msglog);
	}
	//else
	//{
	//	DPRINTF("m_sCurrentHub == 0\n");
	//}

	client = new CSearchClient();
	
	// set default values for search
	client->m_tSearchTimeout = time(0);

	// client settings
	client->SetNick(CConfig::Instance()->GetSearchNick());
	client->SetComment(CConfig::Instance()->GetDescription(!m_bEnableTag));
	client->SetConnectionType(CConfig::Instance()->GetSpeed());
	client->SetEMail(CConfig::Instance()->GetEMail());
	client->SetVersion(VERSION);
	client->SetShareSize(CString().setNum(CFileManager::Instance()->GetShareSize()));
	client->SetMode(CConfig::Instance()->GetMode());
	client->SetHandleUserList(m_bHandleUserList);
	client->SetHandleSearch(FALSE);
	client->SetHandleMyinfo(FALSE);
	client->SetHandleForceMove(FALSE);
	client->SetHandleTransfer(FALSE);
	client->SetCallBackFunction( new CCallback<CSearchManager>( this, &CSearchManager::CallBackClient ) );

//	SendDebug( "Start search on: [" + CString().setNum(m_nCurrentHub) + "] " + server );

	// add client to client list
	m_pClientList->Add(client);

	// final connect
	client->Connect(*m_sCurrentHub);
	
	return TRUE;
}

/** */
void CSearchManager::UpdateClients()
{
	m_Mutex.Lock();

	// sanity check
	if ( m_pClientList != 0 )
	{
		CSearchClient * client = 0;

		// update all clients from the client list
		while ( ( (client = m_pClientList->Next(client)) != NULL) )
		{
			client->Thread(0);
			
			// connection timeout
			if ( (client->m_bSearchRemove == FALSE) &&
			     (client->m_bSearchEnable == FALSE) )
			{
				if ( (time(0) - client->m_tSearchTimeout) >= SEARCH_CLIENT_TIMEOUT )
				{
					DPRINTF("remove client timeout\n");
					
					// remove client, connection timeout
					client->m_bSearchRemove = TRUE;
					client->m_tSearchTimeout = 0;
				}
			}
			// send search
			else if ( client->m_bSearchEnable == TRUE )
			{
				if ( (time(0) - client->m_tSearchTimeout) >= (CConfig::Instance()->GetAutoSearchInterval()) )
				{
					if ( DoSearch(client) == FALSE )
					{
						// no more searches
						client->m_bSearchEnable = FALSE;
						client->m_bSearchRemove = TRUE;
					}
					
					// update timeout
					client->m_tSearchTimeout = time(0);
					
					DPRINTF("search enabled\n");
				} 
			}
		}
	}
	
	m_Mutex.UnLock();
}

/** */
bool CSearchManager::RemoveClients()
{
	bool res;
	
	m_Mutex.Lock();

	res = FALSE;

	// sanity check
	if ( m_pClientList != 0 )
	{
		CSearchClient * client = 0;

		// remove clients from the client list
		while ( ( (client = m_pClientList->Next(client)) != NULL) )
		{
			if ( client->m_bSearchRemove == TRUE )
			{
				if ( client->m_tSearchTimeout == 0 )
				{
					client->SetCallBackFunction(0);
				
					m_pClientList->Remove(client);
					delete client;
				
					client = 0;
				}
				// remove timeout
				else
				{
					if ( ((m_eClientMode == ecmACTIVE) &&
					      ((time(0) - client->m_tSearchTimeout) >= SEARCH_ACTIVE_CLIENT_REMOVE_TIMEOUT)) ||
					     ((m_eClientMode == ecmPASSIVE) &&
					      ((time(0) - client->m_tSearchTimeout) >= SEARCH_PASSIVE_CLIENT_REMOVE_TIMEOUT)) )
					{
						client->m_tSearchTimeout = 0;
						client->Disconnect(TRUE);
					}
				}
			}
		}
		
		// ready on empty client list
		if ( m_pClientList->Count() == 0 )
		{
			res = TRUE;
		}
	}
	else
	{
		res = TRUE;
	}
		
	m_Mutex.UnLock();
	
	return res;
}

/** */
void CSearchManager::DisconnectClients()
{
	m_Mutex.Lock();

	// sanity check
	if ( m_pClientList != 0 )
	{
		CSearchClient * client = 0;

		// disconnect all clients
		while ( ( (client = m_pClientList->Next(client)) != NULL) )
		{
			// disconnect client
			if ( (client->GetConnectionState() != estNONE) &&
			     (client->GetConnectionState() != estDISCONNECTING) )
			{
				client->Disconnect(TRUE);
			}
		}
	}
	
	m_Mutex.UnLock();
}

/** send search to the client */
bool CSearchManager::DoSearch( CSearchClient * HubSearchClient )
{
	bool res = FALSE;

	if ( HubSearchClient != 0 )
	{
		// active search, get current search object
		m_pCurrentSearchObject = HubSearchClient->m_pCurrentSearchObject;
	}
	
	while ( (res == FALSE) && ((m_pCurrentSearchObject=m_pSearchList->Next(m_pCurrentSearchObject)) != 0) )
	{
		switch(((CDCMessage*)m_pCurrentSearchObject)->m_eType)
		{
			case DC_MESSAGE_SEARCH_USER:
			{
				CMessageSearchUser * msg = (CMessageSearchUser*)m_pCurrentSearchObject;

				if ( !HubSearchClient )
				{
					CList<DCHubObject> list;
					DCHubObject * HubObject;
					CString s = "";
					
					if ( m_eSearchMode == esmCONNECTEDSINGLE )
					{
						m_sCurrentHub = 0;
						if ( m_pHubList->Next((CObject*&)m_sCurrentHub) != 0 )
							s = *m_sCurrentHub;
					}

					if ( CConnectionManager::Instance()->IsUserOnline( msg->m_sNick, s, "", &list ) == TRUE )
					{
						HubObject = 0;

						while ( (HubObject=list.Next(HubObject)) != 0 )
						{
							CMessageSearchResultUser * MessageSearchResultUser = new CMessageSearchResultUser();

							MessageSearchResultUser->m_eType    = DC_MESSAGE_SEARCHRESULT_USER;
							MessageSearchResultUser->m_sHubName = HubObject->m_sHubName;
							MessageSearchResultUser->m_sNick    = msg->m_sNick;

							if ( !SendObject(MessageSearchResultUser) )
								delete MessageSearchResultUser;
						}

						list.Clear();
					}
				}
				else
				{
					if ( HubSearchClient->UserList()->IsUserOnline( msg->m_sNick ) == TRUE )
					{
						CMessageSearchResultUser * MessageSearchResultUser = new CMessageSearchResultUser();

						MessageSearchResultUser->m_eType    = DC_MESSAGE_SEARCHRESULT_USER;
						MessageSearchResultUser->m_sHubName = HubSearchClient->GetHubName();
						MessageSearchResultUser->m_sNick    = msg->m_sNick;

						if ( !SendObject(MessageSearchResultUser) )
							delete MessageSearchResultUser;
					}
				}
				
				break;
			}

			case DC_MESSAGE_SEARCH_FILE:
			{
				CMessageSearchFile * msg = (CMessageSearchFile*)m_pCurrentSearchObject;

				if ( m_eSearchType == estyMULTI )
					msg->m_bMulti = TRUE;
				else
					msg->m_bMulti = FALSE;
				
				DPRINTF("send search\n");
				
				if ( HubSearchClient )
				{
					HubSearchClient->SendSearch(msg);
				}
				else
				{
					if ( m_eSearchMode == esmCONNECTEDSINGLE )
					{
						m_sCurrentHub = 0;
						
						if ( m_pHubList->Next((CObject*&)m_sCurrentHub) != 0 )
						{
							CConnectionManager::Instance()->SendSearchToConnectedServers(msg,*m_sCurrentHub);
						}
					}
					else
					{
						CConnectionManager::Instance()->SendSearchToConnectedServers(msg);
					}
				}
				
				res = TRUE;

				break;
			}

			default:
			{
				break;
			}
		}

	}

	if ( HubSearchClient != 0 )
	{
		// active search, set last search object
		HubSearchClient->m_pCurrentSearchObject = m_pCurrentSearchObject;
	}
	
	// check for last search
	if ( !m_pCurrentSearchObject )
		res = FALSE;
	else if ( !m_pSearchList->Next(m_pCurrentSearchObject) )
		res = FALSE;
	
	return res;
}

/** */
bool CSearchManager::SendObject( CObject * object )
{
	bool res = FALSE;

	if ( SearchType() == estyEXTERNAL )
	{
		res = CDownloadManager::Instance()->DLM_HandleSearch((CMessageSearchResult *)object);
	}
	else if ( m_pParentCallback != 0 )
	{
		if ( m_pParentCallback->notify( 0, object ) == 0 )
			res = TRUE;
	}

	return res;
}

/** */
bool CSearchManager::HandleSearch( CObject * object )
{
	return SendObject(object);
}

/** */
int CSearchManager::CallBackSearchSocket( CObject *, CObject * Object )
{
	int err = -1;
	
	if ( SendObject(Object) )
	{
		err = 0;
	}

	return err;
}

/** */
int CSearchManager::CallBackClient( CObject * Client, CObject * Object )
{
	CSearchClient * c = (CSearchClient*)Client;
	CDCMessage * cmsg = (CDCMessage*) Object;
	
	if ( (c == 0) || (cmsg == 0) )
	{
		return -1;
	}

	switch ( cmsg->m_eType )
	{
		case DC_MESSAGE_CONNECTION_STATE:
		{
			CMessageConnectionState *msg = (CMessageConnectionState*) cmsg;

			switch(msg->m_eState)
			{
				case estCONNECTED:
				{
					break;
				}

				case estSOCKETERROR:
				{
					DPRINTF("socket error\n");
					m_nHubError++;
					break;
				}

				case estDISCONNECTED:
				{
					DPRINTF("disconnect\n");
					c->m_bSearchRemove = TRUE;
					c->m_tSearchTimeout = 0;
				}

				default:
				{
					break;
				}
			}

			break;
		}

		case DC_MESSAGE_VALIDATEDENIDE:
		{
			//SendDebug( "Validate denide " + c->GetIP() + CString(":") + CString().setNum(c->GetPort()) );
			c->Disconnect(TRUE);
			break;
		}

		case DC_MESSAGE_HUBISFULL:
		{
			//SendDebug( "Hub is full on " + c->GetIP() + CString(":") + CString().setNum(c->GetPort()) );
			c->Disconnect(TRUE);
			break;
		}

		case DC_MESSAGE_FORCEMOVE:
		{
			//SendDebug( "Force move on " + c->GetIP() + CString(":") + CString().setNum(c->GetPort()) );
			c->Disconnect(TRUE);
			break;
		}

		case DC_MESSAGE_MYINFO:
		{
			CMessageMyInfo * MessageMyInfo = (CMessageMyInfo *)cmsg;
			
			if ( MessageMyInfo->m_sNick == CConfig::Instance()->GetSearchNick() )
			{
				DPRINTF("enable search\n");
				c->m_bSearchEnable  = TRUE;
				c->m_tSearchTimeout = 0;
			}
			
			break;
		}
		
		default:
			break;
	}

	delete cmsg;

	return 0;
}
