/* 
 * $Id: ctknotebook.c,v 1.25 2000/07/25 03:14:43 terpstra Exp $
 *
 * CTK - Console Toolkit
 *
 * Copyright (C) 1998-2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Authors: Kevin Lindsay, Wesley Terpstra
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ctk.h"
#include "ctkstack.h"
#include "ctkcolor.h"

gboolean ctk_notebook_key_press   (CtkWidget* widget, CdkEventKey*    event, gpointer data);
gboolean ctk_notebook_button_press(CtkWidget* widget, CdkEventButton* event, gpointer data);
gboolean ctk_notebook_focus_in    (CtkWidget* widget, CdkEventFocus*  event, gpointer data);
gboolean ctk_notebook_focus_out   (CtkWidget* widget, CdkEventFocus*  event, gpointer data);
gpointer ctk_notebook_destroy(CtkObject *object);

/* Calculate the minimum size the notebook needs */
void ctk_notebook_min_size(CtkWidget* widget)
{
	GSList*         tab_list;
	CtkNotebook*    notebook;
	CtkNotebookTab* tab;
	
	gint tab_accum_x = 0;
	gint tab_accum_y = 0;
	
	gint page_biggest_x = 0;
	gint page_biggest_y = 0;
	
	notebook = CTK_NOTEBOOK(widget);
	
	for (tab_list = notebook->tab_labels; tab_list; tab_list = tab_list->next)
	{
		tab = (CtkNotebookTab*)tab_list->data;
		
		if (tab->label)
		{
			if (notebook->show_tabs && tab->label->show)
			{
				switch (notebook->tab_pos)
				{
					case CTK_POS_TOP:
					case CTK_POS_BOTTOM:
						tab_accum_x += tab->label->min_width + 3;
						if (tab->label->min_height > tab_accum_y)
							tab_accum_y = tab->label->min_height;
						break;
					case CTK_POS_LEFT:
					case CTK_POS_RIGHT:
						tab_accum_y += tab->label->min_height + 1;
						if (tab->label->min_width > tab_accum_x)
							tab_accum_x = tab->label->min_width;
						break;
				}
			}
		}
		
		if (tab->page)
		{
			if (tab->page->min_height > page_biggest_y)
				page_biggest_y = tab->page->min_height;
			if (tab->page->min_width  > page_biggest_x)
				page_biggest_x = tab->page->min_width;
		}
	}
	
	switch (notebook->tab_pos)
	{
	case CTK_POS_TOP:
	case CTK_POS_BOTTOM:
		widget->min_height = page_biggest_y + tab_accum_y +
			CTK_CONTAINER(widget)->border_width * 2;
		widget->min_width = ctk_helper_max(
			page_biggest_x + CTK_CONTAINER(widget)->border_width*2,
			tab_accum_x);
		break;
	case CTK_POS_LEFT:
	case CTK_POS_RIGHT:
		widget->min_width = page_biggest_x + tab_accum_x +
			CTK_CONTAINER(widget)->border_width * 2;
		widget->min_height = ctk_helper_max(
			page_biggest_y + CTK_CONTAINER(widget)->border_width*2,
			tab_accum_y);
		break;
	}

	notebook->tab_width  = tab_accum_x;
	notebook->tab_height = tab_accum_y;
}

/* Assumption: min_size was already called to set the tab_width & height */
void ctk_notebook_real_size(CtkWidget* widget)
{
	GSList*         tab_list;
	CtkContainer*   container;
	CtkNotebook*    notebook;
	CtkNotebookTab* tab;
	CtkWidget*      child;
	
	gint tab_accum_x = 0;
	gint tab_accum_y = 0;
	
	container = CTK_CONTAINER(widget);
	notebook = CTK_NOTEBOOK(widget);
	
	notebook->tab_row = widget->row;
	notebook->tab_col = widget->col;
	notebook->page_col    = widget->col + container->border_width;
	notebook->page_row    = widget->row + container->border_width;
	notebook->page_width  = widget->width  - container->border_width * 2;
	notebook->page_height = widget->height - container->border_width * 2;
	
	switch (notebook->tab_pos)
	{
		case CTK_POS_TOP:
			notebook->page_height -= notebook->tab_height;
			notebook->page_row    += notebook->tab_height;
			break;
		case CTK_POS_BOTTOM:
			notebook->page_height -= notebook->tab_height;
			notebook->tab_row     =  widget->row + widget->height - notebook->tab_height;
			break;
		case CTK_POS_LEFT:
			notebook->page_width -= notebook->tab_width;
			notebook->page_col   += notebook->tab_width;
			break;
		case CTK_POS_RIGHT:
			notebook->page_width -= notebook->tab_width;
			notebook->tab_col    =  widget->col + widget->width - notebook->tab_width;
			break;
	}
	
	for (tab_list = notebook->tab_labels; tab_list; tab_list = tab_list->next)
	{
		tab = (CtkNotebookTab*)tab_list->data;
		
		if (tab->label)
		{
			child = CTK_WIDGET(tab->label);
			
			if (notebook->show_tabs && tab->label->show)
			{
				switch (notebook->tab_pos)
				{
					case CTK_POS_TOP:
					case CTK_POS_BOTTOM:
						child->row = notebook->tab_row;
						child->col = notebook->tab_col + tab_accum_x;
						child->width  = child->min_width;
						child->height = notebook->tab_height;
						tab_accum_x += tab->label->width + 3;
						break;
					case CTK_POS_LEFT:
					case CTK_POS_RIGHT:
						child->row = notebook->tab_row + tab_accum_y;
						child->col = notebook->tab_col;
						child->width  = notebook->tab_width;
						child->height = child->min_height;
						tab_accum_y += tab->label->height + 1;
						break;
				}
			}
			else
			{
				child->width  = 0;
				child->height = 0;
			}
		}
		
		if (tab->page)
		{
			child = CTK_WIDGET(tab->page);
			
			if (tab->selected)
			{
				child->col = notebook->page_col;
				child->row = notebook->page_row;
				child->width  = notebook->page_width;
				child->height = notebook->page_height;
			}
			else
			{
				child->width  = 0;
				child->height = 0;
			}
		}
	}
}

/* CTK Notebook Initializer */
void ctk_notebook_init(CtkNotebook* notebook)
{
	CtkContainer* container;
	
	ctk_container_init(&notebook->container);
	((CtkObject *)notebook)->type = CtkTypeNotebook;
	
	container = CTK_CONTAINER(notebook);
	container->set_child_flags = &ctk_notebook_set_default_child_flags;
	container->lost_child      = &ctk_notebook_lost_child;
	container->border_width    = 1;
	
	notebook->tab_labels = NULL;
	notebook->tab_pos = CTK_POS_TOP;
	notebook->show_tabs = TRUE;
	ctk_signal_new("switch_page",CtkTypeNotebook);

	/* Initialize Other Variables */

	((CtkWidget *)notebook)->width = 2;
	((CtkWidget *)notebook)->height = 2;
	((CtkWidget *)notebook)->orig_width = 2;
	((CtkWidget *)notebook)->orig_height = 2;
	CTK_WIDGET(notebook)->sensitive = TRUE;

	/* Also setup Colors */
	((CtkWidget *)notebook)->main_col = 
	    ctk_calculate_palette(CTK_COLOR_BLACK,CTK_COLOR_LIGHT_GRAY);
	((CtkWidget *)notebook)->title_col = 
	    ctk_calculate_palette(CTK_COLOR_BLACK,CTK_COLOR_LIGHT_GRAY);
	((CtkWidget *)notebook)->selected_col = 
	    ctk_calculate_palette(CTK_COLOR_LIGHT_GRAY,CTK_COLOR_BLUE);
	((CtkWidget *)notebook)->inverse_col =
		ctk_calculate_palette(CTK_COLOR_WHITE,CTK_COLOR_RED);

	((CtkObject *)notebook)->destroy_func = ctk_notebook_destroy;

	CTK_WIDGET(notebook)->set_min_size  = &ctk_notebook_min_size;
	CTK_WIDGET(notebook)->set_real_size = &ctk_notebook_real_size;
	
	ctk_signal_connect(CTK_OBJECT(notebook), "focus_in_event",
			   CTK_SIGNAL_FUNC(&ctk_notebook_focus_in), NULL);
	ctk_signal_connect(CTK_OBJECT(notebook), "focus_out_event",
			   CTK_SIGNAL_FUNC(&ctk_notebook_focus_out), NULL);
	ctk_signal_connect(CTK_OBJECT(notebook), "button_press_event",
			   CTK_SIGNAL_FUNC(&ctk_notebook_button_press), NULL);
	ctk_signal_connect(CTK_OBJECT(notebook), "key_press_event",
			   CTK_SIGNAL_FUNC(&ctk_notebook_key_press), NULL);
}

void ctk_notebook_set_default_child_flags_noop(CtkContainer* container, CtkWidget* child)
{
      // noop
}

void ctk_notebook_set_default_child_flags(CtkContainer* container, CtkWidget* child)
{
	CtkWidget*      self     = CTK_WIDGET(container);
	CtkNotebook*    notebook = CTK_NOTEBOOK(container);
	CtkNotebookTab* tab;
	
	child->xexpand = TRUE;
	child->yexpand = TRUE;
	child->xfill = TRUE;
	child->yfill = TRUE;
	
	tab = g_malloc(sizeof(CtkNotebookTab));
	tab->page = child;
	tab->label = NULL;
	
	if (!self->node || !self->node->children)
	{
		tab->selected = TRUE;
	}
	else
	{
		tab->selected = FALSE;
	}
	
	notebook->tab_labels = g_slist_append(notebook->tab_labels, tab);
}

void ctk_notebook_lost_child(CtkContainer* container, CtkWidget* widget)
{
	CtkNotebook*    notebook;
	CtkNotebookTab* tab;
	GSList*         tab_list;
	gint            i;
	gint            kill_page;

	notebook = CTK_NOTEBOOK(container);
	
	kill_page = -1;
	i = 0;
	for (tab_list = notebook->tab_labels; tab_list; tab_list = tab_list->next)
	{
		tab = (CtkNotebookTab *)tab_list->data;
		
		if (tab->page == widget)
		{
			tab->page = NULL;
			kill_page = i;
		}
		if (tab->label == widget)
		{
			tab->label = NULL;
		}
		
		i++;
	}
	
	if (kill_page != -1)
	{
		ctk_notebook_remove_page(notebook, kill_page);
	}
}	

/* CTK Notebook New - Constructor */
CtkWidget* ctk_notebook_new(void)
{
	CtkNotebook *notebook;

	/* Init CtkBin */

	notebook = g_malloc(sizeof(CtkNotebook));

	ctk_notebook_init(notebook);
	return((CtkWidget *)notebook);
}

/* Get Notebook Page Number */
gint ctk_notebook_page_num(CtkNotebook* notebook, CtkWidget* child)
{
	CtkNotebookTab* tab;
	GSList*         tab_list;
	gint            page_num;

	if (!notebook || !notebook->tab_labels)
	    return 0;

	page_num = 0;
	for (tab_list = notebook->tab_labels; tab_list; tab_list = tab_list->next)
	{
		tab = (CtkNotebookTab *)tab_list->data;
		
		if (tab->page == child)
		    break;
		
		page_num++;
	}		
	
	return page_num;
}

CtkWidget* ctk_notebook_get_nth_page(CtkNotebook* notebook, gint page_num)
{
	CtkNotebookTab* tab;
	GSList*         tab_list;

	if (!notebook || !notebook->tab_labels)
	    return NULL;

	for (tab_list = notebook->tab_labels; tab_list; tab_list = tab_list->next)
	{
		tab = (CtkNotebookTab *)tab_list->data;
		
		if (page_num == 0)
			return tab->page;
		page_num--;
	}		
	
	return NULL;
}

/* Set the page label */
void ctk_notebook_set_tab_label(CtkNotebook* notebook,
			       CtkWidget* child,
			       CtkWidget* tab_label)
{
	GSList *tab_list;
	CtkNotebookTab *tab;
	
	if (!notebook || !child || !notebook->tab_labels)
	    return;
	 
	for (tab_list = notebook->tab_labels; tab_list; tab_list = tab_list->next)
	{
		tab = (CtkNotebookTab *)tab_list->data;
		
		if (tab->page == child)
		{
			if (tab->label)
			{
				ctk_container_remove(CTK_CONTAINER(notebook), tab->label);
			}
			
			tab->label = tab_label;
			CTK_CONTAINER(notebook)->set_child_flags = 
				&ctk_notebook_set_default_child_flags_noop;
			ctk_container_add(CTK_CONTAINER(notebook), tab->label);
			CTK_CONTAINER(notebook)->set_child_flags = 
				&ctk_notebook_set_default_child_flags;
			
			tab->label->selected = tab->selected;
		}
	}
	 
	ctk_size_mark_changed(CTK_WIDGET(notebook));
}

/* Set the page */
void ctk_notebook_set_page(CtkNotebook* notebook, gint page_num)
{
	GSList*         tab_list;
	CtkNotebookTab* tab;
	gint            i;
	CtkNotebookTab* cur_sel;
	CtkNotebookTab* new_sel;
	CtkWindow*      window;
	CtkWidget*      widget;

	if (!notebook || !notebook->tab_labels)
	    return;

	i = 0;
	
	widget = CTK_WIDGET(notebook);
	window = ctk_window_container(widget);
	
	if (window && window == ctk_window_get_focus_window() &&
		window->focus_widget == widget)
	{
		ctk_notebook_focus_out(CTK_WIDGET(notebook), NULL, NULL);
	}
	
	cur_sel = NULL;
	new_sel = NULL;
	
	for (tab_list = notebook->tab_labels; tab_list; tab_list = tab_list->next)
	{
		tab = (CtkNotebookTab *)tab_list->data;

		if (i == page_num)
			new_sel = tab;
		else if (tab->selected) 
		    cur_sel = tab;
		    
		i++;
	}
	
	if (new_sel)
	{
		if (cur_sel) 
		{
			cur_sel->selected = FALSE;
			if (cur_sel->label)
				cur_sel->label->selected = FALSE;
		}
		
		new_sel->selected = TRUE;
		if (new_sel->label)
			new_sel->label->selected = TRUE;
		
		ctk_signal_emit_by_name(CTK_OBJECT(notebook),"switch_page");
		ctk_draw_mark_changed(CTK_WIDGET(notebook));
	}

	if (window && window == ctk_window_get_focus_window() &&
		window->focus_widget == widget)
	{
		ctk_notebook_focus_in(CTK_WIDGET(notebook), NULL, NULL);
	}
}

/* Get Label from x y coord's */
CtkNotebookTab* ctk_notebook_get_yx_label(CtkNotebook* notebook, gint y, gint x)
{
	CtkNotebookTab* tab;
	GSList*         tab_list;
	CtkWidget*      widget;
	gint            cur_col;
	gint            cur_row;

	if (!notebook || !notebook->tab_labels)
	    return NULL;

	widget = CTK_WIDGET(notebook);
	cur_col = widget->col;
	cur_row = widget->row;
	
	for (tab_list = notebook->tab_labels; tab_list; tab_list = tab_list->next)
	{
		tab = (CtkNotebookTab*)tab_list->data;
		
		/* Did the user click on this label? */
		if (tab->label && ctk_check_clipping_xy(tab->label, x, y))
			break;
	}

	if (!tab_list)
	    tab = NULL;

	return tab;
}

/* Next Page - Switch to next page. Does nothing if current page is the
 * last page */
void ctk_notebook_next_page(CtkNotebook* notebook)
{
	CtkNotebookTab* tab;
	GSList*         tab_list;

	if (!notebook || !notebook->tab_labels)
	    return;
	    
	for (tab_list = notebook->tab_labels; tab_list; tab_list = tab_list->next)
	{
		tab = (CtkNotebookTab*)tab_list->data;
		
		if (tab->selected)
		{
			if (!tab_list->next)
			    break;
			
			tab = (CtkNotebookTab*)tab_list->next->data;
			ctk_notebook_set_page(notebook,
					      ctk_notebook_page_num(notebook, tab->page));			
			break;
		}
	}
}

/* Previous Page - Switch to previous page. Does nothing if current page is the
 * first page */
void ctk_notebook_prev_page(CtkNotebook* notebook)
{
	CtkNotebookTab* tab;
	CtkNotebookTab* tab_next;
	GSList*         tab_list;

	if (!notebook || !notebook->tab_labels)
	    return;

	for (tab_list = notebook->tab_labels; tab_list; tab_list = tab_list->next)
	{
		tab = (CtkNotebookTab *)tab_list->data;
		
		if (tab_list->next)
		    tab_next = (CtkNotebookTab *)tab_list->next->data;
		else
		    tab_next = NULL;

		if (tab_next && tab_next->selected)
		{
			ctk_notebook_set_page(notebook,
					      ctk_notebook_page_num(notebook, tab->page));
			break;
		}
	}	
}

/* Button key handle events */
gboolean ctk_notebook_key_press(CtkWidget* widget, CdkEventKey* event, gpointer data)
{
      if ((event->keyval == AK_ARROW_RIGHT) || (event->keyval == AK_ARROW_DOWN))
	    ctk_notebook_next_page(CTK_NOTEBOOK(widget));
      else if ((event->keyval == AK_ARROW_LEFT) || (event->keyval==AK_ARROW_UP))
	    ctk_notebook_prev_page(CTK_NOTEBOOK(widget));
      
      return TRUE;
}

/* Button mouse handle events */
gboolean ctk_notebook_button_press(CtkWidget* widget, CdkEventButton* event, gpointer data)
{	
      CtkNotebookTab* tab;
      
      tab = ctk_notebook_get_yx_label(CTK_NOTEBOOK(widget), event->y, event->x);
      if (tab)
      {
	    ctk_notebook_set_page(CTK_NOTEBOOK(widget),
				  ctk_notebook_page_num(CTK_NOTEBOOK(widget),tab->page));
      }
      
      return TRUE;
}

/* Set Tab Position */
void ctk_notebook_set_tab_pos(CtkNotebook* notebook, CtkPositionType pos)
{
	notebook->tab_pos = pos;
	
	ctk_size_mark_changed(CTK_WIDGET(notebook));
}

/* Append a Page */
void ctk_notebook_append_page(CtkNotebook* notebook,
			     CtkWidget* child,
			     CtkWidget* tab_label)
{
	if (!notebook || !child || !tab_label)
	    return;

	ctk_container_add(CTK_CONTAINER(notebook), child);
	ctk_notebook_set_tab_label(notebook, child, tab_label);
}

/* Prepend a Page 
 * Not yet implemented so just appending it for now */
void ctk_notebook_prepend_page(CtkNotebook* notebook,
			      CtkWidget* child,
			      CtkWidget* tab_label)
{
	ctk_notebook_append_page(notebook,child,tab_label);
}

/* Insert a Page
 * Note yet implemented so just appending it for now */
void ctk_notebook_insert_page(CtkNotebook* notebook,
			     CtkWidget* child,
			     CtkWidget* tab_label,
			     gint position)
{
	ctk_notebook_append_page(notebook,child,tab_label);
}

/* Remove a Page */
void ctk_notebook_remove_page(CtkNotebook* notebook, gint page_num)
{
	GSList*         tab_list;
	CtkNotebookTab* tab_item;
	gint            i;
	
	if (!notebook)
	    return;
	
	i = 0;
	for (tab_list = notebook->tab_labels; tab_list; tab_list = tab_list->next)
	{
		if (i == page_num) break;
		i++;
	}
	
	if (!tab_list)
		return;
	
	tab_item = (CtkNotebookTab*)tab_list->data;
	
	if (tab_item->selected)
	{
		if (tab_list->next)
		{
			ctk_notebook_next_page(notebook);
		}
		else
		{
			ctk_notebook_prev_page(notebook);
		}
	}

	notebook->tab_labels = g_slist_remove_link(notebook->tab_labels, tab_list);
	g_slist_free_1(tab_list);

	if (tab_item->label) ctk_widget_destroy(tab_item->label);
	if (tab_item->page)  ctk_widget_destroy(tab_item->page);

	ctk_size_mark_changed(CTK_WIDGET(notebook));
}

/* Set Show Tabs */
void ctk_notebook_set_show_tabs(CtkNotebook* notebook, gboolean show_tabs)
{
	if (!notebook)
	    return;

	notebook->show_tabs = show_tabs;
	ctk_size_mark_changed(CTK_WIDGET(notebook));
}

/* Set Show Border */
void ctk_notebook_set_show_border(CtkNotebook* notebook, gboolean show_border)
{
	if (!notebook)
	    return;
	
	if (show_border)
		CTK_CONTAINER(notebook)->border_width = 1;
	else
		CTK_CONTAINER(notebook)->border_width = 0;
		
	ctk_size_mark_changed(CTK_WIDGET(notebook));
}

/* Set Tabs Scrollable */
void ctk_notebook_set_scrollable(CtkNotebook* notebook, gboolean scrollable)
{
	if (!notebook)
	    return;
	
// Not implemented:
//	notebook->tabs_scrollable = scrollable;
//	ctk_size_mark_changed(CTK_WIDGET(notebook));
}

/* Destroy notebook data */
gpointer ctk_notebook_destroy(CtkObject* object)
{
	GSList*         list;
	CtkNotebook*    notebook;

	notebook = CTK_NOTEBOOK(object);
	
	for (list = notebook->tab_labels; list; list = list->next)
		g_free(list->data);

	if (notebook->tab_labels)
	    g_slist_free(notebook->tab_labels);
	
	return NULL;
}

gboolean ctk_notebook_focus_in(CtkWidget* widget, CdkEventFocus* event, gpointer data)
{
	CtkNotebook*    notebook;
	CtkNotebookTab* tab;
	GSList*         tab_list;
	
	notebook = CTK_NOTEBOOK(widget);

	if (!notebook || !notebook->tab_labels)
	    return FALSE;

	for (tab_list = notebook->tab_labels; tab_list; tab_list = tab_list->next)
	{
		tab = (CtkNotebookTab *)tab_list->data;
		
		if (tab->selected)
			tab->label->inversed = TRUE;
	}
	
	ctk_draw_mark_changed(CTK_WIDGET(notebook));
	return TRUE;
}

gboolean ctk_notebook_focus_out(CtkWidget* widget, CdkEventFocus* event, gpointer data)
{
	CtkNotebook*    notebook;
	CtkNotebookTab* tab;
	GSList*         tab_list;
	
	notebook = CTK_NOTEBOOK(widget);

	if (!notebook || !notebook->tab_labels)
	    return FALSE;

	for (tab_list = notebook->tab_labels; tab_list; tab_list = tab_list->next)
	{
		tab = (CtkNotebookTab *)tab_list->data;
		
		if (tab->selected)
			tab->label->inversed = FALSE;
	}
	
	ctk_draw_mark_changed(CTK_WIDGET(notebook));
	return TRUE;
}
