/* 
 * $Id: ctkbutton.c,v 1.26 2000/07/18 22:17:53 terpstra Exp $
 *
 * CTK - Console Toolkit
 *
 * Copyright (C) 1998-2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Authors: Kevin Lindsay, Wesley Terpstra
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <glib.h>

#include "ctk.h"
#include "ctkcolor.h"

gboolean ctk_button_button_press  (CtkWidget* widget, CdkEventButton*   event, gpointer data);
gboolean ctk_button_button_release(CtkWidget* widget, CdkEventButton*   event, gpointer data);
gboolean ctk_button_leave         (CtkWidget* widget, CdkEventCrossing* event, gpointer data);
gboolean ctk_button_enter         (CtkWidget* widget, CdkEventCrossing* event, gpointer data);
gboolean ctk_button_key_press     (CtkWidget* widget, CdkEventKey*      event, gpointer data);

/* Initialize the Button */
void ctk_button_init(CtkButton *button)
{
	CtkWidget* self;
	
	ctk_bin_init(&button->bin);	
	CTK_OBJECT(button)->type = CtkTypeButton;

	self = CTK_WIDGET(button);

	((CtkContainer*)button)->border_width = 0;
		
	button->pressed = FALSE;
	button->entered = FALSE;

	self->sensitive = TRUE;

	ctk_signal_new("clicked",  CtkTypeButton);

	ctk_signal_new("pressed",  CtkTypeButton);
	ctk_signal_new("released", CtkTypeButton);
	
	ctk_signal_new("enter",    CtkTypeButton);
	ctk_signal_new("leave",    CtkTypeButton);
		
	self->orig_width  = 5;
	self->orig_height = 1;
	 
	self->main_col           = ctk_calculate_palette(CTK_COLOR_CYAN,
	                                                 CTK_COLOR_BLUE);
	self->inverse_col        = ctk_calculate_palette(CTK_COLOR_WHITE,
	                                                 CTK_COLOR_RED);
	self->insensitive_col    = ctk_calculate_palette(CTK_COLOR_GRAY,
	                                                 CTK_COLOR_LIGHT_GRAY);

	ctk_signal_connect(CTK_OBJECT(button), "key_press_event",
			   CTK_SIGNAL_FUNC(&ctk_button_key_press), NULL);
	ctk_signal_connect(CTK_OBJECT(button), "button_press_event",
			   CTK_SIGNAL_FUNC(&ctk_button_button_press), NULL);
	ctk_signal_connect(CTK_OBJECT(button), "button_release_event",
			   CTK_SIGNAL_FUNC(&ctk_button_button_release), NULL);
	ctk_signal_connect(CTK_OBJECT(button), "enter_notify_event",
			   CTK_SIGNAL_FUNC(&ctk_button_enter), NULL);
	ctk_signal_connect(CTK_OBJECT(button), "leave_notify_event",
			   CTK_SIGNAL_FUNC(&ctk_button_leave), NULL);

	self->set_min_size  = &ctk_button_min_size;
	self->set_real_size = &ctk_button_real_size;
}

/* Create a Button Widget */
CtkWidget* ctk_button_new()
{
	CtkButton *button;
	
	button = g_malloc(sizeof(CtkButton));
	
	ctk_button_init(button);
	return ((CtkWidget *)button);
}

/* Button key handle events */
gboolean ctk_button_key_press(CtkWidget* widget, CdkEventKey* event, gpointer data)
{
      CtkButton* button = CTK_BUTTON(widget);
      gboolean   state;

      if (event->keyval == AK_ENTER || event->keyval == ' ')
      {
	    state = button->entered;
	    button->entered = TRUE;

	    ctk_button_button_press(widget, NULL, NULL);
	    ctk_redraw_screen(CTK_REDRAW_CHANGED);
	    usleep(50000); /* give 1/20th a second button display */
	    ctk_button_button_release(widget, NULL, NULL);

	    button->entered = state;
	    return TRUE;
      }
      
      return ctk_signal_return;
}

/* Pressed event callback */
gboolean ctk_button_button_press(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
      CTK_BUTTON(widget)->pressed = TRUE;
      widget->inverse_col = ctk_calculate_palette(CTK_COLOR_BRIGHT_RED, CTK_COLOR_RED);
      ctk_draw_mark_changed(widget);

      ctk_signal_emit_by_name(CTK_OBJECT(widget), "pressed");
      return TRUE;
}

void ctk_button_clicked(CtkButton* button)
{
	ctk_signal_emit_by_name(CTK_OBJECT(button), "clicked");
}

/* Released event callback */
gboolean ctk_button_button_release(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
      CtkButton* button = CTK_BUTTON(widget);

      if (button->pressed && button->entered)
            ctk_button_clicked(button);

      button->pressed = FALSE;
      widget->inverse_col = ctk_calculate_palette(CTK_COLOR_WHITE, CTK_COLOR_RED);
      ctk_draw_mark_changed(widget);

      ctk_signal_emit_by_name(CTK_OBJECT(widget), "released");
      return TRUE;
}

gboolean ctk_button_enter(CtkWidget* widget, CdkEventCrossing* event, gpointer data)
{
      CtkButton* button = CTK_BUTTON(widget);

      button->entered = TRUE;
      
      if (button->pressed)
      {
	    widget->inverse_col = ctk_calculate_palette(CTK_COLOR_BRIGHT_RED, CTK_COLOR_RED);
	    ctk_draw_mark_changed(widget);
      }

      ctk_signal_emit_by_name(CTK_OBJECT(widget), "enter");
      return TRUE;
}

gboolean ctk_button_leave(CtkWidget* widget, CdkEventCrossing* event, gpointer data)
{
      CtkButton* button = CTK_BUTTON(widget);

      button->entered = FALSE;

      if (button->pressed)
      {
	    widget->inverse_col = ctk_calculate_palette(CTK_COLOR_WHITE, CTK_COLOR_RED);
	    ctk_draw_mark_changed(widget);
      }

      ctk_signal_emit_by_name(CTK_OBJECT(widget), "leave");
      return TRUE;
}

CtkWidget* ctk_button_new_with_label(const gchar* text)
{
	CtkWidget *button;
	CtkWidget *label;

	button = ctk_button_new();

	label = ctk_label_new(text);

	ctk_container_add(CTK_CONTAINER(button), label);
	
	ctk_widget_show(label);
	
	return button;
}

/* NOTE: min_height is either == 1 or >= 3 */
void ctk_button_min_size(CtkWidget* widget)
{
	/* We are basically a bin, but if we're 1 row high we have a special case */
	
	((CtkContainer*)widget)->border_width = 0;
	ctk_bin_min_size(widget);
	
	if (widget->min_height <= 0) widget->min_height = 1;
	
	if (widget->min_height == 1)
	{
		/* make way for the []s */
		widget->min_width += 2;
	}
	else
	{
		/* Throw on the border */
		((CtkContainer*)widget)->border_width = 1;
		ctk_bin_min_size(widget);
	}
}

void ctk_button_real_size(CtkWidget* widget)
{
	switch (widget->height)
	{
		case 1:
			/* Remove space for []s, be a bin, restore space */
			widget->col++;
			widget->width -= 2;
			
			ctk_bin_real_size(widget);
			
			widget->col--;
			widget->width += 2;
			break;
			
		case 2:
			/* This happened b/c we said we needed 1 row only.
			 * Hence we have the extra [] space and should draw
			 * child on the first row of the border
			 */
			widget->col++;
			widget->width -= 2;
			widget->height--;
			
			ctk_bin_real_size(widget);
			
			widget->col--;
			widget->width += 2;
			widget->height++;
			break;
			
		default:
			/* Throw on the border */
			((CtkContainer*)widget)->border_width = 1;
			ctk_bin_real_size(widget);
	}
}
