//LabPlot : PlotQWT3D.cc

#include <iostream>
#include <fstream>
#include <qregexp.h>
#include <kmessagebox.h>
#include "TitleDialog.h"
#include "LegendDialog.h"
#include "AxesDialog.h"
#include "GraphListDialog.h"
#include "PlotQWT3D.h"
#include "Label.h"
#ifdef HAVE_GL
#include "qwt3d_function.h"
#include "qwt3d_io.h"
#endif

using namespace std;

PlotQWT3D::PlotQWT3D(Worksheet *p)
	: Plot(p)
#ifdef HAVE_GL
	, Qwt3D::SurfacePlot((QWidget *)p)
#endif
{
	QFont font = p->getMainWin()->defaultFont();
	font.setPointSize((int)(0.7*font.pointSize()));	// for axes label

	// set default axis settings for all axes
	for(int i=0;i<12;i++)
		readAxisSettings(&axis[i],PQWT3D,i);

	// for order see qwt3d_types.h
	axis[0].setLabel(new Label(i18n("x-Axis"),font));
	axis[1].setLabel(new Label(i18n("y-Axis"),font));
	axis[2].setLabel(new Label(i18n("z-Axis"),font));
	axis[3].setLabel(new Label(i18n("x2-Axis"),font));
	axis[4].setLabel(new Label(i18n("x3-Axis"),font));
	axis[5].setLabel(new Label(i18n("x4-Axis"),font));
	axis[6].setLabel(new Label(i18n("y4-Axis"),font));
	axis[7].setLabel(new Label(i18n("y3-Axis"),font));
	axis[8].setLabel(new Label(i18n("y2-Axis"),font));
	axis[9].setLabel(new Label(i18n("z2-Axis"),font));
	axis[10].setLabel(new Label(i18n("z4-Axis"),font));
	axis[11].setLabel(new Label(i18n("z3-Axis"),font));

	setGraphBackground(Qt::black);
#ifdef HAVE_GL
	plotstyle = Qwt3D::FILLEDMESH;
	coordinatestyle = Qwt3D::BOX;
	floorstyle = Qwt3D::NOFLOOR;

	// use default surface color
	KConfig *config = p->getMainWin()->Config();
	config->setGroup( "Plot Surface Style" );
	QString colorscale=config->readEntry("Colorscale",0);

	cv.clear();
	Qwt3D::RGBA rgb;

	ifstream file((const char*)colorscale.local8Bit());

	if(file) {
		while ( file ) {
			file >> rgb.r >> rgb.g >> rgb.b;
			file.ignore(1000,'\n');
			if (!file.good())
				break;
			else {
				rgb.a = 1;
				rgb.r /= 255;
				rgb.g /= 255;
				rgb.b /= 255;
				cv.push_back(rgb);
			}
		}
	}
	else {
		for(int i=0;i<255;i++) {
			rgb.a = 1;
			rgb.r = i/255.0;
			rgb.g = 40.0/255.0;
			rgb.b = 1.0-i/255.0;
			cv.push_back(rgb);
		}
	}

	// initial values
	setRotation(30,0,15);
	aspect_ratio=1;	// X-Y-Z

	font.setPointSize((int)(0.7*font.pointSize()));	// for tic label
	for (int i=0;i<12;i++) {
		axis[i].enableMajorGrid(false);
		axis[i].setTickLabelFont(font);
		axis[i].setMajorTickLength(0.05);
		axis[i].setMinorTickLength(0.02);
	}

	//TODO : set legend box ?
	Plot::legend.setPosition(.9,.05);

	mouse_disabled=false;
	resolution=1;

	show();
#endif
}

PlotQWT3D::~PlotQWT3D() {
	delete graphlist;

	delete title;
	delete markx;
	delete marky;

	delete region;
}

QStringList PlotQWT3D::Info() {
	QStringList s;
	s<<"QWT";

	s<<QString::number(position.X())+QString(" , ")+QString::number(position.Y());
	s<<QString::number(Plot::size.X())+QString(" X ")+QString::number(Plot::size.Y());
	if (transparent)
		s<<QString("yes");
	else
		s<<QString("no");
	s<<bgcolor.color().name();
	s<<gbgcolor.color().name();

	return s;
}

void PlotQWT3D::mouseDoubleClickEvent(QMouseEvent *e) {
	kdDebug()<<"PlotQWT3D::mouseDoubleClickEvent()"<<endl;
	int x=e->x(), y=e->y();
	kdDebug()<<"	MOUSE @ "<<x<<' '<<y<<endl;

	MainWin *mw = worksheet->getMainWin();
	int X=worksheet->getX(), Y=worksheet->getY();

	if (title && title->inside(x,y,position,Plot::size,X,Y))
		(new TitleDialog(mw,0))->show();
	else if (insideLegend(x,y))
		 (new LegendDialog(mw,0))->show();
	else if (x/(double)X>0.25 &&x/(double)X<0.75 && y/(double)Y>0.25 &&y/(double)Y<0.75 )
		(new GraphListDialog(mw))->show();
	else
		(new AxesDialog(mw,0,0))->show();
}

void PlotQWT3D::Export(QPainter *p, QString fn,QString format, int w, int h) {
	kdDebug()<<"PlotQWT3D::Export() : w/h="<<w<<' '<<h<<endl;
	kdDebug()<<"	File/FORMAT = "<<fn<<' '<<format<<endl;
#ifdef HAVE_GL
#if KDE_VERSION > 0x030104
	kdDebug()<<"	supported : "<<Qwt3D::IO::outputFormatList()<<endl;
#endif
	draw(p,w,h);
	Qwt3D::Plot3D::save(fn,format);
//	kdDebug()<<"save() done"<<endl;
#endif
}

void PlotQWT3D::draw(QPainter *p, int w, int h) {
	kdDebug()<<"PlotQWT3D::draw() : w/h="<<w<<' '<<h<<endl;
#ifdef HAVE_GL
	resize(w,h);

	kdDebug()<<"PlotStyle = "<<plotstyle<<endl;
	Qwt3D::Plot3D::setPlotStyle(plotstyle);
	// TODO : Enrichments : Qwt3D::Plot3D::setPlotStyle(Bar(0.007,0.5));

	// TODO : set style of legend too
	kdDebug()<<"CoordinateStyle = "<<coordinatestyle<<endl;
	Qwt3D::Plot3D::setCoordinateStyle(coordinatestyle);
	kdDebug()<<"FloorStyle = "<<floorstyle<<endl;
	Qwt3D::SurfacePlot::setFloorStyle(floorstyle);

	QColor c = Background().color();
	setBackgroundColor(Qwt3D::RGBA(c.red()/255.0,c.green()/255.0,c.blue()/255.0));
	c = graphBackground().color();
	setMeshColor(Qwt3D::RGBA(c.red()/255.0,c.green()/255.0,c.blue()/255.0));

	// Title
	// TODO : boxed, rotation, richtext ?
	QString t = title->simpleTitle();
	kdDebug()<<"title = "<<t<<endl;
	setTitle(t);
	setTitlePosition(1.0-title->Y(),title->X());		// TODO : Qwt3D::Anchor
	QFont f = title->Font();
	setTitleFont(f.family(),f.pointSize(),f.weight(),f.italic());
	c = title->Color();
	setTitleColor(Qwt3D::RGBA(c.red()/255.0,c.green()/255.0,c.blue()/255.0));

	// set data color
	Qwt3D::StandardColor *col = new Qwt3D::StandardColor(this);
	col->setColorVector(cv);
	setDataColor(col);

	// resolution
	setResolution(resolution);

	// Legend
	// TODO : set isolines of legend
	showColorLegend(Plot::legend.Enabled());
	Qwt3D::ColorLegend *l = Qwt3D::Plot3D::legend();
	// set legend position
	double lx = Plot::legend.X(), ly = 1.0-Plot::legend.Y();
	kdDebug()<<"Legend X/Y "<<lx<<'/'<<ly<<endl;
	// for LeftRight :

	GraphM *graph = graphlist->getGraphM(0);
	if(graph)
		l->setTitleString(graph->getLabel()->simpleTitle());		// TODO : position ?
	f = Plot::legend.Font();
	l->setTitleFont(f.family(),f.pointSize(),f.weight(),f.italic());
	//TODO : Top,Bottom,Left,Right
	if(Plot::legend.getOrientation()) {	// LeftRight
		l->setRelPosition(Qwt3D::Tuple(lx-.25,ly-.05),Qwt3D::Tuple(lx+.05,ly));
		l->setOrientation(Qwt3D::ColorLegend::LeftRight,Qwt3D::ColorLegend::Bottom);
	}
	else {	// BottomTop
		l->setRelPosition(Qwt3D::Tuple(lx,ly-.3),Qwt3D::Tuple(lx+.05,ly));
		l->setOrientation(Qwt3D::ColorLegend::BottomTop,Qwt3D::ColorLegend::Left);
	}

	Qwt3D::SCALETYPE scale = Qwt3D::LINEARSCALE;
	switch(axis[2].Scale()) {
		case LINEAR : scale = Qwt3D::LINEARSCALE; break;
		case LOG10 : scale = Qwt3D::LOG10SCALE; break;
		case LOG2 : case LN : case SQRT: case SX2: break;
	}
	l->setScale(scale);
	// TODO : what if delta==0 ?
	l->setLimits(Plot::Ranges()[2].rMin(),Plot::Ranges()[2].rMax());

	drawCurves(p,w,h);

	updateData();
	updateGL();
#endif
	kdDebug()<<"PlotQWT3D::draw() DONE"<<endl;
}

void PlotQWT3D::drawCurves(QPainter *, int, int) {
	kdDebug()<<"PlotQWT3D::drawCurves()"<<endl;
#ifdef HAVE_GL
	double minx=actrange[0].rMin(), maxx=actrange[0].rMax();
	double miny=actrange[1].rMin(), maxy=actrange[1].rMax();
	double minz=actrange[2].rMin(), maxz=actrange[2].rMax();

	int item=0;	// only one graph possible
	Graph *graph = graphlist->getGraph(item);
	if(graph == 0 || graph->isShown() == false)
		return;

	//Style *style=0;
	//Symbol *symbol=0;
	GRAPHType s = graphlist->getType(item);
	LRange rx, ry, rz;
	int rows, cols;
	if(s == GRAPHM) {
		GraphM *g = graphlist->getGraphM(item);
		rx = g->Range(0);
		ry = g->Range(1);
		rz = g->Range(2);

		double *data = g->Data();

		rows = g->NY();
		cols = g->NX();

		kdDebug()<<"	QWT SURFACE :"<<endl;
		kdDebug()<<"	DATA min/max x y = "<<rx.rMin()<<' '<<rx.rMax()<<' '<<ry.rMin()<<' '<<ry.rMax()<<endl;
		kdDebug()<<"	PLOT min/max x y = "<<minx<<' '<<maxx<<' '<<miny<<' '<<maxy<<endl;

		kdDebug()<<"	DATA ROWS/COLS :"<<rows<<' '<<cols<<endl;
		kdDebug()<<"	REAL ROWS/COLS :"<<rows*(maxy-miny)/(ry.rMax()-ry.rMin())<<' '<<cols*(maxx-minx)/(rx.rMax()-rx.rMin())<<endl;
		int rcols = (int)(cols*(maxx-minx)/(rx.rMax()-rx.rMin()));
		int scols = (int)(cols*(minx-rx.rMin())/(rx.rMax()-rx.rMin()));
		int rrows = (int)(rows*(maxy-miny)/(ry.rMax()-ry.rMin()));
		int srows = (int)(rows*(miny-ry.rMin())/(ry.rMax()-ry.rMin()));
		kdDebug()<<"	PLOT ROWS/COLS :"<<rrows<<' '<<rcols<<endl;
		kdDebug()<<"	START ROWS/COLS :"<<srows<<' '<<scols<<endl;

		if((minx-miny>0) && (rrows<0 || srows<0) ) {
			kdDebug()<<"rrows = "<<rrows<<" / srows = "<<srows<<endl;
			kdDebug()<<"rcols = "<<rcols<<" / scols = "<<scols<<endl;
			KMessageBox::error(this,i18n("Sorry. Something went wrong. Giving up!"));
			return;
		}

		double** sdata = new double* [rcols];

		for ( int i = 0; i < rcols; ++i)
			sdata[i] = new double [rrows];

		// copy data
		for(int i=0;i<rrows;i++) {
			for(int j=0;j<rcols;j++) {
				if(i<abs(srows) || j<abs(scols) || j>=cols-scols || i>=rows-srows)
					sdata[j][i]=0;
				else
					sdata[j][i] = data[j+(i+srows)*cols+scols];
			}
		}

		// load data into surface plot
		if(rcols >0 && rrows > 0)
			loadFromData(sdata,rcols,rrows,minx,maxx,miny,maxy);
	}
	else if(s == GRAPH3D) {
		Graph3D *g = graphlist->getGraph3D(item);
		rx = g->Range(0);
		ry = g->Range(1);
		rz = g->Range(2);

		Point3D *data = g->Data();
		rows = g->NY();
		cols = g->NX();

		kdDebug()<<"	DATA ROWS/COLS :"<<rows<<' '<<cols<<endl;
		int rcols = (int) round(cols*(maxx-minx)/(rx.rMax()-rx.rMin()));
		int scols = (int) round(cols*(minx-rx.rMin())/(rx.rMax()-rx.rMin()));
		int rrows = (int) round(rows*(maxy-miny)/(ry.rMax()-ry.rMin()));
		int srows = (int) round(rows*(miny-ry.rMin())/(ry.rMax()-ry.rMin()));
		kdDebug()<<"	PLOT ROWS/COLS :"<<rrows<<' '<<rcols<<endl;
		kdDebug()<<"	START ROWS/COLS :"<<srows<<' '<<scols<<endl;

		if(rrows<0 || srows<0) {
			kdDebug()<<"	Could not estimate dimensions. Giving up!"<<endl;
			KMessageBox::error(this,i18n("Sorry. Something went wrong. Giving up!"));
			return;
		}

		Qwt3D::Triple **sdata = new Qwt3D::Triple* [rcols];
		for ( int i = 0; i < rcols; ++i)
			sdata[i] = new  Qwt3D::Triple[rrows];

		// copy data
		// TODO : zoom in 3d data
//		kdDebug()<<"	FILLING DATA"<<endl;
//		kdDebug()<<"	rrows="<<rrows<<"	rcols="<<rcols<<endl;
//		kdDebug()<<"	srows="<<srows<<"	scols="<<scols<<endl;
		for(int i=0;i<rrows;i++) {
			for(int j=0;j<rcols;j++) {
				if(i<abs(srows))
					sdata[j][i]=Qwt3D::Triple(minx,miny,0);
				else if(j<abs(scols))
					sdata[j][i]=Qwt3D::Triple(minx,miny,0);
				else if( j>=cols-scols )
					sdata[j][i]=Qwt3D::Triple(maxx,maxy,0);
				else if( i>=rows-srows )
					sdata[j][i]=Qwt3D::Triple(maxx,maxy,0);
				else {
					int item = j+(i+srows)*cols+scols;
//					kdDebug()<<"i/j/item "<<i<<' '<<j<<' '<<item<<"	: x/y/z "<<data[item].X()<<' '<<data[item].Y()<<' '<<data[item].Z()<<endl;
					sdata[j][i] = Qwt3D::Triple(data[item].X(),data[item].Y(),data[item].Z());
				}
			}
		}

		// load data into surface plot
		loadFromData(sdata,rcols,rrows);

		//TODO : convert for showing surface, etc. ???
		Qwt3D::Plot3D::setPlotStyle(Qwt3D::POINTS);
	}
	else
		return;

	// scaling
	double d = (hull().maxVertex-hull().minVertex).length();
	double xrange = maxx-minx;
	double yrange = maxy-miny;
	double zrange = maxz-minz;
	if(zrange == 0)
		zrange = 1.0;
	double xymax = fmax(xrange,yrange);
	double rmax = fmax(xymax,zrange);
	switch(aspect_ratio) {
		case 0: setScale(1.0/rmax,1.0/rmax,1.0/rmax); break;
		case 1: setScale(1.0/xrange,1.0/yrange,1.0/zrange); break;
		case 2: setScale(1.0/xrange,1.0/yrange,1.0/xymax); break;
		case 3: setScale(1.0/rmax,1.0/rmax,1.0/zrange); break;
	}
	setZoom(d / sqrt(3.0));

	if(mouse_disabled)
		Plot3D::disableMouse();
	else
		Plot3D::enableMouse();

	// use color of first axis
	QColor c = axis[0].TickColor();
	coordinates()->setAxesColor(Qwt3D::RGBA(c.red()/255.0,c.green()/255.0,c.blue()/255.0));

	// grid lines
	int sum = Qwt3D::NOSIDEGRID;
	if(axis[2].MajorGridEnabled() || axis[10].MajorGridEnabled())
		sum |= Qwt3D::LEFT;
	if(axis[4].MajorGridEnabled() || axis[8].MajorGridEnabled())
		sum |= Qwt3D::RIGHT;
	if(axis[3].MajorGridEnabled() || axis[7].MajorGridEnabled())
		sum |= Qwt3D::CEIL;
	if(axis[9].MajorGridEnabled() || axis[1].MajorGridEnabled())
		sum |= Qwt3D::FLOOR;
	if(axis[0].MajorGridEnabled() || axis[5].MajorGridEnabled())
		sum |= Qwt3D::FRONT;
	if(axis[6].MajorGridEnabled() ||axis[11].MajorGridEnabled())
		sum |= Qwt3D::BACK;

	coordinates()->setGridLines(sum, sum, sum);

	// grid + border color (global !)
	for(int i=0;i<12;i++) {
		if(axis[i].majorGridColor() != Qt::black) {
			c = axis[i].majorGridColor();
			coordinates()->setGridLinesColor(Qwt3D::RGBA(c.red()/255.0,c.green()/255.0,c.blue()/255.0));
		}
		if(axis[i].BorderColor() != Qt::black) {
			c = axis[i].BorderColor();
			coordinates()->setAxesColor(Qwt3D::RGBA(c.red()/255.0,c.green()/255.0,c.blue()/255.0));
		}
	}

	// TODO : scale data too!
	// TODO : USERSCALE for LOG2, etc. ?
	// use scale from first x,y,z axis
	Qwt3D::SCALETYPE scale = Qwt3D::LINEARSCALE;
	switch(axis[0].Scale()) {
		case LINEAR : scale = Qwt3D::LINEARSCALE; break;
		case LOG10 :
			      scale = Qwt3D::LOG10SCALE;
			      coordinates()->axes[Qwt3D::X1].setLimits(log10(minx),log10(maxx));
			      coordinates()->axes[Qwt3D::X2].setLimits(log10(minx),log10(maxx));
			      coordinates()->axes[Qwt3D::X3].setLimits(log10(minx),log10(maxx));
			      coordinates()->axes[Qwt3D::X4].setLimits(log10(minx),log10(maxx));
			      break;
		case LOG2 : case LN : case SQRT: case SX2: break;
	}
	coordinates()->axes[Qwt3D::X1].setScale(scale);
	coordinates()->axes[Qwt3D::X2].setScale(scale);
	coordinates()->axes[Qwt3D::X3].setScale(scale);
	coordinates()->axes[Qwt3D::X4].setScale(scale);
	switch(axis[1].Scale()) {
		case LINEAR : scale = Qwt3D::LINEARSCALE; break;
		case LOG10 :
			      scale = Qwt3D::LOG10SCALE;
			      coordinates()->axes[Qwt3D::Y1].setLimits(log10(miny),log10(maxy));
			      coordinates()->axes[Qwt3D::Y2].setLimits(log10(miny),log10(maxy));
			      coordinates()->axes[Qwt3D::Y3].setLimits(log10(miny),log10(maxy));
			      coordinates()->axes[Qwt3D::Y4].setLimits(log10(miny),log10(maxy));
			      break;
		case LOG2 : case LN : case SQRT: case SX2: break;
	}
	coordinates()->axes[Qwt3D::Y1].setScale(scale);
	coordinates()->axes[Qwt3D::Y2].setScale(scale);
	coordinates()->axes[Qwt3D::Y3].setScale(scale);
	coordinates()->axes[Qwt3D::Y4].setScale(scale);
	switch(axis[2].Scale()) {
		case LINEAR : 	scale = Qwt3D::LINEARSCALE; break;
		case LOG10 :
			      scale = Qwt3D::LOG10SCALE;
			      coordinates()->axes[Qwt3D::Z1].setLimits(log10(minz),log10(maxz));
			      coordinates()->axes[Qwt3D::Z2].setLimits(log10(minz),log10(maxz));
			      coordinates()->axes[Qwt3D::Z3].setLimits(log10(minz),log10(maxz));
			      coordinates()->axes[Qwt3D::Z4].setLimits(log10(minz),log10(maxz));
			      break;
		case LOG2 : case LN : case SQRT: case SX2: break;
	}
	coordinates()->axes[Qwt3D::Z1].setScale(scale);
	coordinates()->axes[Qwt3D::Z2].setScale(scale);
	coordinates()->axes[Qwt3D::Z3].setScale(scale);
	coordinates()->axes[Qwt3D::Z4].setScale(scale);

	// all axes
	for(int i=0;i<12;i++) {
		Label *l = axis[i].getLabel();
		QString t = l->Title();
		QRegExp rx("<.*>");		// remove tags
		rx.setMinimal(true);
		t.replace(rx,"");
		t.replace(QRegExp(QString("\n")),"");	// replace newlines

		// TODO : more axes settings
		// also : axes[Z1].setLabelString(QChar(0x38f);  // Omega - see http://www.unicode.org/charts/
		// dont use Qwt3D::Axis axes !
		coordinates()->axes[i].setTicLength(axis[i].majorTickLength(),axis[i].minorTickLength());

		if(axis[i].TickPos()==2)
			coordinates()->axes[i].setSymmetricTics(true);
		else
			coordinates()->axes[i].setSymmetricTics(false);

		coordinates()->axes[i].setNumbers(axis[i].Enabled());
		coordinates()->axes[i].setNumberFont(axis[i].TickLabelFont());
		c = axis[i].TickLabelColor();
		coordinates()->axes[i].setNumberColor(Qwt3D::RGBA(c.red()/255.0,c.green()/255.0,c.blue()/255.0));
		coordinates()->axes[i].adjustNumbers((int)(axis[i].TickLabelPosition()/2.0));

		coordinates()->axes[i].setLabelString(t);
		coordinates()->axes[i].setLabelFont(l->Font());
		c = l->Color();
		coordinates()->axes[i].setLabelColor(Qwt3D::RGBA(c.red()/255.0,c.green()/255.0,c.blue()/255.0));

		int bw=axis[i].borderWidth();
		coordinates()->axes[i].setLineWidth(bw,
				axis[i].majorTickWidth()/(double)bw,axis[i].minorTickWidth()/(double)bw);

		// tics
		int major=0, minor=0;
		if(axis[i].MajorTicksEnabled())
			major=(int) axis[i].MajorTicks();
		if(axis[i].MinorTicksEnabled())
			minor=axis[i].MinorTicks();

		if(major==-1) {
			double min, max;
			coordinates()->axes[i].limits(min,max);
			major=autoTicks(min,max);
		}
//		kdDebug()<<"	Axis "<<i<<" MAJOR/MINOR "<<major<<' '<<minor<<endl;
		coordinates()->axes[i].setMajors(major);
		coordinates()->axes[i].setMinors(minor);
	}
#endif
}

void PlotQWT3D::saveXML(QDomDocument doc,QDomElement plottag) {
	QDomElement tag;

	for (int i = 0; i < 12; i++) {
		tag = axis[i].saveXML(doc,i);
   		plottag.appendChild( tag );
	}

#ifdef HAVE_GL
	tag = doc.createElement( "PlotStyle" );
   	plottag.appendChild( tag );
  	QDomText t = doc.createTextNode( QString::number(plotstyle) );
    	tag.appendChild( t );
	tag = doc.createElement( "CoordinateStyle" );
   	plottag.appendChild( tag );
  	t = doc.createTextNode( QString::number(coordinatestyle) );
    	tag.appendChild( t );
	tag = doc.createElement( "FloorStyle" );
   	plottag.appendChild( tag );
  	t = doc.createTextNode( QString::number(floorstyle) );
    	tag.appendChild( t );

	QDomElement cvtag = doc.createElement( "ColorVector" );
	plottag.appendChild( cvtag );
	for(unsigned int i=0;i<cv.size();i++) {
		tag = doc.createElement( "Color" );
		tag.setAttribute("r",QString::number(cv[i].r));
		tag.setAttribute("g",QString::number(cv[i].g));
		tag.setAttribute("b",QString::number(cv[i].b));
		tag.setAttribute("a",QString::number(cv[i].a));
		cvtag.appendChild( tag );
	}

	tag = doc.createElement( "AspectRatio" );
   	plottag.appendChild( tag );
  	t = doc.createTextNode( QString::number(aspect_ratio) );
    	tag.appendChild( t );
	tag = doc.createElement( "MouseDisabled" );
   	plottag.appendChild( tag );
  	t = doc.createTextNode( QString::number(mouse_disabled) );
    	tag.appendChild( t );
	tag = doc.createElement( "Resolution" );
   	plottag.appendChild( tag );
  	t = doc.createTextNode( QString::number(resolution) );
    	tag.appendChild( t );
#endif
}

void PlotQWT3D::openXML(QDomElement e) {
	if(e.tagName() == "Axis")
		axis[e.attribute("id").toInt()].openXML(e.firstChild());
#ifdef HAVE_GL
	else if(e.tagName() == "PlotStyle")
		plotstyle = (Qwt3D::PLOTSTYLE) e.text().toInt();
	else if(e.tagName() == "CoordinateStyle")
		coordinatestyle = (Qwt3D::COORDSTYLE) e.text().toInt();
	else if(e.tagName() == "FloorStyle")
		floorstyle = (Qwt3D::FLOORSTYLE) e.text().toInt();
	if(e.tagName() == "ColorVector") {
		QDomNode node = e.firstChild();
		cv.clear();
		Qwt3D::RGBA rgb;
		while(!node.isNull()) {
			QDomElement e = node.toElement();
			rgb.r = e.attribute("r").toDouble();
			rgb.g = e.attribute("g").toDouble();
			rgb.b = e.attribute("b").toDouble();
			rgb.a = e.attribute("a").toDouble();
			// kdDebug()<<"RGBA : "<<rgb.r<<' '<<rgb.g<<' '<<rgb.b<<' '<<rgb.a<<endl;
			cv.push_back(rgb);
			node = node.nextSibling();
		}
	}
	else if(e.tagName() == "AspectRatio")
		aspect_ratio = e.text().toInt();
	else if(e.tagName() == "MouseDisabled")
		mouse_disabled = (bool) e.text().toInt();
	else if(e.tagName() == "Resolution")
		resolution = e.text().toInt();
#endif
}
