/*   grep-dctrl - grep Debian control files
     Copyright (C) 1999, 2000  Antti-Juhani Kaijanaho
  
     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.
  
     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details. 
  
     You should have received a copy of the GNU General Public License
     along with this program; see the file COPYING.  If not, write to
     the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
  
     The author can be reached via mail at (ISO 8859-1 charset for the city)
        Antti-Juhani Kaijanaho
        Helvintie 2 e as 9
        FIN-40500 JYVSKYL
        FINLAND
        EUROPE
     and via electronic mail from
        gaia@iki.fi
     If you have a choice, use the email address; it is more likely to
     stay current.

*/

#include <assert.h>
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include "buffer.h"
#include "matcher.h"
#include "msg.h"
#include "strutil.h"

static void
str_tolower (char * target, const char * source)
{
  size_t i;

  for (i = 0; i == 0 || source [i-1] != 0; i++)
    {
      unsigned char c = source [i];

      c = isupper (c) ? tolower (c) : c;
      target [i] = c;
    }
}

/* strstr ignoring case */
static char *
strcasestr (const char * haystack, const char * needle)
     /* This implemetation is suboptimal.  Ugh. */
{
  char * lower_hs;  /* haystack in lower case */
  char * lower_ne;  /* needle in lower case */
  char * rv;

  debug_message ("strcasestr", 0);

  lower_hs = malloc (strlen (haystack) + 1);
  lower_ne = malloc (strlen (needle) + 1);
  if (lower_hs == 0 || lower_ne == 0)
    {
      enomem (0);
      return 0;
    }

  str_tolower (lower_hs, haystack);
  str_tolower (lower_ne, needle);

  rv = strstr (lower_hs, lower_ne);
  if (rv != 0)
    rv = (char *) ( (rv - lower_hs) + haystack);

  free (lower_hs);
  free (lower_ne);

  return rv;
}

/* Copy the body of the field "field" from "s" to "body". */
static void
find_field (buffer body, const char * field, const char *s, const char * fname)
{
  const char * p;

  p = strcasestr (s, field);
  while (p != 0 && !((p == s || *(p-1) == '\n')
                     && (p [strlen (field)] == ':')))
    p = strcasestr (p + 1, field);

  if (p == 0)
    return;

  /* Skip whitespace at start of the field body. */
  for (p += strlen (field) + 1; *p == ' ' || *p == '\t'; p++);

  for (/* p is inherited */;
       *p != 0 && ( (p [0] == '\n') ? (p [1] == ' ' || p [1] == '\t') : 1);
       p++)
    {
      debug_message (_("find_field: copying a char"), fname);
      if (!buffer_append (body, *p))
        {
          buffer_free (body);
          fatal_enomem (fname);
        }          
    }
}


static void
show (const char * para, char * fieldv[],
      int show_fieldname, const char * fname)
{
  int i;
  buffer body;
  static int separatorp = 0;

  if (separatorp)
    puts ("");

  separatorp = fieldv [1] != 0;

  if (fieldv [0] == 0)
    {
      puts (para);
      return;
    }

  body = new_buffer ();
  if (body == buffer_nomem)
    fatal_enomem (fname);

  for (i = 0; fieldv [i] != 0; i++)
    {
      find_field (body, fieldv [i], para, fname);

      if (show_fieldname)
        printf ("%s: %s\n", fieldv[i], left_trimmed (buffer_c_str (body)));
      else
        printf ("%s\n", left_trimmed (buffer_c_str (body)));

      buffer_clear (body);
    }
  buffer_free (body);
}

char *
get_regerror (int errcode, regex_t *compiled)
{
  size_t length = regerror (errcode, compiled, NULL, 0);
  char * buffer = malloc (length);

  if (buffer == 0)
    return 0;

  (void) regerror (errcode, compiled, buffer, length);
  return buffer;
}

static int
grep_regex (regex_t * regex, const char * para, const char * fname)
{
  int regex_errcode;
  char * s;

  s = strdup (para);
  if (s == 0)
    {
      enomem (fname);
      return 0;
    }
  
  regex_errcode = regexec (regex, s, 0, 0, 0);
  free (s);
  if (regex_errcode == 0 || regex_errcode == REG_NOMATCH)
    return (regex_errcode == 0);
  
  /* Error handling be here. */
  assert (regex_errcode != 0 && regex_errcode != REG_NOMATCH);
  s = get_regerror (regex_errcode, regex);
  if (s == 0)
    {
      enomem (fname);
      return 0;
    }
  message (L_IMPORTANT, s, fname);
  free (s);
  return 0;
}

static int
grep_all (enum match_type_t type, int case_sensitive,  int exact,
          union pattern_t * pattern, const char * para,
          const char * fname)
{
  switch (type)
    {
    case MATCH_EXACT:
      {
        int (*matcher) (const char *, const char *);
        
        if (case_sensitive)
          matcher = strcmp;
        else
          matcher = strcasecmp;
        
        return matcher (para, pattern->fixed) == 0;
      }
      
    case MATCH_FIXED:
      {
        char * (*substring) (const char *, const char *);
        
        if (case_sensitive)
          substring = strstr;
        else
          substring = strcasestr;
        
        return substring (para, pattern->fixed) != 0;
      }

    case MATCH_REGEX:
      return grep_regex (pattern->regex, para, fname);
    }
  message (L_FATAL, _("I'm broken - please report this to <gaia@iki.fi>"), "matcher.c");
  abort ();
}

static int
grep (struct matcher_t * matcher, const char * para,
      const char * fname)
{
  buffer body;
  int rv;
  size_t i;

  assert (matcher != 0);
  assert (para != 0);

  /* Handle case 1: no fields restriction */
  if (matcher->fields == 0 || matcher->numfields == 0)
    return grep_all (matcher->type, matcher->case_sensitive,
                     matcher->exact_match,
                     &(matcher->pattern), para, fname);

  /* Now handle case 2: search restricted to given fields */

  assert (matcher->fields != 0);
  assert (matcher->numfields > 0);

  body = new_buffer ();
  if (body == buffer_nomem)
    fatal_enomem (fname);

  for (i = 0; i < matcher->numfields; i++)
    {
      debug_message (_("grep: finding field"), fname);  
      find_field (body, matcher->fields[i], para, fname);
  
      /* Grep there. */
      rv = grep_all (matcher->type, matcher->case_sensitive, 
                     matcher->exact_match,                 
                     &(matcher->pattern), buffer_c_str (body),
                     fname);
      buffer_clear(body);
      if (rv) break;
    }

  buffer_free (body);
  return rv;
}

/* Return true on success. */
static int
slurp_paragraph (buffer buf, FILE * f, const char * fname)
{
  register int c;
  register enum { in_para, skipping_blanks } state = in_para;

  while ( (c = getc (f)) != EOF)
    {
      switch (state)
        {
        case in_para:
          if (c == '\n')
            state = skipping_blanks;
          break;
        case skipping_blanks:
          if (c == '\n')
            goto out_of_loop;
          else if (c != '\t' && c != ' ')
            state = in_para;
          break;
        default:
          message (L_INFORMATIONAL, _("Illegal state in slurp_paragraph"), 0);
          message (L_FATAL, _("I'm broken - please report this "
                          "to <gaia@iki.fi>"), 0);
          abort ();
        }
      if (!buffer_append (buf, c))
        {
          debug_message ( _("slurp_paragraph: buffer is full"), fname);
          buffer_free (buf);
          enomem (fname);
          return 0;
        }
    }
 out_of_loop:

  if (ferror (f))
    {
      message (L_INFORMATIONAL, _("slurp_paragraph: error reading file"), fname);
      message (L_IMPORTANT, strerror (errno), fname);
      return 0;
    }
  if (feof (f))
    debug_message (_("slurp_paragraph: hit end of file"), fname);

  return 1;
}

/* Return true if all went well. */
int
grep_control (struct matcher_t * matcher, FILE * f, char * show_fields[],
              const char * fname)
{
  buffer paragraph;

  message (L_INFORMATIONAL, _("grep_control: processing new file"), fname);
  debug_message (_("grep_control: allocating new buffer"), fname);
  paragraph = new_buffer ();
  if (paragraph == buffer_nomem)
    {
      enomem (fname);
      return 0;
    }

  while (1)
    {
      int grep_result;

      debug_message (_("grep_control: slurping a new paragraph"), fname);
      if (!slurp_paragraph (paragraph, f, fname))
        return 0;
      
      if (feof(f) && buffer_len(paragraph) == 0)
        return 1;

      debug_message (_("grep_control: grepping the paragraph"), fname);
      grep_result = grep (matcher, buffer_c_str (paragraph), fname);
      if (matcher->inverse ? !grep_result : grep_result)
        {
          if (!matcher->suppress)
            {
              show (buffer_c_str (paragraph), show_fields,
                    matcher->show_fieldnames, fname);
            }
          ++matcher->match_count;
        }
      
      debug_message (_("grep_control: trimming and clearing the buffer"),
                     fname);
      buffer_trim (paragraph);
      buffer_clear (paragraph);
    }

  message (L_INFORMATIONAL, _("grep_control: done with file"), fname);
  return 1;
}

