/*
 * GooCanvas. Copyright (C) 2005 Damon Chaplin.
 * Released under the GNU LGPL license. See COPYING for details.
 *
 * goocanvasellipseview.c - view for ellipse item.
 */

/**
 * SECTION:goocanvasellipseview
 * @Title: GooCanvasEllipseView
 * @Short_Description: a view for a #GooCanvasEllipse item.
 *
 * #GooCanvasEllipseView represents a view of a #GooCanvasEllipse item.
 *
 * It implements the #GooCanvasItemView interface, so you can use the
 * #GooCanvasItemView functions such as goo_canvas_item_view_get_item()
 * and goo_canvas_item_view_get_bounds().
 *
 * Applications do not normally need to create item views themselves, as
 * they are created automatically by #GooCanvasView when needed.
 *
 * To respond to events such as mouse clicks in the ellipse view you can
 * connect to one of the #GooCanvasItemView signals such as
 * #GooCanvasItemView::button-press-event. You can connect to these signals
 * when the view is created. (See goo_canvas_view_get_item_view() and
 * #GooCanvasView::item-view-created.)
 */
#include <config.h>
#include <math.h>
#include <gtk/gtk.h>
#include "goocanvasellipseview.h"
#include "goocanvasellipse.h"


static void goo_canvas_ellipse_view_create_path (GooCanvasItemSimple *simple,
						 cairo_t             *cr);

G_DEFINE_TYPE_WITH_CODE (GooCanvasEllipseView, goo_canvas_ellipse_view,
			 GOO_TYPE_CANVAS_ITEM_VIEW_SIMPLE,
			 G_IMPLEMENT_INTERFACE (GOO_TYPE_CANVAS_ITEM_VIEW,
						NULL))


static void
goo_canvas_ellipse_view_class_init (GooCanvasEllipseViewClass *klass)
{
  GooCanvasItemViewSimpleClass *simple_view_class = (GooCanvasItemViewSimpleClass*) klass;

  simple_view_class->create_path = goo_canvas_ellipse_view_create_path;
}


static void
goo_canvas_ellipse_view_init (GooCanvasEllipseView *ellipse_view)
{

}


/**
 * goo_canvas_ellipse_view_new:
 * @canvas_view: the canvas view.
 * @parent_view: the parent view.
 * @ellipse: the ellipse item.
 * 
 * Creates a new #GooCanvasEllipseView for the given #GooCanvasEllipse item.
 *
 * This is not normally used by application code, as the views are created
 * automatically by #GooCanvasView.
 * 
 * Returns: a new #GooCanvasEllipseView.
 **/
GooCanvasItemView*
goo_canvas_ellipse_view_new      (GooCanvasView     *canvas_view,
				  GooCanvasItemView *parent_view,
				  GooCanvasEllipse  *ellipse)
{
  GooCanvasItemViewSimple *view;

  view = g_object_new (GOO_TYPE_CANVAS_ELLIPSE_VIEW, NULL);
  view->canvas_view = canvas_view;
  view->parent_view = parent_view;
  view->item = g_object_ref (ellipse);

  goo_canvas_item_view_simple_setup_accessibility (view);

  g_signal_connect (ellipse, "changed",
		    G_CALLBACK (goo_canvas_item_view_simple_item_changed),
		    view);

  return (GooCanvasItemView*) view;
}


static void
goo_canvas_ellipse_view_create_path (GooCanvasItemSimple *simple,
				     cairo_t             *cr)
{
  GooCanvasEllipse *ellipse = (GooCanvasEllipse*) simple;

  cairo_new_path (cr);
  cairo_save (cr);
  cairo_translate (cr, ellipse->center_x, ellipse->center_y);
  cairo_scale (cr, ellipse->radius_x, ellipse->radius_y);
  cairo_arc (cr, 0.0, 0.0, 1.0, 0.0, 2.0 * M_PI);
  cairo_restore (cr);
}
