/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
     satupdate.c:  Functions to update the TLE files.                    

  Copyright (C)  2003  Alexandru Csete.

  Authors:  Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <gnome.h>
#include <libgnomevfs/gnome-vfs.h>
#include <gconf/gconf-client.h>
#include "satdata.h"
#include "engine.h"
#include "satupdate.h"
#include "sat-ui-utils.h"
#include "fileio.h"
#include "satlog.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


/* needed globally */
static GtkWidget *source,*dialog,*updb,*cons,*addb,*delb;


/* Private function prototypes */
static void satupdate_add_file_cb          (GtkWidget *button, gpointer list);
static void satupdate_add_file_entered_cb  (gchar *string, gpointer list);
static void satupdate_delete_file_cb       (GtkWidget *button, gpointer list);
static void satupdate_update_files_cb      (GtkWidget *button, gpointer list);
static gint satupdate_progress_cb          (GnomeVFSXferProgressInfo *, gpointer);
static void satupdate_list_select_row_cb   (GtkCList *list, gint row, gint column,
					    GdkEventButton *event, gpointer delb);
static void satupdate_list_unselect_row_cb (GtkCList *list, gint row, gint column,
					    GdkEventButton *event, gpointer delb);
static void satupdate_add_local_files      (GtkCList *clist);

void
satupdate_start    ()
{
	GtkWidget *label;
	GtkWidget *frame,*table,*list,*swin;
	GtkTooltips *tips;
	gint button;

	/* warning label and frame */
	label = gtk_label_new (_("This function will only work well, if you have not manually "\
				 "altered the .tle files in your data directory, which is "\
				 "'$(HOME)/.gpredict/tle' by default. Otherwise, your files "\
				 "will be replaced by the files downloaded from the specified "\
				 "HTTP server."));
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_FILL);
	gtk_misc_set_padding (GTK_MISC (label), 5, 5);
	frame = gtk_frame_new (_("Important notice"));
	gtk_container_add (GTK_CONTAINER (frame), label);

	/* Server */
	source = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY (source), "http://www.celestrak.com/NORAD/elements");
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, source,
			      _("Enter full path where the source files can be found"),
			      _("Enter full path where the source files can be found"));
	/* Files */
	list = gtk_clist_new (1);
	gtk_clist_column_titles_hide (GTK_CLIST (list));
	gtk_clist_set_column_auto_resize (GTK_CLIST (list), 0, TRUE);
	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	satupdate_add_local_files (GTK_CLIST (list));
	gtk_container_add (GTK_CONTAINER (swin), list);
	addb = sat_ui_utils_pixmap_button (_("Add"), NULL, _("Add a new file"));
	gtk_signal_connect (GTK_OBJECT (addb), "clicked",
			    GTK_SIGNAL_FUNC (satupdate_add_file_cb), list);
	delb = sat_ui_utils_pixmap_button (_("Remove"), NULL, _("Remove selected file"));
	gtk_signal_connect (GTK_OBJECT (delb), "clicked",
			    GTK_SIGNAL_FUNC (satupdate_delete_file_cb), list);
	gtk_widget_set_sensitive (delb, FALSE);
	updb = sat_ui_utils_pixmap_button (_("Update"), NULL, _("Download the files"));
	gtk_signal_connect (GTK_OBJECT (updb), "clicked",
			    GTK_SIGNAL_FUNC (satupdate_update_files_cb), list);
	/* connect CList signals as well */
	gtk_signal_connect (GTK_OBJECT (list), "select-row",
			    GTK_SIGNAL_FUNC (satupdate_list_select_row_cb), delb);
	gtk_signal_connect (GTK_OBJECT (list), "unselect-row",
			    GTK_SIGNAL_FUNC (satupdate_list_unselect_row_cb), delb);

	/* console */
	cons = gtk_text_new (NULL, NULL);
	gtk_text_set_editable (GTK_TEXT (cons), FALSE);

	/* create the table */
	table = gtk_table_new (6, 5, FALSE);
	gtk_table_set_col_spacings (GTK_TABLE (table), 5);
	gtk_table_set_row_spacings (GTK_TABLE (table), 5);
	gtk_table_attach_defaults (GTK_TABLE (table), gtk_label_new (_("1.")),
				   0, 1, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table), gtk_label_new (_("Source:")),
				   1, 2, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table), source,
				   2, 6, 0, 1);
	gtk_table_attach_defaults (GTK_TABLE (table), gtk_label_new (_("2.")),
				   0, 1, 1, 2);
	gtk_table_attach_defaults (GTK_TABLE (table), gtk_label_new (_("Files:")),
				   1, 2, 1, 2);
	gtk_table_attach_defaults (GTK_TABLE (table), addb, 1, 2, 2, 3);
	gtk_table_attach_defaults (GTK_TABLE (table), delb, 1, 2, 3, 4);
	gtk_table_attach_defaults (GTK_TABLE (table), swin, 2, 6, 1, 5);
	gtk_table_attach_defaults (GTK_TABLE (table), gtk_label_new (_("3.")),
				   0, 1, 5, 6);
	gtk_table_attach_defaults (GTK_TABLE (table), updb,
				   1, 2, 5, 6);

	/* Create the dialog */
	dialog = gnome_dialog_new (_("Update TLE Files"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);
	gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 0, FALSE);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), frame, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), gtk_hseparator_new (), FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), table, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), gtk_hseparator_new (), FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), cons, TRUE, TRUE, 0);

	gtk_widget_show_all (dialog);

	button = gnome_dialog_run (GNOME_DIALOG (dialog));

	if (GNOME_IS_DIALOG (dialog)) {
		/* The user clicked a button and not just
		   closed/destroyed the dialog.
		*/
		if (!button) {
			GnomeVFSResult result;
			GnomeVFSDirectoryFilter *vfsfilter;
			GnomeVFSFileInfo *fileinfo;
			GList *list,*node;
			gchar *path,**vbuff,*vfsuri,*in,*out,*buff;

			/* Get list of downloaded files */
			path = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, ".gpredict",
					    G_DIR_SEPARATOR_S, "tle", NULL);
			vfsuri = gnome_vfs_get_uri_from_local_path (path);

			vfsfilter = gnome_vfs_directory_filter_new (GNOME_VFS_DIRECTORY_FILTER_SHELLPATTERN,
								    GNOME_VFS_DIRECTORY_FILTER_NODIRS |
								    GNOME_VFS_DIRECTORY_FILTER_NODOTFILES |
								    GNOME_VFS_DIRECTORY_FILTER_NOSELFDIR |
								    GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR |
								    GNOME_VFS_DIRECTORY_FILTER_NOBACKUPFILES,
								    "*.txt");
			result = gnome_vfs_directory_list_load (&list, vfsuri,
								GNOME_VFS_FILE_INFO_DEFAULT,
								vfsfilter);
			g_free (vfsuri);
			gnome_vfs_directory_filter_destroy (vfsfilter);

			/* Copy existing .txt files to .tle files */
			if (result == GNOME_VFS_OK) {
				/* load each file */
				for (node = list; node != NULL; node = node->next) {
					fileinfo = node->data;   /* fileinfo->name contains file name */
					vbuff = g_strsplit (fileinfo->name, ".", 0);
					in = g_strconcat (path, G_DIR_SEPARATOR_S, fileinfo->name, NULL); 
					out = g_strconcat (path, G_DIR_SEPARATOR_S, vbuff[0], ".tle", NULL);
					g_strfreev (vbuff);
					if (fileio_file_copy (in, out)) {
						buff = g_strconcat (__FUNCTION__, ": ", _("Error copying"),
								    " ", in, _("to"), " ", out, NULL);
						satlog_log (SAT_LOG_CRITICAL, buff);
						g_free (buff);
					}
					g_free (in);
					g_free (out);
				}
				g_list_free (list);
			}
			g_free (path);

			/* Holly Mother! */
			engine_update ();

		}
		gnome_dialog_close (GNOME_DIALOG (dialog));
	}
}


void
satupdate_start_cb (GtkWidget *widget, gpointer data)
{
	/* This is just a convenience callback function
	   which wraps the satupdate_start ().
	*/
	satupdate_start ();
}



static void
satupdate_add_file_cb     (GtkWidget *button, gpointer list)
{
	/* This function is called when the user clicks on the
	   add satellite button.
	*/
	GtkWidget *win;

	win = gnome_request_dialog (FALSE, _("Enter file name on remote server"),
				    NULL, 50, satupdate_add_file_entered_cb,
				    list, GTK_WINDOW (dialog));
	gtk_window_set_modal (GTK_WINDOW (win), TRUE); /* must have, otherwise we don't get focus */
	gtk_widget_show_all (win);
}


static void
satupdate_add_file_entered_cb (gchar *string, gpointer list)
{
	/* callback function to the satupdate_add_file_cb
	   dialog.
	*/
	gchar *new[1];

	if ((string != NULL) && (g_strcasecmp (string, ""))) {
		new[0] = g_strdup (string);
		gtk_clist_append (GTK_CLIST (list), new);
		g_free (new[0]);
		/* re-enable update button */
		gtk_widget_set_sensitive (updb, TRUE);
	}
}


static void
satupdate_delete_file_cb  (GtkWidget *button, gpointer list)
{
	/* This function is called when the user clicks on
	   the delete satellite button.
	*/
	GList *selection = (GTK_CLIST (list))->selection;
	
	if (selection) {
		gtk_clist_remove (GTK_CLIST (list), GPOINTER_TO_UINT (selection->data));
		/* Disable update button if list is empty */
		if (!GTK_CLIST (list)->rows)
			gtk_widget_set_sensitive (updb, FALSE);
	}
}


static void
satupdate_update_files_cb (GtkWidget *button, gpointer list)
{
	/* This function is called when the user clicks on
	   the update button.
	*/
	GnomeVFSResult result;
	GnomeVFSURI *source_uri,*target_uri;
	GList *selection = (GTK_CLIST (list))->selection;
	gchar *buff,*msg;
	guint i;

	/* Disable input widgets to prevent the user doing
	   bad things...
	*/
	if (selection)
		gtk_widget_set_sensitive (delb, FALSE);
	gtk_widget_set_sensitive (updb, FALSE);
	gtk_widget_set_sensitive (addb, FALSE);
	gtk_widget_set_sensitive (source, FALSE);
	gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 0, FALSE);
	gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 1, FALSE);
	gtk_editable_delete_text (GTK_EDITABLE (cons), 0, -1);
	for (i=0; i<GTK_CLIST (list)->rows; i++) {
		if (gtk_clist_get_text (GTK_CLIST (list), i, 0, &buff)) {
			
			msg = g_strconcat (_("Fetching "), buff, NULL);
			gtk_text_insert (GTK_TEXT (cons), NULL, &cons->style->black, NULL, msg, -1);
			g_free (msg);
			msg = g_strconcat (gtk_entry_get_text (GTK_ENTRY (source)),
					   G_DIR_SEPARATOR_S, buff, NULL);
			source_uri = gnome_vfs_uri_new (msg);
			g_free (msg);
			msg = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, ".gpredict",
					   G_DIR_SEPARATOR_S, "tle", G_DIR_SEPARATOR_S, buff, NULL);
			target_uri = gnome_vfs_uri_new (msg);
			g_free (msg);
			result = gnome_vfs_xfer_uri (source_uri, target_uri,
						     GNOME_VFS_XFER_DEFAULT,
						     GNOME_VFS_XFER_ERROR_MODE_ABORT,
						     GNOME_VFS_XFER_OVERWRITE_MODE_REPLACE,
						     satupdate_progress_cb,
						     NULL);
			if (result == GNOME_VFS_OK) {
				gtk_text_insert (GTK_TEXT (cons), NULL, &cons->style->black, NULL, "OK\n", -1);
			}
			else {
				gtk_text_insert (GTK_TEXT (cons), NULL, &cons->style->black, NULL,
						 gnome_vfs_result_to_string (result), -1);
				gtk_text_insert (GTK_TEXT (cons), NULL, &cons->style->black, NULL, "\n", -1);
			}

			gnome_vfs_uri_unref (source_uri);
			gnome_vfs_uri_unref (target_uri);
		}
	}
	/* re-enable widgets */
	if (selection)
		gtk_widget_set_sensitive (delb, TRUE);
	gtk_widget_set_sensitive (updb, TRUE);
	gtk_widget_set_sensitive (addb, TRUE);
	gtk_widget_set_sensitive (source, TRUE);
	gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 0, TRUE);
	gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 1, TRUE);
}


static gint
satupdate_progress_cb (GnomeVFSXferProgressInfo *info, gpointer data)
{
	/* This function is called automatically by the GnomeVFS Xfer
	   function.
	*/
	gtk_text_insert (GTK_TEXT (cons), NULL, &cons->style->black, NULL, ".", -1);
	return TRUE;
}

static void
satupdate_list_select_row_cb (GtkCList *list, gint row, gint column,
			      GdkEventButton *event, gpointer delb)
{
	/* This function is called when a row is selected in the
	   files list.
	*/

	gtk_widget_set_sensitive (GTK_WIDGET (delb), TRUE);

}


static void
satupdate_list_unselect_row_cb (GtkCList *list, gint row, gint column,
				GdkEventButton *event, gpointer delb)
{
	/* This function is called when a row is unselected in the
	   files list.
	*/
	GList *selection = list->selection;

	if (!selection)
		gtk_widget_set_sensitive (GTK_WIDGET (delb), FALSE);

}


static void
satupdate_add_local_files      (GtkCList *clist)
{
	/* This function lists the files in the local
	   directory $(HOME)/.gpredict/tle/ and appends
	   them to the GtkCList.
	*/
	GnomeVFSResult result;
	GnomeVFSDirectoryFilter *vfsfilter;
	GnomeVFSFileInfo *fileinfo;
	GList *list,*node;
	gchar *path,**vbuff,*vfsuri,*new[1];

	path = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, ".gpredict",
			    G_DIR_SEPARATOR_S, "tle", NULL);

	/* convert to VFS URI */
	vfsuri = gnome_vfs_get_uri_from_local_path (path);
	g_free (path);

	/* create a VFS filter and load the directory */
	vfsfilter = gnome_vfs_directory_filter_new (GNOME_VFS_DIRECTORY_FILTER_SHELLPATTERN,
						    GNOME_VFS_DIRECTORY_FILTER_NODIRS |
						    GNOME_VFS_DIRECTORY_FILTER_NODOTFILES |
						    GNOME_VFS_DIRECTORY_FILTER_NOSELFDIR |
						    GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR |
						    GNOME_VFS_DIRECTORY_FILTER_NOBACKUPFILES,
						    "*.tle");
	result = gnome_vfs_directory_list_load (&list, vfsuri,
						GNOME_VFS_FILE_INFO_DEFAULT,
						vfsfilter);
	/* free some memory */
	g_free (vfsuri);
	gnome_vfs_directory_filter_destroy (vfsfilter);

	if (result == GNOME_VFS_OK) {
		/* load each file */
		for (node = list; node != NULL; node = node->next) {
			fileinfo = node->data;   /* fileinfo->name contains file name */
			vbuff = g_strsplit (fileinfo->name, ".", 0);
			new[0] = g_strconcat (vbuff[0], ".txt", NULL);
			g_strfreev (vbuff);
			gtk_clist_append (clist, new);
			g_free (new[0]);
		}
		g_list_free (list);
	}
}

