/*
 * configdump.c - Sets up dump configuration parameters
 * Created by:  Gary Hicks  (gary.r.hicks@hp.com)
 *
 * This mini version:  Check dump configuration file and set OS parms
 * Modified by: Bob Montgomery (bobm@fc.hp.com)
 * Modified by: Troy Heber (troy.heber@hp.com)
 *
 * Copyright 2004 Hewlett-Packard Development Company, L.P.
 * 
 * Adapted from lkcdutils
 * Created by: Matt D. Robinson (yakker@aparity.com)
 * Copyright 2001 Matt D. Robinson (yakker@aparity.com), all rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */ 

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/param.h>
#include <netinet/ether.h>

#include "parseconf.h"
#include "version.h"


#define DUMP_DEVICE "/dev/dump"
#define PANIC_PATH "/proc/sys/kernel/panic"

#define TOK_SIZE 132

// Kernel 2.4 LKCD ioctrl's
#define DIOSDUMPDEV_4		1       /* set the dump device              */
#define DIOGDUMPDEV_4		2       /* get the dump device              */
#define DIOSDUMPLEVEL_4		3       /* set the dump level               */
#define DIOGDUMPLEVEL_4		4       /* get the dump level               */
#define DIOSDUMPFLAGS_4		5       /* set the dump flag parameters     */
#define DIOGDUMPFLAGS_4		6       /* get the dump flag parameters     */
#define DIOSDUMPCOMPRESS_4	7       /* set the dump compress level      */
#define DIOGDUMPCOMPRESS_4	8       /* get the dump compress level      */

// Kernel 2.6 LKCD ioctrl's
// dump ioctl() control options
#define DIOSDUMPDEV_6     _IOW('p', 0xA0, unsigned int)  /* set the dump device            */
#define DIOGDUMPDEV_6     _IOR('p', 0xA1, unsigned int)  /* get the dump device            */
#define DIOSDUMPLEVEL_6   _IOW('p', 0xA2, unsigned int)  /* set the dump level            */
#define DIOGDUMPLEVEL_6   _IOR('p', 0xA3, unsigned int)  /* get the dump level            */
#define DIOSDUMPFLAGS_6   _IOW('p', 0xA4, unsigned int)  /* set the dump flag parameters     */
#define DIOGDUMPFLAGS_6   _IOR('p', 0xA5, unsigned int)  /* get the dump flag parameters     */
#define DIOSDUMPCOMPRESS_6 _IOW('p', 0xA6, unsigned int) /* set the dump compress level      */
#define DIOGDUMPCOMPRESS_6 _IOR('p', 0xA7, unsigned int) /* get the dump compress level      */

// these ioctls are used only by netdump module 
#define DIOSTARGETIP_6    _IOW('p', 0xA8, unsigned int)  /* set the target m/c's ip           */
#define DIOGTARGETIP_6    _IOR('p', 0xA9, unsigned int)  /* get the target m/c's ip           */
#define DIOSTARGETPORT_6  _IOW('p', 0xAA, unsigned int) /* set the target m/c's port          */
#define DIOGTARGETPORT_6  _IOR('p', 0xAB, unsigned int) /* get the target m/c's port          */
#define DIOSSOURCEPORT_6  _IOW('p', 0xAC, unsigned int) /* set the source m/c's port          */
#define DIOGSOURCEPORT_6  _IOR('p', 0xAD, unsigned int) /* get the source m/c's port          */
#define DIOSETHADDR_6     _IOW('p', 0xAE, unsigned int) /* set ethernet address    */
#define DIOGETHADDR_6     _IOR('p', 0xAF, unsigned int) /* get ethernet address     */
#define DIOGDUMPOKAY_6    _IOR('p', 0xB0, unsigned int) /* check if dump is configured      */
#define DIOSDUMPTAKE_6    _IOW('p', 0xB1, unsigned int) /* Take a manual dump           */



char *prog;

int
get_version()
{
    FILE *fp;
    char buffer[20];
    char *p;
    int res, ans;
    struct stat buf;

    if((fp = fopen("/proc/sys/kernel/osrelease", "r")) == NULL)
	return -1;

    if((res = fread(buffer, 1, sizeof(buffer), fp)) == 0)
	return -2;

    fclose(fp);

    buffer[3]='\0';
    p = buffer + 2;

    res = atoi(p);

    if ((ans = stat("/sys/dump/version", &buf)) == 0)
	    res=61;
    else{
	    if ((fp = fopen(DUMP_DEVICE, "r")) == NULL) {
		    fprintf(stderr, "%s: Error - can\'t open DUMP_DEVICE %s "
				    "or /sys/dump/version: \n\t", prog, DUMP_DEVICE);
		    perror("");
		    fprintf(stderr, "Are you sure that you have LKCD configured" 
				    " in your kernel and the dump modules loaded?\n");
		    exit(1);
	    }

	    fclose(fp);
    }


    return res;
}


void usage(FILE *fp, char *prog, char *config_file)
{
    fprintf(fp, "usage: %s [-h] [-l level]\n", prog);
    fprintf(fp, "\tconfigure the kernel crash dump driver using %s\n",
	    config_file);
    fprintf(fp, "\t-l level: override DUMP_LEVEL value with level\n");
}

void version(FILE *fp, char *prog)
{
#ifdef VERSION
    fprintf(fp, "%s %s\n", prog, VERSION);
#else
    fprintf(fp, "%s undefined\n", prog);
#endif 
}

int
twoSix_One(int level_override, struct pconf *conf)
{

	FILE *pfp, *fp;

	int gfd;
	int err;

	int tmp_level;
	struct hostent *host;
	struct ether_addr *ether;
	char devstr[512];
	struct stat devdump_buf;

	printf("Configuring for 2.6 based kernel via sysfs\n");

	if ((strncmp(conf->dumpdev, "eth", 3) == 0) | 
			(strncmp(conf->dumpdev, "ath", 3) == 0) |
			(strncmp(conf->dumpdev, "wlan", 4) == 0)){

		/* DUMPDEV for netdev is all of the required net parameters
		 */
		if ((fp = fopen("/sys/dump/dumpdev","w")) == NULL) {
			fprintf(stderr, "%s: Error - can\'t open /sys/dump/dumpdev: ",
					prog);
			perror("");
			fprintf(stderr, "Are you sure that you have LKCD configured" 
					"in your kernel?\n");
			exit(1);
		}

		host =  gethostbyname(conf->target_host);
		if (!host) {
			fprintf(stderr, "gethostbyname: %s\n",
					hstrerror(h_errno));
			return (-1);
		}

		sprintf(conf->target_host, "%lx", 
				(unsigned long)*(unsigned int *)*host->h_addr_list);

		if (!(ether = ether_aton(conf->eth_address))) {
			fprintf(stderr, "Invalid ethernet address %s\n",
					conf->eth_address);
			return (-1);
		}

		sprintf(conf->eth_address, "%llx", *ether);

		strcpy(devstr, conf->dumpdev);
		strcat(devstr, ",");
		strcat(devstr, conf->target_host);
		strcat(devstr, ",");
		strcat(devstr, conf->target_port);
		strcat(devstr, ",");
		strcat(devstr, conf->source_port);
		strcat(devstr, ",");
		strcat(devstr, conf->eth_address);

		fprintf(fp, "%s\n", devstr);
		fclose(fp);

	} else {
		/* DUMPDEV is the name of the partition to store the dump
		 * temporarily if a dump occurs - Make sure it exists.
		 */
		if ((gfd = open(conf->dumpdev,O_RDONLY)) < 0) {
			fprintf(stderr, "%s: Error - can\'t open DUMPDEV %s: ",
					prog, conf->dumpdev); 
			perror("");
			exit(1);
		}

		if ((err = fstat(gfd,&devdump_buf)) < 0) {
			fprintf(stderr, "%s: Error - can\'t stat DUMPDEV %s: ",
					prog, conf->dumpdev);
			perror("");
			close(gfd);
			exit(1);
		}

		close(gfd);

		if ((fp = fopen("/sys/dump/dumpdev","w")) == NULL) {
			fprintf(stderr, "%s: Error - can\'t open /sys/dump/dumpdev: ",
					prog);
			perror("");
			fprintf(stderr, "Are you sure that you have LKCD configured" 
					"in your kernel?\n");
			exit(1);
		}
		fprintf(fp, "%lx\n", devdump_buf.st_rdev);
		fclose(fp);
	}

	/* Set the DUMP_LEVEL parameter */
	if (level_override >= 0) {
		tmp_level = level_override;
	} else {
		tmp_level = strtol(conf->dump_level, NULL, 0);
	}

	if ((fp = fopen("/sys/dump/level","w")) == NULL) {
		fprintf(stderr, "%s: Error - can\'t open /sys/dump/level: ", prog);
		perror("");
		exit(1);
	}

	fprintf(fp, "%lx\n", (unsigned long)tmp_level);
	fclose(fp);

	if ((fp = fopen("/sys/dump/compress","w")) == NULL) {
		fprintf(stderr, "%s: Error - can\'t open /sys/dump/compress: ", prog);
		perror("");
		exit(1);
	}

	fprintf(fp, "%lx\n", strtoul(conf->dump_compress, NULL, 0));
	fclose(fp);

	/* Put a timeout value in /proc PANIC_PATH, based
	 * upon the PANIC_TIMEOUT value
	 */
	if ((pfp = fopen(PANIC_PATH,"w")) == NULL) {
		fprintf(stderr, "%s: Error - can\'t open %s: ", 
				prog, PANIC_PATH);
		perror("");
		exit(1);
	}
	fprintf(pfp,"%s",conf->panic_timeout);
	fclose(pfp);

	printf("%s: successful dump configuration\n", prog);

	return 0;
}


int
twoSix(int level_override, struct pconf *conf)
{

    FILE *pfp;

    int dfd;
    int gfd;
    int err;

    int tmp_level;
    struct stat devdump_buf;

    printf("Configuring for 2.6 based kernel\n");


    /* DUMP_DEVICE is the name of the kernel created dump device */
    if ((dfd = open(DUMP_DEVICE, O_RDWR)) < 0) {
	fprintf(stderr, "%s: Error - can\'t open DUMP_DEVICE %s: "
			" or /sys/dump/version: ", prog, DUMP_DEVICE);
	perror("");
	fprintf(stderr, "Are you sure that you have LKCD configured in your kernel?\n");
	exit(1);
    }

    /* DUMPDEV is the name of the partition to store the dump
     * temporarily if a dump occurs - Make sure it exists.
     */
    if ((gfd = open(conf->dumpdev,O_RDONLY)) < 0) {
	fprintf(stderr, "%s: Error - can\'t open DUMPDEV %s: ",
		prog, conf->dumpdev); 
	perror("");
	close(dfd);
	exit(1);
    }

    if ((err = fstat(gfd,&devdump_buf)) < 0) {
	fprintf(stderr, "%s: Error - can\'t stat DUMPDEV %s: ",
		prog, conf->dumpdev);
	perror("");
	close(dfd);
	close(gfd);
	exit(1);
    }

    close(gfd);


    /* Set the DUMP_FLAGS parameter */
    /* In the 2.6 version, this must come before DIOSDUMPDEV
     * to establish the dump target type.
     */
    if ((err=ioctl(dfd,DIOSDUMPFLAGS_6, 
		    strtoul(conf->dump_flags, NULL, 0)) < 0)) {
	fprintf(stderr, "%s: Error - can\'t set dump flags: ", prog);
	perror("");
	fprintf(stderr, "Are you sure that you have LKCD configured in your kernel?\n");
	close(dfd);
	exit(1);
    }

    /* Set the DUMP_DEV parameter */
    if ((err=ioctl(dfd,DIOSDUMPDEV_6, devdump_buf.st_rdev)) < 0) {
	fprintf(stderr, "%s: Error - can\'t set dump device: ",
		prog);
	perror("");
	close(dfd);
	exit(1);
    }	

    // DUMPDIR could be FTP, don't try to check it here.
    /* DUMPDIR is the location where the dump will be stored
     * after the system is booted.  
    if (access(conf->dumpdir, F_OK) != 0) {
	fprintf(stderr, "%s: Warning - can\'t access DUMPDIR %s: ",
		prog, conf->dumpdir);
	perror("");
    }
    */

    /* Set the DUMP_LEVEL parameter */
    if (level_override >= 0) {
	tmp_level = level_override;
    } else {
	tmp_level = strtol(conf->dump_level, NULL, 0);
    }

    if ((err=ioctl(dfd,DIOSDUMPLEVEL_6, (unsigned long)tmp_level) < 0)) {
	fprintf(stderr, "%s: Error - can\'t set dump level: ", prog);
	perror("");
	close(dfd);
	exit(1);
    }

    /* Set the DUMP_COMPRESS parameter */
    if ((err=ioctl(dfd, DIOSDUMPCOMPRESS_6, 
		    strtoul(conf->dump_compress, NULL, 0)) < 0)) {
	fprintf(stderr, "%s: Error - can\'t set dump "
		"compression level: ", prog);
	perror("");
	close(dfd);
	exit(1);
    }

    /* Put a timeout value in /proc PANIC_PATH, based
     * upon the PANIC_TIMEOUT value
     */
    if ((pfp = fopen(PANIC_PATH,"w")) == NULL) {
	fprintf(stderr, "%s: Error - can\'t open %s: ", 
		prog, PANIC_PATH);
	perror("");
	close(dfd);
	exit(1);
    }
    fprintf(pfp,"%s",conf->panic_timeout);
    fclose(pfp);

    printf("%s: successful dump configuration\n", prog);

    close(dfd);
    return 0;
}

int
twoFour(int level_override, struct pconf *conf)
{

    FILE *pfp;

    int dfd;
    int gfd;
    int err;

    int tmp_level;
    struct stat devdump_buf;
    
    printf("Configuring for 2.4 based kernel\n");

    /* DUMP_DEVICE is the name of the kernel created dump device */
    if ((dfd = open(DUMP_DEVICE,O_RDWR)) < 0) {
	fprintf(stderr, "%s: Error - can\'t open DUMP_DEVICE %s: ",
		prog, DUMP_DEVICE);
	perror("");
	fprintf(stderr, "Are you sure that you have LKCD configured in your kernel?\n");
	exit(1);
    }

    /* DUMPDEV is the name of the partition to store the dump
     * temporarily if a dump occurs - Make sure it exists.
     */
    if ((gfd = open(conf->dumpdev,O_RDONLY)) < 0) {
	fprintf(stderr, "%s: Error - can\'t open DUMPDEV %s: ",
		prog, conf->dumpdev); 
	perror("");
	close(dfd);
	exit(1);
    }

    if ((err = fstat(gfd,&devdump_buf)) < 0) {
	fprintf(stderr, "%s: Error - can\'t stat DUMPDEV %s: ",
		prog, conf->dumpdev);
	perror("");
	close(dfd);
	close(gfd);
	exit(1);
    }

    close(gfd);


    /* Set the DUMP_DEV parameter */
    if ((err=ioctl(dfd,DIOSDUMPDEV_4, devdump_buf.st_rdev)) < 0) {
	fprintf(stderr, "%s: Error - can\'t set dump device: ",
		prog);
	perror("");
	close(dfd);
	exit(1);
    }	

    /* Set the DUMP_FLAGS parameter */
    if ((err=ioctl(dfd,DIOSDUMPFLAGS_4, 
		    (unsigned long)atoi(conf->dump_flags)) < 0)) {
	fprintf(stderr, "%s: Error - can\'t set dump flags: ", prog);
	perror("");
	fprintf(stderr, "Remember DUMP_FLAGS changed to hex for 2.6 kernels!");
	close(dfd);
	exit(1);
    }

    /* Set the DUMP_LEVEL parameter */
    if (level_override >= 0) {
	tmp_level = level_override;
    } else {
	tmp_level = atoi(conf->dump_level);
    }

    if ((err=ioctl(dfd,DIOSDUMPLEVEL_4, (unsigned long)tmp_level) < 0)) {
	fprintf(stderr, "%s: Error - can\'t set dump level: ", prog);
	perror("");
	close(dfd);
	exit(1);
    }

    /* Set the DUMP_COMPRESS parameter */
    if ((err=ioctl(dfd, DIOSDUMPCOMPRESS_4, 
		    (unsigned long)atoi(conf->dump_compress)) < 0)) {
	fprintf(stderr, "%s: Error - can\'t set dump "
		"compression level: ", prog);
	perror("");
	close(dfd);
	exit(1);
    }

    /* Put a timeout value in /proc PANIC_PATH, based
     * upon the PANIC_TIMEOUT value
     */
    if ((pfp = fopen(PANIC_PATH,"w")) == NULL) {
	fprintf(stderr, "%s: Error - can\'t open %s: ", 
		prog, PANIC_PATH);
	perror("");
	close(dfd);
	exit(1);
    }
    fprintf(pfp,"%s",conf->panic_timeout);
    fclose(pfp);

    printf("%s: successful dump configuration\n", prog);

    close(dfd);
    return 0;
}


void 
checkconf(struct pconf *conf, char *config_file, int ver)
{
	/* Use DUMP_ACTIVE to determine whether to modify the dump device */
	if (atoi(conf->dump_active) != 1) {
		printf("%s: Configuration stopped because " 
				"DUMP_ACTIVE is set to %d\n", prog,
				atoi(conf->dump_active));
		exit(1);
	}

	/* Verify that all configuration parameters were found */
	int err = 0;
	if (!conf->dump_active[0]) {
		err++;
		fprintf(stderr, "%s: Error - no DUMP_ACTIVE parameter\n", prog);
	}
	if (!conf->dumpdev[0]) {
		err++;
		fprintf(stderr, "%s: Error - no DUMPDEV parameter\n", prog);
	}
	if (!conf->dumpdir[0]) {
		err++;
		fprintf(stderr, "%s: Error - no DUMPDIR parameter\n", prog);
	}
	if (!conf->dump_level[0]) {
		err++;
		fprintf(stderr, "%s: Error - no DUMP_LEVEL parameter\n", prog);
	}
	if (!conf->dump_compress[0]) {
		err++;
		fprintf(stderr, "%s: Error - no DUMP_COMPRESS parameter\n", prog);
	}
	if (!conf->panic_timeout[0]) {
		err++;
		fprintf(stderr, "%s: Error - no PANIC_TIMEOUT parameter\n", prog);
	}
	if ((ver == 4) || (ver == 6)){
		if (!conf->dump_flags[0]) {
			err++;
			fprintf(stderr, "%s: Error - no DUMP_FLAGS parameter\n",
					prog);
		}
    }

	if (ver == 61){ /* netdev based dump */
		if ((strncmp(conf->dumpdev, "eth", 3) == 0) | 
				(strncmp(conf->dumpdev, "ath", 3) == 0) |
				(strncmp(conf->dumpdev, "wlan", 4) == 0)){
			if (!conf->target_host[0]) {
				err++;
				fprintf(stderr, "%s: Error - no TARGET_HOST parameter\n",
						prog);
			}
			if (!conf->target_port[0]) {
				err++;
				fprintf(stderr, "%s: Error - no TARGET_PORT parameter\n",
						prog);
			}
			if (!conf->source_port[0]) {
				err++;
				fprintf(stderr, "%s: Error - no SOURCE_PORT parameter\n",
						prog);
			}
			if (!conf->eth_address[0]) {
				err++;
				fprintf(stderr, "%s: Error - no ETH_ADDRESS parameter\n",
						prog);
			}
		}
	}

	if (err) {
		fprintf(stderr, "%s: Error - incomplete configuration in %s\n", 
				prog, config_file);
		exit(1);
	}
}


int 
main(int argc, char *argv[])
{
    char config_file[] = "/etc/dumputils.conf";
    int ver, c, level_override=-1;
    struct pconf *conf;

    if  ((conf=malloc(sizeof(struct pconf))) == NULL){
	perror("FATAL ERROR: Couldn't allocate enough memory");
	exit(-1);
    }


    prog = argv[0];

    /* Get the command line options */

    while ((c=getopt(argc,argv,"hvl:")) != EOF) {
	switch (c) {
	    case 'h':
		usage(stdout, prog, config_file);
		exit(0);
		break;
	    case 'v':
		version(stdout, prog);
		exit(0);
		break;
	    case 'l':
		level_override = strtol(optarg, NULL, 0);
		switch (level_override) {
		    case 0:
		    case 1:
		    case 2:
		    case 4:
		    case 8:
			break;
		    default:
			fprintf(stderr, "%s: Error - bad level %d\n",
				prog, level_override);
			exit(1);
		}
		break;
	    default:
		usage(stderr,prog, config_file);
		exit(1);
		break;
	}
    }
    if (optind < argc) {
	usage(stderr, prog, config_file);
	exit(1);
    }

    if ((conf = parseconf(config_file)) == NULL){
	fprintf(stderr, "Error could not read: %s\n", config_file);
	exit(1);
    }

    ver = get_version();

    checkconf(conf, config_file, ver);


    /* Use DUMP_ACTIVE to determine whether to modify the dump device */
    if (atoi(conf->dump_active) != 1) {
	printf("%s: Configuration stopped because " 
		"DUMP_ACTIVE is set to %d\n", prog,
		atoi(conf->dump_active));
	exit(1);
    }
    switch (ver){
	    case 4: // Kernel 2.4 
		    twoFour(level_override, conf);
		    break;
	    case 6: // Kernel 2.6 (ioctl style)
		    twoSix(level_override, conf);
		    break;
	    case 61: // Kernel 2.6 (newer sysfs style)
		    twoSix_One(level_override, conf);
		    break;
	    default:
		    printf("Fatal Error: Can not read or unknown kernel version." 
				    "Is proc mounted?\n");
		    return -1;

    }

    free(conf);

    return 0;
}
